#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

package Lib::App::Topic;

use strict;
use warnings;

use constant {
    LOCK_SH => 1,
    LOCK_EX => 2,
    LOCK_NB => 4,
    LOCK_UN => 8,
};

use Lib::Logger;
use Lib::User;
use Lib::DateTime;
use Lib::String;
use Lib::Calendar;
use Lib::App::Ads;
use Lib::App::Topic;
use Lib::App::Archive;
use Lib::App::Keyword;
use Lib::App::Conf;

my $logger = Lib::Logger->new();

# コンストラクタ
sub new {
    my $self = {};

    use Lib::App::Conf;
    my $conf = Lib::App::Conf->new();

    $self->{system_dir} = $conf->get_system_dir();  # システムディレクトリ
    $self->{data_dir} = $conf->get_data_dir();      # データディレクトリ
    $self->{docroot_dir} = $conf->get_docroot_dir();# ドキュメントルートディレクトリ
    $self->{site_title} = $conf->get_site_title();
    $self->{sub_title} = $conf->get_sub_title();
    $self->{link_target} = $conf->get_link_target();
    $self->{trackback_target} = $conf->get_trackback_target();
    $self->{comment_target} = $conf->get_comment_target();
    $self->{anonymous} = $conf->get_anonymous();      # 匿名の場合の代替文字列
    $self->{static_html} = $conf->get_static_html();  # 静的ファイルを生成するか

    $self->{topicid} = '';    # トピックID
    $self->{lastdate} = '';   # 最終更新日 YYYY/MM/DD HH:MM:SS
    $self->{status} = 0;      # 状態
    $self->{editor} = '';     # 編集者
    $self->{handle} = '';     # ハンドル名
    $self->{title} = '';      # タイトル
    $self->{text} = '';       # 本文
    $self->{commentnum} = 0;  # コメント数
    $self->{trackbacknum} = 0;# トラックバック数
    $self->{linknum} = 0;     # 関連リンク数
    $self->{ad1} = '';        # 広告コード1（幅468、記事上）
    $self->{ad2} = '';        # 広告コード2（幅468、記事下）
    $self->{ad3} = '';        # 広告コード3（幅300、サイドバー）
    $self->{point} = 0;       # 累計ポイント
    $self->{keyword} = '';    # キーワード
    $self->{ipaddr} = '';     # 投稿者のIPアドレス

    $self->{links} = [];      # 関連リンクの配列
    $self->{comments} = [];   # コメントの配列
    $self->{trackbacks} = []; # トラックバックの配列

    bless($self);
    return $self;
}

# トピックIDの取得
sub get_topicid {
    my $self = shift;
    return $self->{topicid};
}

# トピックIDの設定
sub set_topicid {
    my $self = shift;
    $self->{topicid} = shift;
}

# トピックの最終更新日の取得
sub get_lastdate {
    my $self = shift;
    return $self->{lastdate};
}

# トピックの最終更新日の設定
sub set_lastdate {
    my $self = shift;
    $self->{lastdate} = shift;
}

# 状態の取得
sub get_status {
    my $self = shift;
    return $self->{status};
}

# 状態の設定
sub set_status {
    my $self = shift;
    $self->{status} = shift;
}

# 編集者の取得
sub get_editor {
    my $self = shift;
    return $self->{editor};
}

# 編集者の設定
sub set_editor {
    my $self = shift;
    $self->{editor} = shift;
}

# ハンドル名の取得
sub get_handle {
    my $self = shift;
    return $self->{handle};
}

# ハンドル名の設定
sub set_handle {
    my $self = shift;
    $self->{handle} = shift;
}

# タイトルの取得
sub get_title {
    my $self = shift;
    return $self->{title};
}

# タイトルの設定
sub set_title {
    my $self = shift;
    $self->{title} = shift;
}

# 本文の取得
sub get_text {
    my $self = shift;
    return $self->{text};
}

# 本文の設定
sub set_text {
    my $self = shift;
    $self->{text} = shift;
}

# コメント数の取得
sub get_commentnum {
    my $self = shift;
    return $self->{commentnum};
}

# コメント数の設定
sub set_commentnum {
    my $self = shift;
    $self->{commentnum} = shift;
}

# トラックバック数の取得
sub get_trackbacknum {
    my $self = shift;
    return $self->{trackbacknum};
}

# トラックバック数の設定
sub set_trackbacknum {
    my $self = shift;
    $self->{trackbacknum} = shift;
}

# 関連リンク数の取得
sub get_linknum {
    my $self = shift;
    return $self->{linknum};
}

# 関連リンク数の設定
sub set_linknum {
    my $self = shift;
    $self->{linknum} = shift;
}

# 広告コード1の取得
sub get_ad1 {
    my $self = shift;
    return $self->{ad1};
}

# 広告コード1の設定
sub set_ad1 {
    my $self = shift;
    $self->{ad1} = shift;
}

# 広告コード2の取得
sub get_ad2 {
    my $self = shift;
    return $self->{ad2};
}

# 広告コード2の設定
sub set_ad2 {
    my $self = shift;
    $self->{ad2} = shift;
}

# 広告コード3の取得
sub get_ad3 {
    my $self = shift;
    return $self->{ad3};
}

# 広告コード3の設定
sub set_ad3 {
    my $self = shift;
    $self->{ad3} = shift;
}

# ポイントの取得
sub get_point {
    my $self = shift;
    return $self->{point};
}

# ポイントの設定
sub set_point {
    my $self = shift;
    $self->{point} = shift;
}

# キーワードの取得
sub get_keyword {
    my $self = shift;
    return $self->{keyword};
}

# キーワードの設定
sub set_keyword {
    my $self = shift;
    $self->{keyword} = shift;
}

# IPアドレスの取得
sub get_ipaddr {
    my $self = shift;
    return $self->{ipaddr};
}

# IPアドレスの設定
sub set_ipaddr {
    my $self = shift;
    $self->{ipaddr} = shift;
}

# 関連リンクの取得
sub get_links {
    my $self = shift;
    return @{ $self->{links} };
}

# 関連リンクの設定
sub set_links {
    my $self = shift;
    @{ $self->{links} } = @_;
}

# コメントの取得
sub get_comments {
    my $self = shift;
    return @{ $self->{comments} };
}

# コメントの設定
sub set_comments {
    my $self = shift;
    @{ $self->{comments} } = @_;
}

# トラックバックの取得
sub get_trackbacks {
    my $self = shift;
    return @{ $self->{trackbacks} };
}

# トラックバックの設定
sub set_trackbacks {
    my $self = shift;
    @{ $self->{trackbacks} } = @_;
}

# トピックデータの読込み
sub load {
    my $self = shift;
    my ($topicid, $mode) = @_;

    # トピックデータを読み込む
    my $datafile = "$self->{data_dir}/$topicid" . "tp.data";
    return $self->read($datafile, $mode);
}

# ファイル名指定して読込み
sub read {
    my $self = shift;
    my ($datafile, $mode) = @_;
    my $result = 0;

    # トピックデータを読み込む
    if (-f $datafile) {  # ファイルが存在する場合
        my @links = ();
        my @comments = ();
        my @trackbacks = ();

        open my $datafh, '<', $datafile;
        while (my $data = <$datafh>) {
            chop($data); # 改行を取り除く
            if ($data ne '') {
                my $rectype = substr($data, 0, 1);
                if ($rectype == 0) {    # 現行トピックデータ
                    ($self->{topicid},
                     $self->{lastdate},
                     $self->{status},
                     $self->{editor},
                     $self->{handle},
                     $self->{title},
                     $self->{text},
                     $self->{commentnum},
                     $self->{linknum},
                     $self->{trackbacknum},
                     $self->{ad1},
                     $self->{ad2},
                     $self->{ad3},
                     $self->{point},
                     $self->{keyword},
                     $self->{ipaddr}) = (split(/\,/, $data))[1..16];
                }
                elsif ($rectype == 2) { # 関連リンク
                    push @links, substr($data, 2);
                }
                elsif ($rectype == 3) { # コメント
                    push @comments, substr($data, 2);
                }
                elsif ($rectype == 4) { # トラックバック
                    push @trackbacks, substr($data, 2);
                }
            }
            last if (!$mode);
        }
        close $datafh;
        $result = 1;

        @{ $self->{links} } = @links;
        @{ $self->{comments} } = @comments;
        @{ $self->{trackbacks} } = @trackbacks;

        $self->{title} =~ s/enc_conma/,/g;
        $self->{text}  =~ s/enc_conma/,/g;
        $self->{text} =~ s/enc_crlf/\n/g;

        $self->{ad1} =~ s/enc_conma/,/g;
        $self->{ad2} =~ s/enc_conma/,/g;
        $self->{ad3} =~ s/enc_conma/,/g;
        $self->{ad1} =~ s/enc_crlf/\n/g;
        $self->{ad2} =~ s/enc_crlf/\n/g;
        $self->{ad3} =~ s/enc_crlf/\n/g;
    }

    return $result;
}

# トピックデータの更新（必ずmode='1'で読み込んでから更新する）
sub update {
    my $self = shift;

    my $datafile = "$self->{data_dir}/$self->{topicid}tp.data";

    my @links = @{ $self->{links} };
    my @comments = @{ $self->{comments} };
    my @trackbacks = @{ $self->{trackbacks} };

    $self->{title} =~ s/,/enc_conma/g;
    $self->{text}  =~ s/,/enc_conma/g;
    $self->{text} =~ s/\r?\n/enc_crlf/g;

    $self->{ad1} =~ s/,/enc_conma/g;
    $self->{ad2} =~ s/,/enc_conma/g;
    $self->{ad3} =~ s/,/enc_conma/g;
    $self->{ad1} =~ s/\r?\n/enc_crlf/g;
    $self->{ad2} =~ s/\r?\n/enc_crlf/g;
    $self->{ad3} =~ s/\r?\n/enc_crlf/g;

    $self->{commentnum} = scalar(@comments);
    $self->{linknum} = scalar(@links);
    $self->{trackbacknum} = scalar(@trackbacks);

    my $topicdata = "0,$self->{topicid},$self->{lastdate},$self->{status},$self->{editor},$self->{handle},";
    $topicdata .= "$self->{title},$self->{text},$self->{commentnum},$self->{linknum},$self->{trackbacknum},";
    $topicdata .= "$self->{ad1},$self->{ad2},$self->{ad3},$self->{point},$self->{keyword},$self->{ipaddr}";

    # トピックデータファイルを更新
    open my $datafh, '>', $datafile;
    flock $datafh, LOCK_EX;
    print {$datafh} "$topicdata\n";
    if (scalar(@links) > 0) { # 関連リンク
        foreach my $link (@links) {
            print {$datafh} "2,$link\n";
        }
    }
    if (scalar(@comments) > 0) { # コメント
        foreach my $comment (@comments) {
            print {$datafh} "3,$comment\n";
        }
    }
    if (scalar(@trackbacks) > 0) { # トラックバック
        foreach my $trackback (@trackbacks) {
            print {$datafh} "4,$trackback\n";
        }
    }
    flock $datafh, LOCK_UN;
    close $datafh;

    chmod 0766, $datafile;
}

# トピック詳細ページのhtmlソース作成
sub get_html {
    my $self = shift;
    my ($static, $mode) = @_;

    my $docrootdir = $self->{docroot_dir};
    my $sitetitle = $self->{site_title};
    my $subtitle = $self->{sub_title};

    my $topicid = $self->{topicid};
    $self->load($topicid, 1);

    my $topictitle = $self->{title};
    if ($self->{status} == 9) { # ステータスが削除の場合
        $topictitle = "削除されました";
    }
    my $caption = "$topictitle：$sitetitle";
    $sitetitle = "<a class=\"sitetitle\" href=\"$docrootdir/\">$sitetitle</a>";

    my $permalink = "$docrootdir/?act=view_topic&id=$topicid";
    my $rssurl = "$docrootdir/rss.xml";
    my $trackbackurl = "$docrootdir/rcvtb.cgi/$topicid";

    my $topicdetail = $self->_topic_detail();           # トピック詳細を取得
    my $trackbacklist = $self->_trackback_list($mode);  # トラックバック一覧を取得
    my $commentlist = $self->_comment_list($mode);      # コメント一覧を取得
    my $linklist = $self->_link_list($mode);            # リンク一覧を取得

    my $keywordlist ='';
    my $text = $self->{text};
    my $keyword = $self->{keyword};
    my $lastdate = $self->{lastdate};

    my $menu = <<"END_MENU";
<a href="$rssurl"><img src="feed-icon-16x16.jpg" border="0"></a> <a href="$rssurl">更新情報の購読</a>&nbsp;&nbsp;
END_MENU

    if ($static) {
        $menu .= "<a href=\"./?act=input_register\">ユーザー登録</a>&nbsp;&nbsp;";
        $menu .= "<a href=\"./?act=new_topic&topicid=$topicid\">新規投稿</a>&nbsp;&nbsp;";
        $menu .= "<a href=\"./?act=input_session\">ログイン</a>";
    }
    else {
        if ($mode == 1) { # 編集
            $menu .= <<"END_MENU";
<a href="./?act=logout_session">ログアウト</a>
<br /><a href="./?act=input_topic&topicid=$topicid">新規投稿</a>&nbsp;&nbsp;
<a href="./?act=edit_topic&topicid=$topicid">編集</a>&nbsp;&nbsp;
<a href="./?act=input_link&topicid=$topicid">関連リンク</a>
END_MENU
        }
        elsif ($mode == 2) { # ログイン中
            $menu .= "<a href=\"./?act=input_topic&topicid=$topicid\">新規投稿</a>&nbsp;&nbsp;";
            $menu .= "<a href=\"./?act=logout_session\">ログアウト</a>";
        }
        else { # 閲覧
            $menu .= "<a href=\"./?act=input_register\">ユーザー登録</a>&nbsp;&nbsp";
            $menu .= "<a href=\"./?act=new_topic&topicid=$topicid\">新規投稿</a>&nbsp;&nbsp;";
            $menu .= "<a href=\"./?act=input_session\">ログイン</a>";
        }
    }

    # メタタグ"description"の設定
    my $string = Lib::String->new();
    $string->set_tail('');
    my $descstr = $string->cut($text, 250);
    my $metadisc = "<meta name=\"description\" content=\"$descstr\">";

    my $metakeywords = '';
    my $keywordobj = Lib::App::Keyword->new();

    # キーワードをメタタグの"keywords"を設定
    if ($keyword ne '') {
        $metakeywords = $keyword;
        $metakeywords =~ s/[ ]/,/g;
        $metakeywords = "<meta name=\"keywords\" content=\"$metakeywords\">";

        # キーワードリストの作成
        my @ketwords = split(/ /, $keyword);
        $keywordlist = $keywordobj->list(@ketwords);
    }

    # カレンダーを作成
    my ($year, $month, $day) = (split(/\//, substr($lastdate, 0, 10)));
    my $cal = Lib::Calendar->new();
    my $calendar = $cal->get_calendar($year, $month);

    # 最新アーカイブ一覧を取得
    my $archive = Lib::App::Archive->new();
    my $archivelist = $archive->archive_list();
    my $newtopiclist = $archive->new_topiclist();

    # 広告の設定
    my $ad1 = $self->{ad1};
    my $ad2 = $self->{ad2};
    my $ad3 = $self->{ad3};
    my $editor = $self->{editor};
    my $status = $self->{status};
    my $blogparts = '';

    my $ads = Lib::App::Ads->new();
    if ($editor ne '') {
        # ユーザー広告の取得
        my $user = Lib::User->new();
        $user->load($editor);
        if ($ad1 eq '') {
            $ad1 = $user->get_ad1();
        }
        if ($ad2 eq '') {
            $ad2 = $user->get_ad2();
        }
        if ($ad3 eq '') {
            $ad3 = $user->get_ad3();
        }
        $blogparts = $user->get_parts();
    }
    else {
        # デフォルト広告の取得
        if ($ad1 eq '') {
            $ad1 = $ads->get_ad1();
        }
        if ($ad2 eq '') {
            $ad2 = $ads->get_ad2();
        }
        if ($ad3 eq '') {
            $ad3 = $ads->get_ad3();
        }
    }
    my $adlist = $ads->get_adlist_table();

#    $logger->write("blogparts:$blogparts");  # for debug

    # クラウドを取得
    my $clouds = $keywordobj->clouds(1);

    # 検索ボックスの取得
    my $searchbox = $self->_search_box();
    $searchbox =~ s/\$PREVQ\$//g;

    my $systemdir = $self->{system_dir};

    my $path = <<"END_PATH";
<strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
$topictitle
</strong>
END_PATH

    my $commentform = '';
    if ($status == 1 || $status == 3) {
        $commentform = $self->_comment_form();
    }

    open my $templatefh, '<', "$systemdir/tmpl/detail.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$METADISC\$/$metadisc/g;
    $template =~ s/\$METAKEYWORDS\$/$metakeywords/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$RSSURL\$/$rssurl/g;
    $template =~ s/\$MENU\$/$menu/g;
    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$TOPICTITLE\$/$topictitle/g;
    $template =~ s/\$TOPICDETAIL\$/$topicdetail/g;
    $template =~ s/\$COMMENTLIST\$/$commentlist/g;
    $template =~ s/\$COMMENTFORM\$/$commentform/g;
    $template =~ s/\$TRACKBACKURL\$/$trackbackurl/g;
    $template =~ s/\$TRACKBACKLIST\$/$trackbacklist/g;
    $template =~ s/\$LINKLIST\$/$linklist/g;
    $template =~ s/\$AD1\$/$ad1/g;
    $template =~ s/\$AD2\$/$ad2/g;
    $template =~ s/\$AD3\$/$ad3/g;
    $template =~ s/\$ADLIST\$/$adlist/g;
    $template =~ s/\$NEWTOPICLIST\$/$newtopiclist/g;
    $template =~ s/\$CALENDAR\$/$calendar/g;
    $template =~ s/\$ARCHIVELIST\$/$archivelist/g;
    $template =~ s/\$SEARCHBOX\$/$searchbox/g;
    $template =~ s/\$KEYWORDLIST\$/$keywordlist/g;
    $template =~ s/\$CLOUDS\$/$clouds/g;
    $template =~ s/\$BLOGPARTS\$/$blogparts/g;
    $template =~ s/\$PERMALINK\$/$permalink/g;

    if ($static) {
        $template =~ s/\$AUTHOR\$//g;
        $template =~ s/\$HPURL\$//g;

        # HTMLファイルを出力
        my $topicurl = "detail$topicid.html";
        open my $htmlfh, '>', $topicurl;
        flock $htmlfh, LOCK_EX;
        print {$htmlfh} $template;
        flock $htmlfh, LOCK_UN;
        close $htmlfh;
        chmod 0766, $topicurl;
    }

    return $template;
}

# トピック詳細を作成し、作成したHTMLを返す、loadが前の処理で呼び出されていることが前提
sub _topic_detail {
    my $self = shift;

    my $topicid = $self->{topicid};
    my $topicurl = "./?act=view_topic&id=$topicid";
    my $status = $self->{status};
    my $text = $self->{text};
    my $lastdate = $self->{lastdate};
    my $commentnum = $self->{commentnum};
    my $trackbacknum = $self->{trackbacknum};
    my $point = $self->{point};
    my $editor = $self->{editor};
    my $handle = $self->{handle};

    if ($status == 9) { # ステータスが削除の場合
        $text = "削除されました";
    }

    if ($editor eq '') {
        $editor = $self->{anonymous};
        if ($handle ne '') {
            $editor = $handle;
            $editor =~ s/enc_conma/,/g;
        }

        # URLをリンク
        $text =~ s/(http:\/\/[^()<>\[\]{}"'\x7F-\xFF\x00-\x20]+)/<a href="$1" target="_blank" rel="nofollow">$1<\/a>/g;

        # キーワードをリンク
        my $keyword = Lib::App::Keyword->new();
        $text = $keyword->link_keyword($self->{keyword}, $text);

    }
    else {
        my $user = Lib::User->new();
        $user->load($editor);
        my $mailaddr = $user->get_mailaddr();
        my $hpurl = $user->get_hpurl();
        my $username = $user->get_username();
        if ($username ne '') {
            $editor = $username;
            $editor =~ s/enc_conma/,/g;
        }
        if ($hpurl ne '') {
            $editor = "<a href=\"$hpurl\">$username</a>";
        }
        else {
            if ($mailaddr ne '') {
                $editor = "<a href=\"mailto:$mailaddr\">$username</a>";
            }
        }
    }

    my $fontcolor = "Black";
    if ($status == 0) {
        $fontcolor = "Orange";
    }
    elsif ($status == 1) {
        $fontcolor = "Darkgreen";
    }
    elsif ($status == 2) {
        $fontcolor = "Darkred";
    }
    elsif ($status == 3) {
        $fontcolor = "Darkblue";
    }

    my $detail = <<"END_DETAIL";
<div>$text</div>
<div>
<font color="$fontcolor">[$topicid]</font> posted by <strong>$editor</strong> at <font class="datetime">$lastdate</font> |
  <a href="$topicurl#comment">Comment ($commentnum)</a> |
  <a href="$topicurl#trackback">Trackback ($trackbacknum)</a> |
  <b><font class="point" size="+1">$point</font> point</b>
</div>
END_DETAIL

    return $detail;
}

# トピックの関連リンクを取得
sub get_linklist {
    my $self = shift;

    my $list = $self->_link_list(0);

    return $list;
}

# トピックの関連リンク一覧を作成する
sub _link_list {
    my $self = shift;
    my ($mode) = @_;

    my $list = '';
    my $docrootdir = $self->{docroot_dir} . "/";
    my $link_target = $self->{link_target};
    my $topicid = $self->{topicid};

    my @links = @{ $self->{links} };

    for my $i (0 .. $#links) {
        my $link = $links[$i];
        my ($linkid, $title, $url, $summary) = split(/\,/, $link);

        $url =~ s/\A(detail[0-9][.]html)\z/$docrootdir$1/g;
        $url =~ s/\A(\.\/\?act=view_topic.+)\z/$docrootdir$1/g;

        # 改行をデコード
        $summary =~ s/enc_crlf//g;

        # コンマをデコード
        $title =~ s/enc_conma/,/g;
        $url =~ s/enc_conma/,/g;
        $summary =~ s/enc_conma/,/g;

        if ($summary !~ m/<a .+>.+<\/a>/i) {
            # キーワードをリンク
            my $keyword = Lib::App::Keyword->new();
            $summary = $keyword->link_keyword($self->{keyword}, $summary);
        }

        my $menu = '';
        if ($mode == 1) {   # 編集モード
            $menu = <<"END_MENU";
<a href="./?act=edit_link&topicid=$topicid&linkid=$linkid">編集</a>&nbsp;&nbsp;<a href="./?act=confirm_link&topicid=$topicid&linkid=$linkid">削除</a>
END_MENU
        }

        $list .= <<"END_LIST";
<div class="link">
<table border="0" width="100%" cellpadding="3" cellspacing="0">
<tr>
  <td><a href="$url" $link_target>$title</a></td>
  <td width="90" align="right">$menu</td></tr>
<tr>
  <td colspan="2">$summary</td>
</tr>
</table>
</div>
END_LIST
    }

    if ($list ne '') {
       my $topictitle = $self->{title};
       $topictitle =~ s/enc_conma/,/g;
       $list = <<"END_LIST";
<a name="link"></a>
<h3>$topictitleの関連リンク</h3>
$list
END_LIST
    }

    return $list;
}

# トピックのトラックバック一覧を作成する
sub _trackback_list {
    my $self = shift;
    my ($mode) = @_;

    my $list = '';
    my $trackback_target = $self->{trackback_target};
    my $topicid = $self->{topicid};

    my @trackbacks = @{ $self->{trackbacks} };

    for my $i (0 .. $#trackbacks) {
        my $trackback = $trackbacks[$i];
        my ($trackbackid, $status, $recvdate, $url, $blogname, $title, $excerpt, $ipaddr) = split(/\,/, $trackback);

        # ステータスが無効なトラックバックは表示しない
        next if ($status == 0);

        # 編集モード以外ではステータスが承認待ちのトラックバックは表示しない
        next if ($mode != 1 && $status == 2);

        # コンマをデコード
        $title =~ s/enc_conma/,/g;
        $excerpt =~ s/enc_conma/,/g;
        $url =~ s/enc_conma/,/g;
        $blogname =~ s/enc_conma/,/g;

        # キーワードをリンク
        my $keyword = Lib::App::Keyword->new();
        $excerpt = $keyword->link_keyword($self->{keyword}, $excerpt);

        my $menu = '';
        if ($mode == 1) {   # 編集モード
            $menu = "<a href=\"./?act=confirm_trackback&topicid=$topicid&id=$trackbackid\">削除</a>";
            if ($status == 2) {
                $menu .= "&nbsp;&nbsp;<a href=\"./?act=recognize_trackback&topicid=$topicid&id=$trackbackid\">承認</a>";
            }
        }

        $list .= <<"END_LIST";
<div class="trackback">
<table border="0" width="100%" cellpadding="3" cellspacing="0">
<tr>
  <td><a name="trackback$trackbackid"></a>Linked from <b>$blogname</b> : <a href="$url" $trackback_target>$title</a> at $recvdate</td>
  <td width="90" align="right" nowrap>$menu</td></tr>
<tr>
  <td colspan="2">$excerpt</td>
</tr>
</table>
</div>
END_LIST
    }

    return $list;
}

# トピックのコメント一覧を作成する
sub _comment_list {
    my $self = shift;
    my ($mode) = @_;

    my $list = '';
    my $comment_target = $self->{comment_target};
    my $anonymous = $self->{anonymous};
    my $topicid = $self->{topicid};
    my $topicurl = "./?act=view_topic&id=$topicid";

    my @comments = @{ $self->{comments} };

    for my $i (0 .. $#comments) {
        my $comment = $comments[$i];
        my ($commentid, $status, $postdate, $author, $mailaddr, $hpurl, $text, $evaluate, $ipaddr) = split(/\,/, $comment);

        # ステータスが無効なトラックバックは表示しない
        next if ($status == 0);

        # 編集モード以外ではステータスが承認待ちのコメントは表示しない
        next if ($mode != 1 && $status == 2);

        # 匿名の置き換え
        if ($author eq '') {
            $author = $anonymous;
        }

        # 改行を削除
        $text =~ s/<br.+?\/>enc_crlf/enc_crlf/g;
        $text =~ s/enc_crlf/<br \/>/g;

        # コンマをデコード
        $text =~ s/enc_conma/,/g;

        # >>$commentidに対してリンクを張る。
        $text =~ s/>>([0-9]+)/<a href="$topicurl#comment$1">&gt;&gt;$1<\/a>/g;
        $text =~ s/&gt;&gt;([0-9]+)/<a href="$topicurl#comment$1">&gt;&gt;$1<\/a>/g;

        # URLに対してリンクを張る。
        $text =~ s/(http:\/\/[^()<>\[\]{}"'\x7F-\xFF\x00-\x20]+)/<a href="$1" $comment_target rel="nofollow">$1<\/a>/g;

        # キーワードをリンク
        my $keyword = Lib::App::Keyword->new();
        $text = $keyword->link_keyword($self->{keyword}, $text);

        # メールアドレスやホームページURLが入力されていた場合はリンク
        if ($hpurl ne '') {
            $author = "<a href=\"$hpurl\" $comment_target rel=\"nofollow\">$author</a>";
        }
        else {
            if ($mailaddr ne '') {
                $author = "<a href=\"mailto:$mailaddr\">$author</a>";
            }
        }

        my $menu = '';
        if ($mode == 1) {   # 編集モード
            $menu = "<a href=\"./?act=confirm_comment&topicid=$topicid&id=$commentid\">削除</a>";
            if ($status == 2) {
                $menu .= "&nbsp;&nbsp;<a href=\"./?act=recognize_comment&topicid=$topicid&id=$commentid\">承認</a>";
            }
        }

        $list .= <<"END_LIST";
<div class="comment">
<table border="0" width="100%" cellpadding="3" cellspacing="0">
<tr>
  <td><a name="comment$commentid"></a><b>$commentid</b> posted by <b>$author</b> at $postdate</td>
  <td width="90" align="right" nowrap>$menu</td></tr>
<tr>
  <td colspan="2">$text</td>
</tr>
</table>
</div>
END_LIST
    }

    return $list;
}

# コメント入力用フォームの生成
sub _comment_form {
    my $self = shift;

    my $topicid = $self->{topicid};

    my $form = <<"END_FORM";
<a name="commentform"></a>
<div class="commentform">
<div id="toggleHide" style="display:block">
  <form name="hideform" method="post" action="./">
  <input type="hidden" name="act" value="add_comment">
  <input type="hidden" name="topicid" value="$topicid">
  <a href="javascript:;" onmousedown="toggleInput('toggleShow','toggleHide');">＋</a>&nbsp;をクリックすると名前、メールアドレス、ホームページURLの入力フィールドが表示されます
  <br /><br />
  <input type="checkbox" name="evaluate" value="1"><strong>おすすめ</strong> &nbsp;(チェックしてコメントすると更新情報に掲載)<br />
  <input type="hidden" name="author" value="\$AUTHOR\$">
  <input type="hidden" name="mailaddr" value="">
  <input type="hidden" name="hpurl" value="\$HPURL\$">
  <b>コメント：</b><br />
  <textarea name="text" rows="10" cols="60"></textarea>
  <br />
  <input type="submit" value="投稿">
  </form>
</div>
<div id="toggleShow" style="display:none">
  <form name="showform" method="post" action="./">
  <input type="hidden" name="act" value="add_comment">
  <input type="hidden" name="topicid" value="$topicid">
  <a href="javascript:;" onmousedown="toggleInput('toggleShow','toggleHide');">－</a>&nbsp;をクリックすると名前、メールアドレス、ホームページURLの入力フィールドが非表示になります
  <br /><br />
  名前：<br />
  <input type="text" name="author" size="60" value="\$AUTHOR\$"><br />
  メールアドレス：<br />
  <input type="text" name="mailaddr" size="60"><br />
  ホームページURL：<br />
  <input type="text" name="hpurl" size="60" value="\$HPURL\$"><br />
  <br />
  <input type="checkbox" name="evaluate" value="1"><strong>おすすめ</strong> &nbsp;(チェックしてコメントすると更新情報に掲載)<br />
  <b>コメント：</b><br />
  <textarea name="text" rows="10" cols="60"></textarea>
  <br />
  <input type="submit" value="投稿">
  </form>
</div>
</div>
END_FORM

    return $form;
}

# トピック詳細ページ用検索ボックスの作成
sub _search_box {
    my $self = shift;

    my $form = <<"END_FORM";
<div class="searchbox">
<form method="post" action="./">
  <input type="text" name="q" size="60" value="\$PREVQ\$">&nbsp;
  <input type="submit" value="検索">
</form>
</div>
END_FORM

    return $form;
}

1;
# End of Topic.pm
