// Gokurakuシリーズ共通の関数ライブラリー
// measure, merge, visualizeで完全同一

//// 定数
final String Version          = "3.6";
final String JapaneseFont     = "Meiryo";
final int    Delay_message    = 2000;          // ショートメッセージの表示時間（ミリ秒）

File _fp = new File("");
final String Separator = _fp.separator;  // システム依存のパス区切り子 (UNIX: /, Windwos: \\)

// グローバル変数
ShortMessage shortMessage;
InputInteger  inputInteger;
InputFloat    inputFloat;
InputFilename inputFilename;
InputString   inputString;
KeyStatus     keyStatus;

//// 変換関係
// 浮動小数点を小数点以下n桁で切り捨てる関数
float floorN(float f, int n) {
  return float(floor(f * pow(10, n))) / pow(10, n);
}

// RGBの色情報を文字列に変換する関数
String color2str(int r, int g, int b) {
  String s;
  s = "[" + r + "," + g + "," + b + "]";
  return s;
}

// 画像のアスペクト比を調べて文字列で返す
String aspect_ratio(int w, int h) {
  float ratio;
  ratio = (float)h / (float)w;
  if (ratio == 3.0/4.0)  return "4:3(Warning)";
  if (ratio == 9.0/16.0) return "16:9";
  return "1:" + nf(ratio) + "(Warning)";
}

//// 文字列関係の関数
// StringList版
final String [] extensionListOfImage = {"jpg", "jpeg", "gif", "tga", "png"};  // TIFFはsaveFrame()できるが，loadImage()はできない．
final String [] extensionListOfText  = {"txt"};                               // 読み込めるテキストファイルの拡張子リスト

// File [] 内の全ファイル名を比較してソーティング（昇順）を行うクラス
class SortingFileList {
  StringList fList;  // 一時的な利用（ソートとリストからの削除）
  String     path;   // フルパス名（親ディレクトリ）
  // コンストラクタ
  // Fileのリストを受け取り，StringListに格納
  SortingFileList(File [] fs, String [] extList) {
    fList = new StringList();
    path = fs[0].getParent();
    // FileのリストをStringListに移し替える
    for (int i = 0; i < fs.length; i++) {
      if (fs[i].isDirectory()) continue;  // フォルダーはスキップ
      fList.append(fs[i].getName());      // ファイル名だけをStringListに格納（後でpathを追加
    }
    // 指定された拡張子のリスト以外のファイルをStringListから削除
    for (int i = fList.size() - 1; i >= 0; i--) {
      boolean flag = false;
      for (int j = 0; j < extList.length; j++) {
        String ext;
        String fname;
        fname = fList.get(i);
        ext = getExtension(fList.get(i)).toUpperCase(); 
        if (ext.equals(extList[j].toUpperCase())) {
          flag = true;
          break;
        }
      }
      if (flag == false) fList.remove(i);
      else               fList.set(i, path + Separator + fList.get(i));
    }
  }
  // 右から拡張子を取得する
  String getExtension(String str) {
    if (str.indexOf(".") == -1)               return "";
    if (str.indexOf(".") == str.length() - 1) return "";
    for (int i = str.length() - 1; i >= 0; i--) {
      if (str.charAt(i) == '.')               return str.substring(i + 1); 
    }
    return "";
  }
  // ソートして，フルパス名に戻した上でFileのリストに戻す
  File [] sort() {
    // ソート
    fList.sort();
    // Fileのリストに戻す
    File [] f = {};
    // Fileのリストに戻す
    f = (File [])expand(f, fList.size());
    for (int i = 0; i < fList.size(); i++) f[i] = new File(fList.get(i));
    return f;
  }
  // ファイルリストのコンソールへの出力：デバッグ用
  void showFileList() {
    for (int i = 0; i < fList.size(); i++) {
      println(i + ":" + fList.get(i));
    }
  }
}

// 文字列から数字を得て比較するクラス
//（merge中でメモリー内のデータを画像ファイル名や日時でソーティングするために用いる）
// 文字列はファイル名も可
// ANK文字列+数字列.拡張子を想定
// ピリオドは１つのみ（sample.jpg.txtはＮＧ）
// 数字列は0001のように左に0があっても構わない
class CompareString {
  String number1, number2;
  CompareString() {
    number1 = "";
    number2 = "";
  }
  // 比較（str1が大きければ 1，小さければ -1，等しければ 0 を返す）
  int compare(String str1, String str2) {
    // 念のための前処理
    if (str1 == null) str1 = "";
    if (str2 == null) str2 = "";
    // 正規化
    str1 = removeExtension(str1);
    str2 = removeExtension(str2);
    str1 = getNumbers(str1);
    str2 = getNumbers(str2);
    str1 = trimLeft(str1);
    str2 = trimLeft(str2);
    // 自明な場合
    if (str1.equals(str2))             return  0;  // 両方共 "" の場合を含む
    if (str1.length() > str2.length()) return  1;  // str2が "" の場合を含む
    if (str1.length() < str2.length()) return -1;  // str1が "" の場合を含む
    // 桁数が等しい場合
    for (int i = 0; i < str1.length(); i++) {
      int n1 = str1.charAt(i);
      int n2 = str2.charAt(i);
      if (n1 > n2) return  1;
      if (n1 < n2) return -1;
    }
    return 0;
  }
  // 右から拡張子を取得する
  String getExtension(String str) {
    if (str.indexOf(".") == -1)               return "";
    if (str.indexOf(".") == str.length() - 1) return "";
    for (int i = str.length() - 1; i >= 0; i--) {
      if (str.charAt(i) == '.')               return str.substring(i + 1); 
    }
    return "";
  }
  // 右から拡張子のみを取り除く
  String removeExtension(String str) {
    for (int i = str.length() - 1; i >= 0; i--) {
      if (str.charAt(i) == '.') return str.substring(0,i); 
    }
    return str;
  }
  // 右から数字のみの文字列を探して返す
  String getNumbers(String str) {
    String ret = "";
    char   c;
    for (int i = str.length() - 1; i >= 0; i--) {
      c = str.charAt(i);
      if ((c > '9') || (c < '0')) break;
      ret = c + ret;
    }
    return ret;
  }
  // 数字だけの文字列だとして，頭に付いている0を取り除く
  String trimLeft(String str) {
    for (int i = 0; i < str.length(); i++) {
      if (str.charAt(i) != '0') return str.substring(i);
    }
    return "0";
  }
};

//// ファイル関係のクラス
// CSV（タブ区切りの整数）のテキストを整数値に切り分けるクラス
class CsvStrings {
  String text;
  String str;
  float  value;
  // コンストラクター
  CsvStrings(String text) {
    this.set(text);
  }
  void set(String text) {
    this.text = text + "\t";
    value     =  float("NaN");  // floatの形式ではないのでNaNになる
  }
  // 次の値を取得する：正常に終了したらtrueを返し，valueに値が設定される
  boolean next() {
    if (text.indexOf("\t") == -1) return false;
    if (text.length() == 0)       return false;
    if (text.equals("\t"))        return false;
    str   = text.substring(0, text.indexOf("\t"));
    value = float(str);
    text  = text.substring(text.indexOf("\t") + 1);
    return true;
  }
  boolean is_float() {
    if (value != value) return false;  // NaN, ちょっとトリッキーな方法
    return true;
  }
}

// 指定されたデータファイルから１画像分のデータを取得してメモリーに保存するクラス
class ImportMeasuredFile {
  String import_folder;
  // コンストラクター
  ImportMeasuredFile() {
    import_folder = sketchPath();
  }
  
  // ファイル形式のチェック
  boolean validation(String fname) {
    File         fp;
    String[]     lines;  // データファイルから読み出した全テキストデータ

    // 存在するか？
    fp = new File(fname);
    if (  fp.isFile() != true) return false;
    // 読み込めるか？
    if ((lines = loadStrings(fname)) == null) return false;
    // 少なくとも２行以上か？
    if ((lines.length < 2)) return false;
    // １行目はscale, org_width, org_heightか？
    CsvStrings csv = new CsvStrings(lines[0]);
    csv.next();
    if (! csv.is_float() || csv.value <= 0.0)     return false;  // scale (float)
    csv.next();
    if (! csv.is_float() || int(csv.value) <= 0)  {              // org_width (int)
      if (int(csv.value) < 0) return false;
      shortMessage.set_message("[" + fname + "] Warning: org_width == 0");  // measureで出力する座標ファイル中の画像幅と高さの値が0になることがある(Ver.3.6以前で記録したデータ）
    }
    csv.next();
    if (! csv.is_float() || int(csv.value) <= 0) {               // org_height (int)
      if (int(csv.value) < 0) return false;
      shortMessage.set_message("[" + fname + "] Warning: org_height == 0"); // measureで出力する座標ファイル中の画像幅と高さの値が0になることがある(Ver.3.6以前で記録したデータ）
    }
    // 互換性（Ver. 1.*のGokuraku_measureは画像ファイル名を記録していない）ので，この後に文字列があるかどうかはチェックしない
    // 互換性（Ver. 2.1以降のgookuraku_measureは計測日時を記録していないので，この後に文字列があるかどうかはチェックしない
    // ２行目は0で始まるか？
    csv.set(lines[1]);
    csv.next();
    if (! csv.is_float() || int(csv.value) != 0)  return false;

    return true;
  }
  
  // 実際にファイルのデータを読み込む
  OneData read(String fname) {
    return read(fname, false);
  }
  OneData read(String fname, boolean noConvert) {
    File         fp;
    CsvStrings   csv;
    OneData      onedata;
    String[]     lines;

    // ファイルチェック
    if (! validation(fname)) return null; 
    // ファイル読み込み
    if ((lines = loadStrings(fname)) == null) return null;

    // Scale, org_width, org_heightの読み出し
    onedata = new OneData();
    csv     = new CsvStrings(lines[0]);
    csv.next();
    onedata.scale      = csv.value;
    csv.next();
    onedata.org_width  = int(csv.value);
    csv.next();
    onedata.org_height = int(csv.value);
    csv.next();  // Ver.2 から，Gokuraku_maesureで画像ファイル名を記録するように変更
    if (csv.str.length() > 0) {
      onedata.filename      = csv.str;
    } else onedata.filename = "";
    csv.next();  // Ver.2.1 から，Gokuraku_maesureで計測日時を記録するように変更
    if (csv.str.length() > 0) {
      onedata.dateString    = csv.str;
    } else onedata.dateString = "";
    // Ver.3.0から計測データ数とスキップ数を記録するように変更したが，それは読み飛ばす（実際に読みこんだデータで再集計する）

    // データ本体の読み込み
    for (int i = 1; i < lines.length; i++) {
      int px, py, col_R, col_G, col_B;
      csv.set(lines[i]);
      csv.next();
      if (int(csv.value) != (i - 1)) {
        shortMessage.set_message("Error: データ形式が正しくない（No.が一致しない）");
        return null;
      }
      // X座標
      csv.next();
      px = int(csv.value);
      if (noConvert == false) int(px / (float)onedata.scale); 
      // Y座標
      csv.next();
      py = int(csv.value);
      if (noConvert == false) int(py / (float)onedata.scale);
      // R, G の空読み
      csv.next();
      col_R = int(csv.value);
      csv.next();
      col_G = int(csv.value);
      // B が存在するかどうかで一行のデータ数を確認
      if (csv.next() == false) {
        shortMessage.set_message("Error: データ形式が正しくない（一行のデータ数が少ない)");
        return null;
      }
      col_B = int(csv.value);
      onedata.add_point(px, py, col_R, col_G, col_B);
    }
    fp = new File(fname);  // ファイル名の分離と，エクスポートフォルダー未指定時に自動設定するため
    if (onedata.filename.length() == 0) onedata.filename = fp.getName();
    return onedata;
  }     
}

//// データに関するクラス
// 計測するデータに関するクラス
// １ファイル分（１計測点分）のデータ
class OneData {  // リスト構造
  String filename;
  String dateString;  // Ver.2.1から追加
  int    org_width, org_height;
  double scale;
  int    num;
  int [] px    = {};
  int [] py    = {};
  int [] col_R = {};
  int [] col_G = {};
  int [] col_B = {};
  
  // コンストラクター
  OneData() {
    filename   = "";
    org_width  = 0; 
    org_height = 0;
    scale      = 1.0;
    num        = 0;
  }
  // 座標データの追加（色は無視）
  void add_point(int _px, int _py) {
    add_point(_px, _py, 0, 0, 0);
  }
  // 座標データの追加
  void add_point(int _px, int _py, int _col_R, int _col_G, int _col_B) {
    px    = (int [])append(px, _px);
    py    = (int [])append(py, _py);
    col_R = (int [])append(col_R, _col_R);
    col_G = (int [])append(col_G, _col_G);
    col_B = (int [])append(col_B, _col_B);
    num   = px.length;
  }
  // 座標データの削除（measure専用）
  void delete_point() {
    px    = shorten(px);
    py    = shorten(py);
    col_R = shorten(col_R);
    col_G = shorten(col_G);
    col_B = shorten(col_B);
  }
  // 座標データのみ消去（measure専用：計測のRetry用）
  void points_clear() {
    num        = 0;
    px         = (int [])expand(px, 0);
    py         = (int [])expand(py, 0);
    col_R      = (int [])expand(col_R, 0);
    col_G      = (int [])expand(col_G, 0);
    col_B      = (int [])expand(col_B, 0);
  }
  // 全データ消去
  void all_clear() {
    filename   = "";
    org_width  = 0; 
    org_height = 0;
    points_clear();
    // scale = 1.0;  Ver.3.0からスケールはクリアしない仕様に変更
  }
  // 計測点の色情報を格納する（measure専用）
  void get_color(PImage img, int x, int y) {
    if (num <= 0) return;
    int idx = x + y * img.width;
    if (idx >= img.pixels.length) return;
    int c = img.pixels[idx];
    col_R[num - 1] = int(red(c));
    col_G[num - 1] = int(green(c));
    col_B[num - 1] = int(blue(c));
  }
  String col2str(int n) {
    if (n >= num) return "";
    return color2str(col_R[n], col_G[n], col_B[n]); 
  }
  // 計測されたデータの中のスキップされた計測点の数を返す
  int countSkip() {
    int skip = 0;
    for (int i = 0; i < num; i++) {
      if (px[i] == -1) skip++;
    }
    return skip;
  }
}

//// プロットに関するクラス
// プロットの種類
final String [] plotName = {"none", "cross", "circle", "circlefill", "rect", "rectfill", "scope", "scopefill"}; 
// プロットを描画
class Plot {
  char mode;       // (0: none), 1: cross,  2: circle, 3: circlefill, 4: rect, 5: rectfill, 6, scope, 7. scopefill
  int  size;
  int  p_color;    // 0, 1, 2
  int  lineWidth;

  // コンストラクター
  Plot() {
    mode      = 1;  // default: cross
    size      = Plot_size;
    lineWidth = 1;
  }
  // 全プロットの表示
  void draw_plot(OneData data) {  // measureでは計測した座標にscaleが掛かっているので変換不要
    draw_plot(data, 1.0);
  }
  void draw_plot(OneData data, float scale) {
    strokeWeight(lineWidth);
    for (int i = 0; i < data.num; i++) {
      int x = data.px[i];
      int y = data.py[i];
      if ((x == -1) || (y == -1)) continue; // スキップされた計測点への対応

      if (ci.enable_scale && (scale != 1.0)) {
        x = (int)(x * scale);
        y = (int)(y * scale);
      }
      plot(x, y);
    }
    strokeWeight(1);
  }
  // プロットの形状を変更
  void change_shape() {
    if (++mode >= plotName.length) mode = 1;
  }
  String get_shape() {
    return plotName[mode];
  }
  // プロットの色を変更
  void change_color() {
    if (++p_color > Plot_color.length - 1) p_color = 0; 
  }
  // プロットを描く
  void plot(int x, int y) {
    stroke(this.red(), this.green(), this.blue());
    noFill();
    switch (mode) {
      case 0 : break;
      case 1 : // cross
        cross(x, y);
        break;
      case 3 : // circlefill
        fill(this.red(), this.green(), this.blue());
      case 2 : // circle 
        circle(x, y);
        break;
      case 5 : // rectfill
        fill(this.red(), this.green(), this.blue());
      case 4 : // rect 
        rectangle(x, y); 
        break;
      case 7 : // scopefill
        fill(this.red(), this.green(), this.blue());
      case 6 : // scope      
        cross(x, y, true);
        circle(x, y);
        break;
      default : mode = 0;
    }
  }
  // 十文字
  void cross(int x, int y) {
    cross(x, y, false);
  }
  // 十文字（照準マーク対応）
  void cross(int x, int y, boolean large) {
    int s = size;
    if (large) s *= 2;
    line(x - s, y, x + s, y);
    line(x, y - s, x, y + s);
  }
  // 円
  void circle(int x, int y) {
    ellipse(x, y, size * 2, size * 2);
  }
  // 四角
  void rectangle(int x, int y) {
    rectMode(CENTER); rect(x, y, size, size);
  }
  // 線分を描く（太さとカラーは事前に指定すること）
  void stick(int x0, int y0, int x1, int y1, float scale) {
    if (scale != 1.0) {
      x0 = (int)(x0 * scale);
      x1 = (int)(x1 * scale);
      y0 = (int)(y0 * scale);
      y1 = (int)(y1 * scale);
    }
    noFill();
    line(x0, y0, x1, y1);
  }
  // 水平線
  void h_line() {
    stroke(this.red(), this.green(), this.blue());
    line(0, mouseY, width, mouseY);    
  }  
  // 垂直線
  void v_line() {
    stroke(this.red(), this.green(), this.blue());
    line(mouseX, 0, mouseX, height);
  }
  // 色情報のテキスト化
  String color_str() {
    return color2str(this.red(), this.green(), this.blue());
  }
  int red() {
    return Plot_color[p_color][0];
  }
  int green() {
    return Plot_color[p_color][1];
  }
  int blue() {
    return Plot_color[p_color][2];
  }
}

// 線分描画のクラス
class Stick {
  boolean     enable_stick;  // 線分表示を有効化するか
  int [][]    define_stick;  // Stick picture(line segment)の定義

  // コンストラクター
  Stick() {
    define_stick = Define_stick;
    enable_stick = false;
  }
  // 線分定義領域のクリア
  void clearDefineStick() {
    define_stick = (int [][])expand(define_stick, 0);
  }
  // 線分定義ファイルの読み込み
  void readDefineStick(String fname) {
    File      fp;
    String [] lines;
    int [][]  def_stick = {};

    // メモリーへの読み込み
    if ((lines = loadStrings(fname)) == null) {
      shortMessage.set_message("Can't read define_stick file");
      return;
    }
    // 書式のチェック
    for (int i = 0; i < lines.length; i++) {
      CsvStrings csv = new CsvStrings(lines[i]);
      for (int j = 0; j < 6; j++) {
        if (!csv.next() || ! csv.is_float() || int(csv.value) < 0)  {  // start, end, R, G, B, Width
          shortMessage.set_message("Error: format error (not integer or negative)");
          return;
        }
      }
      if (csv.next()) {
        shortMessage.set_message("Error: format error (too much, 6 parameters)");
        return;  // 7項目以上あったらエラー
      }
    }
    // 格納
    for (int i = 0; i < lines.length; i++) {
      int [] onedef = {0, 0, 0, 0, 0, 0};
      CsvStrings csv = new CsvStrings(lines[i]);
      for (int j = 0; j < 6; j++) {
        csv.next();
        onedef[j] = int(csv.value);
      }
      if ((onedef[0] < 0) || (onedef[1] < 0)) {
        shortMessage.set_message("Error: format error (Start and End must be large equal 0)");
        return;
      }
      if ((onedef[2] > 255) || (onedef[3] > 255) || (onedef[4] > 255)) {
        shortMessage.set_message("Error: format error (color code must be 0 - 255)");
        return;
      }
      def_stick = (int [][])append(def_stick, onedef);
    }
    define_stick = def_stick;
    shortMessage.set_message("Imported define_stick file :" + define_stick.length + " data");
    enable_stick = true;  // 読み込みに成功したら自動的に有効にする
  }
  // 線分描画（measure専用）
  void draw_stick(OneData data) { // measureでは計測した座標にscaleが掛かっているので変換不要
    draw_stick(data, 1.0);
  }
  // 線分描画
  void draw_stick(OneData data, float scale) {
    if (enable_stick) {
      int x0, y0, x1, y1;
      // 未定義あるいは読み込み失敗時（０から順番に線で結ぶ）
      if (define_stick.length == 0) {
        stroke(ci.plotMark.red(), ci.plotMark.green(), ci.plotMark.blue());
        strokeWeight(ci.plotMark.lineWidth);
        for (int i = 0; i < data.num - 1; i++) {
          x0 = data.px[i]; y0 = data.py[i]; x1 = data.px[i+1]; y1 = data.py[i+1];
          if ((x0 == -1) || (y0 == -1) || (x1 == -1) || (y1 == -1)) continue;  // スキップされた計測点への対応
          if (ci.enable_scale && (scale != 1.0)) ci.plotMark.stick(x0, y0, x1, y1, scale);
          else                                   ci.plotMark.stick(x0, y0, x1, y1, 1.0);
        }
      }else {
        // 定義済みあるいは読み込み成功時
        for (int i = 0; i < define_stick.length; i++) {
          int start, end, r, g, b, weight;
          start  = define_stick[i][0];
          end    = define_stick[i][1];
          r      = define_stick[i][2];
          g      = define_stick[i][3];
          b      = define_stick[i][4];
          weight = define_stick[i][5];
          if ((start >= data.num) || (end >= data.num)) continue;

          x0 = data.px[start];
          y0 = data.py[start];
          x1 = data.px[end];
          y1 = data.py[end];
          if ((x0 == -1) || (y1 == -1) || (x1 == -1) || (y1 == -1)) continue;  // スキップされた計測点への対応

          stroke(r, g, b);
          strokeWeight(weight);
          if (ci.enable_scale && (scale != 1.0)) {
                 ci.plotMark.stick(x0, y0, x1, y1, scale);
          } else ci.plotMark.stick(x0, y0, x1, y1, 1.0);
        }
      }
      strokeWeight(1);
    }
  }
  // 線分表示（stick picture）の無効／有効切り替え
  void switch_enable_stick() {
    enable_stick = (enable_stick ? false : true);
  }
}

//// メニュー関係のユーザーインターフェース
// 色の列挙型の宣言
enum TextColor {normal, option, input};
enum Thema {dark, light};
// メニュー画面の記述
class MenuScreen {
  int         x,  y;         // 左上の座標
  private int dx, dy;        // メニューの説明のタブ位置，行送りのステップ
  private int x_org, y_org;  // xとyの初期値
  Gui_thema   thema;         // テーマ関係: Ver.3.5以降
  // コンストラクター
  MenuScreen(int x0, int y0, int dx0, int dy0) {
    x_org = x  = x0;
    y_org = y  = y0;
    dx = dx0;
    dy = dy0;
    thema = new Gui_thema(Dark_thema);
  }
  // リセット（draw()の度に座標値を元に戻す
  void reset() {
    x = x_org;
    y = y_org;
  }
  // fill()の再設定
  void fill_color(int col[]) {
    fill(col[0], col[1], col[2]);
  }
  // 画面消去
  void cls() {
    fill_color(thema.bg);
    rect(0, 0, Scr_w - 1, Scr_h - 1);
  }
  // タイトル表示
  void display_title(String title) {
    fill_color(thema.title);
    textSize(28);
    text(title + " (@tatsuva) Ver." + Version, x, y);
    y += dy;
  }
  // タイトル下のサブタイトル表示
  void display_subtitle(String subtitle) {
    textSize(18);
    text(subtitle, x, y);
    y += dy;
  }
  // メニュー項目の色設定
  void text_color(TextColor col) {
    fill_color(thema.text_normal);
    if (col == TextColor.option) fill_color(thema.text_option);
    if (col == TextColor.input)  fill_color(thema.text_input);
  }
  // メニュー項目
  void display_menu(String keyStr, String description) {
    textSize(24);
    text(keyStr,             x,      y);
    text(": " + description, x + dx, y);
    y += dy;
  }
  // キー入力プロンプト
  void display_keyInputPrompt(String prompt) {
    fill_color(thema.text_input);
    textSize(24);
    if      (inputInteger.inputing_now) text(prompt + " = " + inputInteger.value, x, y); 
    else if (inputFloat.inputing_now)   text(prompt + " = " + inputFloat.value,   x, y); 
    else if (inputFilename.inputing_now)text(prompt + " = " + inputFilename.value,x, y); 
    else if (inputString.inputing_now)  text(prompt + " = " + inputString.value,  x, y); 
    y += dy;
  }
  // ノート表示
  void display_note(String note) {
    textSize(16);
    fill_color(thema.note);
    text(note, x, y);
    y += 20;
  }
}

// 画面モード（テーマ）: Ver.3.5以降
class Gui_thema {
  boolean dark; // ダークテーマならばtrue, ライトテーマならばfalse
  int     title[];
  int     bg[];
  int     text_normal[], text_option[], text_input[];
  int     note[];
  int     shortmsg[];
  int     picbg[];
  
  Gui_thema(boolean dark_thema) {
    dark = dark_thema;
    if (dark) change2dark(); else change2light();
  }
  void change2dark() {
    title       = Dark_title;
    bg          = Dark_bg;
    text_normal = Dark_textcolor_normal;
    text_option = Dark_textcolor_option;
    text_input  = Dark_textcolor_input;
    note        = Dark_note;
    shortmsg    = Dark_shortmsg;
    picbg       = Dark_picbg;
  }
  void change2light() {
    title       = Light_title;
    bg          = Light_bg;
    text_normal = Light_textcolor_normal;
    text_option = Light_textcolor_option;
    text_input  = Light_textcolor_input;
    note        = Light_note;
    shortmsg    = Light_shortmsg;
    picbg       = Light_picbg;
  }
  void switch_thema() {
    dark = dark ? false : true;
    if (dark) change2dark(); else change2light();
  }
}

// 正の整数をキーボードから入力するためのクラス
class InputInteger {
  boolean inputing_now;  // 入力中（未確定）
  boolean halt;          // 中断（確定）
  int     value;
  
  // コンストラクター
  InputInteger() {
    inputing_now = false;
    halt         = false;
    value        = 0;
  }
  // 入力開始
  void start() {
    start(0);
  }
  void start(int init) {
    inputing_now = true;
    halt         = false;
    value        = init;
  }
  // 終了
  void end() {
    inputing_now = false;
    if (value < 0) value = 0;
  }
  // キーが入力された場合
  boolean keyin(char c) {
    int code;  // キャラクターコード
    code = int(c) - 48;  // 0 - 9
    if ((code >= 0) && (code <= 9)) {
      value = value * 10 + code;
    } else if (c == BACKSPACE) {
      value /= 10;
    } else if (c == ENTER) {
      end();
    } else if (c == ' ') {
      value = 0;
    } else if (c == ESC) {
      halt = true;
      end();
      shortMessage.set_message("Canceled");
    }
    return inputing_now;
  }
}

// InputString関連
// 浮動小数点の入力
class InputFloat extends InputString {
  InputFloat() {
    super();
  }
  boolean validation(char c) {
    if ((c >= '0') && (c <= '9')) return true;
    if (c == '-') return true;
    if (c == '.') {
      if (Float.isNaN(float(value + c))) return false;
      return true;
    }
    return false;
  }
}
class InputFilename extends InputString {
  InputFilename() {
    super();
  }
  boolean validation(char c) {
    if ((c >= 'a') && (c <= 'z')) return true;
    if ((c >= 'A') && (c <= 'Z')) return true;
    if ((c >= '0') && (c <= '9')) return true;
    if ((c == '-') || (c == '(') || (c == ')')) return true;
    return false;
  }
}
// 文字列をキーボードから入力するためのクラス
class InputString {
  boolean inputing_now;  // 入力中（未確定）
  boolean halt;          // 中断（確定）
  String  value;
  
  // コンストラクター
  InputString() {
    inputing_now = false;
    halt         = false;
    value        = "";
  }
  // 入力開始
  void start() {
    start("");
  }
  void start(String init) {
    inputing_now = true;
    halt         = false;
    value        = init;
  }
  // 終了
  void end() {
    inputing_now = false;
  }
  // OKならばtrue, NGならばfalse
  boolean validation(char c) {  // overrideされる
    return true;
  }
  // キーが入力された場合
  boolean keyin(char c) {
    boolean err = true;
    if ((c > 33) && (c < 126)) err = false;
    if (validation(c) == false) err = true;
    if (! err) value += c;
    else {
      if (c == BACKSPACE) {
        if (value.length() > 0) value = value.substring(0, value.length() - 1);
      } else if (c == ENTER) {
        end();
      } else if (c == ' ') {
        value = "";
      } else if (c == ESC) {
        halt = true;
        end();
        shortMessage.set_message("Canceled");
      }
    }
    return inputing_now;
  }
}

// キーの同時押しに対応するためのクラス
class KeyStatus {
  boolean left, right, shift, ctrl;
  KeyStatus() {
    left = right = shift = ctrl = false;
  }
  void push(int kc) {
    switch (kc) {
      case LEFT    : left  = true; break;
      case RIGHT   : right = true; break;
      case SHIFT   : shift = true; break;
      case CONTROL : ctrl  = true; break;
    }
  }
  void release(int kc) {
    switch (kc) {
      case LEFT    : left  = false; break;
      case RIGHT   : right = false; break;
      case SHIFT   : shift = false; break;
      case CONTROL : ctrl  = false; break;
    }
  }
};

// ウインドウとコンソールエリアに短いメッセージを一時的に表示するためのクラス
class ShortMessage {
private  int     start;
private  String  message;
private  boolean autoclear;
boolean busy;  // コメント表示中（マウスポインタのリアルタイム表示など，緊急性の低いコメントを抑制するために用いる）
  ShortMessage() {
    start     = millis();
    message   = "";
    autoclear = true;
  }
  void clear() {
    message = "";
  }
  void set_message(String str) {
    set_message(str, false, false);
  }
  void set_message_realtime(String str) {
    set_message(str, true, true);
  }
  // noconsole==trueで，Processing統合開発環境のコンソール領域への出力を抑制
  void set_message(String str, boolean noconsole, boolean realtime) {
    message = str;
    if (! noconsole) println(str);
    start   = millis();
    if (realtime != true) busy = true;
  }
  // draw()から呼ばれる
  void display_short_message(int x, int y) {
    if (millis() - start >= Delay_message) busy = false;
    if (autoclear) {
      if (! busy) return;  // 自動消去時は一定時間が経過したら表示しない
    }
    menuScreen.fill_color(menuScreen.thema.shortmsg);
    textSize(16);
    text(message, x, y);
  }
  // メッセージ自動消去を有効にする際には enable = trueとする
  void set_autoclear(boolean enable) {
    autoclear = enable;
  }
}

// 小数点より左の桁数を返す（0以下の場合は１を返す），負数の場合は +1（符号分）
int order(float value) {
  int v, n = 1;
  v = int(value);
  if (v < 0) v = -v;
  
  for (; (v /= 10) > 0; n++);
  
  if (n <= 1) n = 1;
  if (value < 0.0) n++;
  return n;
}
// 1.0より小さな値の小数点以下の桁数を返す（1.0以上は0, 0.0も0，0.1は1，0.01は2，0.00123でも3．負数は無関係）
int porder(double value) {
  int i;
  if (value < 0.0) value = -value;
  if (value >= 1.0) return 0;
  if (value == 0.0) return 0;
  for (i = 0; int((float)value) < 1; i++) {
    value *= 10.0;
  }
  return i;
}

// Quick measure関係
class QuickMeasure {
  boolean enable;       // Quick measure機能が有効な時はtrue
  boolean colorpicker;  // カラーピッカーモードの時はtrue（その時はenableもtrue）：後付けなので…
  int     status;       // -1: 未入力，0：起点記録済み，１：終点記録中，２：終点（連続）記録中
  boolean guideline           = Guide_line;           // ガイドラインの表示
  boolean mousepoint_realtime = Mousepoint_realtime;  // マウスポインターのx, y座標をリアルタイム表示する
  int []  px = {-1, -1}, py = {-1, -1};
  String  filename;
  int     dx, dy;
  float   distance, angle;
  float   calibration = Calibration;  // 校正表示専用
  String  cal_unit    = Cal_unit;
  Plot    plotMark;
  int     xorder, yorder, disorder;   // x,y,distanceの最大桁数

  QuickMeasure(Plot p) {
    enable      = Quick_measure_mode;  // 有効な場合は true
    colorpicker = Colorpicker_mode;    // 有効な場合は true;
    status      = -1;
    filename    = "";
    plotMark    = p;
    if (colorpicker) enable = true;    // 後付けなので…
  }
  // 初期化
  void init(String fname) {
    status = -1;
    px[0] = px[1] = py[0] = py[1] = -1;
    filename = fname;
    xorder   = order(width);
    yorder   = order(height);
    if (calibration == 0.0) disorder = order(sqrt(pow(width, 2) + pow(height, 2)));
      else                  disorder = order(sqrt(pow(width * calibration, 2) + pow(height * calibration, 2)));
  }
  // 計測モードの状態確認
  boolean is_normal() {
    if (enable == false) {
      colorpicker = false;
      return true;
    }
    return false;
  }
  boolean is_quickmeasure() {
    if ((enable == true) && (colorpicker == false)) return true;
    return false;
  }
  boolean is_colorpicker() {
    if (colorpicker == true) {
      if (enable == false) enable = true;
      return true;
    }
    return false;
  }
  // 計測モードの切り替え（通常計測 → クイックメジャー → カラーピッカー）
  void switch_measure_mode() {
    if (is_colorpicker()) { //<>//
      // カラーピッカー　→　通常計測
      enable      = false;
      colorpicker = false;
    } else if (is_quickmeasure()) {
      // クイックメジャー　→　カラーピッカー
      enable      = true;
      colorpicker = true;
    } else {
      // 通常計測　→　クイックメジャー
      enable      = true;
      colorpicker = false;
    }
  }
  // クリップボードにコピーする文字列（タブ区切り）
  // リアルタイム表示ではdx, dyを表示するが，クリップボードには余計なデータは送らない
  // start : 始点の時はtrue
  String get_quick_measure_results(boolean start) { 
    String str;
    // 共通する情報
    str  = filename;
    str += "\t" + px[0] + "\t" + py[0];
    // カラーピッカーの場合
    if (colorpicker) {
      int [] col;
      col = xy_color(px[0],py[0]);
      if (col != null) str += "\t" + col[0] + "\t" + col[1] + "\t" + col[2];
      return str;
    }
    // クイックメジャーの場合
    if (start != true)  str += "\t" + px[1] + "\t" + py[1] + "\t" + distance + "\t" + angle;
    return str;
  }
  // 0で埋めないnf()の整数版
  String nfi(int v, int order) {
    String str;
    str = nf(v);
    for (int i = str.length(); i < order; i++) str = " " + str;
    return str;
  }
  // 0で埋めないnf()の浮動小数点版
  String nff(float v, int order, int point) {
    String str;
    str = nf(v);
    for (int i = 0; i < point; i++) str += " ";
    str = str.substring(0, str.indexOf(".") + point + 1);
    for (int i = str.length(); i < order + point + 1; i++) str = " " + str;
    return str;
  }
  // (x, y)形式の文字列
  String xy_str(int x, int y) {
    String str;
    str = "(" + nfi(x, xorder) + ", " + nfi(y, yorder) + ")";
    return str;
  }
  // 色情報
  int [] xy_color(int x, int y) {
    color  c;
    int [] col = {0, 0, 0};
    if (ci.img == null) return null;
    if ((y >= ci.img.height) || (y < 0)) return null;
    if ((x >= ci.img.width)  || (x < 0)) return null;
    c = ci.img.pixels[x + y * ci.img.width];
    col[0] = c >> 16 & 0xff;
    col[1] = c >>  8 & 0xff;
    col[2] = c       & 0xff;
    return col;
  }
  String xyc_str(int x, int y) {
    String str;
    int [] col;
    str = xy_str(x, y);
    if ((col = xy_color(x, y)) == null) return str;
    return str + " R:G:B = " + nfi(col[0], 3) + ":" + nfi(col[1], 3) + ":" + nfi(col[2], 3);
  }
  // continuous : 右クリックによる相対座標の連続計測時にtrue
  void clicked(int x, int y, boolean continuous) {
    String str = "Quick: ";
    if (colorpicker || (status == -1)) {
      // 始点の保存
      px[0] = x;
      py[0] = y;
      CopyToClipboard(get_quick_measure_results(true));
      if (colorpicker) shortMessage.set_message(      "Picked =" + xyc_str(x, y));
        else           shortMessage.set_message(str + "Start ="  + xyc_str(x, y));
      if (! colorpicker) status = 0;
    } else if ((status == 0) || (status == 1)) {
      // 終点の保存
      px[1] = x;
      py[1] = y;
      // 計算
      str += calculate();
      CopyToClipboard(get_quick_measure_results(false));
      shortMessage.set_message(str);
      if (continuous == true) status =  1;  // 連続計測時
        else                  status = -1;
    } else {
      shortMessage.set_message("System error: wrong status of quick measure:" + status);
      status = -1;
    }
    return;
  }
  // 距離と傾きの計算
  String calculate()             { return calculate(-1, -1); }
  String calculate(int x, int y) {
    String str;
    if (x < 0) x = px[1];
    if (y < 0) y = py[1];
    dx = x - px[0];
    dy = y - py[0];
    distance = sqrt(dx * dx + dy * dy);
    angle = degrees(atan2(-dy, dx));
    str  = xy_str(px[0], py[0]) +" to " + xy_str(x, y);
    str += " -> " + xy_str(dx, dy) + " dist.= " + get_distance_str() +  ", angle = " + nff(angle, 4, 2) + "[deg]";
    return str;
  }
  // 距離表示の単位文字列
  String get_distance_str() {
    int    order, porder;
    String str;
    if (calibration == 0.0) {
      return nff(distance, disorder, 1) + "[pixel]";
    }
    porder = porder(calibration);
    if (porder <= 0) porder = 1;
    return nff(distance * calibration, disorder, porder) + "[" + cal_unit + "]";    
  }
  // プロットの描画
  void plot(boolean guideline) {
    // 起点のマーク表示
    if (status == 0) plotMark.plot(px[0], py[0]);
    // 終点のマークとラインの表示
    if ((status == -1) || (status == 1)) {
      if (px[0] > 0) {
        plotMark.plot(px[0], py[0]);
        plotMark.plot(px[1], py[1]);
        if (! colorpicker) plotMark.stick(px[0], py[0], px[1], py[1], 1.0);
      }
    }
    if (guideline) {
      ci.plotMark.h_line();  // 水平線
      ci.plotMark.v_line();  // 垂直線
      if (colorpicker) return;
      if ((status == 0) || (status == 1)) plot_guideline();
    }
  }
  // ガイドラインの表示
  void plot_guideline() {
    float x0, y0, x1, y1;
    float a, b;
    
    x0 = px[0];  y0 = py[0];
    x1 = mouseX; y1 = mouseY;
    if (x1 - x0 == 0) {
      //print("divide by zero\n");
    } else {
      a = (y1 - y0) / (x1 - x0);
      b = y0 - a * x0;

      int px0, py0, px1, py1;
      px0 = 0;
      px1 = width;
      py0 =  int(a * px0 + b);
      py1 =  int(a * px1 + b);
      plotMark.stick(px0, py0, px1, py1, 1.0);
    }
  }
  void switch_guide_line() {
    if (guideline) guideline = false; else guideline = true;
  }
  // 標準測定，クイックメジャー時のリアルタイムなマウス座標表示
  void mousepoint_realtime_display(PImage img) {
    String str;
    if (mouseY < img.height) { 
      str = xyc_str(mouseX, mouseY);
    } else str = xy_str(mouseX, mouseY);
    if (enable) {
      if ((status == 0) || (status == 1)) {
        str = calculate(mouseX, mouseY);
      }
    }
    if (shortMessage.busy != true) shortMessage.set_message_realtime(str);
  }
  void switch_mousepoint_realtime() {
    if (mousepoint_realtime) mousepoint_realtime = false; else mousepoint_realtime = true;
  }
}

//// D&D関係
// 参考URL（ドラッグ＆ドロップに関して
// https://qiita.com/rain-lotus/items/258fa3defd41846bfd50
// https://sites.google.com/site/gutugutu30/other/processingdedoragguandodoroppu

import java.awt.datatransfer.*;  
import java.awt.dnd.*;  
import java.io.File;  
import java.io.IOException;  
import java.awt.Component;
import java.util.List;

DropTarget dropTarget;
Component  component;

void drop_init() {  
  component = (Component)this.surface.getNative();
  dropTarget = new DropTarget(component, new DropTargetListener() {
    public void dragEnter(DropTargetDragEvent dtde) {
    }  
    public void dragOver(DropTargetDragEvent dtde) {
    }  
    public void dropActionChanged(DropTargetDragEvent dtde) {
    }  
    public void dragExit(DropTargetEvent dte) {
    }  
    public void drop(DropTargetDropEvent dtde) {  
      dtde.acceptDrop(DnDConstants.ACTION_COPY_OR_MOVE);  
      Transferable trans = dtde.getTransferable();  
      List<File> fileNameList = null;  
      if (trans.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {  
        try {  
          fileNameList = (List<File>)  
            trans.getTransferData(DataFlavor.javaFileListFlavor);
        } 
        catch (UnsupportedFlavorException ex) {
        } 
        catch (IOException ex) {
        }
      }  
      if (fileNameList == null) return;  
      fileSelected(fileNameList);
    }
  }
  );
}

// クリップボード関係
// 引用元： https://romly.com/blog/processing_copy_to_clipboard/
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.StringSelection;
import java.awt.Toolkit;
void CopyToClipboard(String s)
{
    Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
    StringSelection data = new StringSelection(s);
    clipboard.setContents(data, data);
}

// 言語パック
// ハッシュマップ（連想配列）
import java.util.Map;
HashMap<String,String> lang_string = new HashMap<String,String>();

class Get_string {
  String get_lang;  // 文字列を取得する言語
  String set_key;   // 文字列を設定するキーワード

  // コンストラクタ
  Get_string(String str) {
    get_lang = str;
  }
  // 記述を設定するためのキーワードをセット
  void set_key(String str, String description) {
    set_key = str;
    lang_string.put(str, description);
  }
  // 記述を設定
  void set_str(String set_lang, String description) {
    if (! set_lang.equalsIgnoreCase(get_lang)) return; 
    lang_string.put(set_key, description);
  }
  String get_str(String key) {
    String str = "";
    str = lang_string.get(key);
    if (str == null) return "<<no description>>";
    return str;
  }
}

// 少しだけ楽に読み出し
String get_str(String index) {
  return gs.get_str(index);
}
// パラメータ付き
String get_str(String index, String param) {
  return gs.get_str(index) + " (" + param + ")";
}
String get_str(String index, int param) {
  return get_str(index, nf(param));
}
String get_str(String index, float param) {
  return get_str(index, nf(param));
}
String get_str(String index, boolean flag) {
  return gs.get_str(index) + " (" + (flag ? "true" : "false") + ")";
}
// 宣言
// （1) インデックスの宣言とデフォルト（en）の設定
void set_key(String index, String description) {
  gs.set_key(index, description);
}
// (2) 指定された言語で今のindexの記述を上書き
void set_str(String lang, String description) {
  gs.set_str(lang, description);
}
