//// データ関係のクラス

// プレビュー：線分＋プロット表示と静止画表示（visualize）のクラス
enum PreviewMode {stop, playing, pause};
// Preview関係の再生のコントロール
class Preview {
  PreviewMode playing;
  int         count;
  boolean     resized;
  int         delay_time;       // イメージ間の遅れ時間計測用（millis()で差を得る）
  int         delay_per_image;  // イメージ間の遅れ時間（プレビュー専用）
  boolean     enable_image;     // 画像データファイルも読み込むか
  boolean     export_enable;    // プレビューと同時にexportも実行
  
  Preview() {
    playing         = PreviewMode.stop;
    resized         = false;
    enable_image    = true;
    count           = 0;
    export_enable   = false;
    delay_time      = millis();
    delay_per_image = Delay_preview_per_image;
  }
  // 画像読み込みの無効／有効切り替え
  void switch_enable_image() {
    enable_image = (enable_image ? false : true);
  }
  // プレビュー開始
  void start_preview() {
    start_preview(false);
  }
  void start_preview(boolean ex) {  // export時は ex = true
    if ((data == null) || (data.num <= 0)) return;  // データが無い
    playing       = PreviewMode.playing;
    resized       = false;
    count         = 0;
    export_enable = ex;
  }
  void stop_preview() {
    playing       = PreviewMode.stop;
    count         = 0;
    export_enable = false;
  }
  // 一つ前のコマに
  void prev_image() {
    prev_image(1);
  }
  void prev_image(int n) {
    count -= n;
    if (count < 0) count = 0;
    resized = false;
  }
  // 一つ後のコマに
  void next_image() {
    next_image(1);
  }
  void next_image(int n) {
    count += n;
    if (count >= data.num) count = data.num - 1;
    resized = false;
  }
  // 最初のコマに
  void first_image() {
    count   = 0;
    resized = false;
  }
  void last_image() {
    count   = data.num - 1;
    resized = false;
  }
  // ディレイタイマーセット
  void set_timer() {
    delay_time = millis();
  }
  // 後処理（主にディレイ）
  void post_process(Data data) {
    int past_time;

    if (! resized) return;
    if (playing == PreviewMode.stop) return;

    past_time = millis() - delay_time;
    if (count + 1 >= data.num) {    // 最後のコマの後
      if (!export_enable) {  // プレビュー
        playing = PreviewMode.pause;  // 強制的に一時停止モードに遷移
        return;
      } else {               // エクスポート
        if (past_time < Delay_export_after)  return;
        frameCount = count;
        saveFrame(ex_file.fullpathname);
        shortMessage.set_message("Exported image file : " + data.num + " files");
        stop_preview();
      }
    } else {                        // コマ間
      if (! export_enable) {  // プレビュー
        if (keyStatus.shift) {
          if (past_time < delay_per_image * 5)     return;  // SHIFTを押すと速度を1/5に減速
        } else {
          if (past_time < delay_per_image)         return;
        }
        if (playing == PreviewMode.pause)          return;
      } else {                // エクスポート
        if (past_time < Delay_export_per_image)    return;
        frameCount = count;
        saveFrame(ex_file.fullpathname);
      }
      next_image();
    }
    return;
  }
}

// 計測データの可視化に関するクラス
class CurrentImage {
  PImage  img;
  String  fullpathname;            // 開いている画像ファイルのフルパス名
  Preview preview;
  int     bg_color;
  
  int     org_width, org_height;   // 画像のオリジナルサイズ
  File[]  files;                   // フォルダーモード：ファイルリスト
  int     f_no;                    // フォルダーモード時の読みだし位置
  int     filter_no;               // -1 : なし，0 : GRAY, 1 : INVERT
  boolean enable_scale;     // 計測時のスケールを有効化するか
  
  Plot    plotMark;
  Stick   stick;
  
  // コンストラクター
  CurrentImage() {
    img          = null;
    fullpathname = null;
    preview      = new Preview();
    stick        = new Stick();
    bg_color     = 0;
    org_width = org_height = -1;
    enable_scale    = true;
    
    // プロットの初期化
    plotMark = new Plot();
  }
  // １データ１画像を作成
  void renderer(Data data) {
    File   fp;
    PImage img0;
    int    h, w;
    double scale;
    int    index = preview.count;
    if (data.body[index] == null) return;

    // 元データのサイズを保存
    org_width  = w = data.body[index].org_width;
    org_height = h = data.body[index].org_height;
    scale          = data.body[index].scale;
    // もしscaleが有効ならばサイズを変更
    if (enable_scale && scale != 1.0) {
      w = (int)(org_width  * scale);
      h = (int)(org_height * scale); 
    }
    if (! preview.resized) {  // 1サイクル目：スクリーンサイズを変更
      // 画像も読み込む場合
      if (preview.enable_image) {
        fp = new File(data.src_fullpath + Separator + data.body[index].filename);
        img0 = null;
        if (fp.isFile()) {
          if ((img0 = loadImage(fp.getAbsolutePath())) != null) {
            // 読み込み完了
            if (enable_scale && scale != 1.0) img0.resize(w, h);  // 画像を拡大する場合
          }
        }
        img = img0;
      } else img = null;
      if ((width != w) || (height != h)) {  // スクリーンサイズの変更が必要な場合
        if (ci.preview.export_enable) surface.setSize(w, h);
          else                        surface.setSize(w, h + 40);
        surface.setVisible(true); // Processing4.2でリサイズ後にキー操作その他を受け付けなくなる障害に対する無根拠な対策（効果あり）
      }
      preview.resized = true;
      if (img == null) background(BG_color[bg_color][0], BG_color[bg_color][1], BG_color[bg_color][2]);  // チラツキの原因
      preview.set_timer();
      // 画面下のショートメッセージ領域をクリア
      if (! ci.preview.export_enable) {
        fill(72, 72, 72);
        rect(0, height - 40, width, 40);
        shortMessage.set_message(ci.preview.count + ":" + data.body[index].filename, true, false);
      }
    } else {                  // ２サイクル目以降：プロットの描画
      // 画面下のショートメッセージ領域をクリア
      if (! ci.preview.export_enable) {
        fill(72, 72, 72);
        rect(0, height - 40, width, 40);
        shortMessage.set_message(ci.preview.count + ":" + data.body[index].filename, true, false);
      }
      // 画像も読み込む場合
      if (preview.enable_image) {
        if (img != null) image(img, 0, 0);
      }
      // 線分
      ci.stick.draw_stick(data.body[index], (float)scale);
      // プロット
      ci.plotMark.draw_plot(data.body[index], (float)scale);
    }
    return;
  }
  // プレビュー
  void play(Data data) {
    if (preview.playing != PreviewMode.stop) {
      renderer(data);
      preview.post_process(data);
    }
  }
  // 背景色の変更
  void change_bg() {
    if (++bg_color >= BG_color.length) bg_color = 0; 
  }
  // スケールの無効／有効切り替え
  void switch_enable_scale() {
    enable_scale = (enable_scale ? false : true);
  }
}

// 結合されたデータのバッファー
class Data {
  OneData[] body = {};
  String    src_fullpath;  // ファイル名抜きの絶対パス（getParent()）
  private 
  int       num;       // 読み込まれた行数
  // コンストラクター 配列は確保するが，num はstore()ごとに増やしていく
  Data() {
    num          = 0;
    src_fullpath = "";
  }
  // リストを全て削除する
  boolean reallocate(int n) {
    if (body.length != n) {  // サイズの変更が必要な場合（別のmerged.txtを読み込んだ場合）
      body = (OneData [])expand(body, n);
      if (body.length != n) {
        shortMessage.set_message("Error: Can't allocate memory");
        body = (OneData [])expand(body, 0);
        num = 0;
        return false;
      }
    }
    for (int i = 0; i < n; i++) body[i] = new OneData();
    src_fullpath = "";
    num          = 0;
    return true;
  }
  // 読み込んだデータの全リストを表示（コンソール部）：デバッグ用
  void show_data_list() {
    if (data.num <= 0) return;
    for (int i = 0; i < data.num; i++) {
      println(i + ":" + body[i].filename + " " + body[i].num + "data");
    }
  }
}

//// ファイル関係のクラス

// 指定されたデータファイルから１画像分のデータを取得してメモリーに保存するクラス
class ImportFile {
  String  import_folder;
  String  fullpathname;
  boolean noConversion;  // 計測データを読み込む際にscaleによる変換を行わない（Ver.3.0以降）
  
  // コンストラクター
  ImportFile() {
    import_folder = sketchPath();
    fullpathname  = null;
    noConversion  = false; 
  }

  // 与えられたファイルリストに従ってファイルを読み込む
  void open_files(File [] fileList) {
    SortingFileList sortingFileList;

    // ファイルリストのソーティング
    sortingFileList    = new SortingFileList(fileList, extensionListOfText);
    fileList = sortingFileList.sort();

    // 読み込み
    for (int i = 0; i < fileList.length; i++) read(fileList[i].getAbsolutePath());
  }
  void switch_noConversion() {
    noConversion = noConversion ? false : true; 
  }

  // ファイル形式のチェック
  boolean validation(String fname) {
    String[]     lines;  // データファイルから読み出した全テキストデータ
    int          num;    // X,Y座標のデータの組数

    // 読み込めるか？
    if ((lines = loadStrings(fname)) == null) return false;
    // １行はファイル名（ノーチェック），scale, org_width, org_height，num, X座標，Y座標，X座標，Y座標…の繰り返しか？
    for (int i = 0; i < lines.length; i++ ) {
      CsvStrings csv = new CsvStrings(lines[i]);
      csv.next();  // ファイル名
      csv.next();  // scale or 計測日時（Ver. 2.1から）
      if (csv.str.length() == 4+2+2+2+2+2) csv.next();  // Ver.2.1以降の計測日時を読み飛ばす
      if (! csv.is_float() || csv.value <= 0.0)     return false;  // scale (float)
      csv.next();
      if (! csv.is_float() || int(csv.value) <= 0)  return false;  // org_width (int)
      csv.next();
      if (! csv.is_float() || int(csv.value) <= 0)  return false;  // org_height (int)
      csv.next();
      if (! csv.is_float() || int(csv.value) <= 0)  return false;  // num (int)
      csv.next();  // Ver.3.0からスキップ数もmerged.txtに書き込まれるが，それは読み飛ばす

      num = int(csv.value);
      // X座標，Y座標の繰り返しのチェック
      for (int j = 0; j < num; j++) {
        if (! csv.next())                             return false;                              
        if (! csv.is_float() || int(csv.value) < -1)  return false;  // X座標 (int), -1: skipped measuring
        if (! csv.next())                             return false;                              
        if (! csv.is_float() || int(csv.value) < -1)  return false;  // Y座標 (int), -1: skipped measuring
      }
    }
    return true;
  }
  
  // 実際に統合されたデータファイルを読み込む（失敗時はnullを返す）
  boolean read(String fname) {
    CsvStrings   csv;
    String[]     lines;
    OneData      onedata;
    int          num;
    boolean noConversion;

    noConversion = im_file.noConversion;
    // ファイルチェック
    if (! validation(fname))                  return false; 
    // ファイル読み込み
    if ((lines = loadStrings(fname)) == null) return false;

    fullpathname = fname;  // 後で画像をインポートする時にパスを使うかも知れない
    // データ領域をクリアする．
    if (data.reallocate(lines.length) == false) return false;

    // 各行のデータを読み込む
    for (int i = 0; i < lines.length; i++) {
      // filename, scale, org_width, org_heightの読み出し
      csv = new CsvStrings(lines[i]);
      csv.next();
      data.body[i].filename   = csv.str;          // ファイル名
      csv.next();
      if (csv.str.length() == 4+2+2+2+2+2) csv.next();  // Ver.2.1以降の計測日時を読み飛ばす
      data.body[i].scale      = csv.value;        // scale
      csv.next();
      data.body[i].org_width  = int(csv.value);   // org_width
      csv.next();
      data.body[i].org_height = int(csv.value);   // org_height
      csv.next();
      num                     = int(csv.value);   // num
      csv.next();  // スキップ数の読み飛ばし（Ver.3.0以降）
  
      // X,Y座標データの読み込み
      int x, y;
      for (int j = 0; j < num; j++) {
        csv.next(); x = int(csv.value); // X座標
        if (noConversion == false) x = int(x / (float)data.body[i].scale);
        csv.next(); y = int(csv.value); // Y座標
        if (noConversion == false) y = int(y / (float)data.body[i].scale);
        data.body[i].add_point(x, y);
      }
      data.body[i].num = num;
    }
    data.num = lines.length;
    // エクスポート関係
    File fp = new File(fullpathname);
    data.src_fullpath = fp.getParent();  // データファイルの格納されているフォルダーをセット
    // 連番ファイル名のフォーマット用
    ex_file.order10 = int(log(data.num) / log(10)) + 1;
    if (ex_file.order10 < 4) ex_file.order10 = 4;      // デフォルトで４桁以上
    ex_file.set_export_information(data.src_fullpath); // エクスポートフォルダー更新

    shortMessage.set_message("Data imported: " + fullpathname + " " + data.num + " lines");
    return true;
  }     
}

// 作成した静止画をファイルに出力するクラス
class ExportFile {
  String      export_folder;  // 動画データを保存する絶対パス
  String      image_filename; // 連番静止画像のファイル名（"merged-"など）
  int         order10;        // 連番画像ファイルの桁数
  String      fullpathname;   // 最終的なフルパス名（saveFrame（）用）
  int         image_type;     // 出力する静止画像の種類
  
  // コンストラクター
  ExportFile() {
    export_folder  = sketchPath();
    order10        = 4;  // #### : 0001 - 9999．ファイル読み込み時に自動調整される
    image_type     = 0;
    image_filename = Image_filename;
    set_export_information(null);
  }
  void set_export_information(String ex_folder) {
    if (order10 < 3) order10 = 3;
    String format = "";
    for (int i = 0; i < order10; i++) format += "#";
    if (ex_folder != null) export_folder = ex_folder;
    File f = new File(export_folder, image_filename + format + Image_type_ext[image_type]);
    fullpathname = f.getAbsolutePath(); 
  }
  void start_export() {
    if (data.num == 0) {
      shortMessage.set_message("Export aborted : No data");
      return;
    }
    shortMessage.clear();
    ci.preview.start_preview(true);
  }
  void change_image_type() {
    if (++image_type >= Image_type_ext.length) image_type = 0;
    set_export_information(null);
  }
}
