// 極楽画像計測アプリケーション（Gokuraku_measure）
// Created by Tatsuya Shirai
// National Institute of Technology, Suzuka college
// Mechanical Department
//
// Ver. 1.0 : 2022.06.07
// Ver. 1.1 : 2022.06.08 : filter（ヘルプ表示にバグあり：動作に支障なし）
// Ver. 1.2 : 2022.06.08 : 縮小可能に．軽微なバグの修正（ヘルプ表示）
// Ver. 1.3 : 2022.06.08 : データフォーマットの変更（データファイル１行目のScale:の後にタブを追加）
// Ver. 1.4 : 2022.06.10 : データフォーマットの変更（データファイル１行目をscale, org_weight, org_heigtのタブ区切りに変更），ショートメッセージ表示機能の追加
// Ver. 2.0 : 2022.06.21 : ファイル名（パス無し）を記録する
// Ver. 2.1 : 2022.06.21 : 全てD&Dに対応，動的メモリー確保に統一，VisualizerがStick pictureに対応
// Ver. 2.2 : 日本語フォント対応（全アプリ），計測中にも線分表示を有効にする，計測のリトライとスキップに対応
// Ver. 3.0 : 2023.03.02 : クリックしたデータ数を出力，メニューの色分け，画像サイズが記録されないことがあるバグの修正，スケールを1.0に戻さない
// Ver. 3.2 : 2023.03.13 : Quick measureの追加, max_measureの規定値を32から0に変更
// Ver. 3.3 : 2023.03.17 : マウス位置のリアルタイム表示機能（標準計測，クイックメジャーモード共に），校正表示機能（Quick measreuのみ）
// Ver. 4.0 : 未定       : visualizeにもクイックメジャーモードを追加，言語パック化

ImportImageFile    im_file;
CurrentImage       ci;
ExportFile         ex_file;
MenuScreen         menuScreen;

final int Scr_w = 700;
final int Scr_h = 700;

enum Menu_status {main, setting, sizeinputing, maxinputing, linewidthinputing, calibrationinputing, calunitinputing};
Menu_status menu_status;

void setup()
{
  im_file       = new ImportImageFile();
  ci            = new CurrentImage();
  ex_file       = new ExportFile();
  shortMessage  = new ShortMessage();
  inputInteger  = new InputInteger();
  inputFloat    = new InputFloat();
  inputString   = new InputString();
  inputFilename = new InputFilename();  // 使わないのだが，初期化しないとNull pointer assignmentが発生する
  menuScreen    = new MenuScreen(8, 48, 150, 40);
  menu_status   = Menu_status.main;

  //サイズ変更を許可する
  surface.setResizable(true);
  surface.setLocation(Init_PosX, Init_PosY);

  // 日本語フォントを使用する
  PFont font = createFont(JapaneseFont, 50);
  textFont(font);

  drop_init();
}

void draw()
{
  menuScreen.reset();

  noStroke();  
  fill(72, 72, 72);

  shortMessage.set_autoclear(true);  // 測定中以外はショートメッセージの自動消去を有効にする
  if ((ci.img != null) || (im_file.folder_mode && ex_file.recording_now())) {
    // 画像が正常にロードされている場合
    shortMessage.set_autoclear(false);  // 測定中はショートメッセージの自動消去を無効にする

    int cnt = 0;
    surface.setSize(ci.img.width, ci.img.height + 40);
    surface.setVisible(true); // Processing4.2でリサイズ後にキー操作その他を受け付けなくなる障害に対する無根拠な対策（効果あり）
    rect(0, 0, width, height);

    // 画像表示
    if (! ex_file.recording_now()) image(ci.img, 0, 0);
    // 全プロット表示
    ci.plotMark.draw_plot(ci.data);  
    // 線分の表示
    if (ci.stick.enable_stick) ci.stick.draw_stick(ci.data);
    // Quick measureモード
    if (ci.quick_measure) ci.qm.plot(ci.guideline); else {
      if (ci.guideline) {
        ci.plotMark.h_line();
        ci.plotMark.v_line();
      }
    }
    // マウス座標のリアルタイム表示
    if (ci.mousepoint_realtime) ci.mousepoint_realtime_display();

    menuScreen.y = height - 16;
  } else {
    // メインメニュー：画像が選択されていない場合
    surface.setSize(Scr_w, Scr_h);
    rect(0, 0, Scr_w - 1, Scr_h - 1);
    menuScreen.dx = 150;
    menuScreen.display_title("Gokuraku measure", 255, 125, 125);
    menuScreen.text_color(TextColor.normal);
    if (menu_status == Menu_status.sizeinputing) {
      menuScreen.display_keyInputPrompt("Plot size");
    } else if (menu_status == Menu_status.maxinputing) {
      menuScreen.display_keyInputPrompt("max measure");
    } else if (menu_status == Menu_status.linewidthinputing) {
      menuScreen.display_keyInputPrompt("Line width");
    } else if (menu_status == Menu_status.calibrationinputing) {
      menuScreen.display_keyInputPrompt("Calibration");
    } else if (menu_status == Menu_status.calunitinputing) {
      menuScreen.display_keyInputPrompt("Unit");
    } else if (menu_status == Menu_status.main) {
      menuScreen.text_color(TextColor.input);
      menuScreen.display_menu("[I]",          "import image files from folder");
      menuScreen.display_menu("(Right click)","select a image file / end");
      menuScreen.text_color(TextColor.option);
      menuScreen.display_menu("[S]",          "open sub-menu");
      menuScreen.display_menu("[L]",          "import define_stick (" + (ci.stick.define_stick.length == 0 ? "none" : "defined") + ")");
      menuScreen.display_menu("[M]",           "change mode (" + (ci.quick_measure ? "Quick measure" : "Normal measure") + ")");
      menuScreen.text_color(TextColor.normal);
      menuScreen.display_menu("(Left click)", "measure");
      menuScreen.display_menu("(Right click)", "quit measurement");
      menuScreen.display_menu("Shift+(left)", "back");
      menuScreen.display_menu("Ctrl+(left)",  "skip");
      menuScreen.display_menu("[←][→]",     "retry / skip measuring");
      menuScreen.display_menu("[SPACE][C]",   "all clear measured data");
      menuScreen.display_menu("[+][-][0]",    "scale up/down/reset (=" + floorN((float)ci.data.scale, 4) + ")");
      menuScreen.display_menu("[ESC]",        "end of measurement / exit");

      menuScreen.text_color(TextColor.option);
      String str = "None";  // filter_no == -1
      if (ci.filter_no == 0) str = "Gray"; else if (ci.filter_no == 1) str = "Invert"; 
      menuScreen.display_menu("[F]",          "change filter (=" + str + ")");
      menuScreen.display_menu("[D]",          "show data (Console only)");
    } else if (menu_status == Menu_status.setting) {
      menuScreen.dx = 64;
      if (!ci.quick_measure) menuScreen.display_menu("[M]",          "input max measure number (=" + ci.max_measure + ")");
      if (!ci.quick_measure) menuScreen.display_menu("[L]",          "toggle line segment flag ("  + ci.stick.enable_stick + ")");
      menuScreen.display_menu("[T]",          "change plot type  (=" + ci.plotMark.get_shape() + ")");
      menuScreen.display_menu("[C]",          "change plot color (=" + ci.plotMark.color_str() + ")");
      menuScreen.display_menu("[P]",          "input plot size  (=" + ci.plotMark.size + ")");
      menuScreen.display_menu("[W]",          "input line width (=" + ci.plotMark.lineWidth + ")");
      menuScreen.display_menu("[G]",          "toggle guide line (" + ci.guideline + ")");
      menuScreen.display_menu("[R]",          "toggle realtime display (" + ci.mousepoint_realtime + ")");
      if (ci.quick_measure)  menuScreen.display_menu("[N]",          "input calibration param.(=" + ci.qm.calibration + "), 0.0 is pixel");
      if (ci.quick_measure)  menuScreen.display_menu("[U]",          "input unit string (=" + ci.qm.cal_unit + ")");
      menuScreen.text_color(TextColor.option);
      menuScreen.display_menu("[Q]",          "quit sub-menu");

      menuScreen.y += 8;
      if (ci.max_measure == 0) menuScreen.display_note("Max measure number = 0, no limit");
      else                     menuScreen.display_note("Max measure number = " + ci.max_measure);
      menuScreen.y += 8;
    }
  }
  shortMessage.display_short_message(menuScreen.x, menuScreen.y);
}

void mousePressed()
{
  if (mouseButton == RIGHT) {    // 右クリック時
    if (ci.quick_measure == false) {
      ex_file.flush_data();  // データを出力
      if (im_file.folder_mode) {
        // フォルダーモードの場合の
        if (im_file.next_image() == null) im_file.quit_folder_mode (); // もう画像がフォルダー内に無いならば終了
      } else {
        // 単体画像モードの場合はファイル選択ダイアログボックスを表示
        open_fileselector();
      }
    } else {
        // Quick measureモード
        if ((ci.qm.status == 0 || ci.qm.status == 1)) {
          ci.qm.clicked(mouseX, mouseY, true);
        }
    }
  } else if (mouseButton == LEFT) {  // 左クリック時
    if (ci.img != null) {
      if (ci.quick_measure == false) {
        if      (keyPressed && keyCode == SHIFT)   ci.back();                   // 計測のキャンセル（バック）
        else if (keyPressed && keyCode == CONTROL) ci.skip();                   // 計測のスキップ
        else                                       ci.measure(mouseX, mouseY);  // 計測
      } else {
        // Quick measureモード
        ci.qm.clicked(mouseX, mouseY, false);
      }
    }
  }
  mouseButton = 0; // なぜか誤動作して，右クリックしても左クリックが反応する場合があるので．
}

void keyPressed()
{
  int    num;
  float  fnum;
  String str;
  if (menu_status == Menu_status.sizeinputing) {  // plot sizeを修正時
    // plot_size を修正時
    if (inputInteger.keyin(key) == false) {
      if (inputInteger.halt != true) {
        if ((num = inputInteger.value) >= 1) ci.plotMark.size = num;
        shortMessage.set_message("Change plot size to " + ci.plotMark.size);
      }        
      menu_status = Menu_status.setting;
      key = 0;
      return;
    }
  } else if (menu_status == Menu_status.maxinputing) {  // max_measure を修正時
    if (inputInteger.keyin(key) == false) {
      if (inputInteger.halt != true) {
        if ((num = inputInteger.value) >= 0) ci.max_measure = num;
        shortMessage.set_message("Change max measure to " + ci.max_measure);
      }        
      menu_status = Menu_status.setting;
      key = 0;
      return;
    }
  } else if (menu_status == Menu_status.linewidthinputing) {  // lineWidth を修正時
    if (inputInteger.keyin(key) == false) {
      if (inputInteger.halt != true) {
        if ((num = inputInteger.value) >= 1) ci.plotMark.lineWidth = num;
        shortMessage.set_message("Change line width to " + ci.plotMark.lineWidth);
      }        
      menu_status = Menu_status.setting;
      key = 0;
      return;
    }
  } else if (menu_status == Menu_status.calibrationinputing) {  // calibration を修正時
    if (inputFloat.keyin(key) == false) {
      if (inputFloat.halt != true) {
        ci.qm.calibration = float(inputFloat.value);
        shortMessage.set_message("Change calibration to " + ci.qm.calibration);
      }        
      menu_status = Menu_status.setting;
      key = 0;
      return;
    }
  } else if (menu_status == Menu_status.calunitinputing) {  // cal_unit を修正時
    if (inputString.keyin(key) == false) {
      if (inputString.halt != true) {
        if ((str = inputString.value).length() >= 0) ci.qm.cal_unit = str;
        shortMessage.set_message("Change calibration unit to [" + ci.qm.cal_unit + "]");
      }        
      menu_status = Menu_status.setting;
      key = 0;
      return;
    }
  } else if (menu_status == Menu_status.main) {
    switch (keyCode) {
      case 'I' :
        File fp;
        fp = new File(im_file.folder_mode_folder);
        selectFolder("Select a image folder:", "folderSelected", fp);  
        break;
      case 'S' :
        // 設定のサブメニューに移動
        menu_status = Menu_status.setting;
        break;
      case 'L' :
        selectInput("Select a define stick:", "stickSelected");
        break;
      case 'M' :
        ci.switch_measure_mode();
        shortMessage.set_message("Change measure mode to " + (ci.quick_measure ? "Quick measure" : "Normal measure") + " mode");
        break;
      case ESC : 
        if (ci.img == null) terminate(); else {
          // 画像計測中
          ex_file.flush_data();                                // データを出力
          if (im_file.folder_mode) im_file.quit_folder_mode(); // フォルダーモードならば終了
          ci.qm.init();                                        // QuickMeasureを終了
          key = 0;                                             // keyをクリアしないとProcessingが終了してしまう
        }
        // 画像計測時以外は，このままProcessingが終了する
        break;
      case 'F' :
        ci.change_filter();
        String fstr = "none";
        if (ci.filter_no >= 0) fstr = filter_name[ci.filter_no];
        shortMessage.set_message("Change filter to [" + fstr + "]");
        break;
      case 107 :
      case  59 :
      case '+' :
        ci.rescale(Scale_rate);
        break;
      case 109 :
      case '-' :
        ci.rescale(1.0/Scale_rate); 
        break;
      case  96 :
      case '0' :
        ci.rescale( 0.0);
        break;
      case 'D' : 
        show_alldata();
        break;
      case LEFT :
        im_file.retry_image();
        break;
      case RIGHT :
        im_file.skip_image();
        break;
      case ' ' :
      case 'C' :
        ci.data.points_clear();
        shortMessage.set_message("All clear measured data");
        break;      
    }
  } else if (menu_status == Menu_status.setting) {
    switch (keyCode) {
      case 'M' :
        if (ci.quick_measure) break;
        inputInteger.start(ci.max_measure);
        menu_status = Menu_status.maxinputing;
        break;
      case 'L' :
        if (ci.quick_measure) break;
        ci.stick.switch_enable_stick();
        shortMessage.set_message("Toggled enable line segment : " + ci.stick.enable_stick);
        break;
      case 'G' :
        ci.switch_guide_line();
        shortMessage.set_message("Toggled guide line : " + ci.guideline);
        break;
      case 'R' :
        ci.switch_mousepoint_realtime();
        shortMessage.set_message("Toggled mousepoint realtime : " + ci.mousepoint_realtime);
        break;
      case 'N' :
        if (!ci.quick_measure) break;
        inputFloat.start(nf(ci.qm.calibration));
        menu_status = Menu_status.calibrationinputing;
        break;
      case 'U' :
        if (!ci.quick_measure) break;
        inputString.start(ci.qm.cal_unit);
        menu_status = Menu_status.calunitinputing;
        break;
      case 'T' : 
        ci.plotMark.change_shape();
        shortMessage.set_message("Change plot type to " + ci.plotMark.get_shape());
        break;
      case 'C' :
        ci.plotMark.change_color();
        shortMessage.set_message("Change plot color to " + ci.plotMark.color_str());
        break;
      case 'P' :
        inputInteger.start(ci.plotMark.size);
        menu_status = Menu_status.sizeinputing;
        break;
      case 'W' :
        inputInteger.start(ci.plotMark.lineWidth);
        menu_status = Menu_status.linewidthinputing;
        break;
      case 'Q' :
        menu_status = Menu_status.main;
        break;
    }
    key = 0;
  }
}

//// ファイル・フォルダー選択ダイアログボックス関係の関数
// ファイル選択ダイアログボックスの呼び出し
void open_fileselector()
{
  selectInput("Select a image file:", "fileSelected");
}
// ファイル選択後の処理
void fileSelected(File selection) 
{
  ex_file.flush_data();
  if (selection == null) ci.img = null;
  else im_file.load(selection.getAbsolutePath());
}

// フォルダーモード
void folderSelected(File selection) {
  im_file.folderSelected(selection);          // selectFolder()から直接呼び出すことができないので．
}

// Stick定義ファイル選択後の処理
void stickSelected(File selection) 
{
  if (selection == null) {
    ci.stick.clearDefineStick();
    return;
  }
  ci.stick.readDefineStick(selection.getAbsolutePath());
}

//// それ以外のユーザー関数
// 終了時の処理
void terminate()
{
  ex_file.flush_data();  // 計測済みならば出力
  println("Exit program.");
  exit();
}

// 計測済みのデータをコンソールに表示（デバッグ用）
void show_alldata()
{
  if (ci.img == null) return;
  println("[Data]");
  if (ci.data.num <= 0) println("no data");
  else for (int n = 0; n < ci.data.num; n++) println(n + " : " + ci.data.px[n] + " , " + ci.data.py[n]); 
}

//// ドラッグ＆ドロップ関係
// 上記URL参照
// ドロップされたファイルの処理
void fileSelected(List<File> fs) {
  File [] fileList = {};
  File oneDirectory = null;  // 特例のための変数
  for (File f:fs) {
    // ドラッグ＆ドロップされたリストの中にフォルダーが含まれていた場合は無視する
    if (f.isDirectory()) {
      oneDirectory = f;
      continue;
    }
    fileList = (File [])append(fileList, f);  // ListからArrayへ変更
  }
  if (fileList.length == 0) {
    // 特例として，D&Dで単一のフォルダーのみがDropされた時はフォルダーモードとして動作する
    if (oneDirectory != null) {
      im_file.folderSelected(oneDirectory);
      return;
    }
    return;  // ファイルが一つも存在しなかった場合
  }
  im_file.open_files(fileList);
}
