/**********************************************************************
 
	Copyright (C) 2007 Hirohisa MORI <joshua@nichibun.ac.jp>
 
	This program is free software; you can redistribute it 
	and/or modify it under the terms of the GLOBALBASE 
	Library General Public License (G-LGPL) as published by 

	http://www.globalbase.org/
 
	This program is distributed in the hope that it will be 
	useful, but WITHOUT ANY WARRANTY; without even the 
	implied warranty of MERCHANTABILITY or FITNESS FOR A 
	PARTICULAR PURPOSE.

**********************************************************************/

#include	"memory_debug.h"
#include	"mx_format.h"
#include	"xlerror.h"
#include	"win_flame.h"
#include	"pg_scale.h"
#include	"machine/include.h"

int
vmt_create_scale(VIEW_METHOD*,MATRIX_NODE*);
void
vmt_draw_scale_int8(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_scale_int16(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_scale_int32(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_scale_uint32(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_scale_int64(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_scale_uint64(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_scale_float(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);
void
vmt_draw_scale_double(VIEW_METHOD*,GBVIEW_FLAME*,RESOURCE*,DRAW_WORK*);


extern SEM luster_lock,tc_lock;
XL_SEXP *
xl_pg_scale_scale(XLISP_ENV * env,XL_SEXP * s,XLISP_ENV * a,XL_SYM_FIELD * sf);

static int div_list[] = {-2,-2};


#define VIEW_METHOD_TBL_DEF(str,type,vmt_draw_scale)	\
VIEW_METHOD_TBL str = {			\
	type,				\
	2,div_list,0,			\
	vmt_create_scale,		\
	0,				\
	vmt_draw_scale,			\
	0				\
};

VIEW_METHOD_TBL_DEF(vm_scale_tbl_int8,MDT_VECTOR|MDT_INT8,vmt_draw_scale_int8)
VIEW_METHOD_TBL_DEF(vm_scale_tbl_uint8,MDT_VECTOR|MDT_UINT8,vmt_draw_scale_int8)
VIEW_METHOD_TBL_DEF(vm_scale_tbl_int16,MDT_VECTOR|MDT_INT16,vmt_draw_scale_int16)
VIEW_METHOD_TBL_DEF(vm_scale_tbl_uint16,MDT_VECTOR|MDT_UINT16,vmt_draw_scale_int16)
VIEW_METHOD_TBL_DEF(vm_scale_tbl_int32,MDT_VECTOR|MDT_INT32,vmt_draw_scale_int32)
VIEW_METHOD_TBL_DEF(vm_scale_tbl_uint32,MDT_VECTOR|MDT_UINT32,vmt_draw_scale_uint32)
VIEW_METHOD_TBL_DEF(vm_scale_tbl_int64,MDT_VECTOR|MDT_INT64,vmt_draw_scale_int64)
VIEW_METHOD_TBL_DEF(vm_scale_tbl_uint64,MDT_VECTOR|MDT_UINT64,vmt_draw_scale_uint64)
VIEW_METHOD_TBL_DEF(vm_scale_tbl_float,MDT_VECTOR|MDT_FLOAT,vmt_draw_scale_float)
VIEW_METHOD_TBL_DEF(vm_scale_tbl_double,MDT_VECTOR|MDT_DOUBLE,vmt_draw_scale_double)

typedef struct vm_scale_data {
	unsigned		edge:2;
	unsigned		floating:1;
#define EDGE_NONE	0
#define EDGE_IN		1
#define EDGE_OUT	2
	union {
		INTEGER64		i;
		U_INTEGER64		ui;
		double			f;
	} d;
} VM_SCALE_DATA;

typedef struct vm_scale_element {
	struct vm_scale_element * next;
	int			min_color;
	int			max_color;
	VM_SCALE_DATA		max;
	VM_SCALE_DATA		min;
} VM_SCALE_COLOR;

typedef struct vm_scale {
	VM_SCALE_COLOR *	color_list;
} VM_SCALE;


typedef struct vm_scale_header {
	VM_WORK_HEADER		wh;
	int			type;
	VM_SCALE*		scale;
	unsigned		initialize:1;
} VM_SCALE_HEADER;
typedef struct vm_scale_int8_16 {
	VM_SCALE_HEADER		h;
	int			table_size;
	int			table_offset;
	int *			color_table;
} VM_SCALE_INT8_16;

typedef struct vm_scale_int32_64_float {
	VM_SCALE_HEADER		h;
} VM_SCALE_INT32_64_FLOAT;

typedef union vm_scale_type {
	VM_SCALE_HEADER		h;
	VM_SCALE_INT8_16	i8;
	VM_SCALE_INT8_16	i16;
	VM_SCALE_INT32_64_FLOAT	i32;
	VM_SCALE_INT32_64_FLOAT	i64;
	VM_SCALE_INT32_64_FLOAT	f;
	VM_SCALE_INT32_64_FLOAT	d;
} VM_SCALE_TYPE;

void
init_pg_scale(XLISP_ENV * env)
{
XLISP_ENV * pgv_env;

ss_printf("INIT pg_scale\n");
	pgv_env = new_env(env);
	root_tag(env,l_string(std_cm,"pg-scale"),pgv_env);
	set_env(pgv_env,l_string(std_cm,"scale"),
		get_func_prim(xl_pg_scale_scale,
			FO_APPLICATIVE,0,1,1));
}

void
vms_scan_data(VM_SCALE_DATA * sd,L_CHAR * d)
{
char * str;
int sign;
	if ( d[0] == 'F' && d[1] == ':' ) {
		sscanf(n_string(std_cm,d),"%lf",&sd->d.f);
		sd->floating = 1;
	}
	else {
		str = n_string(std_cm,d);
		if ( str[0] == '-' ) {
			sign = -1;
			str ++;
		}
		else	sign = 1;
		sscanf(str,I64_FORMAT,&sd->d.i);
		sd->d.i *= sign;
		sd->floating = 0;
	}
}

VM_SCALE *
vms_initialize(RESOURCE * r)
{
VM_SCALE_TYPE * vms;
VM_SCALE * sc;
	vms = (VM_SCALE_TYPE*)get_vm_work_header(r,&vm_scale_tbl_int8,sizeof(*vms));
	if ( vms->h.initialize )
		return vms->h.scale;
	sc = d_alloc(sizeof(*sc));
	sc->color_list = 0;
	vms->h.type = MDT_INT8;
	vms->h.scale = sc;
	vms->h.initialize = 1;
	vms = (VM_SCALE_TYPE*)get_vm_work_header(r,&vm_scale_tbl_uint8,sizeof(*vms));
	vms->h.type = MDT_UINT8;
	vms->h.scale = sc;
	vms->h.initialize = 1;
	vms = (VM_SCALE_TYPE*)get_vm_work_header(r,&vm_scale_tbl_int16,sizeof(*vms));
	vms->h.type = MDT_INT16;
	vms->h.scale = sc;
	vms->h.initialize = 1;
	vms = (VM_SCALE_TYPE*)get_vm_work_header(r,&vm_scale_tbl_uint16,sizeof(*vms));
	vms->h.type = MDT_UINT16;
	vms->h.scale = sc;
	vms->h.initialize = 1;
	vms = (VM_SCALE_TYPE*)get_vm_work_header(r,&vm_scale_tbl_int32,sizeof(*vms));
	vms->h.type = MDT_INT32;
	vms->h.scale = sc;
	vms->h.initialize = 1;
	vms = (VM_SCALE_TYPE*)get_vm_work_header(r,&vm_scale_tbl_uint32,sizeof(*vms));
	vms->h.type = MDT_UINT32;
	vms->h.scale = sc;
	vms->h.initialize = 1;
	vms = (VM_SCALE_TYPE*)get_vm_work_header(r,&vm_scale_tbl_int64,sizeof(*vms));
	vms->h.type = MDT_INT64;
	vms->h.scale = sc;
	vms->h.initialize = 1;
	vms = (VM_SCALE_TYPE*)get_vm_work_header(r,&vm_scale_tbl_uint64,sizeof(*vms));
	vms->h.type = MDT_UINT64;
	vms->h.scale = sc;
	vms->h.initialize = 1;
	vms = (VM_SCALE_TYPE*)get_vm_work_header(r,&vm_scale_tbl_float,sizeof(*vms));
	vms->h.type = MDT_FLOAT;
	vms->h.scale = sc;
	vms->h.initialize = 1;
	vms = (VM_SCALE_TYPE*)get_vm_work_header(r,&vm_scale_tbl_double,sizeof(*vms));
	vms->h.type = MDT_FLOAT;
	vms->h.scale = sc;
	vms->h.initialize = 1;
	return sc;
}

int
cmp_vm_scale_data(VM_SCALE_DATA * d1,VM_SCALE_DATA * d2)
{
	if ( d1->floating != d2->floating )
		return 1;
	if ( d1->floating == 0 ) {
		if ( d1->d.i < d2->d.i )
			return -1;
		if ( d1->d.i > d2->d.i )
			return 1;
	}
	else {
		if ( d1->d.f < d2->d.f )
			return -1;
		if ( d1->d.f > d2->d.f )
			return 1;
	}
	switch ( d1->edge ) {
	case EDGE_NONE:
		return 1;
	case EDGE_IN:
		switch ( d2->edge ) {
		case EDGE_NONE:
			return 1;
		case EDGE_IN:
			return 0;
		case EDGE_OUT:
			return -1;
		}
		break;
	case EDGE_OUT:
		switch ( d2->edge ) {
		case EDGE_NONE:
			return 1;
		case EDGE_IN:
			return -1;
		case EDGE_OUT:
			return -1;
		}
		break;
	}
	return 0;
}

XL_SEXP *
xl_pg_scale_scale(XLISP_ENV * env,XL_SEXP * s,XLISP_ENV * a,XL_SYM_FIELD * sf)
{
VM_SCALE * sc;
VM_SCALE_COLOR * scc,*scc_prev;
XL_SEXP * ret;
RESOURCE * r;
L_CHAR * start,*start_dot;
L_CHAR * end,*end_dot;
L_CHAR * start_color;
L_CHAR * end_color;
	r = get_resource_ptr(&ret,env,s->h.file,s->h.line);
	if ( r == 0 )
		return ret;
	sc = vms_initialize(r);

	start = get_sf_attribute(sf,l_string(std_cm,"s"));
	start_dot = get_sf_attribute(sf,l_string(std_cm,"s."));
	end = get_sf_attribute(sf,l_string(std_cm,"e"));
	end_dot = get_sf_attribute(sf,l_string(std_cm,"e."));
	start_color = get_sf_attribute(sf,l_string(std_cm,"s-color"));
	end_color = get_sf_attribute(sf,l_string(std_cm,"e-color"));

	for ( scc_prev = sc->color_list ; scc_prev && scc_prev->next ; scc_prev = scc_prev->next );
	scc = d_alloc(sizeof(*scc));
	memset(scc,0,sizeof(*scc));
	if ( start == 0 && start_dot == 0 ) {
		if ( scc_prev == 0 ) {
			scc->min.edge = EDGE_NONE;
			if ( end_color ) {
				sscanf(n_string(std_cm,end_color),"%i",&scc->min_color);
			}
			else {
				ret = get_error(
					s->h.file,
					s->h.line,
					XLE_PROTO_INV_PARAM,
					l_string(std_cm,"pg-scale/scale"),
					List(n_get_string("e-color attribute is required"),
						-1));
				goto end;
			}
		}
		else {
			switch ( scc_prev->max.edge ) {
			case EDGE_NONE:
				ret = get_error(
					s->h.file,
					s->h.line,
					XLE_PROTO_INV_PARAM,
					l_string(std_cm,"pg-scale/scale"),
					List(n_get_string("previous scale edge NONE"),
						-1));
				goto end;
			case EDGE_IN:
				scc->min = scc_prev->max;
				scc->min.edge = EDGE_OUT;
				break;
			case EDGE_OUT:
				scc->min = scc_prev->max;
				scc->min.edge = EDGE_IN;
				break;
			default:
				er_panic("xl_pg_scale_scale");
			}
			if ( start_color ) {
				sscanf(n_string(std_cm,start_color),"%i",&scc->min_color);
			}
			else {
				scc->min_color = scc_prev->max_color;
			}
		}
	}
	else if ( start && start_dot ) {
		ret = get_error(
			s->h.file,
			s->h.line,
			XLE_PROTO_INV_PARAM,
			l_string(std_cm,"pg-scale/scale"),
			List(n_get_string("s and s. select one"),
				-1));
		goto end;
	}
	else if ( start ) {
		vms_scan_data(&scc->min,start);
		scc->min.edge = EDGE_IN;
		goto start_color_setup;
	}
	else {
		vms_scan_data(&scc->min,start_dot);
		scc->min.edge = EDGE_OUT;
	start_color_setup:
		if ( start_color ) {
			sscanf(n_string(std_cm,start_color),"%i",&scc->min_color);
		}
		else if ( scc_prev )
				scc->min_color = scc_prev->max_color;
		else if ( end_color )
			sscanf(n_string(std_cm,end_color),"%i",&scc->min_color);
		else {
			ret = get_error(
				s->h.file,
				s->h.line,
				XLE_PROTO_INV_PARAM,
				l_string(std_cm,"pg-scale/scale"),
				List(n_get_string("e-color is required"),
					-1));
			goto end;
		}
	}
	
	if ( end == 0 && end_color == 0 ) {
		scc->max.edge = EDGE_NONE;
	}
	else if ( end && end_dot ) {
		ret = get_error(
			s->h.file,
			s->h.line,
			XLE_PROTO_INV_PARAM,
			l_string(std_cm,"pg-scale/scale"),
			List(n_get_string("e and e. select one"),
				-1));
		goto end;
	}
	else if ( end ) {
		vms_scan_data(&scc->max,end);
		scc->max.edge = EDGE_IN;
	}
	else {
		vms_scan_data(&scc->max,end_dot);
		scc->max.edge = EDGE_OUT;
	}
	if ( end_color ) {
		sscanf(n_string(std_cm,end_color),"%i",&scc->max_color);
	}
	else {
		scc->max_color = scc->min_color;
	}

	scc->next = 0;
	if ( scc_prev ) {
		if ( cmp_vm_scale_data(&scc_prev->max,&scc->min) >= 0 ) {
			ret = get_error(
				s->h.file,
				s->h.line,
				XLE_PROTO_INV_PARAM,
				l_string(std_cm,"pg-scale/scale"),
				List(n_get_string("overwrapped with prev scale"),
					-1));
			goto end;
		}
		scc_prev->next = scc;
	}
	else {
		sc->color_list = scc;
	}

	return 0;
end:
	d_f_ree(scc);
	return ret;
/*
inv_param:
	return get_error(
		s->h.file,
		s->h.line,
		XLE_PROTO_INV_PARAM,
		l_string(std_cm,"pg-scale/scale"),
		List(n_get_string("attribute d is required"),
			-1));
type_missmatch:
	return get_error(
		s->h.file,
		s->h.line,
		XLE_SEMANTICS_TYPE_MISSMATCH,
		l_string(std_cm,"pg-vector/Arg"),
		List(n_get_string("type missmatch"),
			n_get_string(e_msg),
			e_sexp,
			-1));
*/
}


void
get_cc(unsigned int * rgba_list,int rgba)
{
	rgba_list[0] = ((rgba>>24)&0xff);
	rgba_list[1] = ((rgba>>16)&0xff);
	rgba_list[2] = ((rgba>>8)&0xff);
	rgba_list[3] = ((rgba)&0xff);
}

int
convert_color_int(VM_SCALE * sc,INTEGER64 v)
{
VM_SCALE_COLOR * scc;
unsigned int min_cc[4],max_cc[4],result_cc[4];
U_INTEGER64 base_min,base_max,base;
int i;
int ret;
	for ( scc = sc->color_list ; scc ; scc = scc->next ) {
		switch ( scc->min.edge ) {
		case EDGE_NONE:
			break;
		case EDGE_IN:
			if ( v < scc->min.d.i )
				continue;
			break;
		case EDGE_OUT:
			if ( v <= scc->min.d.i )
				continue;
			break;
		default:
			er_panic("convert_color_int");
		}
		switch ( scc->max.edge ) {
		case EDGE_NONE:
			break;
		case EDGE_IN:
			if ( v > scc->max.d.i )
				continue;
			break;
		case EDGE_OUT:
			if ( v >= scc->max.d.i )
				continue;
			break;
		default:
			er_panic("convert_color_int");
		}
		base_min = v - scc->min.d.i;
		base_max = scc->max.d.i - v;
		base = scc->max.d.i - scc->min.d.i;
		get_cc(min_cc,scc->min_color);
		get_cc(max_cc,scc->max_color);
		if ( base ) {
			for ( i = 0 ; i < 4 ; i ++ )
				result_cc[i] = (min_cc[i]*base_max + max_cc[i]*base_min)/base;
		}
		else {
			for ( i = 0 ; i < 4 ; i ++ )
				result_cc[i] = (min_cc[i] + max_cc[i])/2;
		}
		ret = convert_rgba_to_cc(
				((result_cc[0])<<24)|
				((result_cc[1])<<16)|
				((result_cc[2])<<8)|
				(result_cc[3]));
		return ret;
	}
	return 0;
}


int
convert_color_uint(VM_SCALE * sc,U_INTEGER64 v)
{
VM_SCALE_COLOR * scc;
unsigned int min_cc[4],max_cc[4],result_cc[4];
U_INTEGER64 base_min,base_max,base;
int i;
int ret;
	for ( scc = sc->color_list ; scc ; scc = scc->next ) {
		switch ( scc->min.edge ) {
		case EDGE_NONE:
			break;
		case EDGE_IN:
			if ( v < scc->min.d.ui )
				continue;
			break;
		case EDGE_OUT:
			if ( v <= scc->min.d.ui )
				continue;
			break;
		default:
			er_panic("convert_color_int");
		}
		switch ( scc->max.edge ) {
		case EDGE_NONE:
			break;
		case EDGE_IN:
			if ( v > scc->max.d.ui )
				continue;
			break;
		case EDGE_OUT:
			if ( v >= scc->max.d.ui )
				continue;
			break;
		default:
			er_panic("convert_color_int");
		}
		base_min = v - scc->min.d.i;
		base_max = scc->max.d.i - v;
		base = scc->max.d.i - scc->min.d.i;
		get_cc(min_cc,scc->min_color);
		get_cc(max_cc,scc->max_color);
		for ( i = 0 ; i < 4 ; i ++ )
			result_cc[i] = (min_cc[i]*base_max + max_cc[i]*base_min)/base;
		ret = convert_rgba_to_cc(
				((result_cc[0])<<24)|
				((result_cc[1])<<16)|
				((result_cc[2])<<8)|
				(result_cc[3]));
		return ret;
	}
	return 0;
}


int
convert_color_float(VM_SCALE * sc,double v)
{
VM_SCALE_COLOR * scc;
unsigned int min_cc[4],max_cc[4],result_cc[4];
double base_min,base_max,base;
int i;
int ret;
	for ( scc = sc->color_list ; scc ; scc = scc->next ) {
		switch ( scc->min.edge ) {
		case EDGE_NONE:
			break;
		case EDGE_IN:
			if ( v < scc->min.d.f )
				continue;
			break;
		case EDGE_OUT:
			if ( v <= scc->min.d.f )
				continue;
			break;
		default:
			er_panic("convert_color_int");
		}
		switch ( scc->max.edge ) {
		case EDGE_NONE:
			break;
		case EDGE_IN:
			if ( v > scc->max.d.f )
				continue;
			break;
		case EDGE_OUT:
			if ( v >= scc->max.d.f )
				continue;
			break;
		default:
			er_panic("convert_color_int");
		}
		base_min = v - scc->min.d.i;
		base_max = scc->max.d.i - v;
		base = scc->max.d.i - scc->min.d.i;
		get_cc(min_cc,scc->min_color);
		get_cc(max_cc,scc->max_color);
		for ( i = 0 ; i < 4 ; i ++ )
			result_cc[i] = (min_cc[i]*base_max + max_cc[i]*base_min)/base;
		ret = convert_rgba_to_cc(
				((result_cc[0])<<24)|
				((result_cc[1])<<16)|
				((result_cc[2])<<8)|
				(result_cc[3]));
		return ret;
	}
	return 0;
}

void
setup_color_table(VM_SCALE_TYPE * vst,int bits,int sign)
{
int * ct;
int v;
	ct = d_alloc((1<<bits)*sizeof(int));
	for ( v = 0 ; v < (1<<(bits-1)) ; v ++ ) {
		ct[v] = convert_color_int(vst->h.scale,v);
//ss_printf("V=%i %x\n",v,ct[v]);
	}
	if ( sign ) {
		for ( ; v < (1<<bits) ; v ++ ) {
			ct[v] = convert_color_int(vst->h.scale,v-(1<<bits));
//ss_printf("V=%i %x\n",v,ct[v]);
		}
	}
	else {
		for ( ; v < (1<<bits) ; v ++ ) {
			ct[v] = convert_color_uint(vst->h.scale,v);
		}
	}
	vst->i8.color_table = ct;
}


int
vmt_create_scale(VIEW_METHOD* vm,MATRIX_NODE* n)
{
VM_SCALE_TYPE * vst;
VM_SCALE * sc;
VM_SCALE_COLOR * scc;
	vst = (VM_SCALE_TYPE*)_get_vm_work_header(vm->r,vm->tbl,sizeof(*vst));
	vm->work = vst;
	if ( vst->h.initialize == 0 ) {
		sc = vms_initialize(vm->r);
		scc = d_alloc(sizeof(*scc));
		memset(scc,0,sizeof(*scc));
		scc->min.edge = EDGE_IN;
		scc->max.edge = EDGE_IN;
		scc->min_color = 0xffffff00;
		scc->max_color = 0;
		switch ( vst->h.type ) {
		case MDT_INT8:
			scc->min.d.i = -0x80;
			scc->max.d.i = 0x7f;
			break;
		case MDT_UINT8:
			scc->min.d.i = 0;
			scc->max.d.i = 0xff;
			break;
		case MDT_INT16:
			scc->min.d.i = -0x8000;
			scc->max.d.i = 0x7fff;
			break;
		case MDT_UINT16:
			scc->min.d.i = 0;
			scc->max.d.i = 0xffff;
			break;
		case MDT_INT32:
			scc->min.d.i = -0x80000000;
			scc->max.d.i = 0x7fffffff;
			break;
		case MDT_UINT32:
			scc->min.d.i = 0;
			scc->max.d.i = 0xffffffff;
			break;
		case MDT_INT64:
			scc->min.d.i = -0x8000000000000000LL;
			scc->max.d.i = 0x7fffffffffffffffLL;
			break;
		case MDT_UINT64:
			scc->min.d.i = 0;
			scc->max.d.i = 0xffffffffffffffffLL;
			break;
		case MDT_FLOAT:
			scc->min.floating = 1;
			scc->max.floating = 1;
			scc->min.d.f = __FLT_MIN__;
			scc->max.d.f = __FLT_MAX__;
			break;
		case MDT_DOUBLE:
			scc->min.floating = 1;
			scc->max.floating = 1;
			scc->min.d.f = __DBL_MIN__;
			scc->max.d.f = __DBL_MAX__;
			break;
		default:
			er_panic("vmt_create_scale");
		}
		sc->color_list = scc;
	}
	switch ( vst->h.type ) {
	case MDT_INT8:
		setup_color_table(vst,8,1);
		break;
	case MDT_UINT8:
		setup_color_table(vst,8,0);
		break;
	case MDT_INT16:
		setup_color_table(vst,16,1);
		break;
	case MDT_UINT16:
		setup_color_table(vst,16,0);
		break;
	default:
		break;
	}
	return 0;
}


typedef struct mtx_cache_int8 {
	int		tl_x;
	int		tl_y;
	int		br_x;
	int		br_y;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	unsigned char *	ds_ptr;
	unsigned	leaf;
} MTX_CACHE_INT8;


unsigned long
get_mtx_scale_int8(VIEW_METHOD * vm,MTX_CACHE_INT8 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 y,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[3];
int xx,yy;
INTEGER64 mx,my;
int clr;
VM_SCALE_TYPE * vst;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x && cache->tl_y <= y &&
			cache->br_x > x && cache->br_y > y )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
	dim_code[2] = y;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return C_TRANSPARENT;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	dim_code[2] = cache->n->dim_code[2];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->tl_y = dim_code[2] & (my=(-(1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	cache->br_y = cache->tl_y + (1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]));
if ( (x & mx) != cache->tl_x || (y & my) != cache->tl_y )
er_panic("get_mtx_data");
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	if ( cache->br_y > m->pixel_size[1] )
		cache->br_y = m->pixel_size[1];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	yy = (y - cache->tl_y)>>(cache->level*m->dim_divide[1]);
	vst = vm->work;
	return vst->i8.color_table[cache->ds_ptr[xx + yy * cache->ds.ix[0]]];
}



void
vmt_draw_scale_int8(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx,yy;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
unsigned long cc;
INTEGER64 lev;
INTEGER64 lev_cnt;
MTX_CACHE_INT8 cache;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;
VM_SCALE_TYPE * vst;


	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];
	vst = vm->work;

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
//	lock_task(tc_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		yy = rint(p.y*r->pr64.dpm);
		if ( xx < 0 || yy < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( yy >= mtx->pixel_size[1] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx,ly;
			if (		cache.tl_x <= xx && xx < cache.br_x &&
					cache.tl_y <= yy && yy < cache.br_y ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = vst->i8.color_table[cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]];
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = vst->i8.color_table[cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]];
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
//			cc = _tc_get_pixel(&r->pr64.tc,level,xx,yy,&lev);
			cc = get_mtx_scale_int8(vm,&cache,r,level,xx,yy,&lev);
//ss_printf("CC %lli %lli %lli %x\n",xx,yy,level,cc);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
//			cache = r->pr64.tc;
			lev_cnt ++;
		}
		dw->pixels[i] = cc;
		dw->flags |= WFF_DRAW;
	}
//	unlock_task(tc_lock,"draw_pixel");

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

}



typedef struct mtx_cache_int16 {
	int		tl_x;
	int		tl_y;
	int		br_x;
	int		br_y;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	unsigned short * ds_ptr;
	unsigned	leaf;
} MTX_CACHE_INT16;


unsigned long
get_mtx_scale_int16(VIEW_METHOD * vm,MTX_CACHE_INT16 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 y,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[3];
int xx,yy;
INTEGER64 mx,my;
int clr;
VM_SCALE_TYPE * vst;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x && cache->tl_y <= y &&
			cache->br_x > x && cache->br_y > y )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
	dim_code[2] = y;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return C_TRANSPARENT;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	dim_code[2] = cache->n->dim_code[2];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->tl_y = dim_code[2] & (my=(-(1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	cache->br_y = cache->tl_y + (1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]));
if ( (x & mx) != cache->tl_x || (y & my) != cache->tl_y )
er_panic("get_mtx_data");
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	if ( cache->br_y > m->pixel_size[1] )
		cache->br_y = m->pixel_size[1];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	yy = (y - cache->tl_y)>>(cache->level*m->dim_divide[1]);
	vst = vm->work;
//ss_printf("=== CC %lli %lli \n",x,y);
	return vst->i16.color_table[cache->ds_ptr[xx + yy * cache->ds.ix[0]]];
}



void
vmt_draw_scale_int16(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx,yy;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
unsigned long cc;
INTEGER64 lev;
INTEGER64 lev_cnt;
MTX_CACHE_INT16 cache;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;
VM_SCALE_TYPE * vst;


	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];
	vst = vm->work;

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
//	lock_task(tc_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		yy = rint(p.y*r->pr64.dpm);
		if ( xx < 0 || yy < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( yy >= mtx->pixel_size[1] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx,ly;
			if (		cache.tl_x <= xx && xx < cache.br_x &&
					cache.tl_y <= yy && yy < cache.br_y ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = vst->i16.color_table[cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]];
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = vst->i16.color_table[cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]];
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
//			cc = _tc_get_pixel(&r->pr64.tc,level,xx,yy,&lev);
			cc = get_mtx_scale_int16(vm,&cache,r,level,xx,yy,&lev);
//ss_printf("CC %lli %lli %lli %x\n",xx,yy,level,cc);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
//			cache = r->pr64.tc;
			lev_cnt ++;
		}
		dw->pixels[i] = cc;
		dw->flags |= WFF_DRAW;
	}
//	unlock_task(tc_lock,"draw_pixel");

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

}




typedef struct mtx_cache_int32 {
	int		tl_x;
	int		tl_y;
	int		br_x;
	int		br_y;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	unsigned int *	ds_ptr;
	unsigned	leaf;
} MTX_CACHE_INT32;



unsigned long
get_mtx_scale_int32(VIEW_METHOD * vm,MTX_CACHE_INT32 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 y,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[3];
int xx,yy;
INTEGER64 mx,my;
int clr;
VM_SCALE_TYPE * vst;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x && cache->tl_y <= y &&
			cache->br_x > x && cache->br_y > y )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
	dim_code[2] = y;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return C_TRANSPARENT;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	dim_code[2] = cache->n->dim_code[2];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->tl_y = dim_code[2] & (my=(-(1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	cache->br_y = cache->tl_y + (1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]));
if ( (x & mx) != cache->tl_x || (y & my) != cache->tl_y )
er_panic("get_mtx_data");
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	if ( cache->br_y > m->pixel_size[1] )
		cache->br_y = m->pixel_size[1];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	yy = (y - cache->tl_y)>>(cache->level*m->dim_divide[1]);
	vst = vm->work;
	return convert_color_int(vst->h.scale,cache->ds_ptr[xx + yy * cache->ds.ix[0]]);
}


void
vmt_draw_scale_int32(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx,yy;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
unsigned long cc;
INTEGER64 lev;
INTEGER64 lev_cnt;
MTX_CACHE_INT32 cache;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;
VM_SCALE_TYPE * vst;


	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];
	vst = vm->work;

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
//	lock_task(tc_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		yy = rint(p.y*r->pr64.dpm);
		if ( xx < 0 || yy < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( yy >= mtx->pixel_size[1] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx,ly;
			if (		cache.tl_x <= xx && xx < cache.br_x &&
					cache.tl_y <= yy && yy < cache.br_y ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = convert_rgba_to_cc(
						convert_color_int(vst->h.scale,cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]));
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = convert_rgba_to_cc(
						convert_color_int(vst->h.scale,cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]));
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
//			cc = _tc_get_pixel(&r->pr64.tc,level,xx,yy,&lev);
			cc = get_mtx_scale_int32(vm,&cache,r,level,xx,yy,&lev);
//ss_printf("CC %lli %lli %lli %x\n",xx,yy,level,cc);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
//			cache = r->pr64.tc;
			lev_cnt ++;
		}
		dw->pixels[i] = cc;
		dw->flags |= WFF_DRAW;
	}
//	unlock_task(tc_lock,"draw_pixel");

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

}




typedef struct mtx_cache_uint32 {
	int		tl_x;
	int		tl_y;
	int		br_x;
	int		br_y;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	unsigned int * ds_ptr;
	unsigned	leaf;
} MTX_CACHE_UINT32;



unsigned long
get_mtx_scale_uint32(VIEW_METHOD * vm,MTX_CACHE_UINT32 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 y,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[3];
int xx,yy;
INTEGER64 mx,my;
int clr;
VM_SCALE_TYPE * vst;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x && cache->tl_y <= y &&
			cache->br_x > x && cache->br_y > y )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
	dim_code[2] = y;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return C_TRANSPARENT;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	dim_code[2] = cache->n->dim_code[2];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->tl_y = dim_code[2] & (my=(-(1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	cache->br_y = cache->tl_y + (1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]));
if ( (x & mx) != cache->tl_x || (y & my) != cache->tl_y )
er_panic("get_mtx_data");
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	if ( cache->br_y > m->pixel_size[1] )
		cache->br_y = m->pixel_size[1];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	yy = (y - cache->tl_y)>>(cache->level*m->dim_divide[1]);
	vst = vm->work;
	return convert_color_uint(vst->h.scale,cache->ds_ptr[xx + yy * cache->ds.ix[0]]);
}


void
vmt_draw_scale_uint32(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx,yy;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
unsigned long cc;
INTEGER64 lev;
INTEGER64 lev_cnt;
MTX_CACHE_UINT32 cache;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;
VM_SCALE_TYPE * vst;


	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];
	vst = vm->work;

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
//	lock_task(tc_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		yy = rint(p.y*r->pr64.dpm);
		if ( xx < 0 || yy < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( yy >= mtx->pixel_size[1] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx,ly;
			if (		cache.tl_x <= xx && xx < cache.br_x &&
					cache.tl_y <= yy && yy < cache.br_y ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = convert_rgba_to_cc(
						convert_color_uint(vst->h.scale,cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]));
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = convert_rgba_to_cc(
						convert_color_uint(vst->h.scale,cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]));
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
//			cc = _tc_get_pixel(&r->pr64.tc,level,xx,yy,&lev);
			cc = get_mtx_scale_uint32(vm,&cache,r,level,xx,yy,&lev);
//ss_printf("CC %lli %lli %lli %x\n",xx,yy,level,cc);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
//			cache = r->pr64.tc;
			lev_cnt ++;
		}
		dw->pixels[i] = cc;
		dw->flags |= WFF_DRAW;
	}
//	unlock_task(tc_lock,"draw_pixel");

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

}





typedef struct mtx_cache_int64 {
	int		tl_x;
	int		tl_y;
	int		br_x;
	int		br_y;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	U_INTEGER64 *	ds_ptr;
	unsigned	leaf;
} MTX_CACHE_INT64;



unsigned long
get_mtx_scale_int64(VIEW_METHOD * vm,MTX_CACHE_INT64 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 y,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[3];
int xx,yy;
INTEGER64 mx,my;
int clr;
VM_SCALE_TYPE * vst;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x && cache->tl_y <= y &&
			cache->br_x > x && cache->br_y > y )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
	dim_code[2] = y;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return C_TRANSPARENT;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	dim_code[2] = cache->n->dim_code[2];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->tl_y = dim_code[2] & (my=(-(1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	cache->br_y = cache->tl_y + (1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]));
if ( (x & mx) != cache->tl_x || (y & my) != cache->tl_y )
er_panic("get_mtx_data");
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	if ( cache->br_y > m->pixel_size[1] )
		cache->br_y = m->pixel_size[1];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	yy = (y - cache->tl_y)>>(cache->level*m->dim_divide[1]);
	vst = vm->work;
	return convert_color_int(vst->h.scale,cache->ds_ptr[xx + yy * cache->ds.ix[0]]);
}


void
vmt_draw_scale_int64(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx,yy;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
unsigned long cc;
INTEGER64 lev;
INTEGER64 lev_cnt;
MTX_CACHE_INT64 cache;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;
VM_SCALE_TYPE * vst;


	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];
	vst = vm->work;

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
//	lock_task(tc_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		yy = rint(p.y*r->pr64.dpm);
		if ( xx < 0 || yy < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( yy >= mtx->pixel_size[1] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx,ly;
			if (		cache.tl_x <= xx && xx < cache.br_x &&
					cache.tl_y <= yy && yy < cache.br_y ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = convert_rgba_to_cc(
						convert_color_int(vst->h.scale,cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]));
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = convert_rgba_to_cc(
						convert_color_int(vst->h.scale,cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]));
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
//			cc = _tc_get_pixel(&r->pr64.tc,level,xx,yy,&lev);
			cc = get_mtx_scale_int64(vm,&cache,r,level,xx,yy,&lev);
//ss_printf("CC %lli %lli %lli %x\n",xx,yy,level,cc);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
//			cache = r->pr64.tc;
			lev_cnt ++;
		}
		dw->pixels[i] = cc;
		dw->flags |= WFF_DRAW;
	}
//	unlock_task(tc_lock,"draw_pixel");

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

}




typedef struct mtx_cache_uint64 {
	int		tl_x;
	int		tl_y;
	int		br_x;
	int		br_y;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	U_INTEGER64 * ds_ptr;
	unsigned	leaf;
} MTX_CACHE_UINT64;



unsigned long
get_mtx_scale_uint64(VIEW_METHOD * vm,MTX_CACHE_UINT64 * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 y,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[3];
int xx,yy;
INTEGER64 mx,my;
int clr;
VM_SCALE_TYPE * vst;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x && cache->tl_y <= y &&
			cache->br_x > x && cache->br_y > y )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
	dim_code[2] = y;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return C_TRANSPARENT;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	dim_code[2] = cache->n->dim_code[2];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->tl_y = dim_code[2] & (my=(-(1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	cache->br_y = cache->tl_y + (1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]));
if ( (x & mx) != cache->tl_x || (y & my) != cache->tl_y )
er_panic("get_mtx_data");
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	if ( cache->br_y > m->pixel_size[1] )
		cache->br_y = m->pixel_size[1];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	yy = (y - cache->tl_y)>>(cache->level*m->dim_divide[1]);
	vst = vm->work;
	return convert_color_uint(vst->h.scale,cache->ds_ptr[xx + yy * cache->ds.ix[0]]);
}


void
vmt_draw_scale_uint64(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx,yy;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
unsigned long cc;
INTEGER64 lev;
INTEGER64 lev_cnt;
MTX_CACHE_UINT64 cache;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;
VM_SCALE_TYPE * vst;


	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];
	vst = vm->work;

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
//	lock_task(tc_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		yy = rint(p.y*r->pr64.dpm);
		if ( xx < 0 || yy < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( yy >= mtx->pixel_size[1] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx,ly;
			if (		cache.tl_x <= xx && xx < cache.br_x &&
					cache.tl_y <= yy && yy < cache.br_y ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = convert_rgba_to_cc(
						convert_color_uint(vst->h.scale,cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]));
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = convert_rgba_to_cc(
						convert_color_uint(vst->h.scale,cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]));
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
//			cc = _tc_get_pixel(&r->pr64.tc,level,xx,yy,&lev);
			cc = get_mtx_scale_uint64(vm,&cache,r,level,xx,yy,&lev);
//ss_printf("CC %lli %lli %lli %x\n",xx,yy,level,cc);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
//			cache = r->pr64.tc;
			lev_cnt ++;
		}
		dw->pixels[i] = cc;
		dw->flags |= WFF_DRAW;
	}
//	unlock_task(tc_lock,"draw_pixel");

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

}






typedef struct mtx_cache_float {
	int		tl_x;
	int		tl_y;
	int		br_x;
	int		br_y;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	float*		ds_ptr;
	unsigned	leaf;
} MTX_CACHE_FLOAT;



unsigned long
get_mtx_scale_float(VIEW_METHOD * vm,MTX_CACHE_FLOAT * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 y,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[3];
int xx,yy;
INTEGER64 mx,my;
int clr;
VM_SCALE_TYPE * vst;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x && cache->tl_y <= y &&
			cache->br_x > x && cache->br_y > y )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
	dim_code[2] = y;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return C_TRANSPARENT;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	dim_code[2] = cache->n->dim_code[2];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->tl_y = dim_code[2] & (my=(-(1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	cache->br_y = cache->tl_y + (1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]));
if ( (x & mx) != cache->tl_x || (y & my) != cache->tl_y )
er_panic("get_mtx_data");
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	if ( cache->br_y > m->pixel_size[1] )
		cache->br_y = m->pixel_size[1];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	yy = (y - cache->tl_y)>>(cache->level*m->dim_divide[1]);
	vst = vm->work;
	return convert_color_float(vst->h.scale,cache->ds_ptr[xx + yy * cache->ds.ix[0]]);
}


void
vmt_draw_scale_float(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx,yy;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
unsigned long cc;
INTEGER64 lev;
INTEGER64 lev_cnt;
MTX_CACHE_FLOAT cache;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;
VM_SCALE_TYPE * vst;


	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];
	vst = vm->work;

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
//	lock_task(tc_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		yy = rint(p.y*r->pr64.dpm);
		if ( xx < 0 || yy < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( yy >= mtx->pixel_size[1] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx,ly;
			if (		cache.tl_x <= xx && xx < cache.br_x &&
					cache.tl_y <= yy && yy < cache.br_y ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = convert_rgba_to_cc(
						convert_color_float(vst->h.scale,cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]));
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = convert_rgba_to_cc(
						convert_color_float(vst->h.scale,cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]));
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
//			cc = _tc_get_pixel(&r->pr64.tc,level,xx,yy,&lev);
			cc = get_mtx_scale_float(vm,&cache,r,level,xx,yy,&lev);
//ss_printf("CC %lli %lli %lli %x\n",xx,yy,level,cc);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
//			cache = r->pr64.tc;
			lev_cnt ++;
		}
		dw->pixels[i] = cc;
		dw->flags |= WFF_DRAW;
	}
//	unlock_task(tc_lock,"draw_pixel");

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

}


typedef struct mtx_cache_double {
	int		tl_x;
	int		tl_y;
	int		br_x;
	int		br_y;
	int		level;
	int		rect_logsize;
	MATRIX_DH_SET	ds;
	MATRIX_NODE *	n;
	double*		ds_ptr;
	unsigned	leaf;
} MTX_CACHE_DOUBLE;



unsigned long
get_mtx_scale_double(VIEW_METHOD * vm,MTX_CACHE_DOUBLE * cache,RESOURCE * r,INTEGER64 lev,INTEGER64 x,INTEGER64 y,INTEGER64 * lev_p)
{
int err;
void * d;
MATRIX * m;
MATRIX_NODE * nn;
INTEGER64 dim_code[3];
int xx,yy;
INTEGER64 mx,my;
int clr;
VM_SCALE_TYPE * vst;
	m = r->h.mtx;
	clr = 0;
	if ( cache->n ) {
		if ( cache->level == lev &&
			cache->tl_x <= x && cache->tl_y <= y &&
			cache->br_x > x && cache->br_y > y )
			goto end;
		clr = 1;
		matrix_node_channel_unlock(cache->n,0);
		unlock_node(cache->n,0);
	}
	nn = cache->n;
	
	dim_code[0] = lev;
	dim_code[1] = x;
	dim_code[2] = y;
//ss_printf("get_mtx_data-1\n");

	if ( r->h.mtx->p.flags & MPF_NODE_DIPENDENT ) {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_NODE_CREATE,0,r->h.loop_no);
	}
	else {
		d = get_matrix_node_channel
				(&err,&cache->n,r->h.mtx,
				&dim_code[0],vm->view_channel[0],
				GN_TREE,GN_STEP_NODE_CREATE,0,r->h.loop_no);
	}
	if ( d == 0 ) {
		if ( nn && clr == 0 ) {
			matrix_node_channel_unlock(nn,0);
			unlock_node(nn,0);
		}
		memset(cache,0,sizeof(*cache));
		*lev_p = m->total_levels;
		return C_TRANSPARENT;
	}
	get_matrix_dh_set(&cache->ds,d);
	dim_code[0] = cache->n->dim_code[0];
	dim_code[1] = cache->n->dim_code[1];
	dim_code[2] = cache->n->dim_code[2];
	cache->tl_x = dim_code[1] & (mx=(-(1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]))));
	cache->tl_y = dim_code[2] & (my=(-(1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]))));
	cache->br_x = cache->tl_x + (1<<(dim_code[0]*m->dim_divide[0] + m->block_size[0]));
	cache->br_y = cache->tl_y + (1<<(dim_code[0]*m->dim_divide[1] + m->block_size[1]));
if ( (x & mx) != cache->tl_x || (y & my) != cache->tl_y )
er_panic("get_mtx_data");
	if ( cache->br_x > m->pixel_size[0] )
		cache->br_x = m->pixel_size[0];
	if ( cache->br_y > m->pixel_size[1] )
		cache->br_y = m->pixel_size[1];
	cache->level = dim_code[0];
	cache->ds_ptr = cache->ds.offset;
	cache->rect_logsize = m->block_size[0];
	*lev_p = dim_code[0];

end:
	xx = (x - cache->tl_x)>>(cache->level*m->dim_divide[0]);
	yy = (y - cache->tl_y)>>(cache->level*m->dim_divide[1]);
	vst = vm->work;
	return convert_color_float(vst->h.scale,cache->ds_ptr[xx + yy * cache->ds.ix[0]]);
}


void
vmt_draw_scale_double(
	VIEW_METHOD * vm,
	GBVIEW_FLAME * gf,
	RESOURCE* r,
	DRAW_WORK * dw)
{
int i;
GB_POINT p;
REAL1 res;
INTEGER64 xx,yy;
REAL1 dpm1,dpm2;
INTEGER64 level;
REAL1 d;
unsigned long cc;
INTEGER64 lev;
INTEGER64 lev_cnt;
MTX_CACHE_DOUBLE cache;
INTEGER64 tim,tim2;

MATRIX * mtx;
REAL1 divide;
INTEGER64 int_divide;
VM_SCALE_TYPE * vst;


	if ( r->pr64.exit_lock )
		return;
	if ( get_mov_flag(gf,dw->wfid) != 0 )
		return;
	if ( r->h.mtx == 0 ) {
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
	}
	mtx = r->h.mtx;
	if ( mtx->flags & MXF_ERR_DETECTED ) {
		r->h.mtx = 0;
		close_matrix(mtx);
		gv_new_luster_option(r,0);
		if ( r->h.mtx == 0 )
			return;
		mtx = r->h.mtx;
	}

	divide = 1;
	for ( i = 0 ; i < mtx->dim_divide[0] ; i ++ )
		divide *= 2;
	int_divide = mtx->dim_divide[0];
	vst = vm->work;

	dw->limit_reso = r->pr64.dpm;
	dw->limit_ptr = dw->pt_list[dw->size/2];
	r->h.loop_no ++;
	dw->flags |= WFF_LUSTER; 
	dpm1 = r->pr64.dpm;
	dpm2 = dpm1;
	level = 0;

	res = -1;
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( res < dw->pt_reso[i] )
			res = dw->pt_reso[i];
	}

/*
	if ( res > dpm1*LIMIT_RESOLUTION_RATE )
		return;
*/

	if ( dpm2 < res || res < dpm2/divide ) {
		if ( res > dpm1 ) {
			dpm2 = dpm1;
		level = 0;
		}
		else {
			level = 0;
			d = -1;
			for ( dpm2 = dpm1 ;
				res <= dpm2 &&
				level < r->pr64.max_level;
				d = dpm2,
				dpm2 /= divide,
				level ++ );
			dpm2 = d;
			level --;
		}
	}

	lock_task(luster_lock);
//	lock_task(tc_lock);
	lev_cnt = 0;
	memset(&cache,0,sizeof(cache));
	lev = level;
	tim = get_xltime();
	for ( i = 0 ; i < dw->size ; i ++ ) {
		if ( (i % 1000) == 0 && wf_active_check_2(gf,dw->wfid) < 0 )
{ss_printf("FREE=RETURN-PIXEL\n");
			break;
}
		if ( r->pr64.exit_lock )
			break;
		tim2 = get_xltime();
		if ( tim != tim2 ) {
			if ( get_mov_flag(gf,dw->wfid) != 0 )
				break;
			tim = tim2;
		}
		p = dw->pt_list[i];
		if ( dw->pt_reso[i] < 0 )
			continue;
		xx = rint(p.x*r->pr64.dpm);
		yy = rint(p.y*r->pr64.dpm);
		if ( xx < 0 || yy < 0 )
			continue;
		if ( xx >= mtx->pixel_size[0] )
			continue;
		if ( yy >= mtx->pixel_size[1] )
			continue;
		if ( lev_cnt && cache.ds_ptr ) {
		int lx,ly;
			if (		cache.tl_x <= xx && xx < cache.br_x &&
					cache.tl_y <= yy && yy < cache.br_y ) {
				if ( cache.level == lev ) {

					lx = (xx>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(lev*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = convert_rgba_to_cc(
						convert_color_float(vst->h.scale,cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]));
				}
				else if ( cache.leaf ) {
					lx = (xx>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[0])-1);
					ly = (yy>>(cache.level*int_divide))&((((INTEGER64)1)<<mtx->block_size[1])-1);
					if ( lx >= cache.ds.ix[0] ) {
						cc = C_TRANSPARENT;
					}
					else if ( ly >= cache.ds.ix[1] ) {
						cc = C_TRANSPARENT;
					}
					else cc = convert_rgba_to_cc(
						convert_color_float(vst->h.scale,cache.ds_ptr[lx + 
						ly*cache.ds.ix[0]]));
				}
				else	goto non_cache;
			}
			else goto non_cache;
			lev_cnt ++;
			if ( lev_cnt >= 100 )
				lev_cnt = 0;
		}
		else {
		non_cache:
//			cc = _tc_get_pixel(&r->pr64.tc,level,xx,yy,&lev);
			cc = get_mtx_scale_double(vm,&cache,r,level,xx,yy,&lev);
//ss_printf("CC %lli %lli %lli %x\n",xx,yy,level,cc);
			if ( level != lev && cache.leaf == 0 )
				dw->data_request ++;
//			cache = r->pr64.tc;
			lev_cnt ++;
		}
		dw->pixels[i] = cc;
		dw->flags |= WFF_DRAW;
	}
//	unlock_task(tc_lock,"draw_pixel");

	if ( cache.n ) {
		matrix_node_channel_unlock(cache.n,0);
		unlock_node(cache.n,0);
	}
	
	unlock_task(luster_lock,"draw_pixel");

}







