#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2016 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
#   $Id: SubOpt.pm,v 1.3 2002/07/30 17:40:56 gaou Exp $
#
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@sfc.keio.ac.jp> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#

package SubOpt;

use Carp;
require G::IO;

use strict;
use base qw(Exporter);

our @EXPORT = qw(
		 opt_inputType
		 opt_default
		 opt_val
		 opt_get
		 opt_as_gb
);

our $VERSION = '0.62';

#::::::::::::::::::::::::::::::
#          Variables
#::::::::::::::::::::::::::::::

my %data;
my $opt_list = 0;

#::::::::::::::::::::::::::::::
#           Methods
#::::::::::::::::::::::::::::::


sub set_opt_list{
    $opt_list = shift;
}

sub opt_inputType{

}

sub opt_as_gb {
    return if ($opt_list);

    if ($#_ >= 1){
	croak("Error 1 in SubOpt::opt_as_gb():\n  Given value is not a valid sequence.\n");
	return '';
    }elsif (length $_[0] < 1){
	return lastInstance G::IO();
    }

    my $unknown = shift @_;
    my $ref = \$unknown;
    my $ref2 = ref $unknown;

    if ("$unknown" =~ /\=/){
	if (defined $unknown->{SEQ}){
	    return $unknown;
	}else{
	    croak("Error 3 in SubOpt::opt_as_gb():\n  Given value is not a valid sequence.\n");
	    return '';
	}
    } elsif (ref $unknown eq 'SCALAR') {
	my $new_gb = new G::IO('blessed');
	$new_gb->{SEQ} = lc($$unknown);
	return $new_gb;
    } elsif (ref $unknown eq 'REF') {
	my $new_gb = new G::IO('blessed');
	$new_gb->{SEQ} = lc($$unknown);
	return $new_gb;
    } elsif (ref $ref eq 'SCALAR'){
	if(-e $unknown){
	    return new G::IO($unknown, 'no msg');
	}elsif(length($unknown) < 25 && $unknown =~ /[^a-zA-Z\/]/){
	    return new G::IO($unknown, 'no msg');
	}else{
	    my $new_gb = new G::IO('blessed');
	    $new_gb->{SEQ} = lc($unknown);
	    return $new_gb;
	}
    }else{
	croak("Error 4 in SubOpt::opt_as_gb():\n  Given value is not a valid sequence.\n");
	return '';
    }
}


sub opt_default{
    %data = (@_);

    croak if ($opt_list);
}

sub opt_val{
    my $key = shift;

    if(length($key)){
	return $data{$key};
    }else{
	return %data;
    }
}

sub opt_get{
    my @args = @_;
    my @new_args = ();
    my $i = 0;
    my @bparray = ();

    while(defined $args[$i]){
	if (substr($args[$i], 0, 1) eq '-' && substr($args[$i], 1, 1) !~ /[0-9]/){
	    if(!defined($args[$i + 1]) || substr($args[$i + 1], 0, 1) eq '-' && substr($args[$i + 1], 1, 1) !~ /[0-9]/){
		$data{substr($args[$i], 1)} = 1;
		$i ++;
	    }else{
		$data{substr($args[$i],1)} = $args[$i + 1];
		$i += 2;
	    }
        }else{
	    my $ref = ref $args[$i];
	    push(@new_args, $args[$i]);

	    $i ++;
	}
    }

    mkdir ("data",  0777) if($data{'output'} eq 'f');;
    mkdir ("graph", 0777) if($data{'output'} =~ /g/ || $data{'output'} eq 'show');

    return @new_args;
}


1;
__END__


=head1 NAME

SubOpt - G-language Subroutine Input API

=head1 SYNOPSIS

  use SubOpt;

  &odyssey ($gb, -option=>"neat!", -file=>"fancy.txt", "oops");

  sub odyssey{
      opt_default(option => 'none', file => 'hoge.txt');
      # Set default values. This is optional.

      my @args = opt_get(@_);
      # Parse options.

      my $gb = opt_as_gb(shift @args);
      # Get the input value as a G instance
 
      my $last = shift @args;

      print $gb->{SEQ} , "\n";
      print "last: $last\n";
      print "option: ", opt_val("option"), "\n";
      print "file: ", opt_val("file"), "\n";
      # Option values are accessed via opt_val().
  }



=head1 DESCRIPTION

  SubOpt parses the arguments given to a subroutine in a similar 
  manner as GetOpt module.

  Options are specified in the form: -option=>"hoge"
  The above will input the value "hoge" with a key "option".
  i.e. option with '-' is stored with the value pointed with '=>'.

  supported methods:

  opt_default(<option>=><value>) 
    This mehod sets default values for the options. 
    i.e. if the option is not specified, the value set with this method 
    is used. This method is optional.

    The option name here should be enclosed in quotations, and do not 
    have to have a hyphen in the beginning.

    eg. opt_default("option"=>"hoge");
        #inputs default value of "hoge" for key "option"
        #if the subroutine is called with -option, like
        #&subroutine($gb, -option=>"boo");
        #the option value will be overridden by "boo"

  opt_get(@_) 
    This method parses the option arguments, stores the options in its
    class, and returns the array of arguments that are not options.

    eg. my @args = opt_get(@_);
        #receives all arguments in @args, and all options are stored
        #in SubOpt name space; therefore options are not passed to @args

  opt_val(<option>) 
    This method returns the value of the given option. The option name 
    here should be enclosed in quotations.

    eg. my $key = opt_val("key");
        #gets the value for the option "key".
        #It is recommended to store the option in a local variable
        #instead of calling opt_val() everytime the option is used.
        #This is for the prevention of the confusion of SubOpt name space
        #in case SubOpt is called in cascade.

    When this function is called without any arguments, it returns the
    entire option-value hash.

    eg. my %options = opt_val();

  opt_as_gb(<$gb or $seq>) 
    This method automatically distinguish the variable from
    a scalar sequence, reference of a sequence, and a G instance,
    and returns the value as a G instance. If the input variable is a scalar 
    of reference of sequence, the G instance only contains the $gb->{SEQ} object.

    If only the sequence portion of the G instance is required in a subroutine,
    &subroutine($gb);  and
    &subroutine($gb->{SEQ});  and
    &subroutine(\$gb->{SEQ}); and
    &subroutine("atgc"); and
    &subroutine(\$atgc); #$atgc = "atgc";
    should all be used by the subroutine. In this case, calling opt_as_gb() as
      my @args = opt_get(@_); 
      my $gb = opt_as_gb(shift @args); 
    will store the first argument in G instance form regardless of the type of 
    input argument. Therefore, the sequence information is accessible as
    $gb->{SEQ} in the subroutine.                                       
    

=head1 AUTHOR

  Kazuharu Arakawa, gaou@sfc.keio.ac.jp

=head1 SEE ALSO

perl(1).

=cut
