#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2009 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
#   $Id: G::IO::Handler.pm,v 1.1 2002/07/30 17:44:27 gaou Exp $
#
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@sfc.keio.ac.jp> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#

package G::IO::Handler;

use strict;
use Carp;

use G::Messenger;
use G::Seq::Primitive;
use G::DB::SDB;

use autouse 'File::Temp'=>qw(tempfile);
use autouse 'Bio::Perl'=>qw(read_sequence write_sequence get_sequence);

#::::::::::::::::::::::::::::::
#    Variables
#::::::::::::::::::::::::::::::

my %feat2idHash = ();
my $intergenic = 0;

#::::::::::::::::::::::::::::::
#          Perldoc
#::::::::::::::::::::::::::::::


=head1 NAME

 G::IO::Handler - Internal class with basic sequence manipulation methods

=head1 SYNOPSIS

 require G::IO::Handler;
 use base qw(G::IO::Handler);
   
=head1 DESCRIPTION

 Intended for internal use only. Super class for the core. Provides 
 the native methods.

=head1 AUTHOR

Kazuharu Arakawa, gaou@sfc.keio.ac.jp

=cut


#::::::::::::::::::::::::::::::
#    Let the code begin...
#::::::::::::::::::::::::::::::

#::::::::: Internal :::::::::::

sub set_gene_aliases{
    my $this = shift;

    foreach my $feat ($this->feature()){
	next unless ($this->{$feat}->{type} =~ /CDS|RNA/);
	
	if(length $this->{$feat}->{gene}){
	    $this->{$this->{$feat}->{gene}} = $this->{$feat};
	}
	
	if(length $this->{$feat}->{locus_tag}){
	    $this->{$this->{$feat}->{locus_tag}} = $this->{$feat};
	}
	
	if($this->{$feat}->{type} eq 'CDS'){
	    $this->{'CDS' . $this->{$feat}->{cds}} = $this->{$feat};
	}
    }
}

#::::::::: Class Method ::::::::::

sub clone {
    my $this = shift;
    my $sdbPath = _sdb_path();

    _set_sdb_path('/tmp');

    my $tmpfile = "GINTERNAL-" . time() . rand();
    sdb_save($this,$tmpfile);
    my $new = sdb_load($tmpfile);

    _set_sdb_path($sdbPath);

    return $new;
}

sub disclose {
    my $gb = shift;

    my $return = '';
    foreach my $level1 (keys %{$gb}){
	next if ($level1 eq 'SEQ');
	next unless($level1 =~ /LOCUS|FEATURE|HEADER/);
	if(ref($gb->{$level1}) =~ /HASH/){
	    foreach my $level2 (keys %{$gb->{$level1}}){
		$return .= join("\t", $level1, $level2, $gb->{$level1}->{$level2}) . "\n";
	    }
	}else{
	    $return .= join("\t", $level1, $gb->{$level1}) . "\n";
	}
    }

    return $return;
}


sub find {
    my $this = shift;

    my @args = @_;
    my (@keywords, %keyhash, @results);
    my $i = 0;
    my $print = 0;

    while(defined $args[$i]){
        if (substr($args[$i], 0, 1) eq '-' && substr($args[$i], 1, 1) !~ /[0-9]/){
	    if (substr($args[$i],1) eq 'print'){
		$print = 1;
		$i +=2;
	    }else{
		$keyhash{substr($args[$i],1)} = $args[$i + 1];
		$i += 2;
	    }
        }else{
	    push(@keywords, $args[$i]);
            $i ++;
        }
    }

    foreach my $feat ($this->feature()){
	my $flag = 0;

	foreach my $key (keys %keyhash){
	    my $val = $keyhash{$key};

	    unless($this->{$feat}->{$key} =~ /$val/i){
		$flag = 1;
		last;
	    }
	}

	next if ($flag);

	foreach my $key (@keywords){
	    unless(join('%%%___%%%', values(%{$this->{$feat}})) =~ /$key/i){
		$flag = 1;
		last;
	    }
	}
    
	push(@results, $feat) unless($flag);
    }

    if(msg_ask_interface() eq 'Shell' || $print){
	foreach my $feat (@results){
	    my $gene = $this->{$feat}->{gene} || $this->{$feat}->{locus_tag} || $feat;
	    my $ec = $this->{$feat}->{EC_number};
	    $ec =~ s/\s+/,/g;
	    $ec = '(' . $ec . ')' if (length $ec);

	    msg_send(
		     sprintf("     %s\t%s\t%s\t%s..%s\t%s\t%s %s\n", $feat, $gene, $this->{$feat}->{type}, 
			     $this->{$feat}->{start}, $this->{$feat}->{end}, $this->{$feat}->{direction}, $this->{$feat}->{product}, $ec)
		     );
	}
    }

    return @results;
}



sub getseq {
    if (scalar(@_) < 3){
	carp("Not enough arguments.");
	return;
    }
    my $this = shift;
    my $start = shift;
    my $end = shift;
    my $option = shift;

    if($start < $end){
	return substr($this->{SEQ}, $start, $end-$start+1);
    }else{
	if($option =~ /circ/){
	    return substr($this->{SEQ}, $start) . 
		substr($this->{SEQ}, 0, $end + 1);
	}else{
	    my ($start2, $end2) = sort {$a <=> $b} ($start, $end);
	    return substr($this->{SEQ}, $start, $end-$start+1);
	}
    }
}

sub get_gbkseq {
    if (scalar(@_) < 3){
	carp("Not enough arguments.");
	return;
    }
    return getseq($_[0], $_[1] - 1, $_[2] - 1, $_[3]);
}

sub get_cdsseq {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $this = shift;
    my $object = shift;

    my $cdsseq = '';

    if($this->{$object}->{start} > $this->{$object}->{end}){
	$cdsseq = substr($this->{SEQ}, $this->{$object}->{start} - 1) . 
	    $this->get_gbkseq(1, $this->{$object}->{end});
    }else{
	$cdsseq = $this->get_gbkseq($this->{$object}->{start}, 
				    $this->{$object}->{end});
    }

    $cdsseq = &complement($cdsseq) 
	if ($this->{$object}->{direction} eq 'complement');

    return $cdsseq;
}

sub around_startcodon {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $gb = shift;
    my $cds = shift;
    my $before = shift || 0;
    my $after = shift || 0;
    my $option = shift;

    my $seq  = $gb->before_startcodon($cds, $before);
    $seq .= $gb->startcodon($cds) unless($option =~ /without/);
    $seq .= $gb->after_startcodon($cds, $after);

    return $seq;
}

sub around_stopcodon {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $gb = shift;
    my $cds = shift;
    my $before = shift || 0;
    my $after = shift || 0;
    my $option = shift;

    my $seq  = $gb->before_stopcodon($cds, $before);
    $seq .= $gb->stopcodon($cds) unless($option =~ /without/);
    $seq .= $gb->after_stopcodon($cds, $after);

    return $seq;
}

sub before_startcodon {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $this = shift;
    my $object = shift;
    my $length = shift || 100;

    if ($this->{$object}->{direction} eq 'complement'){
	return complement(substr($this->{SEQ}, $this->{$object}->{end}, $length));
    }else{
	my $start = $this->{$object}->{start} - 1 - $length;
	if ($start < 0){
	    $start = 0;
	    $length = $this->{$object}->{start} - 1;
	}
	return substr($this->{SEQ}, $start, $length);
    }
}

sub after_startcodon {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $this = shift;
    my $object = shift;
    my $length = shift || 100;

    if ($this->{$object}->{direction} eq 'complement'){
	my $start = $this->{$object}->{end} - 1 - 3 - $length + 1;
	if($start < 0){
	    $start = 0;
	    $length = $this->{$object}->{end} - 3;
	}
	return complement(substr($this->{SEQ}, $start, $length));
    }else{
	return substr($this->{SEQ}, $this->{$object}->{start} + 3 - 1, $length);
    }
}

sub before_stopcodon {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $this = shift;
    my $object = shift;
    my $length = shift || 100;

    if ($this->{$object}->{direction} eq 'complement'){
	return complement(substr($this->{SEQ}, $this->{$object}->{start} + 3 - 1, $length));
    }else{
	my $start = $this->{$object}->{end} - 3 - 1 - $length + 1;
	if($start < 0){
	    $start = 0;
	    $length = $this->{$object}->{end} - 3;
	}
	return substr($this->{SEQ}, $start, $length);
    }
}

sub after_stopcodon {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $this = shift;
    my $object = shift;
    my $length = shift || 100;

    if ($this->{$object}->{direction} eq 'complement'){
	my $start = $this->{$object}->{start} - 1 - $length;
	if($start < 0){
	    $start = 0;
	    $length = $this->{$object}->{start};
	}
	return complement(substr($this->{SEQ}, $start, $length));
    }else{
	return substr($this->{SEQ}, $this->{$object}->{end} +1 - 1, $length);
    }
}

sub startcodon {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    return substr($_[0]->get_geneseq($_[1]), 0, 3);
}

sub stopcodon {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    return substr($_[0]->get_geneseq($_[1]), -3, 3);
}

sub pos2feature {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $this = shift;
    my $pos = shift;
    my $pos2 = shift;

    if(length($pos2) > 1){
	my @genes = ();

	foreach my $feat ($this->feature()){
	    next if ($feat eq 'FEATURE0');

	    if ($this->{$feat}->{end} >= $pos && $pos2 >= $this->{$feat}->{start}){
		push(@genes, $feat);
	    }elsif ($pos2 < $this->{$feat}->{start}){
		return @genes;
	    }
	}

    }else{

	foreach my $feat ($this->feature()){
	    next if ($feat eq 'FEATURE0');

	    if ($pos >= $this->{$feat}->{start} && $pos <= $this->{$feat}->{end}){
		return $feat;
	    }elsif ($pos < $this->{$feat}->{start}){
		return '';
	    }
	}
    }
}

sub gene2id {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    return length($_[0]->{$_[1]}->{feature}) ? 'FEATURE' . $_[0]->{$_[1]}->{feature} : '';
}

sub feature2id {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $this = shift;
    my $gene = shift;

    unless(scalar(%feat2idHash)){
	foreach my $feat ($this->feature()){
	    $feat2idHash{$this->{$feat}->{gene}} = $feat;
	    $feat2idHash{$this->{$feat}->{locus_tag}} = $feat;
	}
    }
	   
    return $feat2idHash{$gene};
}

sub pos2gene {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $this = shift;
    my $pos = shift;
    my $pos2 = shift;

    if(length($pos2) > 1){
	my @genes = ();

	foreach my $feat ($this->cds()){
	    if ($this->{$feat}->{end} >= $pos && $pos2 >= $this->{$feat}->{start}){
		push(@genes, $feat);
	    }elsif ($pos2 < $this->{$feat}->{start}){
		return @genes;
	    }
	}

    }else{
	foreach my $feat ($this->cds()){
	    if ($pos >= $this->{$feat}->{start} && $pos <= $this->{$feat}->{end}){
		return $feat;
	    }elsif ($pos < $this->{$feat}->{start}){
		return '';
	    }
	}
    }
}

sub get_geneseq {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $this = shift;
    my $object = shift;

    my $geneseq = $this->get_gbkseq($this->{$object}->{start}, 
				   $this->{$object}->{end});
    if ($this->{$object}->{join}){
	$geneseq = $this->get_exon($object);
    }elsif ($this->{$object}->{direction} eq 'complement'){
	$geneseq = &complement($geneseq);
    }

    return $geneseq;
}

sub get_intron {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $this = shift;
    my $cds = shift;

    return unless (length $this->{$cds}->{join});

    my @join = split(/\.\./, $this->{$cds}->{join});
    shift @join;
    pop @join;
    my @seq;

    foreach my $line (@join){
	$line =~ s/c//g;
	my ($start, $end) = split(/,/, $line, 2);
	my $tmp = $this->get_gbkseq($start + 1, $end - 1);
	$tmp = '' if($end - 2  - $start < 0);
	push (@seq, $tmp);
    }

    return @seq;
}

sub get_exon {
    if (scalar(@_) < 2){
	carp("Not enough arguments.");
	return;
    }
    my $this = shift;
    my $cds = shift;

    return unless (length $this->{$cds}->{join});

    my $seq = '';

    foreach my $line (split(/,/, $this->{$cds}->{join})){
	my $complement = $line =~ tr/c//d;
	my ($start, $end) = split(/\.\./, $line, 2);
	my $tmp = $this->get_gbkseq($start, $end);
	$tmp = complement($tmp) if ($complement);
	$seq .= $tmp;
    }

    $seq = complement($seq) if ($this->{$cds}->{direction} eq 'complement');
    return $seq;
}

sub seq_info {
    my $this = shift;

    my $length = length($this->{SEQ});

    my $a = $this->{SEQ} =~ tr/a/a/;
    my $t = $this->{SEQ} =~ tr/t/t/;
    my $g = $this->{SEQ} =~ tr/g/g/;
    my $c = $this->{SEQ} =~ tr/c/c/;
    my $others = $length - $a - $t - $g - $c;
    my $msg;

    msg_send(sprintf "\n\nAccession Number: %s\n", $this->{LOCUS}->{id}) if(length($this->{LOCUS}->{id}));
    $msg .= sprintf "\n  Length of Sequence : %9d\n" , $length;

    if($length == 0){
	$msg .= sprintf "\n  No Sequence Found.\n\n";
    }elsif($others > $a + $t + $g + $c){
        $msg .= sprintf "\n  This is Amino Acid Sequence. Try amino_info().\n\n";
    }else{
        $msg .= sprintf "           A Content : %9d (%.2f\%)\n" , $a , $a / $length * 100;
        $msg .= sprintf "           T Content : %9d (%.2f\%)\n" , $t , $t / $length * 100;
        $msg .= sprintf "           G Content : %9d (%.2f\%)\n" , $g , $g / $length * 100;
        $msg .= sprintf "           C Content : %9d (%.2f\%)\n" , $c , $c / $length * 100;
        $msg .= sprintf "              Others : %9d (%.2f\%)\n" , $others,  $others / $length * 100;
        $msg .= sprintf "          AT Content :    %.2f\%\n"    , ($a + $t) / $length * 100;
        $msg .= sprintf "          GC Content :    %.2f\%\n\n"  , ($g + $c) / $length * 100;
    }
    &msg_send($msg);

    return ($a, $t, $g, $c);
}

sub disable_pseudogenes {
    my $this = shift;

    foreach my $feature ($this->feature()){
	$this->{$feature}->{on} = 0 if ($this->{$feature}->{pseudo});
    }
}

sub feature {
    my $this = shift;
    my $type  = shift;
    my $opt = shift;

    if($type eq 'all'){
	$opt = 'all';
	$type = '';
    }

    my $i = -1;
    my @feature;

    while(defined($this->{"FEATURE" . ($i + 1)})){
        $i ++;

	if(length($type)){
	    next unless ($this->{"FEATURE$i"}->{type} eq $type);
	}

	if ($opt ne 'all' && defined $this->{"FEATURE$i"}->{on}){
	    next if ($this->{"FEATURE$i"}->{on} == 0);
	}

        push (@feature, "FEATURE$i");
    }

    return @feature;
}

sub rRNA {
    my $this = shift;
    my $option = shift;

    if($option =~ /\d+S|SSU|LSU/){
	my %rRNA;

	foreach ($this->feature('rRNA')){
	    if($this->{$_}->{product} =~ /([0-9\.]+S)/i){
		push(@{$rRNA{uc($1)}}, $_);
	    }elsif($this->{$_}->{gene} =~ /rrf/){
		push(@{$rRNA{'5S'}}, $_);
	    }elsif($this->{$_}->{gene} =~ /rrl/){
		push(@{$rRNA{'23S'}}, $_);
	    }elsif($this->{$_}->{gene} =~ /rrs/){
		push(@{$rRNA{'16S'}}, $_);
	    }elsif($this->{$_}->{gene} =~ /([0-9\.]+S)/i){
		push(@{$rRNA{uc($1)}}, $_);
	    }elsif($this->{$_}->{locus_tag} =~ /([0-9]+S)/i){
		push(@{$rRNA{uc($1)}}, $_);
	    }elsif($this->{$_}->{note} =~ /([0-9\.]+S)+?/){
		push(@{$rRNA{uc($1)}}, $_);
	    }elsif($this->{$_}->{product} =~ /(SSU|small subunit) ribosomal RNA/i){
		push(@{$rRNA{'SSU'}}, $_);
	    }elsif($this->{$_}->{product} =~ /(LSU|large subunit) ribosomal RNA/i){
		push(@{$rRNA{'LSU'}}, $_);
	    }
	}

	if(defined $rRNA{$option}){
	    my @result = sort {length($this->get_geneseq($b))<=>length($this->get_geneseq($a))} @{$rRNA{$option}};
	    return wantarray ? @result : $result[0];
	}
    }else{
	return feature($this, 'rRNA', $option);
    }
}

sub tRNA {
    return feature($_[0], 'tRNA', $_[1]);
}

sub gene {
    return feature($_[0], 'gene', $_[1]);
}

sub cds {
    return feature($_[0], 'CDS', $_[1]);
}


sub next_feature{
    my $this = shift;
    my $feature = shift || 'FEATURE0';
    my $opt = shift;

    $feature = $this->{$feature}->{left} if ($feature =~ /^INTER/);

    my $i = $this->{$feature}->{feature};
    $i ++;

    while(defined(%{$this->{"FEATURE$i"}})){
	my $feat = "FEATURE$i";
	$i ++;

	if(length($opt)){
	    next unless($this->{$feat}->{type} eq $opt);
	}

	return $feat;
    }
}


sub previous_feature{
    my $this = shift;
    my $feature = shift || 'FEATURE0';
    my $opt = shift;

    $feature = $this->{$feature}->{right} if ($feature =~ /^INTER/);

    my $i = $this->{$feature}->{feature} if ($feature =~ /^CDS/);
    $i --;

    while(defined(%{$this->{"FEATURE$i"}})){
	my $feat = "FEATURE$i";
	$i --;

	if(length($opt)){
	    next unless($this->{$feat}->{type} eq $opt);
	}

	return $feat;
    }
}

sub next_cds{
    return next_feature($_[0], $_[1], 'CDS');
}

sub previous_cds{
    return previous_feature($_[0], $_[1], 'CDS');
}

sub intergenic {
    my $this = shift;
    my $opt = shift || '';
    my $i = 0;
    my @cds;

    set_intergenic($this) unless($intergenic);

    while(defined(%{$this->{"INTER" . ($i + 1)}})){
	$i ++;

	if ($opt ne 'all'){
	    next if ($this->{"INTER$i"}->{on} == 0);
	}

	push (@cds, "INTER$i");
    }

    return @cds;
}
	  

sub set_intergenic{
    return if($intergenic);

    my $gb = shift;
    my $num = 1;
    my $i = 0;
    my $interface = msg_ask_interface();
    msg_interface("NULL");
    my @cds = $gb->find(-type=>"CDS|RNA");

    foreach my $feature (@cds){
	if($i == 0 && $gb->{$feature}->{start} < $gb->{$feature}->{end}){
	    $gb->{"INTER$num"}->{start} = 1;
	    $gb->{"INTER$num"}->{end} = $gb->{$feature}->{start} - 1;
	    $gb->{"INTER$num"}->{direction} = "direct";
	    $gb->{"INTER$num"}->{left} = undef;
	    $gb->{"INTER$num"}->{right} = $feature;
	    $gb->{"INTER$num"}->{on} = 1;
	    $num++;
	}elsif($i == $#cds && $gb->{$cds[-1]}->{end} + 1 < $gb->{SEQ}){
	    $gb->{"INTER$num"}->{start} = $gb->{$cds[-1]}->{end} + 1;
	    $gb->{"INTER$num"}->{end} = length $gb->{SEQ};
	    $gb->{"INTER$num"}->{direction} = "direct";
	    $gb->{"INTER$num"}->{left} = $feature;
	    $gb->{"INTER$num"}->{right} = undef;
	    $gb->{"INTER$num"}->{on} = 1;
	    $num++;
	}elsif($i > 0){
	    if($gb->{$cds[$i - 1]}->{end} < $gb->{$feature}->{start}){
		$gb->{"INTER$num"}->{start} = $gb->{$cds[$i - 1]}->{end} + 1;
		$gb->{"INTER$num"}->{end} = $gb->{$feature}->{start} - 1;
		$gb->{"INTER$num"}->{direction} = "direct";
		$gb->{"INTER$num"}->{left} = $cds[$i - 1];
		$gb->{"INTER$num"}->{right} = $feature;
		$gb->{"INTER$num"}->{on} = 1;
		$num++;
	    }
	}
	$i++;
    }

    msg_interface($interface);
    $intergenic = 1;
}


sub seq{
    return $_[0]->{SEQ};
}

sub hairpin_cut {
    system('firefox http://www.toychan.net/afro/');
    return "\n==============\n!!!Afro Man!!!===============\n\n";
}

sub del_key {
    $_[0]->{$_[1]}->{on} = 0;
    return 1;
}


sub relocate_origin{
    require G::IO;
    my $this = new G::IO("blessed");
    my $tmp = shift;
    my $gb = $tmp->clone();
    my $pos = shift;
    croak("New start position \(in Perl coordinate\) must be given.\n") unless($pos =~ /\d/);

    $this->{LOCUS} = $gb->{LOCUS};
    $this->{HEADER} = $gb->{HEADER};
    $this->{COMMENT} = $gb->{COMMENT};
    $this->{SEQ} = substr($gb->{SEQ}, $pos) . substr($gb->{SEQ}, 0, $pos);
    $this->{FEATURE0} = $gb->{FEATURE0};

    my (@before, @after);
    my @features = $gb->feature();
    shift @features;
    foreach my $feature (@features){
	if($gb->{$feature}->{start} >= $pos + 1){
	    push(@after, $feature);
	}else{
	    push(@before, $feature);
	}
    }

    my $f = 0;
    my $c = 0;
    foreach my $feature (@after, @before){
	$f ++;
	$this->{"FEATURE$f"} = $gb->{$feature};
	$this->{"FEATURE$f"}->{feature} = $f;
	if($gb->{$feature}->{type} eq 'CDS'){
	    $c ++;
	    $this->{"FEATURE$f"}->{cds} = $c;
	}

	if($gb->{$feature}->{end} >= $pos + 1 && $gb->{$feature}->{start} < $pos + 1){
	    $this->{"FEATURE$f"}->{start} += length($gb->{SEQ}) - $pos;
	    $this->{"FEATURE$f"}->{end} -= $pos;

	    if(length $this->{"FEATURE$f"}->{join}){
		msg_error("Warning: overriding join definition for FEATURE$f.\n");
		msg_error("  this is likely to destroy positional features of this gene entry.\n");
	    }

	    $this->{"FEATURE$f"}->{join} = sprintf("%d\.\.%d,%d\.\.%d", $this->{"FEATURE$f"}->{start},
						   length($gb->{SEQ}), 1, $this->{"FEATURE$f"}->{end});
	    next;
	}

	my $lng = length($gb->{SEQ}) - $pos;
	if($gb->{$feature}->{start} >= $pos + 1){
	    $lng = -$pos;
	}

	$this->{"FEATURE$f"}->{start} += $lng;
	$this->{"FEATURE$f"}->{end} += $lng;

	if(defined $this->{"FEATURE$f"}->{"join"}){
	    my @join = split(/\,/,$this->{"FEATURE$f"}->{"join"});
	    my @num = ();
	    my @new_join = ();

	    foreach(@join){
		if(tr/c/c/){
		    @num = split(/\.\./,$_);
		    push (@new_join, sprintf ("c%d\.\.%d", $num[0] + $lng, $num[1] + $lng));
		} else {
		    @num = split(/\.\./,$_);
		    push (@new_join, sprintf ("%d\.\.%d",  $num[0] + $lng, $num[1] + $lng));
		}
	    }
	    $this->{"FEATURE$f"}->{join} = join(',', @new_join);
	}
    }
    
    $this->set_gene_aliases();

    return $this;
}


sub reverse_strand{
    require G::IO;
    my $this = new G::IO("blessed");
    my $tmpG = shift;
    my $gb = $tmpG->clone();

    $this->{LOCUS} = $gb->{LOCUS};
    $this->{HEADER} = $gb->{HEADER};
    $this->{COMMENT} = $gb->{COMMENT};
    $this->{SEQ} = complement($gb->{SEQ});
    $this->{FEATURE0} = $gb->{FEATURE0};

    my @feat = $gb->feature();
    shift @feat;
    my (@features, @tmp);
    foreach my $feature (@feat){
	if($gb->{$feature}->{type} eq 'gene'){
	    unshift(@features, @tmp);
	    @tmp = ($feature);
	}else{
	    push(@tmp, $feature);
	}
    }
    unshift(@features, @tmp);

    my $f = 0;
    my $c = 0;
    my $lng = length($gb->{SEQ}) + 1;
    foreach my $feature (@features){
	$f ++;
	$this->{"FEATURE$f"} = $gb->{$feature};
	$this->{"FEATURE$f"}->{feature} = $f;

	if($gb->{$feature}->{type} eq 'CDS'){
	    $c ++;
	    $this->{"FEATURE$f"}->{cds} = $c;
	}

	my ($start, $end) = ($lng - $gb->{$feature}->{end}, $lng - $gb->{$feature}->{start});
	$this->{"FEATURE$f"}->{start} = $start;
	$this->{"FEATURE$f"}->{end}   = $end;
	$this->{"FEATURE$f"}->{direction} = $gb->{$feature}->{direction} eq 'direct' ? 'complement' : 'direct';

	if(defined $this->{"FEATURE$f"}->{"join"}){
	    my @join = split(/\,/,$this->{"FEATURE$f"}->{"join"});
	    my @num = ();
	    my @new_join = ();

	    foreach(@join){
		if(tr/c/c/){
		    @num = split(/\.\./,$_);
		    push (@new_join, sprintf ("c%d\.\.%d", $lng - $num[1], $lng - $num[0]));
		} else {
		    @num = split(/\.\./,$_);
		    push (@new_join, sprintf ("%d\.\.%d",  $lng - $num[1], $lng - $num[0]));
		}
	    }
	    $this->{"FEATURE$f"}->{join} = join(',', reverse(@new_join));
	}
    }
    $this->set_gene_aliases();

    return $this;
}


sub _interpret_format {
    my $filename = shift;
    my $ref = ref $filename;

    if ($ref =~ /Bio::Seq/){
	return 'bioperl';
    }elsif ($filename =~ /^(.*?):(.*)/){
	unless(lc($1) =~ /(swiss|genbank|genpept|embl|refseq)/){
	    croak("Unsupported database name for USA. Supported databases are\n" .
		"swiss, genbank, genpept, embl, or refseq\n");
	}
	return "usa";
    }elsif (lc($filename) =~ /\.(gb|gbk|gbank|genbank)$/i){
	return 'genbank';
    }elsif (lc($filename) =~ /\.(fasta|fast|seq|fst|fa|fsa|nt|aa|ffn|fna)/i){
	return 'fasta';
    }elsif (lc($filename) =~ /\.(fastq|fq)/i){
	return 'fastq';
    }elsif ($filename =~ /^NC_\d+$/i){
	return 'RefSeq';
    }elsif ($filename =~ /^NP_\d+$/i){
	return 'net GenPept';
    }elsif ($filename =~ /^(?:[A-Z]|[A-Z][A-Z]).\d+$/i){
	return 'net GenBank';
    }else{
	require Bio::SeqIO;
	my $format = Bio::SeqIO->_guess_format($filename);

	if (length $format){
	    return $format;
	}else{
	    open(FILE, $filename);
	    while(<FILE>){
		if(/^>/){
		    return 'fasta';
		}elsif(/^LOCUS/){
		    return 'genbank';
		}elsif(/^ID/){
		    return 'embl';
		}else{
		    carp("Unknown file format. Interpreting $filename as GenBank...\n");
		    return 'genbank';
		}
		last;
	    }
	    close(FILE);
	}
    }
}

sub output {
    my $gb = shift;
    my $file = shift;
    my $option = shift;

    $option = _interpret_format($file) unless(length $option);

    if (lc($option) eq 'genbank'){
	$gb->make_gb($file);
    }elsif(length $option){
	my $in;

	if(length $gb->{BIOPERL}){
	    $in = $gb->{BIOPERL};
	}else{
	    my ($fh, $outfile) = tempfile(undef, SUFFIX=>'.gbk');
	    $gb->make_gb($outfile);
	    $in = read_sequence($outfile);
	    unlink($outfile);
	}

	write_sequence(">$file", $option, $in);
    }else{
	&msg_error("G::output - Unknown format to output.");
    }
}

sub bioperl {
    require Bio::SeqIO;
    my $this = shift;
    my ($fh, $outfile) = tempfile(undef, SUFFIX=>'.gbk');
    $this->output($outfile);
    
    my $in = Bio::SeqIO->new(-format => "genbank", -file =>$outfile);
    my $bp = $in->next_seq();

    return $bp;
}



1;



