/************************************************************
* Copyright 2005 Masahiko SAWAI All Rights Reserved. 
************************************************************/

#include "glu4gcj/GLU.h"
#include "glu4gcj/GLUException.h"
#include "glu4gcj/Nurbs.h"
#include "glu4gcj/NurbsListener.h"

#include <java/lang/Object.h>
#include <java/lang/IllegalArgumentException.h>
#include <gcj/cni.h>
#include <GL/glu.h>
#include <stdio.h>

using namespace glu4gcj;
using java::lang::Object;
using java::lang::IllegalArgumentException;
using gnu::gcj::RawData;

// functions
static
inline
GLUException*
createDeletedNurbsException()
{
	return new GLUException(JvNewStringLatin1("This Nurbs was already deleted."));
}

/// callback functions

void
nurbs_begin_callback(GLenum type, Nurbs* nurbs)
{
	NurbsListener* callback = nurbs->getCallback();
	if (callback != NULL)
	{
		callback->nurbsBegin((jint)type, nurbs->getCallbackData());
	}
}

void
nurbs_end_callback(Nurbs* nurbs)
{
	NurbsListener* callback = nurbs->getCallback();
	if (callback != NULL)
	{
		callback->nurbsEnd(nurbs->getCallbackData());
	}
}

void
nurbs_vertex_callback(GLfloat vertex[3], Nurbs* nurbs)
{
	NurbsListener* callback = nurbs->getCallback();
	if (callback != NULL)
	{
		jfloatArray vertexArray = JvNewFloatArray(3);
		jfloat* vertexArrayElements = elements(vertexArray);
		vertexArrayElements[0] = vertex[0];
		vertexArrayElements[1] = vertex[1];
		vertexArrayElements[2] = vertex[2];
		callback->nurbsVertex(vertexArray, nurbs->getCallbackData());
	}
}

void
nurbs_normal_callback(GLfloat normal[3], Nurbs* nurbs)
{
	NurbsListener* callback = nurbs->getCallback();
	if (callback != NULL)
	{
		jfloatArray normalArray = JvNewFloatArray(3);
		jfloat* normalArrayElements = elements(normalArray);
		normalArrayElements[0] = normal[0];
		normalArrayElements[1] = normal[1];
		normalArrayElements[2] = normal[2];
		callback->nurbsVertex(normalArray, nurbs->getCallbackData());
	}
}

void
nurbs_color_callback(GLfloat color[4], Nurbs* nurbs)
{
	NurbsListener* callback = nurbs->getCallback();
	if (callback != NULL)
	{
		jfloatArray colorArray = JvNewFloatArray(4);
		jfloat* colorArrayElements = elements(colorArray);
		colorArrayElements[0] = color[0];
		colorArrayElements[1] = color[1];
		colorArrayElements[2] = color[2];
		colorArrayElements[3] = color[3];
		callback->nurbsVertex(colorArray, nurbs->getCallbackData());
	}
}

void
nurbs_texture_coord_callback(GLfloat *texCoord, Nurbs* nurbs)
{
	NurbsListener* callback = nurbs->getCallback();
	if (callback != NULL)
	{
		jfloatArray texCoordArray = JvNewFloatArray(4);
		jfloat* texCoordArrayElements = elements(texCoordArray);
		texCoordArrayElements[0] = texCoord[0];
		texCoordArrayElements[1] = texCoord[1];
		texCoordArrayElements[2] = texCoord[2];
		texCoordArrayElements[3] = texCoord[3];
		callback->nurbsVertex(texCoordArray, nurbs->getCallbackData());
	}
}

void
nurbs_error_callback(GLenum errno)
{
	throw new GLUException(GLU::gluErrorString(errno));
}

// instance methods

void
Nurbs::beginCurve ()
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	::gluBeginCurve (nativeNurbs);
}

void
Nurbs::endCurve ()
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	::gluEndCurve (nativeNurbs);
}

void
Nurbs::pwlCurve (jint count, jfloatArray dataArray, jint stride, jint type)
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	if (dataArray == NULL) throw new IllegalArgumentException();
	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	GLfloat* data = (GLfloat*)elements(dataArray);
	::gluPwlCurve(nativeNurbs, count, data, stride, type);
}

void
Nurbs::nurbsCurve (jint knotCount, jfloatArray knotsArray,
	jint stride, jfloatArray controlArray, jint order, jint type)
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	if (knotsArray == NULL || controlArray == NULL) throw new IllegalArgumentException();
	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	GLfloat* knots = (GLfloat*)elements(knotsArray);
	GLfloat* control = (GLfloat*)elements(controlArray);
	::gluNurbsCurve(nativeNurbs, knotCount, knots, stride, control, order, type);
}

void
Nurbs::beginSurface ()
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	::gluBeginSurface (nativeNurbs);
}

void
Nurbs::nurbsSurface(
	jint sKnotCount, jfloatArray sKnotsArray,
	jint tKnotCount, jfloatArray tKnotsArray,
	jint sStride, jint tStride, jfloatArray controlArray,
	jint sOrder, jint tOrder, jint type)
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	if (sKnotsArray == NULL || tKnotsArray == NULL || controlArray == NULL)
		throw new IllegalArgumentException();

	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	GLfloat* sKnots = (GLfloat*)elements(sKnotsArray);
	GLfloat* tKnots = (GLfloat*)elements(tKnotsArray);
	GLfloat* control = (GLfloat*)elements(controlArray);
	::gluNurbsSurface(nativeNurbs,
		sKnotCount, sKnots, tKnotCount, tKnots,
		sStride, tStride, control, sOrder, tOrder, type);
}

void
Nurbs::endSurface ()
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	::gluEndSurface (nativeNurbs);
}

void
Nurbs::beginTrim ()
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	::gluBeginTrim (nativeNurbs);
}

void
Nurbs::endTrim ()
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	::gluEndTrim (nativeNurbs);
}

jfloat
Nurbs::getNurbsProperty (jint property)
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	GLfloat value;
	::gluGetNurbsProperty (nativeNurbs, property, &value);
	return value;
}

void
Nurbs::setNurbsProperty (jint property, jfloat value)
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	::gluNurbsProperty (nativeNurbs, property, value);
}

//	public native void loadSamplingMatrices(float[] model, float[] perspective, int[] view);
void
Nurbs::loadSamplingMatrices (jfloatArray modelMatrix, jfloatArray projectionMatrix, jintArray viewport)
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	if (modelMatrix == NULL || projectionMatrix == NULL || viewport == NULL)
		throw new IllegalArgumentException();
	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	GLfloat* modelMatrixElements = (GLfloat*)elements(modelMatrix);
	GLfloat* projectionMatrixElements = (GLfloat*)elements(projectionMatrix);
	GLint* viewportElements = (GLint*)elements(viewport);

	::gluLoadSamplingMatrices (nativeNurbs, modelMatrixElements,
		projectionMatrixElements, viewportElements);
}

void
Nurbs::setCallback (NurbsListener *listener)
{
	if (this->implementation == NULL) throw createDeletedNurbsException();
	GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
	if (callback != NULL)
	{
		this->callback = callback;
		::gluNurbsCallback(nativeNurbs, GLU_NURBS_BEGIN_DATA, (_GLUfuncptr)nurbs_begin_callback);
		::gluNurbsCallback(nativeNurbs, GLU_NURBS_END_DATA, (_GLUfuncptr)nurbs_end_callback);
		::gluNurbsCallback(nativeNurbs, GLU_NURBS_VERTEX_DATA, (_GLUfuncptr)nurbs_vertex_callback);
		::gluNurbsCallback(nativeNurbs, GLU_NURBS_NORMAL_DATA, (_GLUfuncptr)nurbs_normal_callback);
		::gluNurbsCallback(nativeNurbs, GLU_NURBS_COLOR_DATA, (_GLUfuncptr)nurbs_color_callback);
		::gluNurbsCallback(nativeNurbs, GLU_NURBS_TEXTURE_COORD_DATA, (_GLUfuncptr)nurbs_texture_coord_callback);
		::gluNurbsCallback(nativeNurbs, GLU_NURBS_ERROR, (_GLUfuncptr)nurbs_error_callback);
	}
}

void
Nurbs::deleteNurbsRenderer ()
{
	if (this->implementation != NULL)
	{
		GLUnurbs* nativeNurbs = (GLUnurbs*)this->implementation;
		::gluDeleteNurbsRenderer(nativeNurbs);
		this->implementation = NULL; 
	}
}

void
Nurbs::initImplementation ()
{
	GLUnurbs* nativeNurbs = ::gluNewNurbsRenderer();
	this->implementation = (RawData*)nativeNurbs;
}
