/************************************************************
* Copyright 2005 Masahiko SAWAI All Rights Reserved. 
************************************************************/

#include "glu4gcj/GLUException.h"
#include "glu4gcj/Tesselator.h"
#include "glu4gcj/TesselatorListener.h"

#include <java/lang/Object.h>
#include <gcj/cni.h>
#include <GL/glu.h>
#include <stdio.h>

using namespace glu4gcj;
using java::lang::Object;
using gnu::gcj::RawData;
using glu4gcj::GLUException;

// functions
static
inline
GLUException*
createDeletedTesselatorException()
{
	return new GLUException(JvNewStringLatin1("This Tesselator was already deleted."));
}

/// callback functions
void
tess_begin_callback(GLenum type, Tesselator* tesselator)
{
	TesselatorListener* callback = tesselator->getCallback();
	if (callback != NULL)
	{
		callback->tessBegin((jint)type, tesselator->getPolygonObject());
	}
}

void
tess_end_callback(Tesselator* tesselator)
{
	TesselatorListener* callback = tesselator->getCallback();
	if (callback != NULL)
	{
		callback->tessEnd(tesselator->getPolygonObject());
	}
}

void
tess_vertex_callback(Object* vertexObject, Tesselator* tesselator)
{
	TesselatorListener* callback = tesselator->getCallback();
	if (callback != NULL)
	{
		callback->tessVertex(vertexObject, tesselator->getPolygonObject());
	}
}

void
tess_edge_flag_callback(GLboolean flag, Tesselator* tesselator)
{
	TesselatorListener* callback = tesselator->getCallback();
	if (callback != NULL)
	{
		callback->tessEdgeFlag(flag, tesselator->getPolygonObject());
	}
}

void
tess_combine_callback(GLdouble coords[3], Object* data[4], GLfloat weights[4], Object** outData, Tesselator* tesselator)
{
	TesselatorListener* callback = tesselator->getCallback();
	if (callback != NULL)
	{
		jdoubleArray coordsArray = JvNewDoubleArray(3);
		jdouble* coordsArrayElements = elements(coordsArray);
		coordsArrayElements[0] = coords[0];
		coordsArrayElements[1] = coords[1];
		coordsArrayElements[2] = coords[2];

		jobjectArray dataArray = JvNewObjectArray(4, &Object::class$, NULL);
		Object** dataArrayElements = elements(dataArray);
		dataArrayElements[0] = data[0];
		dataArrayElements[1] = data[1];
		dataArrayElements[2] = data[2];
		dataArrayElements[3] = data[3];

		jfloatArray weightsArray = JvNewFloatArray(4);
		jfloat* weightsArrayElements = elements(weightsArray);
		weightsArrayElements[0] = weights[0];
		weightsArrayElements[1] = weights[1];
		weightsArrayElements[2] = weights[2];
		weightsArrayElements[3] = weights[3];

		Object* newVertexObject = callback->tessCombine(coordsArray,
			dataArray, weightsArray, tesselator->getPolygonObject());

		if (newVertexObject == NULL)
			throw new GLUException(JvNewStringLatin1("The combine callback is failed."));

		*outData = newVertexObject;
	}
}

void
tess_error_callback(GLenum errno, Tesselator* tesselator)
{
	TesselatorListener* callback = tesselator->getCallback();
	if (callback != NULL)
	{
		callback->tessError(errno, tesselator->getPolygonObject());
	}
}

// instance methods

void
Tesselator::setCallback (TesselatorListener *callback)
{
	if (this->implementation == NULL) throw createDeletedTesselatorException();
	GLUtesselator* nativeTesselator = (GLUtesselator*)this->implementation;
	if (callback != NULL)
	{
		this->callback = callback;
		::gluTessCallback(nativeTesselator, GLU_TESS_BEGIN_DATA, (_GLUfuncptr)tess_begin_callback);
		::gluTessCallback(nativeTesselator, GLU_TESS_END_DATA, (_GLUfuncptr)tess_end_callback);
		::gluTessCallback(nativeTesselator, GLU_TESS_VERTEX_DATA, (_GLUfuncptr)tess_vertex_callback);
		::gluTessCallback(nativeTesselator, GLU_TESS_EDGE_FLAG_DATA, (_GLUfuncptr)tess_edge_flag_callback);
		::gluTessCallback(nativeTesselator, GLU_TESS_COMBINE_DATA, (_GLUfuncptr)tess_combine_callback);
		::gluTessCallback(nativeTesselator, GLU_TESS_ERROR_DATA, (_GLUfuncptr)tess_error_callback);
	}
}

void
Tesselator::nextContour (jint type)
{
	if (this->implementation == NULL) throw createDeletedTesselatorException();
	GLUtesselator* nativeTesselator = (GLUtesselator*)this->implementation;
	::gluNextContour (nativeTesselator, type);
}

void
Tesselator::beginPolygon (Object* polygonObject)
{
	if (this->implementation == NULL) throw createDeletedTesselatorException();
	this->polygonObject = polygonObject;
	GLUtesselator* nativeTesselator = (GLUtesselator*)this->implementation;
	::gluTessBeginPolygon (nativeTesselator, this);
}

void
Tesselator::endPolygon ()
{
	if (this->implementation == NULL) throw createDeletedTesselatorException();
	GLUtesselator* nativeTesselator = (GLUtesselator*)this->implementation;
	::gluTessEndPolygon (nativeTesselator);
}

void
Tesselator::beginContour ()
{
	if (this->implementation == NULL) throw createDeletedTesselatorException();
	GLUtesselator* nativeTesselator = (GLUtesselator*)this->implementation;
	::gluTessBeginContour (nativeTesselator);
}

void
Tesselator::endContour ()
{
	if (this->implementation == NULL) throw createDeletedTesselatorException();
	GLUtesselator* nativeTesselator = (GLUtesselator*)this->implementation;
	::gluTessEndContour (nativeTesselator);
}

void
Tesselator::normal (jdouble valueX, jdouble valueY, jdouble valueZ)
{
	if (this->implementation == NULL) throw createDeletedTesselatorException();
	GLUtesselator* nativeTesselator = (GLUtesselator*)this->implementation;
	::gluTessNormal (nativeTesselator, valueX, valueY, valueZ);
}

void
Tesselator::vertex (jdoubleArray locationArray, Object* vertexObject)
{
	if (this->implementation == NULL) throw createDeletedTesselatorException();
	if (JvGetArrayLength(locationArray) < 3)
		throw new GLUException(JvNewStringLatin1("Vertex location array is too short."));
	
	GLUtesselator* nativeTesselator = (GLUtesselator*)this->implementation;
	GLdouble* location = (GLdouble*)elements(locationArray);
	::gluTessVertex(nativeTesselator, location, vertexObject);
}

void
Tesselator::setProperty (jint which, jdouble data)
{
	if (this->implementation == NULL) throw createDeletedTesselatorException();
	GLUtesselator* nativeTesselator = (GLUtesselator*)this->implementation;
	::gluTessProperty (nativeTesselator, which, data);
}

jdouble
Tesselator::getProperty (jint which)
{
	if (this->implementation == NULL) throw createDeletedTesselatorException();
	GLUtesselator* nativeTesselator = (GLUtesselator*)this->implementation;
	GLdouble data;

	::gluGetTessProperty (nativeTesselator, which, &data);

	return (jdouble)data;
}

void
Tesselator::deleteTess ()
{
	if (this->implementation != NULL)
	{
		GLUtesselator* nativeTesselator = (GLUtesselator*)this->implementation;
		::gluDeleteTess(nativeTesselator);
		this->implementation = NULL; 
	}
}

void
Tesselator::initImplementation ()
{
	GLUtesselator* nativeTesselator = ::gluNewTess();
	this->implementation = (RawData*)nativeTesselator;
}
