# -*- coding: utf-8 -*-
#
#  textmanager.py - Sakura Script manipulator area for GBottler
#  Copyright (C) 2004-2010 by Atzm WATANABE <atzm@atzm.org>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: textmanager.py,v 1.48 2010/08/23 15:14:02 atzm Exp $
#

import os
import sys

import gobject
import gtk
import pango

from bottleparser import *
from common import *
from hypertextview import HyperTextView
import config

class TextManager(HyperTextView):
	__gsignals__ = {
		'enter-pressed': (gobject.SIGNAL_RUN_LAST|gobject.SIGNAL_ACTION,
						  gobject.TYPE_NONE, (gobject.TYPE_INT,))
	}

	def __init__(self, app, parser=BottleParser('loose')):
		HyperTextView.__init__(self)

		self.app = app
		self.parser = parser
		self.view_style = STYLE_TALK

		# settings
		self.set_wrap_mode(gtk.WRAP_CHAR)
		self.set_editable(True)
		self.set_cursor_visible(True)
		self.connect('anchor-clicked', self._open_browser)

		# create tags
		self.create_tag('script')
		self.create_tag('sakura')
		self.create_tag('kero')
		self.create_tag('sync')
		self.create_tag('error')

		# initialize
		self.receive_preferences_changed()

	def _open_browser(self, widget, text, anchor, button):
		if anchor:
			os.system(config.get('browser', 'browser_command') % anchor + ' &')

	def do_enter_pressed(self, modifier):
		if modifier == gtk.gdk.SHIFT_MASK:
			action = config.get('key', 'shift_enter', 'int')
		elif modifier == gtk.gdk.CONTROL_MASK:
			action = config.get('key', 'ctrl_enter', 'int')
		elif modifier == gtk.gdk.MOD1_MASK:
			action = config.get('key', 'alt_enter', 'int')
		else:
			action = config.get('key', 'enter', 'int')

		if action == config.KEY_ENTER_BACKSLASH_N:
			self.emit('insert-at-cursor', r'\n')
		elif action == config.KEY_ENTER_BACKSLASH_N_NEWLINE:
			self.emit('insert-at-cursor', '\\n\n')
		elif action == config.KEY_ENTER_NEWLINE:
			self.emit('insert-at-cursor', '\n')

	def color_changed(self, over_color='light gray'):
		tag_table = self.get_tag_table()

		tag_table.lookup('script').set_property('foreground', config.get('fgcolor', 'script'))
		tag_table.lookup('sakura').set_property('foreground', config.get('fgcolor', 'sakura'))
		tag_table.lookup('kero').set_property('foreground', config.get('fgcolor', 'unyuu'))
		tag_table.lookup('sync').set_property('foreground', config.get('fgcolor', 'synchronized'))
		tag_table.lookup('error').set_property('foreground', config.get('fgcolor', 'error'))

		self.link['foreground'] = config.get('fgcolor', 'url')
		self.active['foreground'] = config.get('fgcolor', 'url')
		self.hover['foreground'] = config.get('fgcolor', 'url')

	def font_changed(self):
		self.set_style(self.view_style)

	def receive_preferences_changed(self):
		self.update()
		self.color_changed()
		self.font_changed()
		self.coloring()

	def set_style_talk(self):
		self.view_style = STYLE_TALK
		self.modify_font(pango.FontDescription(config.get('font', 'general')))

	def set_style_script(self):
		self.view_style = STYLE_SCRIPT
		self.modify_font(pango.FontDescription(config.get('font', 'script')))

	def set_style_script_with_linefeed(self):
		self.view_style = STYLE_SCRIPT_WITH_LINEFEED
		self.modify_font(pango.FontDescription(config.get('font', 'script')))

	def set_style(self, style):
		if type(style) is not int:
			raise ValueError('Invalid argument: ' + str(style))

		if style == STYLE_TALK:
			self.set_style_talk()
		elif style == STYLE_SCRIPT:
			self.set_style_script()
		else:
			self.set_style_script_with_linefeed()

	def coloring(self, widget=None, event=None):
		pos    = self.get_cursor_position()
		end    = self.get_end_position()
		string = unicode(self.get_text(), 'utf-8', 'replace')

		self.set_text('')
		self.__insert_with_style(string, STYLE_SCRIPT)

		if end != pos:
			if end != 0:
				self.emit('move-cursor', gtk.MOVEMENT_LOGICAL_POSITIONS, end*-1, False)
			if pos != 0:
				self.emit('move-cursor', gtk.MOVEMENT_LOGICAL_POSITIONS, pos, False)

	def insert_with_color_from(self, string, pos=-3):
		self.__insert_with_style(string, self.view_style, '\n')

		if pos != 0:
			self.emit('move-cursor', gtk.MOVEMENT_LOGICAL_POSITIONS, pos, False)

	def __insert_with_style(self, string, style, suffix=None):
		# tag stack for closable (currently only synchronized) sessions
		current = ['sakura']

		for chunk in self.parser.parse(string):
			# SCRIPT_TEXT
			if chunk[0] == SCRIPT_TEXT:
				self.__insert_with_style_text(chunk, style, current)
				continue

			# XXX: SCRIPT_TEXT's PARSE_ERROR
			if type(chunk[1]) == tuple:
				self.__insert_with_style_text(chunk, style, current)
				continue

			# SCRIPT_TAG or PARSE_ERROR
			name, args = chunk[1], chunk[2:]

			if name == r'\h':
				if not current[-1] == 'sync':
					current = ['sakura']
			elif name == r'\u':
				if not current[-1] == 'sync':
					current = ['kero']
			elif name == r'\_s':
				if current[-1] == 'sync' and len(current) > 1:
					# synchronized session closed
					current.pop()
				else:
					# synchronized session opened
					current.append('sync')

			tag = 'script'

			if chunk[0] == PARSE_ERROR:
				tag = 'error'

			if name == r'\s':
				try:
					int(args[0])
				except ValueError:
					tag = 'error'

			# STYLE_TALK
			if style == STYLE_TALK:
				p  = self.get_cursor_position()
				ec = self.get_char_at(self.get_end_position()-1)

				if name in [r'\h', r'\u', r'\_s'] and p and ec != '\n':
					if current[-1] != 'sync' or name == r'\_s':
						self.insert('\n')

				elif name == r'\URL':
					a = []

					for i in args:
						a.append(list(i))

						try:
							if i[0][0] == PARSE_ERROR:
								tag = 'error'
						except IndexError:
							tag = 'error'

					if len(a) == 1:                # contains no label (url only)
						if not a[0]:
							pass
						else:
							self.insert_with_tags_by_name('[', tag)
							self.insert_with_anchor(a[0][0][1], a[0][0][1])
							self.insert_with_tags_by_name(']', tag)

					else:
						if len(a) % 2:             # contains cancel
							text = a.pop(0)[0][1]
							self.insert_with_tags_by_name('[', tag)
							self.insert_with_anchor(text, '')
							self.insert_with_tags_by_name(']', tag)

						while a:
							link = a.pop(0)[0][1]
							try:
								text = a.pop(0)[0][1]
								self.insert_with_tags_by_name('[', tag)
								self.insert_with_anchor(text, link)
								self.insert_with_tags_by_name(']', tag)
							except IndexError:
								self.insert_with_tags_by_name(link, tag)

				continue

			# STYLE_SCRIPT or STYLE_SCRIPT_WITH_LINEFEED
			self.insert_with_tags_by_name(name, tag)

			if name == r'\URL' and chunk[0] == PARSE_ERROR and len(args) >= 2:
				fmt = '[%s]'
			elif name == r'\w':
				fmt = '%s'
			elif name == r'\s' and chunk[0] == PARSE_ERROR and (not args[0] or args[0][0] == '['):
				fmt = '%s'
			else:
				fmt = '[%s]'

			for n in xrange(len(args)):
				if type(args[0]) is tuple:
					try:
						if args[n][0][0] == PARSE_ERROR:
							self.insert_with_tags_by_name('%s' % args[n][0][1], 'error')

						else:
							self.insert_with_tags_by_name(fmt % args[n][0][1], tag)

					except IndexError:
						pass

				else:
					self.insert_with_tags_by_name(fmt % args[n], tag)

			if name == r'\n' and style == STYLE_SCRIPT_WITH_LINEFEED:
				self.insert_with_tags_by_name('\n', tag)

		if suffix:
			self.insert(suffix)

	def __insert_with_style_text(self, chunk, style, current):
		for n in xrange(len(chunk[1])):
			t = chunk[1][n][0]
			c = chunk[1][n][1]

			if style == STYLE_TALK:
				c = c.replace(r'\\', '\\')
				c = c.replace(r'\%', '%')
				c = c.replace(r'\]', ']')

			if t == TEXT_STRING:
				self.insert_with_tags_by_name(c, current[-1])

			elif t in [TEXT_META, PARSE_ERROR]:
				tag = current[-1]

				if t == PARSE_ERROR:
					tag = 'error'

				if c == '%j':
					args = chunk[1][n][2]

					if args[0] == PARSE_ERROR:
						self.insert_with_tags_by_name(c, 'error')

						if args[1]:
							self.insert_with_tags_by_name(args[1], 'error')
					else:
						self.insert_with_tags_by_name('%s[%s]' % (c, args[1]), tag)
				else:
					self.insert_with_tags_by_name(c, tag)


gobject.type_register(TextManager)

gtk.binding_entry_add_signal(TextManager, gtk.keysyms.Return, 0,
							 'enter-pressed', gobject.TYPE_INT, 0)
gtk.binding_entry_add_signal(TextManager, gtk.keysyms.KP_Enter, 0,
							 'enter-pressed', gobject.TYPE_INT, 0)

gtk.binding_entry_add_signal(TextManager, gtk.keysyms.Return, gtk.gdk.SHIFT_MASK,
							 'enter-pressed', gobject.TYPE_INT, gtk.gdk.SHIFT_MASK)
gtk.binding_entry_add_signal(TextManager, gtk.keysyms.KP_Enter, gtk.gdk.SHIFT_MASK,
							 'enter-pressed', gobject.TYPE_INT, gtk.gdk.SHIFT_MASK)

gtk.binding_entry_add_signal(TextManager, gtk.keysyms.Return, gtk.gdk.CONTROL_MASK,
							 'enter-pressed', gobject.TYPE_INT, gtk.gdk.CONTROL_MASK)
gtk.binding_entry_add_signal(TextManager, gtk.keysyms.KP_Enter, gtk.gdk.CONTROL_MASK,
							 'enter-pressed', gobject.TYPE_INT, gtk.gdk.CONTROL_MASK)

gtk.binding_entry_add_signal(TextManager, gtk.keysyms.Return, gtk.gdk.MOD1_MASK,
							 'enter-pressed', gobject.TYPE_INT, gtk.gdk.MOD1_MASK)
gtk.binding_entry_add_signal(TextManager, gtk.keysyms.KP_Enter, gtk.gdk.MOD1_MASK,
							 'enter-pressed', gobject.TYPE_INT, gtk.gdk.MOD1_MASK)
