# -*- coding: utf-8 -*-
#
#  xmllogger.py - XML Log manipulator for GBottler
#  Copyright (C) 2004 by Atzm WATANABE <atzm@atzm.org>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: xmllogger.py,v 1.10 2010/08/28 15:12:17 atzm Exp $
#

import os, sys, time
import re, string, gzip

from logparsers import *
from common     import *

if os.name == 'posix':
	import fcntl
else:
	import msvcrt

XML_HEADER = (
	'<?xml version="1.0" encoding="Shift_JIS"?>\r\n'
	'<?xml-stylesheet type="text/xsl" href="http://bottle.mikage.to/xbtl.xsl"?>\r\n'
	'<!DOCTYPE bottlelog [\r\n'
	'\t<!ELEMENT bottlelog (message*)>\r\n'
	'\t<!ATTLIST bottlelog version CDATA #REQUIRED>\r\n'
	'\t<!ATTLIST bottlelog generator CDATA #REQUIRED>\r\n'
	'\t<!ATTLIST bottlelog saved CDATA #REQUIRED>\r\n'
	'\t<!ELEMENT message (date, channel, script, votes, agrees, ghost)>\r\n'
	'\t<!ATTLIST message mid CDATA #REQUIRED>\r\n'
	'\t<!ELEMENT date (#PCDATA)>\r\n'
	'\t<!ELEMENT channel (#PCDATA)>\r\n'
	'\t<!ELEMENT script (#PCDATA)>\r\n'
	'\t<!ELEMENT votes (#PCDATA)>\r\n'
	'\t<!ELEMENT agrees (#PCDATA)>\r\n'
	'\t<!ELEMENT ghost (#PCDATA)>\r\n'
	']>\r\n'
	)

XML_LINE = (
	'\t<message mid="%s">\r\n'
	'\t\t<date>%s</date>\r\n'
	'\t\t<channel>%s</channel>\r\n'
	'\t\t<script>%s</script>\r\n'
	'\t\t<votes>%s</votes>\r\n'
	'\t\t<agrees>%s</agrees>\r\n'
	'\t\t<ghost>%s</ghost>\r\n'
	'\t</message>\r\n'
	)

class XMLLogger:
	def __init__(self, compress=False):
		self.compress = compress
		self.parsed = []
		self.parser = BottleLogParserExpat()

	def parse(self, filename):
		try:
			try:
				file = gzip.open(filename, 'r')
				lines = file.readlines()[1:] # parser is not supported Shift_JIS code
			except IOError:
				file = open(filename, 'r')
				lines = file.readlines()[1:]
			file.close()

			lines = string.join(lines, '')
			lines = unicode(lines, 'sjis', 'replace').encode('utf-8')
			lines = re.sub('\r\n|\r', '\n', lines)

			self.parsed = self.parser.parse(lines)
			self.parser.initialize()
		except:
			return -1

		return self.parsed

	def conv_xml(self, elem):
		elem = re.sub('&', '&amp;', elem)
		elem = re.sub('<', '&lt;', elem)
		elem = re.sub('>', '&gt;', elem)
		elem = re.sub('"', '&quot;', elem)
		elem = string.strip(elem)
		return elem

	def deconv_xml(self, elem):
		elem = re.sub('&amp;', '&', elem)
		elem = re.sub('&lt;', '<', elem)
		elem = re.sub('&gt;', '>', elem)
		elem = re.sub('&quot;', '"', elem)
		elem = string.strip(elem)
		return elem

	def save_as_xml_from(self, lists, filename):
		# lock start
		if os.name == 'posix':
			lockfunc   = lambda file, filename: fcntl.flock(file.fileno(), fcntl.LOCK_EX|fcntl.LOCK_NB)
			unlockfunc = lambda file, filename: fcntl.flock(file.fileno(), fcntl.LOCK_UN)
		else:
			lockfunc   = lambda file, filename: msvcrt.locking(file.fileno(), msvcrt.LK_NBLCK,
															   os.path.getsize(filename))
			unlockfunc = lambda file, filename: msvcrt.locking(file.fileno(), msvcrt.LK_UNLCK,
															   os.path.getsize(filename))
		count = 5
		lockfilename = os.path.join(open_bottlecase(), '.xmlsavelock')
		lockfile = open(lockfilename, 'w')
		while True:
			try:
				lockfunc(lockfile, lockfilename)
			except IOError:
				if count <= 0:
					sys.stderr.write("Error: cannot write %s\n" % filename)
					return False
				count -= 1
				sys.stderr.write('%s is locked, sleep a second\n' % filename)
				time.sleep(1)
			else:
				break
		# lock end

		try:
			if self.compress:
				if filename.endswith('.gz'):
					file = gzip.open(filename, 'w')
				else:
					file = gzip.open('%s.gz' % filename, 'w')
			else:
				file = open(filename, 'w')
		except:
			unlockfunc(lockfile, lockfilename)
			return False

		saved = time.strftime('%y/%m/%d %H:%M:%S', time.localtime(time.time()))
		generator = '%s ver %s' % (APP, VER)
		version = '1.0'

		opentag = '<bottlelog saved="%s" generator="%s" version="%s">\r\n' % (saved, generator, version)

		file.write(XML_HEADER)
		file.write(opentag)

		for line in lists:
			line = map(self.conv_xml, line)
			datetime, ghost, channel, votes, agrees, script, mid = line
			xmlline = (XML_LINE % (mid, datetime, channel, script, votes, agrees, ghost)).encode('sjis', 'xmlcharrefreplace')
			file.write(xmlline)
		file.write('</bottlelog>\r\n')
		file.close()

		unlockfunc(lockfile, lockfilename)
		return True

	def open_xml_from(self, filename):
		parsed = self.parse(filename)
		if parsed == -1:
			return -1

		lines = []
		for dic in parsed:
			list = map(self.deconv_xml, [dic['date'], dic['mid'], dic['votes'], dic['agrees'],
										 dic['channel'], dic['ghost'], dic['script']])
			lines.insert(0, list)
		return lines

if __name__ == '__main__':
	filename = sys.argv[1]
	mode     = sys.argv[2]

	try:
		file = gzip.open(filename, 'r')
		lines = file.readlines()[1:]
	except IOError:
		file = open(filename, 'r')
		lines = file.readlines()[1:]
	lines = string.join(lines, '')
	lines = unicode(lines, 'sjis', 'replace').encode('utf-8')
	lines = re.sub('\r\n|\r', '\n', lines)

	print 'Mode:   %s' % mode
	if mode == 'expat':
		p = BottleLogParserExpat()
	elif mode == 'sax':
		p = BottleLogParserSAX()
	else:
		raise RuntimeError('invalid mode')

	print 'start: ', time.strftime('%H:%M:%S', time.localtime())
	p.parse(lines)
	print 'end:   ', time.strftime('%H:%M:%S', time.localtime())
