#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
#  gviewer.py - SSTP playing library for PyGTK
#  Copyright (C) 2004-2010 by Atzm WATANABE <atzm@atzm.org>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: gviewer.py,v 1.23 2010/08/16 16:01:10 atzm Exp $
#

import gc
import sys
import time
import datetime
import htmlentitydefs

import gtk
import pango
import gobject

import config

from script import *
from viewercommon import *

class GViewerQueue:
	# milli seconds
	POLLING_INTERVAL = 10

	def __init__(self, sakura, kero, window=None, s=None, cg=None, rest_scripts=None, force_garbage_collect=True,
				 polling_start_at_first=config.get('gviewer', 'use_gviewer', 'boolean')):

		self.sakura_frame, self.t1, self.sw1 = sakura
		self.kero_frame,   self.t2, self.sw2 = kero

		self._window                 = window
		self.sender_label            = s
		self.channel_and_ghost_label = cg
		self.rest_scripts_label      = rest_scripts
		self.force_garbage_collect   = force_garbage_collect

		self._lower_id = []

		# parser
		self.parser = Parser('loose')

		# request queue
		self.queue    = []
		self.playing  = False
		self.timeouts = []

		# one request's each chunk queue
		self.chunk_queue    = []
		self.chunk_playing  = False
		self.chunk_timeouts = []

		# initialize
		self.super_id       = None
		self.super_chunk_id = None

		# initialize for sequential splite
		self.sequential_idx = {SIDE_SAKURA: 0, SIDE_KERO: 0}
		self.sequential_id  = {SIDE_SAKURA: 0, SIDE_KERO: 0}

		if polling_start_at_first:
			self.polling_start()

		self.started_at = time.time()

	def set_force_garbage_collect(self, flag):
		self.force_garbage_collect = bool(flag)

	def polling_start(self):
		if not self.super_id:
			self.super_id = gobject.timeout_add(self.POLLING_INTERVAL, self._insert)
		if not self.super_chunk_id:
			self.super_chunk_id = gobject.timeout_add(self.POLLING_INTERVAL, self._chunk_insert)

	def polling_stop(self):
		self.cancel_all(True)

	def cancel(self):
		for i in self.chunk_timeouts:
			gobject.source_remove(i)

		self.chunk_timeouts = []
		self.chunk_queue    = []

		self.t1.set_text('')
		self.t2.set_text('')

		self.clear_surface(self.sakura_frame, SIDE_SAKURA)
		self.clear_surface(self.kero_frame, SIDE_KERO)
		self.clear_information()

		self.chunk_playing  = False
		self.playing        = False
		self.super_chunk_id = gobject.timeout_add(self.POLLING_INTERVAL, self._chunk_insert)

	def cancel_all(self, stop=False):
		for i in self.chunk_timeouts + self.timeouts:
			gobject.source_remove(i)

		self.timeouts       = []
		self.chunk_timeouts = []
		self.queue          = []
		self.chunk_queue    = []

		self.t1.set_text('')
		self.t2.set_text('')

		self.clear_surface(self.sakura_frame, SIDE_SAKURA)
		self.clear_surface(self.kero_frame, SIDE_KERO)
		self.clear_information()

		self.chunk_playing = False
		self.playing       = False

		if stop:
			self.super_id       = None
			self.super_chunk_id = None
		else:
			self.super_id       = gobject.timeout_add(self.POLLING_INTERVAL, self._insert)
			self.super_chunk_id = gobject.timeout_add(self.POLLING_INTERVAL, self._chunk_insert)

	def clear_information(self):
		if self.sender_label:
			self.sender_label.set_text('')
		if self.channel_and_ghost_label:
			self.channel_and_ghost_label.set_text('')
		if self.rest_scripts_label:
			self.rest_scripts_label.set_text('')

	def clear_surface(self, container, side):
		for image in container.get_children():
			image.hide()
			container.remove(image)
			image.destroy()

		self._clear_timeouts_seq(side)

		if self.force_garbage_collect:
			gc.collect()

	def update_rest(self):
		if self.rest_scripts_label:
			rest = str(len(self.queue))
			self.rest_scripts_label.set_text(rest)

	def enqueue(self, script, ghost, sender=None, channel=None, ifghost=None):
		self.queue.append([script, ghost, sender, channel, ifghost])
		self.update_rest()

	def _raise(self):
		if config.get('gviewer', 'raise_before_play', 'boolean'):
			if isinstance(self._window, gtk.Window) and isinstance(self._window.window, gtk.gdk.Window):
				if self._lower_id:
					for lid in self._lower_id:
						gobject.source_remove(lid)
					self._lower_id = []
				self._window.window.raise_()

	def _lower(self):
		if config.get('gviewer', 'sink_after_play', 'boolean'):
			if isinstance(self._window, gtk.Window) and isinstance(self._window.window, gtk.gdk.Window):
				self._window.window.lower()

	def _insert(self):
		self.timeouts.append(gobject.timeout_add(self.POLLING_INTERVAL, self._insert))

		if not self.queue or self.playing or self.chunk_queue or self.chunk_playing:
			return

		self._raise()

		self.playing = True
		self.t1.set_text('')
		self.t2.set_text('')

		text, ghost, sender, channel, ifghost = self.queue.pop(0)
		self.update_rest()

		self.chunk_queue.append(['', '', 0,  SIDE_SAKURA, 0, [SESSION_QUICK],
								 ghost, sender, channel, ifghost])
		self.chunk_queue.append(['', '', 0, SIDE_KERO, 10, [SESSION_QUICK],
								 ghost, sender, channel, ifghost])

		sessions     = []
		current_side = [SIDE_SAKURA]

		try:
			chunks = self.parser.parse(text)
		except ParserError, e:
			print >> sys.stderr, time.ctime(), str(e)
			return

		for chunk in chunks:
			if chunk[0] == TEXT_STRING:
				for n in xrange(len(chunk[1])):
					if chunk[1][n][0] not in [SCRIPT_TEXT, TEXT_META]:
						continue

					text = chunk[1][n][1]

					if chunk[1][n][0] == TEXT_META:
						if ghost is not None:
							if text == '%username':
								username = ghost.get_user()
								if username: text = username
							elif text == '%selfname':
								sakuraname = ghost.get_sakura()
								if sakuraname: text = sakuraname
							elif text == '%selfname2':
								sakuraname = ghost.get_sakura2()
								if sakuraname: text = sakuraname
							elif text == '%keroname':
								keroname = ghost.get_unyu()
								if keroname: text = keroname

						if text == '%j':
							text = '%s[%s]' % (text, chunk[1][n][2])
						elif text == '%month':
							text = str(datetime.date.today().month)
						elif text == '%day':
							text = str(datetime.date.today().day)
						elif text == '%hour':
							text = str(datetime.datetime.now().hour)
						elif text == '%minute':
							text = str(datetime.datetime.now().minute)
						elif text == '%second':
							text = str(datetime.datetime.now().second)
						elif text == '%screenwidth':
							text = str(gtk.gdk.get_default_root_window().get_geometry()[2])
						elif text == '%screenheight':
							text = str(gtk.gdk.get_default_root_window().get_geometry()[3])
						elif text == '%exh':
							text = str(int(time.time() - self.started_at) / (60 * 60))

					self.chunk_queue.append([text, '', 0, current_side[-1], None,
											 sessions[:], ghost, sender, channel, ifghost])

					while sessions.count(EVENT_CLEAR):
						sessions.remove(EVENT_CLEAR)

			elif chunk[0] == TEXT_META:
				name, args = chunk[1], chunk[2:]

				if name == r'\w':
					wait = int(args[0]) * config.get('gviewer', 'wait_w1', 'int')
					self.chunk_queue.append(['', '', wait, current_side[-1], None,
											 sessions[:], ghost, sender, channel, ifghost])
				elif name == r'\_w':
					self.chunk_queue.append(['', '', int(args[0]), current_side[-1], None,
											 sessions[:], ghost, sender, channel, ifghost])

				elif name in [r'\_u', r'\_m', r'\&']:
					try:
						if name in [r'\_u', r'\_m']:
							t = unichr(int(args[0], 16))
						elif name == r'\&':
							id_ = args[0]

							if id_.startswith('#'):
								id_ = htmlentitydefs.codepoint2name[int(id_[1:])]

							t = unichr(htmlentitydefs.name2codepoint[id_])

						self.chunk_queue.append([t, '', 0, current_side[-1], None,
												 sessions[:], ghost, sender, channel, ifghost])
					except:
						pass

				elif name == r'\s':
					num = range(len(current_side))
					num.reverse()
					for i in num:
						if current_side[i] in [SIDE_SAKURA, SIDE_KERO]:
							self.chunk_queue.append(['', '', 0, current_side[i], int(args[0]), sessions[:],
													 ghost, sender, channel, ifghost])
							break

				elif name == r'\n':
					self.chunk_queue.append(['\n', '', 0, current_side[-1], None,
											 sessions[:], ghost, sender, channel, ifghost])

				elif name in [r'\0', r'\h', r'\1', r'\u', r'\_s']:
					self.chunk_queue.append(['', '', 0, current_side[-1], None,
											 sessions[:], ghost, sender, channel, ifghost])

					while sessions.count(EVENT_CLEAR):
						sessions.remove(EVENT_CLEAR)

					if name in [r'\0', r'\h']:
						if not current_side[-1] == SIDE_SYNC:
							current_side = [SIDE_SAKURA]

					elif name in [r'\1', r'\u']:
						if not current_side[-1] == SIDE_SYNC:
							current_side = [SIDE_KERO]

					elif name == r'\_s':
						if current_side[-1] == SIDE_SYNC and len(current_side) > 1:
							current_side.pop()
						else:
							current_side.append(SIDE_SYNC)

				elif name == r'\URL':
					args = [list(a) for a in args]

					if len(args) == 1:                # contains no label (url only)
						text = args[0][0][1]
						self.chunk_queue.append([text, text, 0, current_side[-1], None,
												 sessions[:] + [EVENT_URL, SESSION_QUICK], ghost, sender,
												 channel, ifghost])
					else:
						if len(args) % 2:             # contains cancel
							text = args.pop(0)[0][1]
							self.chunk_queue.append([text + '\n', '', 0, current_side[-1], None,
													 sessions[:] + [EVENT_URL, SESSION_QUICK], ghost, sender,
													 channel, ifghost])

						while args:
							link = args.pop(0)[0][1]
							text = args.pop(0)[0][1]
							if args: text += '\n'
							self.chunk_queue.append([text, link, 0, current_side[-1], None,
													 sessions[:] + [EVENT_URL, SESSION_QUICK], ghost, sender,
													 channel, ifghost])

				elif name == r'\_q':
					if SESSION_QUICK in sessions:
						while sessions.count(SESSION_QUICK):
							sessions.remove(SESSION_QUICK)
					else:
						sessions.append(SESSION_QUICK)

				elif name == r'\c':
					sessions.append(EVENT_CLEAR)

				elif name == r'\e':
					wait = 5 * config.get('gviewer', 'wait_w1', 'int')
					self.chunk_queue.append(['', '', wait, current_side[-1], None,
											 sessions[:], ghost, sender, channel, ifghost])
					break

	def _chunk_insert(self):
		self.chunk_timeouts.append(gobject.timeout_add(self.POLLING_INTERVAL, self._chunk_insert))
		if not self.chunk_queue or self.chunk_playing:
			return
		self.chunk_playing = True

		chunk_max = len(self.chunk_queue)

		text, link, wait, side, surf, sessions, ghost, sender, channel, ifghost = self.chunk_queue.pop(0)

		if self.sender_label and sender:
			self.sender_label.set_text(sender)

		if self.channel_and_ghost_label:
			if channel is not None:
				self.channel_and_ghost_label.set_text(channel + ' / ')

			ghost_name = None
			if ghost is not None:
				ghost_name = ghost.get_sakura()

			if ghost_name is not None:
				self.channel_and_ghost_label.set_text(self.channel_and_ghost_label.get_text() +
													  ghost_name)
			elif ifghost:
				self.channel_and_ghost_label.set_text(self.channel_and_ghost_label.get_text() +
													  ifghost)

		event = []
		if EVENT_CLEAR in sessions:
			event.append(EVENT_CLEAR)
		if EVENT_URL in sessions:
			event.append(EVENT_URL)

		if text != '':
			if SESSION_QUICK in sessions:
				self.chunk_timeouts.append(gobject.timeout_add(0, self.chunk_insert, text, link, 0, 0,
														   chunk_max, side, surf, event, ghost))
			else:
				# for clear
				self.chunk_timeouts.append(gobject.timeout_add(wait,
															   self.chunk_insert, text[0], link,
															   1, len(text), chunk_max,
															   side, surf, event, ghost))
				for i in xrange(1, len(text)):
					self.chunk_timeouts.append(
						gobject.timeout_add(i * config.get('gviewer', 'wait_char', 'int') + wait,
											self.chunk_insert, text[i], link, i+1, len(text),
											chunk_max, side, surf, [], ghost))
		elif EVENT_CLEAR in sessions:
			self.chunk_timeouts.append(gobject.timeout_add(0, self.chunk_insert, '', link,
														   0, 0, chunk_max, side, surf,
														   [EVENT_CLEAR], ghost))
		else:
			self.chunk_timeouts.append(gobject.timeout_add(wait, self.chunk_insert, '', link,
														   0, 0, chunk_max, side, surf,
														   event, ghost))

	def chunk_insert(self, c, l, i, max, chunk_max, side, surf, event, ghost):
		self.insert(c, l, chunk_max, side, surf, event, ghost)
		if i == max:
			self.chunk_playing = False

	def insert(self, c, l, chunk_max, side, surf, event, ghost):
		self.set_surface(side, surf, ghost)

		if side == SIDE_SAKURA:
			insertee = [[self.t1, self.sw1]]
		elif side == SIDE_KERO:
			insertee = [[self.t2, self.sw2]]
		elif side == SIDE_SYNC:
			insertee = [[self.t1, self.sw1], [self.t2, self.sw2]]

		for buf, sw in insertee:
			if EVENT_CLEAR in event:
				buf.set_text('')
			if EVENT_URL in event:
				buf.insert_with_anchor(c, l)
			else:
				buf.insert(c)
			sw.emit('scroll-child', gtk.SCROLL_END, False)

		if chunk_max == 1:
			gobject.timeout_add(config.get('gviewer', 'wait_char', 'int') + self.POLLING_INTERVAL, self.clear_timeouts)

	def set_number(self, pixbuf, surf_num, side):
		def _expose(drawingarea, event, pixmap, num, side):
			style = drawingarea.get_style()
			fg_gc = style.fg_gc[gtk.STATE_NORMAL]
			bg_gc = style.bg_gc[gtk.STATE_NORMAL]
			drawingarea.window.draw_drawable(bg_gc, pixmap, 0, 0, 0, 0, -1, -1)

			bg = style.bg[gtk.STATE_NORMAL]
			color = '#%04x%04x%04x' % (bg.red, bg.green, bg.blue)
			layout = pango.Layout(drawingarea.get_pango_context())
			layout.set_markup('<span size="x-small" background="%s">%d</span>' % (color, num))

			w, h = layout.get_pixel_size()
			h_pos = SIZE_SURFACE_V - h - 1
			if side == SIDE_SAKURA:
				w_pos = 0
			else:
				w_pos = SIZE_SURFACE_H - w - 3

			drawingarea.window.draw_rectangle(bg_gc, True, SIZE_SURFACE_H, SIZE_SURFACE_V, w, h)
			drawingarea.window.draw_layout(fg_gc, w_pos, h_pos, layout)

			return True

		pixmap = pixbuf.render_pixmap_and_mask()[0]
		drawingarea = gtk.DrawingArea()
		drawingarea.set_events(gtk.gdk.EXPOSURE_MASK)
		drawingarea.connect('expose-event', _expose, pixmap, surf_num, side)
		return drawingarea

	def _get_photoframe(self, side, surf_num, ghost):
		photoframe = gtk.Frame()
		photoframe.set_border_width(0)
		photoframe.set_shadow_type(gtk.SHADOW_OUT)
		photoframe.set_size_request(SIZE_SURFACE_H+1, SIZE_SURFACE_V+1)
		photoframe.show()

		if not ghost:                                        # just label when ghost not found
			l = gtk.Label(str(surf_num))
			l.set_size_request(SIZE_SURFACE_H, SIZE_SURFACE_V)
			l.show()
			photoframe.add(l)
			return photoframe

		item = ghost.get_surface(surf_num, side)             # always pixbuf

		if config.get('gviewer', 'draw_number', 'boolean'):  # overdraw surface numbers
			item = self.set_number(item, surf_num, side)
		else:                                                # widget'ize
			image = gtk.Image()
			image.set_from_pixbuf(item)
			item = image

		item.set_size_request(SIZE_SURFACE_H, SIZE_SURFACE_V)
		item.show()
		photoframe.add(item)
		return photoframe

	def _get_photoframe_seq(self, side, surf_num, ghost):
		def _animation(side, frame, items):
			self.sequential_idx[side] += 1

			if self.sequential_idx[side] >= len(items):
				return False

			if items[self.sequential_idx[side]] == '*':
				self.sequential_idx[side] = 0

			for chld in frame.get_children():
				frame.remove(chld)

			items[self.sequential_idx[side]].show()
			frame.add(items[self.sequential_idx[side]])
			return True

		self._clear_timeouts_seq(side)
		self.sequential_idx[side] = 0

		photoframe = gtk.Frame()
		photoframe.set_border_width(0)
		photoframe.set_shadow_type(gtk.SHADOW_OUT)
		photoframe.set_size_request(SIZE_SURFACE_H+1, SIZE_SURFACE_V+1)
		photoframe.show()

		if not ghost:                                        # just label when ghost not found
			l = gtk.Label(str(surf_num))
			l.set_size_request(SIZE_SURFACE_H, SIZE_SURFACE_V)
			l.show()
			photoframe.add(l)
			return photoframe

		items = ghost.get_surface_seq(surf_num, side)

		if config.get('gviewer', 'draw_number', 'boolean'):  # overdraw surface numbers
			for i in range(len(items)):
				if isinstance(items[i], gtk.gdk.Pixbuf):
					items[i] = self.set_number(items[i], surf_num, side)
		else:                                                # convert pixbuf -> image
			for i in range(len(items)):
				if isinstance(items[i], gtk.gdk.Pixbuf):
					photo = gtk.Image()
					photo.set_from_pixbuf(items[i])
					items[i] = photo

		for item in items:
			if isinstance(item, gtk.gdk.Pixbuf) or isinstance(item, gtk.Image):
				item.set_size_request(SIZE_SURFACE_H, SIZE_SURFACE_V)

		items[0].show()
		photoframe.add(items[0])
		self.sequential_id[side] = gobject.timeout_add(config.get('gviewer', 'seq_wait1', 'int'),
													   _animation, side, photoframe, items)
		return photoframe

	def set_surface(self, side, surf_num, ghost):
		if surf_num is None:
			return

		if side == SIDE_SAKURA:
			surf_container = self.sakura_frame
		elif side == SIDE_KERO:
			surf_container = self.kero_frame
		else:
			return
		self.clear_surface(surf_container, side)

		if config.get('gviewer', 'use_sequential', 'boolean'):
			photoframe = self._get_photoframe_seq(side, surf_num, ghost)
		else:
			photoframe = self._get_photoframe(side, surf_num, ghost)

		fcon = gtk.VBox()
		fcon.set_border_width(0)
		fcon.show()
		fcon.pack_start(photoframe, True, False)

		surf_container.add(fcon)

	def clear_timeouts(self):
		self.timeouts = []
		self.chunk_timeouts = []
		self.playing = False
		self._lower_id.append(gobject.timeout_add(config.get('gviewer', 'sink_timer', 'int'), self._lower))

	def _clear_timeouts_seq(self, side):
		if self.sequential_id[side]:
			gobject.source_remove(self.sequential_id[side])
			self.sequential_id[side]  = 0


if __name__ == "__main__":
	import random, os
	from common import open_bottlecase
	from ghost import Ghost
	from ghostmanager import GhostManager
	from hypertextview import HyperTextView

	SCRIPT_PATTERNS = [
		unicode(r'\t\u\s[10]\h\s[5]みんな、\w5おはよう。\w9\w9\u５月２５日の今日の誕生石だ。\w9\w9\w9\h\_s\c\s[70]イーッ！\w9\w9\_s\_q\n\n\n\n　　　　　　　　　ダイアモンド\w1\c\n\n\n\n　　　　　　　　　　 ルビー\w1\c\n\n\n\n　　　　　　　　　　トパーズ\w1\c\n\n\n\n　　　　　　　　　　　翡翠\w1\c\n\n\n\n　　　　　　　　　　ジルコン\w1\c\n\n\n\n　　　　　　　　　 サファイア\w1\c\n\n\n\n　　　　　　　　　 ガーネット\w1\c\n\n\n\n　　　　　　　　 　アメジスト\w1\c\n\n\n\n　　　　　　　　　 クリスタル\w1\c\n\n\n\n　　　　　　　　　アクアマリン\w1\c\n\n\n\n　　　　　　　　　 エメラルド\w1\c\n\n\n\n　　　　　　　　　ダイアモンド\w1\c\n\n\n\n　　　　　　　　　　オパール\w1\c\n\n\n\n　　　　BTH 小っちゃいってことは便利だねっ\w1\c\n\n\n\n　　　　　　　　　　　真珠\w1\c\n\n\n\n　　　　　　　　　　トルコ石\w1\c\n\n\n\n　　　　　　　　 ムーンストーン\w1\c\n\n\n\n　　　　　　　　ブラッドストーン\w1\c\n\n\n\n　　　　　　　　　 ペリドット\w1\c\n\n\n\n　　　　　　　　　 トルマリン\w1\c\n\n\n\n　　　　　　　　　　　琥珀\w1\c\n\n\n\n　　　　　　　　　　クォーツ\w1\c\n\n\n\n　　　　　　　　　　プラチナ\w1\c\n\n\n\n　　　　　　　　 レッドジルコン\w1\c\n\n\n\n　　　　　　　　　　シルバー\w1\c\n\n\n\n　　　　　　　 　 ラピスラズリ\w9\w9\w9\u\n\n　　　　　　　　　 幸福を得る\w1\c\n\n　　　　　　　　　　　純粋\w1\c\n\n　　　　　　　　　　真実の愛\w1\c\n\n　　　　　　　　　  至福の時\w1\c\n\n　　　　　　　　　　　再開\w1\c\n\n　　　　　　　　　　夢の実現\w1\c\n\n　　　　　　　　　　　情熱\w1\c\n\n　　　　　　　　　　清浄無垢\w1\c\n\n　　　　　　　　　　　希望\w1\c\n\n　　　　　　　　　 夢見るころ\w1\c\n\n　　　　　スカートの中覗き放だ…ウベシッ！\w1\c\n\n　　　　　　　　　 慈愛・誠実\w1\c\n\n　　　　　　　　誰よりもやさしく\w1\c\n\n　　　　　　　　　 健康・長寿\w1\c\n\n　　　　　　　　　　恋の誘惑\w1\c\n\n　　　　　　　　　　　成功\w1\c\n\n　　　　　　　　　 夫婦の幸福\w1\c\n\n　　　　　　　　　　　聡明\w1\c\n\n　　　　　　　　私だけをみつめて\w1\c\n\n　　　　　　　　　　　幸運\w1\c\n\n　　　　　　　　　 永遠の誓い\w9\w9\w9\w9\_s\n\_q　　　　　　　　　　イーッ！\_s\w9\w9\w9\s[10]\c永遠はあるよ…\w9汁親父温泉にあるよ。\w9\w9\h\c\s[4]本当に永遠の世界に逝っちゃうよ…。\e', 'utf-8'),
		unicode(r'\t\u\s[10]\h\s[5]　それでは今日の誕生花です。\n\w9\w9\w9\c\n\n\n\n\s[9]　　　　　　ヒーリングサイバーッ！\w9\w9\w9\c\_q　　　　　　☆　　　　　　　★\n\n\n\n\n\n\n　　　　　　○　　　　　　　●\_q\w2\_q\c　　　　　　　　　　　　★\n　　　　　　☆\n\n\n\n\n　　　　　　　　　　　　　　●\n　　　　　　　○\_q\w2\_q\c\n　　　　　　　　　　　★\n\n　　　　　　　☆\n\n　　　　　　　　　　　　　●\n\n　　　　　　　　　○\n\_q\w2\_q\c\n\n　　　　　　　　　　★\n\n　　　　　　　　☆　　　●\n\n　　　　　　　　　　○\n\n\_q\w2\_q\c\n\n\n　　　　　　　　　★　●\n\n　　　　　　　　　☆　○\n\n\n\_q\w2\_q\c\n\n\n　　　　　　　　　　●○\n　　　　　　　　　　★☆\n\n\n\_q\w2\_q\c\n\n\n[half]　　　　　　　　　　○☆\n　　　　　　　　　　●★\n\n\n\_q\w2\_q\c\n\n　　　　　　　　　　☆★\n　　　　　　　　　　○●\n\n\n\_q\w2\_q\c\n\n[half]　　　　　　　　　　★●\n　　　　　　　　　　☆○\n\n\n\_q\w2\_q\c\n　　　　　　　　　　●○\n　　　　　　　　　　★☆\n\_q\w2\_q\c\n[half]　　　　　　　　　　○☆\n　　　　　　　　　　●★\n\_q\w2\_q\c　　　　　　　　　　☆★\n　　　　　　　　　　○●\n\_q\w2\_q\c　　　　　　　　　　★●\n　　　　　　　　　　☆○\n\_q\w2\_q\c　　　　　　　　　　●○\n　　　　　　　　　　★☆\n\_q\w2\_q\c　　　　　　　　　　○☆\n　　　　　　　　　　●★\n\_q\w2\_q\c　　　　　　　　　　☆★\n　　　　　　　　　　○●\n\_q\w2\_q\c　　　　　　　　　　☆★\n[half]　　　　　　　　　　○●\n\_q\w2\_q\c　　　　　　　　　　 ◎\_q\w9\w9\_q\c　　　　　　　　　　○○\n[half]　　　　　　　　　 ☆　☆\n\_q\w2\_q\c\n[half]　　　　　　　　　 ○　○\n　　　　　　　　 ☆　　　☆\n\_q\w2\_q\c\n　　　　　　　　  ○　　　○\n　　　　　　　 ☆ 　　　　　☆\n\_q\w2\_q\c\n\n[half]　　　　　　　 ○　　　　　○\n\n[half]　　　　　　 ☆　　　　　　　☆\n\_q\w2\_q\c\n\n　　　　　　　○ 　　　　　 ○\n\n　　　　　　☆ 　　　　　　　☆\n\_q\w2\_q\c\n\n\n[half]　　　　　　 ○　　　　　　○\n\n　　　　　 ☆　　　　　　　☆\n\_q\w2\_q\c\n\n\n　　　　　　　○  　　　　　○\n\n　　　　　　☆ 　　　　 　　　☆\n\_q\w2\_q\c\n\n\n　　　　　　　○  　　　　　○\n\n　　　　　　☆ 　　　　 　　　☆\n\_q\w9\w9\w9\_q\c\n\n\n　　　　　　　　○  　　　○\n\n　　　　　　　 ☆　　 　　　☆\n\_q\w2\_q\c\n\n\n　　　　　　　　　○  　○\n\n　　　　　　　　 ☆ 　 　☆\n\_q\w2\_q\c\n\n\n　　　　　　　　　　○○\n\n　　　　　　　　　　☆☆\n\_q\w2\_q\c\n\n\n　　　　　　　　　○,｡･:○\n\n　　　　　　　　　☆:･｡,☆\_q\w2\_q\c\n\n\n　　　　　　　　○★,｡･:*:○\n\n　　　　　　　　☆:*:･｡,★☆\_q\w2\_q\c\n\n\n　　　　　　　○*:･ﾟ★,｡･:*:･ﾟ○\n\n　　　　　　　☆ﾟ･:*:･｡,★ﾟ･:*☆\_q\w2\_q\c\n\n\n　　　　　　○･:*:･ﾟ★,｡･:*:･ﾟ☆○\n\n　　　　　　☆☆ﾟ･:*:･｡,★ﾟ･:*:･☆\_q\w2\_q\c\n\n\n　　　　　○　･:*:･ﾟ★,｡･:*:･ﾟ☆　○\n\n　　　　　☆　☆ﾟ･:*:･｡,★ﾟ･:*:･　☆\_q\w2\_q\c\n\n\n　　　　○　　･:*:･ﾟ★,｡･:*:･ﾟ☆　　○\n\n　　　　☆　　☆ﾟ･:*:･｡,★ﾟ･:*:･　　☆\_q\w2\_q\c\n\n\n　　　○　　　･:*:･ﾟ★,｡･:*:･ﾟ☆　　　○\n\n　　　☆　　　☆ﾟ･:*:･｡,★ﾟ･:*:･　　　☆\_q\w2\_q\c\n\n\n　　○　　　　･:*:･ﾟ★,｡･:*:･ﾟ☆　　　　○\n\n　　☆　　　　☆ﾟ･:*:･｡,★ﾟ･:*:･　　　　☆\_q\w9\s[5]\_q\c　　　　　　　　　えいっ♪\n\n\n　　○　　　　･:*:･ﾟ★,｡･:*:･ﾟ☆　　　　○\n\n　　☆　　　　☆ﾟ･:*:･｡,★ﾟ･:*:･　　　　☆\_q\w9\w9\_q\c　　　　　　　　　えいっ♪\n\n　　　　 〜〜〜〜今日の誕生花〜〜〜〜\n　　○　　　 『デルフィニュウム』 　　　○\n\n　　☆　 『人の心を読む、愛の伝達者』 　☆\_q\w9\w9\w9\n\n　　　　　　　　こんな感じです。\n\w9\w9\n[half]\s[105]　　皆様、\w9今日も一日張り切っていきましょうね。\n\e', 'utf-8'),
		unicode(r'\t\u\s[10]\h\s[5]みんな、\w5おはよう。\w9\w9\u理夢とヨンヨンの今日の誕生石だ。\w9\w9\h\n\n今日の誕生石はこれ！\w9\w9\w9\_s\c\s[70]イーッ！\w9\w9\_s\_q\n\n\n\n　　　　　　　　　ダイアモンド\w1\c\n\n\n\n　　　　　　　　　　 ルビー\w1\c\n\n\n\n　　　　　　　　　　トパーズ\w1\c\n\n\n\n　　　　　　　　　　　翡翠\w1\c\n\n\n\n　　　　　　　　　　ジルコン\w1\c\n\n\n\n　　　　　　　　　 サファイア\w1\c\n\n\n\n　　　　　　　　　 ガーネット\w1\c\n\n\n\n　　　　　　　　 　アメジスト\w1\c\n\n\n\n　　　　　　　　　 クリスタル\w1\c\n\n\n\n　　　　　　　　　アクアマリン\w1\c\n\n\n\n　　　　　　　　　 エメラルド\w1\c\n\n\n\n　　　　　　　　　ダイアモンド\w1\c\n\n\n\n　　　　　　　　　　オパール\w1\c\n\n\n\n　　　　　　　シュークリーム分が足りない\w1\c\n\n\n\n　　　　　　　　　　　真珠\w1\c\n\n\n\n　　　　　　　　　　トルコ石\w1\c\n\n\n\n　　　　　　　　 ムーンストーン\w1\c\n\n\n\n　　　　　　　　ブラッドストーン\w1\c\n\n\n\n　　　　　　　　　 ペリドット\w1\c\n\n\n\n　　　　　　　　　 トルマリン\w1\c\n\n\n\n　　　　　　　　　　　琥珀\w1\c\n\n\n\n　　　　　　　　　　クォーツ\w1\c\n\n\n\n　　　　　　　　　　プラチナ\w1\c\n\n\n\n　　　　　　　　 レッドジルコン\w1\c\n\n\n\n　　　　　　　　　　シルバー\w1\c\n\n\n\n　　　　　　　 　 ダイアモンド\w9\w9\w9\u\n\n　　　　　　　　　 幸福を得る\w1\c\n\n　　　　　　　　　　　純粋\w1\c\n\n　　　　　　　　　　真実の愛\w1\c\n\n　　　　　　　　　  至福の時\w1\c\n\n　　　　　　　　　　　再開\w1\c\n\n　　　　　　　　　　夢の実現\w1\c\n\n　　　　　　　　　　　情熱\w1\c\n\n　　　　ならば代わりにカレーを食べなさい！\w1\c\n\n　　　　　　　　　　　希望\w1\c\n\n　　　　　　　　　 夢見るころ\w1\c\n\n　　　　　　　　　  清浄無垢\w1\c\n\n　　　　　　　　　 慈愛・誠実\w1\c\n\n　　　　　　　　誰よりもやさしく\w1\c\n\n　　　　　　　　　 健康・長寿\w1\c\n\n　　　　　　　　　　恋の誘惑\w1\c\n\n　　　　　　　　　　　成功\w1\c\n\n　　　　　　　　　 夫婦の幸福\w1\c\n\n　　　　　　　　　　　聡明\w1\c\n\n　　　　　　　　私だけをみつめて\w1\c\n\n　　　　　　　　　 永遠の誓い\w1\c\n\n　　　　　　　　　  清浄無垢\w9\w9\w9\w9\_s\n\_q　　　　　　　　　　イーッ！\_s\w9\w9\w9\w9\u\s[10]\c以上、\w5今日の誕生石だ。\w9\w9\h\s[5]\cそれじゃあ、\w5楽しいボトルを！\w9\w9\s[1]\n今日も一日ドキドキだね。\e', 'utf-8'),
		unicode(r'\t\h\s[0]\u\s[10]\_q（カチャ）\h　　　　　　　　　　　■\n　　　　　　　　　　　■\n　　　　　　　　■■■■■■■\n　　　　　　　　　　　■\n　　　　　　　　　　　■\n　　　　　　　　　　　■\n　　　　　　　　　　　■\w4\u（カチャ）\h\c　　　　　　　　　■■■■■\n\n\n\n\n　　　　　　　■■■■■■■■■\w4\u（カチャ）\h\c　　　　　　　■■■■■■■■■\n[half]　　　　　　　■　　　　　　　■\n[half]　　　　　　　■　　　　　　　■\n[half]　　　　　　　■　■■■■■　■\n[half]　　　　　　　■　　　■　　　■\n[half]　　　　　　　■　　　■　　　■\n[half]　　　　　　　■　　■■■　　■\n[half]　　　　　　　■　　　■　　　■\n[half]　　　　　　　■　　　■　■　■\n[half]　　　　　　　■　　　■　　　■\n[half]　　　　　　　■　■■■■■　■\n[half]　　　　　　　■　　　　　　　■\n[half]　　　　　　　■　　　　　　　■\n[half]　　　　　　　■■■■■■■■■\w4\u（カチャ）\h\c　　　　　　　　　■　　■■■■\n[half]　　　　　　　　　■　　　　　■\n[half]　　　　　　　　　　　　　　　■\n[half]　　　　　　　■■■■■　　　■\n[half]　　　　　　　　　　　　　　　■\n[half]　　　　　　　　■■■　　　　■\n[half]　　　　　　　　　　　　■■■■\n[half]　　　　　　　　■■■　■\n[half]　　　　　　　　　　　　■\n[half]　　　　　　　 ■■■■ ■\n[half]　　　　　　　 ■　　■ ■\n[half]　　　　　　　 ■　　■ ■\n[half]　　　　　　　 ■　　■ ■　　　■\n[half]　　　　　　　 ■■■■ 　■■■\w4\c\u\c\_qチャラリララ―――\w9\w9\h\_q\n\n　　　　　　　　　　十二国記\n\n　　　　　　　月の影　影の海　八章\n\n―――――――（ＮＨＫ教育：開始）――――――\_q\uチャラ\w4チャラ\w4チャラ\w4チャラ\w2チャチャチャン！\e', 'utf-8'),
		unicode(r'\t\h\s[0]\u\s[10]\_q（カチャ）\h　　　　　　　　　　　■\n　　　　　　　　　　　■\n　　　　　　　　　　　■\n　　　　　　　　　　　■■\n　　　　　　　　　　　■　■\n　　　　　　　　　　　■　　■\n　　　　　　　　　　　■\n　　　　　　　　　　　■\n　　　　　　　　　　　■\w8\u（カチャ）\h\c　　　　　　　　　　　　　　■\n　　　　　　　　　■　　　　■\n　　　　　　　　　■　　　　■\n　　　　　　　　　■　　　　■\n　　　　　　　　　　　　　　■\n　　　　　　　　　　　　　 ■\n　　　　　　　　　　　　　■\n　　　　　　　　　　　　 ■\n　　　　　　　　　　　　■\w4\u（カチャ）\h\c　　　　　　　　　■　　　　■　■\n　　　　　　　　　■　　　　 ■　■\n　　　　　　　　　■\n[half]　　　　　　　　　　　　　■■\n[half]　　　　　　　　　■■■■\n　　　　　　　　　■\n　　　　　　　　　■\n　　　　　　　　　■\n　　　　　　　　　■\n　　　　　　　　　　■■■■■■\w4\u（カチャ）\h\c　　　　　　　　　■■■■■■■\n[half]\n[half]　　　　　　　　　　　　　　■\n[half]　　　　　　　　　　　 ■\n[half]　　　　　　　　　　　　■■\n　　　　　　　　　　　　■\n　　　　　　　　　　　　■\n　　　　　　　　　　　　■\n　　　　　　　　　　　　■\n　　　　　　　　　　　 ■\n　　　　　　　　　　　■\w4\u（カチャ）\h\c\n\n　　　　　　　　　　■■■\n[half]　　　　　　　 　 ■　■　■\n[half]　　　　　　 　　■ 　■　 ■\n[half]　　　　　　　　■　　■　　■\n[half]　　　　　 　　■ 　　■　　■\n[half]　　　 　　　　■ 　　■　　■\n[half]　　　　　　　　■　 ■　　 ■\n[half]　　　　　　　　 ■ ■　　 ■\n[half]　　　　　　　　　 ■　　 ■\w4\u（カチャ）\h\c　　　　　　　　　　　■\n[half]　　　　　　　　　　■\n[half]　　　　　　　　　■\n[half]　　　　　　　　■■■■■■■\n[half]　　　　　　　　■　　　　　■\n[half]　　　　　　　　■■■■■■■\n[half]　　　　　　　　■　　　　　■\n[half]　　　　　　　　■■■■■■■\n[half]\n[half]　　　　　　　　　　　■　　　■\n[half]　　　　　　　　　　　■　　■\n[half]　　　　　　　■■■■■■■\n[half]　　　　　　　　　　■■■\n[half]　　　　　　　　　 ■ ■ ■\n[half]　　　　　　　　　■　■　■\n[half]　　　　　　　　 ■ 　■　 ■\n[half]　　　　　　　 ■ 　　■　　 ■\n[half]　　　　　　 ■ 　　　■　　　 ■\w4\c\u\c\_qチャラリララ―――\w9\w9\h\_q\n\n　　　　　　　　　トリビアの泉\n\n　　　　　　　 素晴らしきムダ知識\n\n―――――――（フジテレビ：開始）――――――\_q\uチャラ\w4チャラ\w4チャラ\w4チャラ\w2チャチャチャン！\e', 'utf-8'),
		]
	SENDER_PATTERNS = [
		unicode(r'GBottler', 'utf-8'),
		unicode(r'SSTP Bottle Client', 'utf-8'),
		]
	CHANNEL_PATTERNS = [
		unicode(r'更新情報', 'utf-8'),
		unicode(r'駅前繁華街', 'utf-8'),
		unicode(r'海浜公園街', 'utf-8'),
		unicode(r'学園街', 'utf-8'),
		unicode(r'山の温泉街', 'utf-8'),
		]
	GHOST_PATTERNS = [
		unicode(r'理夢', 'utf-8'),
		unicode(r'せりこ', 'utf-8'),
		unicode(r'毒子', 'utf-8'),
		unicode(r'なる', 'utf-8'),
		unicode(r'ねここ', 'utf-8'),
		unicode(r'猫', 'utf-8'),
		]

	# sender, channel and ghost:
	# they must be gtk.Label
	s_l = gtk.Label()
	s_l.show()
	cg_l = gtk.Label()
	cg_l.show()
	r_l = gtk.Label()
	r_l.show()
	i_l = gtk.Label(' Items')
	i_l.show()
	scbox = gtk.HBox()
	scbox.show()
	scbox.pack_start(s_l)
	scbox.pack_start(cg_l)
	scbox.pack_start(r_l)
	scbox.pack_start(i_l)

	# sakura side
	sakura_frame = gtk.Frame()
	sakura_frame.set_shadow_type(gtk.SHADOW_NONE)
	sakura_frame.set_size_request(SIZE_SURFACE_H, SIZE_SURFACE_V)
	sakura_frame.show()

	t1 = HyperTextView()
	t1.show()
	t1.set_wrap_mode(gtk.WRAP_CHAR)
	t1.set_editable(False)
	t1.set_cursor_visible(False)
	sw1 = gtk.ScrolledWindow()
	sw1.show()
	sw1.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_ALWAYS)
	sw1.add(t1)
	f1 = gtk.Frame()
	f1.show()
	f1.set_shadow_type(gtk.SHADOW_IN)
	f1.add(sw1)

	sakurabox = gtk.HBox()
	sakurabox.show()
	sakurabox.pack_start(sakura_frame, False, False, 0)
	sakurabox.pack_start(f1)
	sakurabox.set_size_request(100, SIZE_SURFACE_V * 2 + 20)

	# kero side
	kero_frame = gtk.Frame()
	kero_frame.set_shadow_type(gtk.SHADOW_NONE)
	kero_frame.set_size_request(SIZE_SURFACE_H, SIZE_SURFACE_V)
	kero_frame.show()

	t2 = HyperTextView()
	t2.show()
	t2.set_wrap_mode(gtk.WRAP_CHAR)
	t2.set_editable(False)
	t2.set_cursor_visible(False)
	sw2 = gtk.ScrolledWindow()
	sw2.show()
	sw2.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_ALWAYS)
	sw2.add(t2)
	f2 = gtk.Frame()
	f2.show()
	f2.set_shadow_type(gtk.SHADOW_IN)
	f2.add(sw2)

	kerobox = gtk.HBox()
	kerobox.show()
	kerobox.pack_start(kero_frame, False, False, 0)
	kerobox.pack_start(f2)
	kerobox.set_size_request(100, SIZE_SURFACE_V + 20)

	# play button
	pbutton = gtk.Button(unicode('Test play', 'utf-8'))
	pbutton.show()

	# cancel button
	ibutton = gtk.Button(unicode('Cancel', 'utf-8'))
	ibutton.show()

	# cancel button
	cbutton = gtk.Button(unicode('Cancel all', 'utf-8'))
	cbutton.show()

	# buttons
	bbox = gtk.HBox()
	bbox.show()
	bbox.pack_start(pbutton)
	bbox.pack_start(ibutton)
	bbox.pack_start(cbutton)

	# all
	box = gtk.VBox()
	box.show()
	box.pack_start(scbox)
	box.pack_start(sakurabox)
	box.pack_start(kerobox)
	box.pack_start(bbox)

	# window
	w = gtk.Window()
	w.set_size_request(400, SIZE_SURFACE_V * 2 + 140)
	w.show()
	w.connect('destroy', lambda x: gtk.main_quit())
	w.add(box)

	gv = GViewerQueue([sakura_frame, t1, sw1],
					 [kero_frame, t2, sw2], w, s_l, cg_l, r_l)
	gm = GhostManager(os.path.join(open_bottlecase(), 'gviewer', 'ghost.txt'))

	pbutton.connect('clicked', lambda x: gv.enqueue(random.choice(SCRIPT_PATTERNS),
													gm.nominate_ghost(random.choice(GHOST_PATTERNS)),
													random.choice(SENDER_PATTERNS),
													random.choice(CHANNEL_PATTERNS)))
	ibutton.connect('clicked', lambda x: gv.cancel())
	cbutton.connect('clicked', lambda x: gv.cancel_all())

	gtk.main()
