# -*- coding: utf-8 -*-
#
#  queryluid.py - Query LUID information
#  Copyright (C) 2004 by Atzm WATANABE <atzm@atzm.org>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: queryluid.py,v 1.13 2010/08/28 15:12:17 atzm Exp $
#

import sys, os
import urllib
import htmllib, formatter

import config
from common import *

class LUIDParser(htmllib.HTMLParser):
	def __init__(self):
		htmllib.HTMLParser.__init__(self, formatter.NullFormatter(), 0)

		self.th = False
		self.td = False
		self.tdcount = 0
		self.tddataflag = False

		self.dt = False
		self.dd = False
		self.dtbuf = ''
		self.headres = []
		self.result = []

	def start_dt(self, attrs): self.dt = True
	def end_dt(self):          self.dt = False
	def start_dd(self, attrs): self.dd = True
	def end_dd(self):          self.dd = False
	def start_th(self, attrs): self.th = True
	def end_th(self):          self.th = False
	def start_td(self, attrs): self.td = True
	def end_td(self):
		if not self.tddataflag:
			try:
				self.result[self.tdcount].append('')
			except IndexError:
				self.tdcount = 0
				self.result[self.tdcount].append('')

		self.td = False
		self.tdcount += 1
		self.tddataflag = False

	def handle_data(self, text):
		if self.th:
			self.result.append([text])
		elif self.td:
			self.tddataflag = True
			try:
				self.result[self.tdcount].append(text)
			except IndexError:
				self.tdcount = 0
				self.result[self.tdcount].append(text)

		elif self.dt:
			self.dtbuf += text
		elif self.dd and self.dtbuf:
			self.headres.append([self.dtbuf, text])
			self.dtbuf = ''

	def end_html(self):
		self.format()

	def format(self):
		length = len(self.result[0])
		formatted = [[] for i in xrange(length)]
		for line in self.result:
			if line[1:]:
				for i in xrange(length):
					formatted[i].append(line[i])
		self.result = formatted

		length = len(self.headres[0])
		formatted = [[] for i in xrange(length)]
		for line in self.headres:
			if line[1:]:
				for i in xrange(length):
					formatted[i].append(line[i])
		self.headres = formatted

	def initialize(self):
		self.th = False
		self.td = False
		self.tdcount = 0
		self.tddataflag = False

		self.dt = False
		self.dd = False
		self.dtbuf = ''
		self.headres = []
		self.result = []

class LUIDInfoURLopener(urllib.URLopener):
	version = AGENT_NAME

class LUIDInfo:
	LUID_INFO_GENERAL = 'info'
	LUID_INFO_DETAIL  = 'table'

	def __init__(self, luid):
		self.luid   = luid
		self.parser = LUIDParser()
		self.cache  = LUIDCacheManager()
		self.url    = 'http://bottle.mikage.to/luidstat.cgi'

	def set_luid(self, luid):
		self.luid = luid
	def get_luid(self):
		return self.luid

	def get_html(self):
		proxy = config.get('http_server', 'http_proxy').strip()

		if proxy:
			proxy = {'http': proxy}
		else:
			proxy = None

		try:
			req = self.url + '?luid=' + self.get_luid()
			fil = LUIDInfoURLopener(proxies=proxy).open(req)
			code = fil.getcode()
			data = fil.read()
			fil.close()
		except:
			return

		if code != 200:
			return

		return unicode(data, 'sjis', 'ignore')

	def parse(self):
		html = self.get_html()
		if html:
			self.parser.initialize()
			self.parser.feed(html)
			luid_data = {
				self.LUID_INFO_GENERAL:  self.parser.headres,
				self.LUID_INFO_DETAIL: self.parser.result,
				}
			self.cache.set_luid_data(luid_data)
			self.cache.save_database()

	def get_all(self, refresh=False):
		if refresh and self.luid:
			self.parse()
		return self.cache.get_luid_data()

	def get_info(self, refresh=False):
		if refresh:
			self.parse()
		return self.cache.get_luid_data(self.LUID_INFO_GENERAL)

	def get_table(self, refresh=False):
		if refresh:
			self.parse()
		return self.cache.get_luid_data(self.LUID_INFO_DETAIL)

class LUIDCacheManager:
	DBFILE = "luid_cache"

	def __init__(self):
		self.dbpath = os.path.join(open_bottlecase(), self.DBFILE)
		self.luid_data = {LUIDInfo.LUID_INFO_GENERAL: [],
						  LUIDInfo.LUID_INFO_DETAIL:  []}
		self.load_database()

	def set_luid_data(self, luid_data):
		self.luid_data = luid_data

	def get_luid_data(self, key=None):
		if key == None:
			return self.luid_data
		try:
			return self.luid_data[key]
		except KeyError:
			return []

	def load_database(self):
		try:
			file = open(self.dbpath)
		except IOError:
			return
		current = None
		while 1:
			line = unicode(file.readline(), "utf-8")
			if not line:
				break
			elif line[0] in ["#", ";"]:
				continue
			elif line[0:2] == "//":
				continue
			elif line[-2:] == "\r\n":
				line = line[:-2]
			elif line[-1] in ["\r", "\n"]:
				line = line[:-1]

			if not line:
				continue
			elif line[0] == '[' and line[-1] == ']':
				current = line[1:-1]
				self.luid_data[current] = []
				continue

			line = line.split("\t")
			self.luid_data[current].append(line)
		file.close()

	def save_database(self):
		try:
			file = open(self.dbpath, "w")
		except IOError:
			sys.stderr.write("Error: cannot write %s\n" % self.dbpath)
			return
		file.write("# This is an auto-generated file.\n")
		file.write("# Don't edit this file while running %s!\n" % APP)
		for key in self.luid_data.keys():
			file.write("[%s]\n" % key)
			for line in self.luid_data[key]:
				for i in xrange(len(line)):
					file.write(line[i].encode('utf-8'))
					if i == len(line)-1:
						file.write("\n")
					else:
						file.write("\t")
			file.write("\n")
		file.close()
