# -*- coding: utf-8 -*-
#
#  logwindow.py - Log window for GBottler
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2004-2010 by Atzm WATANABE <atzm@atzm.org>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: logwindow.py,v 1.40 2010/08/22 11:04:07 atzm Exp $
#

import gtk, gobject
import os, time

from logfetchdialog import LogFetchDialog
from textmanager    import TextManager
from logbook        import LogBook
from loglist        import LogList
from logmanager     import XMLLogWriter, XMLLogOpener, LogFetchOperator

from common import *
import config

class LogWindow:
	MENU_STRING = '''
<menubar name="MenuBar">
    <menu name="File" action="File">
        <menuitem name="Open" action="Open"/>
        <menuitem name="Save" action="Save"/>
        <menuitem name="Download" action="Download"/>
        <separator/>
        <menuitem name="Close" action="Close"/>
    </menu>
    <menu name="Edit" action="Edit">
        <menuitem name="CopyTime" action="CopyTime"/>
        <menuitem name="CopyGhost" action="CopyGhost"/>
        <menuitem name="CopyChannel" action="CopyChannel"/>
        <menuitem name="CopyScript" action="CopyScript"/>
        <menuitem name="CopyAll" action="CopyAll"/>
        <separator/>
        <menuitem name="Search" action="Search"/>
        <separator/>
        <menuitem name="HideSystemMessage" action="HideSystemMessage"/>
        <menuitem name="Preferences" action="Preferences"/>
    </menu>
    <menu name="Help" action="Help" position="bot">
        <menuitem name="About" action="About"/>
    </menu>
</menubar>
'''
	POPUP_MENU_STRING = '''
<menubar name="PopupMenuBar">
    <menu name="MainMenu" action="MainMenu">
        <menuitem name="Play" action="Play"/>
        <menuitem name="Volley" action="Volley"/>
        <separator/>
        <menu name="Cancel" action="Cancel">
            <menuitem name="CancelPlay" action="CancelPlay"/>
            <menuitem name="CanselPage" action="CancelPage"/>
            <menuitem name="CancelAll" action="CancelAll"/>
        </menu>
        <separator/>
        <menuitem name="Vote" action="Vote"/>
        <menuitem name="Agree" action="Agree"/>
        <separator/>
        <menuitem name="CopyTime" action="CopyTime"/>
        <menuitem name="CopyGhost" action="CopyGhost"/>
        <menuitem name="CopyChannel" action="CopyChannel"/>
        <menuitem name="CopyScript" action="CopyScript"/>
        <menuitem name="CopyAll" action="CopyAll"/>
    </menu>
</menubar>
	'''

	def __init__(self, app):
# === Definition of Basic Const === #
		self.app = app
		self.volley_canceler = False

# === Create Menubar === #
		self.uimanager = gtk.UIManager()
		self.accel_group = self.uimanager.get_accel_group()
		self.action_group = gtk.ActionGroup('GBottler')
		self.action_group.add_actions([
            ('MenuBar', None, _('_MenuBar')),
            ('File', None, _('_File')),
            ('Open', gtk.STOCK_OPEN, _('_Open...'), '<control>o', _('Open XML log'), self.open_xml_log),
            ('Save', gtk.STOCK_SAVE_AS, _('_Save as...'), '<control>s', _('Save log as XML'), self.save_as_xml),
            ('Download', gtk.STOCK_CONVERT, _('_Download...'), None, _('Download log from bottle server'), self.fetch),
            ('Close', gtk.STOCK_CLOSE, _('_Close'), None, _('Close log window'), self.close),
            ('Edit', None, _('_Edit')),
            ('CopyTime', None, _('Copy time'), None, _('Copy time from selected bottle'), self.edit_copy_time),
            ('CopyGhost', None, _('Copy ghostname'), None, _('Copy ghost name from selected bottle'), self.edit_copy_ghostname),
            ('CopyChannel', None, _('Copy channel'), None, _('Copy channel name from selected bottle'), self.edit_copy_channel),
            ('CopyScript', gtk.STOCK_COPY, _('Copy script'), '<control>c', _('Copy script from selected bottle'), self.edit_copy_script),
            ('CopyAll', None, _('Copy all data'), None, _('Copy all data from selected bottle'), self.edit_copy_all),
            ('Search', gtk.STOCK_FIND, _('Search'), '<control>f', _('Search'), lambda x: self.search_entry.grab_focus()),
            ('Preferences', gtk.STOCK_PREFERENCES, _('_Preferences...'), '<control>p', _('Edit preferences'), self.app.open_preference),
            ('Help', None, _('_Help')),
            ('About', gtk.STOCK_DIALOG_INFO, _('About'), None, _('About GBottler'), self.app.about),
            ])
		self.action_group.add_toggle_actions([
			('HideSystemMessage', None, _('Hide system message'), None, _('Hide system message'),
			 lambda x: self.msgframe.set_property('visible', not x.get_active()))
			])
		self.uimanager.insert_action_group(self.action_group, 0)
		self.ui_merge_id = self.uimanager.add_ui_from_string(self.MENU_STRING)

		main_menubar = self.uimanager.get_widget('/MenuBar')
		main_menubar.show()

# === Create Base Window === #
		self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
		self.window.add_accel_group(self.accel_group)
		self.window.connect("delete_event", self.close)
		self.window.set_title('%s: %s' % (APP, _('Logs')))
		self.window.set_default_size(config.get('logwindow', 'width', 'int'),
									 config.get('logwindow', 'height', 'int'))

# === Create Entry for Search === #
		self.search_entry = gtk.Entry()
		self.search_entry.show()
		self.search_entry.set_text(_('Search'))
		self.search_entry.connect("activate", self.search)
		self.search_entry.connect("key-press-event", self.entry_keypress)
		self.search_entry.connect("focus-in-event", self.set_entry_guide, 'in')
		self.search_entry.connect("focus-out-event", self.set_entry_guide, 'out')

		self.s_arrow = gtk.Arrow(gtk.ARROW_DOWN, gtk.SHADOW_NONE)
		self.s_arrow.show()

		self.s_a_b = gtk.ToggleButton()
		self.s_a_b.show()
		self.s_a_b.add(self.s_arrow)
		self.s_a_b.connect('clicked', lambda x: self.s_arrow.set((not self.s_arrow.get_property('arrow-type')), gtk.SHADOW_NONE))
		self.s_a_b.connect('focus-in-event', lambda x, y: self.search_entry.grab_focus())

		self.sbar = gtk.Statusbar()
		self.sbar.show()
		self.sbar.set_has_resize_grip(True)

		self.rowsbar = gtk.Statusbar()
		self.rowsbar.show()
		self.rowsbar.set_has_resize_grip(False)

		self.curr_rowsbar = gtk.Statusbar()
		self.curr_rowsbar.show()
		self.curr_rowsbar.set_has_resize_grip(False)

		search_hbox = gtk.HBox()
		search_hbox.show()
		search_hbox.pack_start(self.s_a_b, False, False, 0)
		search_hbox.pack_start(self.search_entry)
		search_hbox.pack_start(self.curr_rowsbar)
		search_hbox.pack_start(self.rowsbar)
		search_hbox.pack_start(self.sbar)

# === Create MessageBar === #
		btn = get_icon_button(gtk.STOCK_CLEAR, size=gtk.ICON_SIZE_MENU,
							  relief=gtk.RELIEF_NONE)
		btn.show()
		btn.connect('clicked', lambda x: self.clear_message())
		btn.connect('focus-in-event', self.focus_active_loglist)

		self.messagebar = gtk.Label()
		self.messagebar.show()
		msgbox = gtk.HBox()
		msgbox.show()
		msgbox.pack_start(self.messagebar)
		msgbox.pack_start(btn, False)

		self.msgframe = gtk.Frame()
		self.msgframe.show()
		self.msgframe.set_shadow_type(gtk.SHADOW_ETCHED_IN)
		self.msgframe.add(msgbox)

		bottombox = gtk.VBox()
		bottombox.show()
		bottombox.pack_start(self.msgframe)
		bottombox.pack_start(search_hbox)

# === Create Popup Menu for LogList === #
		self.popup_uimanager = gtk.UIManager()
		self.popup_accel_group = self.popup_uimanager.get_accel_group()
		self.popup_action_group = gtk.ActionGroup('GBottler')
		self.popup_action_group.add_actions([
            ('PopupMenuBar', None, _('PopupMenuBar')),
            ('MainMenu', None, _('MainMenu')),
            ('Play', STATE_BOTTLE_PLAYING, _('Play'), '<control>p', _('Play this bottle'), self.play),
            ('Volley', gtk.STOCK_MEDIA_FORWARD, _('Volley'), None, _('Volley'), self.volley),
            ('Cancel', None, _('Cancel')),
            ('CancelPlay', gtk.STOCK_MEDIA_STOP, _('This bottle'), None, _('Cancel this bottle'), self.cancel_play),
            ('CancelPage', gtk.STOCK_MEDIA_STOP, _('This page'), None, _('Cancel this page'), self.cancel_play_page),
            ('CancelAll', gtk.STOCK_MEDIA_STOP, _('All'), None, _('Cancel all bottle'), self.cancel_play_all),
            ('Vote', gtk.STOCK_YES, _('Vote'), '<control><alt>v', _('Vote this bottle'), self.voting),
            ('Agree', gtk.STOCK_NO, _('Agree'), '<control><alt>a', _('Agree this bottle'), self.agreeing),
            ('CopyTime', None, _('Copy time'), None, _('Copy time from selected bottle'), self.edit_copy_time),
            ('CopyGhost', None, _('Copy ghostname'), None, _('Copy ghost name from selected bottle'), self.edit_copy_ghostname),
            ('CopyChannel', None, _('Copy channel'), None, _('Copy channel name from selected bottle'), self.edit_copy_channel),
            ('CopyScript', gtk.STOCK_COPY, _('Copy script'), '<control>c', _('Copy script from selected bottle'), self.edit_copy_script),
            ('CopyAll', None, _('Copy all data'), None, _('Copy all data from selected bottle'), self.edit_copy_all),
            ])
		self.popup_uimanager.insert_action_group(self.popup_action_group, 0)
		self.popup_ui_merge_id = self.popup_uimanager.add_ui_from_string(self.POPUP_MENU_STRING)
		self.window.add_accel_group(self.popup_accel_group)

		self.popup_menu = self.popup_uimanager.get_widget('/PopupMenuBar/MainMenu').get_submenu()

		# keep MenuItems, vote and agree
		self.vi = self.popup_uimanager.get_widget('/PopupMenuBar/MainMenu/Vote')
		self.ai = self.popup_uimanager.get_widget('/PopupMenuBar/MainMenu/Agree')

# === Create Log Viewer === #
		self.textmanager = TextManager(self)
		self.textmanager.set_style_talk()
		self.textmanager.set_cursor_visible(False)
		self.textmanager.set_editable(False)
		self.textmanager.set_border_width(0)
		self.textmanager.connect("focus-in-event", self.focus_active_loglist)
		self.textmanager.show()

		self.sw2 = gtk.ScrolledWindow()
		self.sw2.show()
		self.sw2.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		self.sw2.set_border_width(0)
		self.sw2.add(self.textmanager)

		text_frame = gtk.Frame()
		text_frame.show()
		text_frame.set_shadow_type(gtk.SHADOW_IN)
		text_frame.set_border_width(0)
		text_frame.add(self.sw2)

# === Create Buttons === #
		fetch_button = get_icon_button(gtk.STOCK_CONVERT, gtk.ICON_SIZE_MENU)
		open_button  = get_icon_button(gtk.STOCK_OPEN,    gtk.ICON_SIZE_MENU)
		save_button  = get_icon_button(gtk.STOCK_SAVE_AS, gtk.ICON_SIZE_MENU)

		self.pb = get_icon_button(STATE_BOTTLE_PLAYING, gtk.ICON_SIZE_MENU)
		self.vb = get_icon_button(gtk.STOCK_YES,     gtk.ICON_SIZE_MENU)
		self.ab = get_icon_button(gtk.STOCK_NO,      gtk.ICON_SIZE_MENU)

		fetch_button.connect("clicked", self.fetch)
		open_button.connect("clicked", self.open_xml_log)
		save_button.connect("clicked", self.save_as_xml)

		self.pb.connect("clicked", self.play)
		self.vb.connect("clicked", self.voting)
		self.ab.connect("clicked", self.agreeing)

		self.vb.set_sensitive(False)
		self.ab.set_sensitive(False)

		for item in [fetch_button, open_button, save_button,
					 self.pb, self.vb, self.ab]:
			item.connect("focus-in-event", self.focus_active_loglist)

		self.tooltips = gtk.Tooltips()
		self.tooltips.set_tip(fetch_button, _('Download log from bottle server'))
		self.tooltips.set_tip(open_button, _('Open XML log'))
		self.tooltips.set_tip(save_button, _('Save log as XML'))

		self.tooltips.set_tip(self.pb, _("Play"))
		self.tooltips.set_tip(self.vb, _("Vote"))
		self.tooltips.set_tip(self.ab, _("Agree"))

# === Create Labels and style === #
		self.style_combo = gtk.combo_box_new_text()
		for text in [_('Talk style'), _('Script style'), _('Script style with linefeed')]:
			self.style_combo.append_text(text)
		self.style_combo.set_active(STYLE_TALK)
		self.style_combo.connect('changed', self.set_style)
		self.style_combo.show()

# === Bring Buttons and Labels together to HBox, Frame === #
		toolbox = gtk.HBox()
		toolbox.show()

		toolbox.pack_start(fetch_button, False, True, 0)
		toolbox.pack_start(open_button, False, True, 0)
		toolbox.pack_start(save_button, False, True, 0)
		vs = gtk.VSeparator()
		vs.show()
		toolbox.pack_start(vs, False, True, 3)

		toolbox.pack_start(self.pb, False, True, 0)
		toolbox.pack_start(self.vb, False, True, 0)
		toolbox.pack_start(self.ab, False, True, 0)
		vs = gtk.VSeparator()
		vs.show()
		toolbox.pack_start(vs, False, True, 3)

		#toolbox.pack_start(style_desc, False, True, 0)
		toolbox.pack_start(self.style_combo, False, True, 0)

		button_frame = gtk.Frame()
		button_frame.show()
		button_frame.set_border_width(0)
		button_frame.set_shadow_type(gtk.SHADOW_OUT)
		button_frame.add(toolbox)

# === Create Notebook === #
		label = _('Current')
		self.logbook = LogBook(label)
		self.tab_pos_changed()
		self.logbook.connect('switch-page', self.tab_changed)

		self.create_tab(label)

# === Bring Log Entry and Log Viewer together to VPaned === #
		self.main_paned = gtk.VPaned()
		self.main_paned.show()
		self.main_paned.pack1(self.logbook, True)
		self.main_paned.pack2(text_frame, False)
		self.main_paned.set_position(config.get('logwindow', 'paned_position', 'int'))

# === Bring all items together to Main Window Finally === #
		main_vbox = gtk.VBox(False, 0)
		main_vbox.show()

		main_vbox.pack_start(main_menubar, False, True, 0)
		main_vbox.pack_start(button_frame, False, True, 0)
		main_vbox.pack_start(self.main_paned)
		main_vbox.pack_start(bottombox, False, False, 0)

		self.window.add(main_vbox)

# === Initialization of LogWindow === #
		button = self.uimanager.get_widget('/MenuBar/Edit/HideSystemMessage')
		button.set_active(config.get('logwindow', 'hide_system_message', 'boolean'))
		self.focus_active_loglist()

	def open(self, widget=None, *args):
		self.window.show()

	def close(self, widget=None, *args):
		self.window.hide()
		return True

	def join(self):
		for item in [self.vb, self.ab, self.vi, self.ai]:
			item.set_sensitive(True)

	def part(self):
		for item in [self.vb, self.ab, self.vi, self.ai]:
			item.set_sensitive(False)

	def set_message(self, text):
		self.messagebar.set_text(text)

	def add_message(self, text):
		self.messagebar.set_text(self.messagebar.get_text() + text)

	def clear_message(self):
		self.messagebar.set_text('')

	def autologging(self):
		list = self.logbook.get_current_loglist()
		if not list:
			return
		filename = time.strftime(config.get('logging', 'filename'), time.localtime(time.time()))

		dir = os.path.dirname(filename)
		if not os.path.exists(dir):
			try:
				os.makedirs(dir)
			except:
				return

		logger = XMLLogWriter(self, list, filename, config.get('logging', 'gzip', 'boolean'), False)
		logger.start()

	def save_as_xml(self, widget=None, list=None):
		ffilter = gtk.FileFilter()
		ffilter.add_pattern('*.xml')
		ffilter.add_pattern('*.xml.gz')
		filew = CompressFileChooser(title=_('Log file selection'),
									parent=self.window,
									action=gtk.FILE_CHOOSER_ACTION_SAVE,
									buttons=(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
											 gtk.STOCK_OK, gtk.RESPONSE_OK))
		filew.set_filter(ffilter)
		filew.set_select_multiple(False)
		filew.connect("destroy", lambda x: filew.destroy())
		filew.set_current_folder(os.path.normpath(open_bottlecase()))

		filew.show()
		while True:
			res = filew.run()
			if not res or res == gtk.RESPONSE_CANCEL:
				filew.destroy()
				return

			filename = filew.get_filename()
			compress = filew.compress.get_active()

			if compress and filename[-3:] != '.gz':
				filename += '.gz'

			if os.path.exists(filename):
				if not open_overwrite_check_dialog(filew, filename):
					continue
			break
		filew.destroy()

		if not list or not isinstance(list, LogList):
			list = self.logbook.get_active_loglist()
		if not list:
			return

		logger = XMLLogWriter(self, list, filename, compress, True)
		logger.start()

	def open_xml_log(self, widget=None, *args):
		ffilter = gtk.FileFilter()
		ffilter.add_pattern('*.xml')
		ffilter.add_pattern('*.xml.gz')
		filew = gtk.FileChooserDialog(title=_('Log file selection'),
									  parent=self.window,
									  action=gtk.FILE_CHOOSER_ACTION_OPEN,
									  buttons=(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
											   gtk.STOCK_OK, gtk.RESPONSE_OK))
		filew.set_filter(ffilter)
		filew.set_select_multiple(False)
		filew.connect("destroy", lambda x: filew.destroy())
		filew.set_current_folder(os.path.normpath(open_bottlecase()))
		filew.show()
		res = filew.run()
		filename = filew.get_filename()
		filew.destroy()

		if res == gtk.RESPONSE_OK:
			if os.path.exists(filename):
				logger = XMLLogOpener(self, filename)
				logger.start()

	def fetch(self, widget=None, *args):
		try:
			channels = self.app.client.channels
		except:
			channels = []

		d = LogFetchDialog(self.window, channels)
		if not d.request or not d.request_name:
			return

		fetcher = LogFetchOperator(self, d.request, d.request_name)
		fetcher.start()

	def popup_from_logtab(self, widget, event, sender):
		if event.button == 3:
			sender.popup(None, None, None, event.button, event.time)

	def tab_drag_data_get(self, widget, context, selection, targetType, eventTime, sender):
		n = self.logbook.page_num(sender)
		selection.set(selection.target, 8, str(n))

	def tab_drop_received(self, widget, context, x, y, selection, targetType, time, receiver):
		sender_n = int(selection.data)
		sender = self.logbook.get_nth_page(sender_n)

		receiver_n = self.logbook.page_num(receiver)

		self.logbook.reorder_child(sender, receiver_n)
		self.logbook.reorder_child(receiver, sender_n)

	def edit_copy_ghostname(self, widget=None, data=None):
		self.edit_copy(widget, data, LogList.LISTSTORE_GHOST)
	def edit_copy_channel(self, widget=None, data=None):
		self.edit_copy(widget, data, LogList.LISTSTORE_CHANNEL)
	def edit_copy_script(self, widget=None, data=None):
		self.edit_copy(widget, data, LogList.LISTSTORE_SCRIPT)
	def edit_copy_time(self, widget=None, data=None):
		self.edit_copy(widget, data, LogList.LISTSTORE_DATETIME)
	def edit_copy_all(self, widget=None, data=None):
		self.edit_copy(widget, data, LogList.LISTSTORE_ALL)
	def edit_copy(self, widget, data, target):
		self.app.dispose_selection()
		self.app.copy_clipboard(self.logbook.edit_copy(widget, data, target))

	def modify_tab_title(self, widget, child):
		title_label = self.logbook.get_tab_label(child).get_children()[0].get_children()[0]
		title = title_label.get_text()
		title = open_simple_entry_dialog(unicode(_('Please input title'), 'utf-8'),
										 unicode(_('Please input title'), 'utf-8'),
										 default=title,
										 parent=self.window)
		if title:
			title_label.set_text(title)

	def connect_for_logtab(self, list, child, eventbox):
		logtab_popup_menu = gtk.Menu()
		logtab_popup_menu.show()

		save_menuitem = gtk.MenuItem(_('Save as XML'))
		save_menuitem.show()
		save_menuitem.connect("activate", self.save_as_xml, list)

		close_menuitem = gtk.MenuItem(_('Close'))
		close_menuitem.show()

		if self.logbook.current_is(list):
			close_menuitem.connect("activate", self.close_current_tab, child)
		else:
			close_menuitem.connect("activate", self.close_tab, child)
			modify_menuitem = gtk.MenuItem(_('Modify tab title'))
			modify_menuitem.show()
			modify_menuitem.connect("activate", self.modify_tab_title, child)
			logtab_popup_menu.append(modify_menuitem)

		logtab_popup_menu.append(save_menuitem)
		logtab_popup_menu.append(close_menuitem)
		eventbox.connect("button-press-event", self.popup_from_logtab, logtab_popup_menu)

		# DND
		eventbox.connect("drag-data-get", self.tab_drag_data_get, child)
		eventbox.drag_source_set(gtk.gdk.BUTTON1_MASK, [ ( "text/plain", 0, 80 ) ], gtk.gdk.ACTION_MOVE)

		eventbox.connect("drag-data-received", self.tab_drop_received, child)
		eventbox.drag_dest_set(gtk.DEST_DEFAULT_MOTION|gtk.DEST_DEFAULT_HIGHLIGHT|gtk.DEST_DEFAULT_DROP,
							   [ ( "text/plain", 0, 80 ) ], gtk.gdk.ACTION_MOVE)

	def create_tab(self, label_text, list=None):
		close_button, eventbox, child = self.logbook.create_tab(label_text, list)

		list = child.get_child()
		self.connect_list(list)

		if self.logbook.current_is(list):
			close_button.connect('clicked', self.close_current_tab, child)
		else:
			close_button.connect('clicked', self.close_tab, child)
		close_button.connect('focus-in-event', self.focus_active_loglist)

		self.connect_for_logtab(list, child, eventbox)
		self.focus_active_loglist()

		if self.logbook.active_is(list):
			self.update_rows(list)
			if list.rows() > 0 and list.is_selected():
				self.select(list)

	def update(self, mid, channel, ghost, script, receive_time):
		datetime = time.strftime("%y/%m/%d %H:%M:%S", time.localtime(receive_time))
		close_button, eventbox, child = self.logbook.update(mid, channel, ghost, script, datetime)

		if child and close_button and eventbox:
			# if Tab named `Current' Created
			close_button.connect('clicked', self.close_current_tab, child)
			close_button.connect('focus-in-event', self.focus_active_loglist)

			list = child.get_child()
			if list:
				self.connect_list(list)

				if list == self.logbook.get_active_loglist():
					if list.is_selected():
						self.select(list)
					self.update_rows(list)
			self.connect_for_logtab(list, child, eventbox)

		elif self.logbook.current_is(self.logbook.get_active_loglist()):
			self.update_rows(self.logbook.get_active_loglist())

		self.focus_active_loglist()

	def close_current_tab(self, widget, child):
		if config.get('logging', 'save_current_when_close_tab', 'boolean'):
			self.autologging()
		self.close_tab(widget, child)

	def close_tab(self, widget, child):
		tablabel = self.logbook.get_tab_label(child)
		n = self.logbook.close_tab(widget, child)
		if n == -1:
			self.remove_status()

		self.volley_canceler = True
		list = child.get_children()[0]
		self.app.notify_play_cancel_page_of(list)

		tablabel.destroy()
		#list.destroy()
		#child.destroy()
		#gc.collect()

	def connect_list(self, list):
		list.connect("select-cursor-row",  self.select)
		list.connect("key-press-event",    self.loglist_keypress)
		list.connect("button-press-event", self.popup)

	def remove_status(self):
		self.sbar.pop(0)
		self.rowsbar.pop(0)
		self.curr_rowsbar.pop(0)
		self.textmanager.set_text('')

	def update_rows(self, list):
		if list and list.rows():
			self.rowsbar.pop(0)
			self.rowsbar.push(0, str(list.rows()) + _('Items'))
		self.update_select_no(list)

	def update_select_no(self, list):
		if list and list.is_selected():
			self.curr_rowsbar.pop(0)
			self.curr_rowsbar.push(0, 'No.' + str(list.selection()+1))

	def tab_changed(self, notebook, page, pagenum):
		list = self.logbook.tab_changed(notebook, page, pagenum)
		if list is None:
			self.remove_status()
		else:
			self.select(list)
			self.update_rows(list)

	def search(self, widget=None, data=None):
		self.logbook.search(self.s_arrow.get_property('arrow-type'),
							self.search_entry.get_text(),
							widget, data)

	def volley(self, widget=None, data=None):
		list = self.logbook.get_active_loglist()
		if not list or not list.is_selected():
			return

		self.volley_canceler = False
		gobject.timeout_add(WAIT_NEXT_REQUEST, self._volley, widget, data, list, list.selection(), 0)

	VOLLEY_INTERVAL = 500 # 0.5 sec
	def _volley(self, widget, data, list, target, min):
		if self.volley_canceler:
			self.volley_canceler = False
			return
		gobject.timeout_add(WAIT_NEXT_REQUEST, self.play, widget, data, list, target)
		if target > min:
			gobject.timeout_add(self.VOLLEY_INTERVAL, self._volley, widget, data, list, target-1, min)

	def play(self, widget=None, data=None, list=None, target=None):
		if list is None or target is None:
			list = self.logbook.get_active_loglist()
			if not list or not list.is_selected():
				return
			target = list.selection()

		script  = unicode(list.get_text(target, LogList.LISTSTORE_SCRIPT), 'utf-8')
		ifghost = unicode(list.get_text(target, LogList.LISTSTORE_GHOST), 'utf-8')
		channel = unicode(list.get_text(target, LogList.LISTSTORE_CHANNEL), 'utf-8')
		mid     = list.mids[target]

		if script is None or ifghost is None or not mid:
			return

		if self.get_bottle_state(mid, list) == STATE_BOTTLE_NONE: # ignore duplex
			self.app.send_local_message(channel, ifghost, script, mid=mid, list=list)

	def cancel_play(self, widget=None, data=None):
		mid = self.logbook.cancel_play(widget, data)
		if not self.app.notify_play_cancel(mid):
			open_error_dialog(unicode(_("Couldn't cancel to play message"), 'utf-8'), self.window)

	def cancel_play_page(self, widget=None, data=None):
		list = self.logbook.cancel_play_page()
		if list is not None:
			self.volley_canceler = True
			self.app.notify_play_cancel_page_of(list)
		else:
			open_error_dialog(unicode(_("Couldn't cancel to play message"), 'utf-8'), self.window)

	def cancel_play_all(self, widget=None, data=None):
		self.volley_canceler = True
		self.logbook.cancel_play_all()
		self.app.notify_play_cancel_all()

	def get_bottle_state(self, mid, list):
		return self.logbook.get_bottle_state(mid, list)

	def set_bottle_state(self, mid, list, stock=STATE_BOTTLE_NONE):
		self.logbook.set_bottle_state(mid, list, stock)

	def set_playing_bottle(self, mid, list, prev_list=None):
		if prev_list is not None:
			self.clear_playing_bottle(prev_list)
		self.logbook.set_playing_bottle(mid, list)

	def set_reserve_bottle(self, mid, list):
		self.set_bottle_state(mid, list, STATE_BOTTLE_RESERVED)

	def clear_playing_bottle(self, list=None):
		self.logbook.clear_playing_bottle(list)

	def log_votes(self, mid, type, num):
		self.logbook.log_votes(mid, type, num)

	def voting(self, widget=None, data=None):
		mid, type = self.logbook.voting(widget, data, self.window)
		if mid is None or type is None:
			return

		self.app.vote_message(mid, type)

	def agreeing(self, widget=None, data=None):
		mid, type = self.logbook.agreeing(widget, data, self.window)
		if mid is None or type is None:
			return
		self.app.vote_message(mid, type)

	#def button_popup(self, widget, event, sender):
	#	if event.button in [1, 3]:
	#		sender.popup(None, widget, None, event.button, event.time)

	def set_entry_guide(self, widget, event, data):
		if data == 'in':
			if widget.get_text() == _('Search'):
				widget.set_text('')
		elif data == 'out':
			if not widget.get_text():
				widget.set_text(_('Search'))

	def popup(self, loglist, event):
		if event.button == 3:
			self.popup_menu.popup(None, loglist, None, event.button, event.time)
		elif event.type == 5:
			self.play(loglist, event)

	def select(self, loglist, se=False):
		self.textmanager.set_text('')
		row = loglist.selection()
		script = unicode(loglist.get_text(row, LogList.LISTSTORE_SCRIPT), "utf-8", 'replace')
		self.textmanager.insert_with_color_from(script)
		self.sw2.emit('scroll-child', gtk.SCROLL_START, False)
		self.sbar.pop(0)
		self.sbar.push(0, str(len(script.encode('sjis', 'xmlcharrefreplace'))) + unicode(_('bytes'), 'utf-8'))
		self.update_select_no(loglist)

	def scroll(self, widget, event):
		if event.state == 1: # shift
			if event.keyval == 65362: # up
				self.sw2.emit('scroll-child', gtk.SCROLL_PAGE_UP, False)
				return True
			elif event.keyval == 65364: # down
				self.sw2.emit('scroll-child', gtk.SCROLL_PAGE_DOWN, False)
				return True
			return False
		return False

	def entry_keypress(self, widget, event):
		return self.scroll(widget, event)

	def loglist_keypress(self, widget, event):
		if self.scroll(widget, event) == False:
			if event.keyval == 65293: # enter
				self.play()
				return True
			return False
		return True

	def set_style(self, widget=None, *args):
		data = self.style_combo.get_active()
		self.textmanager.set_style(data)

		llist = self.logbook.get_active_loglist()
		if not llist or not llist.is_selected():
			return
		self.select(llist)

	def focus_active_loglist(self, widget=None, data=None):
		self.logbook.focus_active_loglist()

	### notify preference changed start
	def tab_pos_changed(self):
		pos = config.get('logwindow', 'tab_placement', 'int')
		if pos == config.LOGWINDOW_TAB_PLACEMENT_TOP:
			gtk_pos = gtk.POS_TOP
		elif pos == config.LOGWINDOW_TAB_PLACEMENT_BOTTOM:
			gtk_pos = gtk.POS_BOTTOM
		elif pos == config.LOGWINDOW_TAB_PLACEMENT_LEFT:
			gtk_pos = gtk.POS_LEFT
		else:
			gtk_pos = gtk.POS_RIGHT
		self.logbook.set_tab_pos(gtk_pos)

	def color_changed(self):
		self.textmanager.color_changed()
		list = self.logbook.get_active_loglist()
		if not list or not list.is_selected():
			return
		self.select(list)

	def font_changed(self):
		self.textmanager.font_changed()

	def receive_preferences_changed(self):
		self.color_changed()
		self.font_changed()
		self.tab_pos_changed()
	### notify preference changed end


class CompressFileChooser(gtk.FileChooserDialog):
	def __init__(self, title='', parent=None, action=gtk.FILE_CHOOSER_ACTION_OPEN, buttons=None, backend=''):
		gtk.FileChooserDialog.__init__(self, title, parent, action, buttons, backend)
		self.compress = gtk.CheckButton(unicode(_('GZIP Compress'), 'utf-8'))
		self.compress.show()
		self.vbox.pack_start(self.compress, False, False)
