# -*- coding: utf-8 -*-
#
#  hypertextview.py - Hyperlink supported TextView
#  Copyright (C) 2004-2010 by Atzm WATANABE <atzm@atzm.org>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: hypertextview.py,v 1.3 2010/08/23 15:14:02 atzm Exp $

import gtk
import gobject
import pango

class HyperTextView(gtk.TextView):
	__gtype_name__ = 'HyperTextView'
	__gsignals__ = {'anchor-clicked': (gobject.SIGNAL_RUN_LAST, None, (str, str, int))}
	__gproperties__ = {
		'link':   (gobject.TYPE_PYOBJECT, 'link color', 'link color of TextView', gobject.PARAM_READWRITE),
		'active': (gobject.TYPE_PYOBJECT, 'active color', 'active color of TextView', gobject.PARAM_READWRITE),
		'hover':  (gobject.TYPE_PYOBJECT, 'link:hover color', 'link:hover color of TextView', gobject.PARAM_READWRITE),
	}

	def do_get_property(self, prop):
		try:
			return getattr(self, prop.name)
		except AttributeError:
			raise AttributeError, 'unknown property %s' % prop.name

	def do_set_property(self, prop, val):
		if hasattr(self.props, prop.name):
			setattr(self, prop.name, val)
		else:
			raise AttributeError, 'unknown property %s' % prop.name

	def __init__(self, buffer=None):
		gtk.TextView.__init__(self, buffer)
		self.link   = {'background': 'white', 'foreground': 'blue', 'underline': pango.UNDERLINE_SINGLE}
		self.active = {'background': 'light gray', 'foreground': 'red', 'underline': pango.UNDERLINE_SINGLE}
		self.hover  = {'background': 'light gray', 'foreground': 'blue', 'underline': pango.UNDERLINE_SINGLE}

		self.set_editable(False)
		self.set_cursor_visible(False)

		self.__tags = []

		self.connect('motion-notify-event', self._motion)
		self.connect('focus-out-event', lambda w, e: self.get_buffer().get_tag_table().foreach(self.__tag_reset, e.window))

	def connect_textbuffer(self, signal, callback, *params):
		return self.get_buffer().connect(signal, callback, *params)

	def get_cursor_position(self):
		b = self.get_buffer()
		return b.get_iter_at_mark(b.get_insert()).get_offset()

	def get_end_position(self):
		return self.get_buffer().get_end_iter().get_offset()

	def create_tag(self, name=None, **prop):
		return self.get_buffer().create_tag(name, **prop)

	def get_tag_table(self):
		return self.get_buffer().get_tag_table()

	def get_char_at(self, pos):
		return self.get_buffer().get_iter_at_offset(pos).get_char()

	def get_text(self):
		b = self.get_buffer()
		return b.get_text(*b.get_bounds())

	def set_text(self, text):
		return self.get_buffer().set_text(text)

	def insert(self, text, _iter=None):
		b = self.get_buffer()
		if _iter is None:
			_iter = b.get_end_iter()
		return b.insert(_iter, text)

	def insert_with_tags_by_name(self, text, tag, _iter=None):
		b = self.get_buffer()
		if _iter is None:
			_iter = b.get_end_iter()
		return b.insert_with_tags_by_name(_iter, text, tag)

	def insert_with_anchor(self, text, anchor=None, _iter=None):
		b = self.get_buffer()
		if _iter is None:
			_iter = b.get_end_iter()
		if anchor is None:
			anchor = text

		tag = b.create_tag(None, **self.get_property('link'))
		tag.set_data('is_anchor', True)
		tag.connect('event', self._tag_event, text, anchor)
		self.__tags.append(tag)
		return b.insert_with_tags(_iter, text, tag)

	def update(self):
		for tag in self.__tags:
			self.__tag_reset(tag, self.window)

	def _motion(self, view, ev):
		window = ev.window
		x, y, _ = window.get_pointer()
		x, y = view.window_to_buffer_coords(gtk.TEXT_WINDOW_TEXT, x, y)
		tags = view.get_iter_at_location(x, y).get_tags()
		for tag in tags:
			if tag.get_data('is_anchor'):
				for t in set(self.__tags) - set([tag]):
					self.__tag_reset(t, window)
				self.__set_anchor(window, tag, gtk.gdk.Cursor(gtk.gdk.HAND2),
								  self.get_property('hover'))
				break
		else:
			tag_table = self.get_buffer().get_tag_table()
			tag_table.foreach(self.__tag_reset, window)

	def _tag_event(self, tag, view, ev, _iter, text, anchor):
		_type = ev.type
		if _type == gtk.gdk.MOTION_NOTIFY:
			return
		elif _type in [gtk.gdk.BUTTON_PRESS, gtk.gdk.BUTTON_RELEASE]:
			button = ev.button
			cursor = gtk.gdk.Cursor(gtk.gdk.HAND2)
			if _type == gtk.gdk.BUTTON_RELEASE:
				self.emit('anchor-clicked', text, anchor, button)
				self.__set_anchor(ev.window, tag, cursor, self.get_property('hover'))
			elif button in [1, 2]:
				self.__set_anchor(ev.window, tag, cursor, self.get_property('active'))

	def __tag_reset(self, tag, window):
		if tag.get_data('is_anchor'):
			self.__set_anchor(window, tag, None, self.get_property('link'))

	def __set_anchor(self, window, tag, cursor, prop):
		window.set_cursor(cursor)
		for key, val in prop.iteritems():
			tag.set_property(key, val)

gobject.type_register(HyperTextView)

if __name__ == '__main__':
	def clicked(widget, text, anchor, button):
		print widget, text, anchor, button

	t = HyperTextView()
	t.connect('anchor-clicked', clicked)
	t.link['foreground'] = 'dark blue'
	t.insert_with_anchor('Google', 'http://www.google.com/')
	t.insert('\n')
	t.insert_with_anchor('Yahoo!', 'http://www.yahoo.com/')

	w = gtk.Window()
	w.set_default_size(200, 100)
	w.connect('destroy', lambda w: gtk.main_quit())
	w.add(t)

	w.show_all()
	gtk.main()
