# -*- coding: utf-8 -*-
#
#  ghostmanager.py - The Manager of Ghosts
#  Copyright (C) 2004 by Takuya KAWAHARA <num@sann.ne.jp>
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: ghostmanager.py,v 1.28 2005/09/27 05:59:49 atzm Exp $
#

__version__ = '$Revision: 1.28 $'
__author__ = '$Author: atzm $'

import gobject
import os, sys, time
import re, string

from ghost import Ghost
from viewercommon import *

class GhostManager:
	OPTIMIZE_INTERVAL = 1000*60*5 # 5 minutes

	def __init__(self, textfile, ghostdir, callback_generate_ghost_list):
		self.ghost_dic = OrderedDictionary() # key: path
		self.fmo_ghost_list = []
		self.callback_generate_ghost_list = callback_generate_ghost_list

		self.textfile = textfile
		self.ghostdir = ghostdir

		self.ghost_cache = {} # key: sakura_name
		self._optimize_id = gobject.timeout_add(self.OPTIMIZE_INTERVAL, self.optimize_ghost_cache, priority=gobject.PRIORITY_LOW)

		self.ignore_list = []
		# ghosts listed in here don't exist
		# but it's cleared when saves modified GhostList or regenerates dict.
		# for fast proccess and consistency

		self.initialize()

	def get_ghost_text(self): # returns abspath of ~/.bottlecase/gviewer/ghost.txt as default
		return self.textfile
	def get_ghost_dir(self):  # returns abspath of ~/.bottlecase/gviewer/Ghost as default
		return self.ghostdir
	def get_base_dir(self):   # returns abspath of ~/.bottlecase/gviewer as default
		return os.path.dirname(self.textfile)

	def initialize(self):
		if os.path.exists(self.textfile):
			self.generate_dictionary()
		else:
			basedir = os.path.dirname(self.textfile)
			try:
				os.makedirs(basedir)
			except OSError:
				pass

			try:
				self.save_dictionary()
			except:
				print >> sys.stderr, '!!! WARNING: cannot touch %s' % self.textfile
			else:
				print >> sys.stderr, '!!! %s created.' % self.textfile

		if not os.path.isdir(self.ghostdir):
			try:
				os.makedirs(self.ghostdir)
			except OSError, e:
				print >> sys.stderr, '!!! WARNING: cannot mkdir %s' % self.ghostdir
				print >> sys.stderr, '!!! %s' % e
			else:
				print >> sys.stderr, '!!! %s created.' % self.ghostdir

	def optimize_ghost_cache(self):
		glist = self.callback_generate_ghost_list()
		for name in self.ghost_cache.keys():
			if name not in glist:
				del self.ghost_cache[name]
			elif __debug__:
				print >> sys.stderr, '!!! GhostManager: Cache Optimization: hit %s' % name.encode('euc-jp')

		return True

	def search_ghost(self, ifghost): # FIXME : NOT SMART
		if ifghost not in self.ignore_list:

			# returns ghost information if in cache
			try:
				if self.ghost_cache[ifghost][0]:
					return self.ghost_cache[ifghost]
			except KeyError:
				pass

			# search ghost information and create cache
			for path in self.ghost_dic.keys():
				ghost = self.nominate_ghost_txt(os.path.basename(expand_quote(path)))
				if ghost is None:
					continue
				sakura_name = ghost.get_sakura()[:]

				data = self.ghost_dic[path]
				self.ghost_cache[sakura_name] = data

				if sakura_name == ifghost and data[0]:
					return data

			# ghost does not exist if process comes here
			self.ignore_list.append(ifghost)
			msg = ''
		else:
			msg = '(cached)'

		if __debug__:
			now = time.strftime("%y/%m/%d %H:%M:%S", time.localtime(time.time()))
			print >> sys.stderr, '!!! no such ghost: %s (at %s) in search_ghost %s' % (ifghost.encode('euc-jp'), now, msg)
		return None

	def get_sakura_path(self): # FIXME : NOT SMART
		for sakura in [NAME_SAKURA_KAWAKAMI, NAME_SAKURA_KUROI]: #, NAME_SAKURA_DOT, NAME_SAKURA_DOT_]:
			try:
				data = self.search_ghost(sakura)
				if data is None:
					return ''
				[define, name, fmo, path] = data
				if define:
					return path
			except KeyError:
				pass
		return ''

	def nominate_ghost(self, ifghost, ghostdir=None):
		if ghostdir is None:
			ghostdir = self.ghostdir

		try:
			#if ifghost == NAME_SAKURA:
			#	path = expand_quote(self.get_sakura_path())
			#else:
			data = self.search_ghost(ifghost)
			if data is None:
				return None
			path = expand_quote(data[3])

			if not path:
				return None

			if not os.path.isabs(path):
				path = os.path.join(ghostdir, os.path.basename(path))

			if not os.path.isfile(path):
				return None

			return Ghost(path, ghostdir)
		except KeyError:
			if __debug__:
				print >> sys.stderr, '%s' % ('-' * 20)
				print >> sys.stderr, time.strftime('Nominated at %m/%d %H:%M:%S / ', time.localtime())
				print >> sys.stderr, 'no such ghost: %s in nominate_ghost' % ifghost.encode('euc-jp')
			return None

	def nominate_ghost_txt(self, svg_txt, ghostdir=None):
		if ghostdir is None:
			ghostdir = self.ghostdir

		if not svg_txt:
			return None

		if not os.path.isabs(svg_txt):
			svg_txt = os.path.join(ghostdir, svg_txt)

		if not os.path.isfile(svg_txt):
			return None

		try:
			return Ghost(svg_txt, ghostdir)
		except KeyError:
			if __debug__:
				print >> sys.stderr, '%s' % ('-' * 20)
				print >> sys.stderr, time.strftime('Nominated at %m/%d %H:%M:%S / ', time.localtime())
				print >> sys.stderr, 'no such svg_txt: %s in nominate_ghost_txt' % svg_txt
			return None

	def unicoding(self, text):
		try:
			return unicode(text, 'utf-8')
		except:
			pass
		return text

	def get_fmo_ghostnames(self):
		return self.fmo_ghost_list

	def save_dictionary(self, list=[], textfile=None): # gtk.ListStore can use as the List type
		if textfile is None:
			textfile = self.textfile

		try:
			file = open(textfile, 'w')
		except IOError:
			return -1

		self.ghost_dic.clear() #= OrderedDictionary()
		self.fmo_ghost_list[:] = []
		self.ghost_cache.clear()
		self.ignore_list[:] = []

		file.write('INSTALL,,,"Ghost\\"\r\n')
		for i in xrange(len(list)):
			[define, name, fmo, path] = list[i]

			name = self.unicoding(name)
			fmo  = self.unicoding(fmo)
			path = self.unicoding(path)

			self.ghost_dic[path] = [define, name, fmo, path]
			if define:
				if fmo[:3] == 'FMO':
					fmoghost = self.nominate_ghost_txt(os.path.basename(expand_quote(path)))
					if fmoghost is not None:
						fmoghostname = fmoghost.get_sakura()
						self.fmo_ghost_list.append(fmoghostname)

				define = 'GHOST'
			else:
				define = '#GHOST'

			path = path.replace('/', '\\')
			line = string.join([define, name, fmo, path], ',').encode('sjis')
			file.write(line + '\r\n')
		file.write('\r\n')
		file.write('/EOF\r\n')
		file.close()

	def get_dict(self):
		return self.ghost_dic

	def generate_dictionary(self, textfile=None):
		if textfile is None:
			textfile = self.textfile

		try:
			file = open(textfile)
		except IOError:
			return -1

		self.ghost_dic.clear() #= OrderedDictionary()
		self.fmo_ghost_list[:] = []
		self.ghost_cache.clear()
		self.ignore_list[:] = []

		base_dir = os.path.dirname(textfile)
		read_start = False
		while 1:
			line = file.readline()
			if not line:
				print >> sys.stderr, '!!! /EOF not found: break reading.'
				break
			if line[:4] == '/EOF':
				break
			if line[:7] == 'INSTALL':
				read_start = True
				continue
			if not read_start:
				continue

			line = line.strip()
			if not line:
				continue

			[define, ghostname, fmo, path] = line.split(',')
			if define[:5] == 'GHOST':
				define = True
			else:
				define = False

			path = os.path.normpath(path.replace('\\', r'/'))
			ghostname = unicode(ghostname, 'sjis', 'replace')
			self.ghost_dic[path] = [define, ghostname, fmo, path]

			expanded_path = expand_quote(path)
			if fmo[:3] == "FMO" and define and os.path.isfile(os.path.join(base_dir, expanded_path)):
				ghost = self.nominate_ghost_txt(os.path.basename(expanded_path))
				if ghost is not None:
					name = ghost.get_sakura()
					self.fmo_ghost_list.append(name)


# from Python Cookbook
#   URL: http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/107747
class OrderedDictionary(dict):
	def __init__(self, dictionary={}):
		self._keys = []
		dict.__init__(self, dictionary)

	def __delitem__(self, key):
		dict.__delitem__(self, key)
		self._keys.remove(key)

	def __setitem__(self, key, item):
		dict.__setitem__(self, key, item)
		if key not in self._keys:
			self._keys.append(key)

	def copy(self):
		newdict = OrderedDictionary()
		newdict.update(self)
		return newdict

	def clear(self):
		dict.clear(self)
		self._keys[:] = []

	def items(self):
		return zip(self._keys, self.values())

	def keys(self):
		return self._keys[:]

	def popitem(self):
		try:
			key = self._keys[-1]
		except IndexError:
			raise KeyError('dictionary is empty')

		val = self[key]
		del self[key]

		return (key, val)

	def setdefault(self, key, failobj=None):
		dict.setdefault(self, key, failobj)
		if key not in self._keys:
			self._keys.append(key)

	def update(self, dictionary):
		for (key, val) in dictionary.items():
			self.__setitem__(key, val)

	def values(self):
		return map(self.get, self._keys)
