# -*- coding: ascii -*-
#
#  prefs.py - Preference manipulator for GBottler
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: prefs.py,v 1.56 2004/09/29 21:58:11 atzm Exp $
#

import os, sys, re, string
import gtk

from queryluid import LUIDInfo
from common    import *

class PreferenceDialog(gtk.Dialog):
	def __init__(self, app, prefmanager, luid=''):
		self.prefs = {}
		self.app = app
		self.luid_info = None
		if luid:
			self.luid_info = LUIDInfo(luid)

		general_tab = self.create_general_tab()
		log_tab     = self.create_log_tab()
		net_tab     = self.create_network_tab()
		font_tab     = self.create_font_tab()
		viewer_tab  = self.create_gviewer_tab()

		if self.luid_info:
			luid_data = self.luid_info.get_all()

			if luid_data[LUIDInfo.LUID_INFO_GENERAL]:
				self.luidinfolist = gtk.CList(len(luid_data[LUIDInfo.LUID_INFO_GENERAL][0]),
											  luid_data[LUIDInfo.LUID_INFO_GENERAL][0])
			else:
				self.luidinfolist = gtk.CList()
			self.luidinfolist.column_titles_passive()
			self.luidinfolist.set_column_auto_resize(1, gtk.TRUE)
			self.luidinfolist.set_shadow_type(gtk.SHADOW_IN)
			self.luidinfolist.set_selection_mode(gtk.SELECTION_BROWSE)
			self.luidinfolist.set_reorderable(gtk.TRUE)
			self.luidinfolist.set_column_width(0, 100)
			self.luidinfolist.set_column_auto_resize(1, gtk.TRUE)

			self.luidinfolist.freeze()
			self.luidinfolist.clear()
			for line in luid_data[LUIDInfo.LUID_INFO_GENERAL][1:]:
				self.luidinfolist.append(tuple(line))
			self.luidinfolist.sort()
			self.luidinfolist.columns_autosize()
			self.luidinfolist.thaw()

			self.luidinfosw = gtk.ScrolledWindow()
			self.luidinfosw.set_size_request(300, 80)
			self.luidinfosw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
			self.luidinfosw.set_border_width(5)
			self.luidinfosw.add(self.luidinfolist)

			if luid_data[LUIDInfo.LUID_INFO_DETAIL]:
				self.luidlist = gtk.CList(len(luid_data[LUIDInfo.LUID_INFO_DETAIL][0]),
										  luid_data[LUIDInfo.LUID_INFO_DETAIL][0])
			else:
				self.luidlist = gtk.CList()
			self.luidlist.column_titles_passive()
			self.luidlist.set_column_auto_resize(1, gtk.TRUE)
			self.luidlist.set_shadow_type(gtk.SHADOW_IN)
			self.luidlist.set_selection_mode(gtk.SELECTION_BROWSE)
			self.luidlist.set_reorderable(gtk.TRUE)
			self.luidlist.set_column_width(0, 100)
			self.luidlist.set_column_auto_resize(1, gtk.TRUE)

			self.luidlist.freeze()
			self.luidlist.clear()
			for line in luid_data[LUIDInfo.LUID_INFO_DETAIL][1:]:
				self.luidlist.append(tuple(line))
			self.luidlist.sort()
			self.luidlist.columns_autosize()
			self.luidlist.thaw()

			self.luidsw = gtk.ScrolledWindow()
			self.luidsw.set_size_request(300, 100)
			self.luidsw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
			self.luidsw.set_border_width(5)
			self.luidsw.add(self.luidlist)

			luidbutton = gtk.Button(unicode(_('Update')))
			luidbutton.connect('clicked', self.refresh_luid_info)

			luidbox = gtk.VBox()
			luidbox.pack_start(self.luidinfosw, gtk.FALSE, gtk.FALSE, 0)
			luidbox.pack_start(self.luidsw, gtk.TRUE, gtk.TRUE, 0)
			luidbox.pack_end(luidbutton, gtk.FALSE, gtk.FALSE, 3)

# === Bring all Tabs together to Notebook === #
		main_notebook = gtk.Notebook()
		main_notebook.popup_enable()
		main_notebook.set_scrollable(gtk.TRUE)
		main_notebook.set_tab_pos(gtk.POS_TOP)

		for item in [
			[general_tab, gtk.Label(unicode(_('General')))],
			[log_tab,     gtk.Label(unicode(_('Logs')))],
			[net_tab,     gtk.Label(unicode(_('Network')))],
			[font_tab,     gtk.Label(unicode(_('Fonts')))],
			[viewer_tab,  gtk.Label(unicode(_('GViewer')))],
			]:
			main_notebook.append_page(item[0], item[1])

		if self.luid_info:
			main_notebook.append_page(luidbox, gtk.Label(unicode(_('LUID'))))

# === Bring all items together to Main Window Finally === #
		gtk.Dialog.__init__(self, ('%s: ' % APP + unicode(_('Preferences'))), self.app.window,
							gtk.DIALOG_DESTROY_WITH_PARENT|gtk.DIALOG_MODAL|gtk.DIALOG_NO_SEPARATOR,
							(gtk.STOCK_APPLY, gtk.RESPONSE_APPLY,
							 gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
							 gtk.STOCK_OK, gtk.RESPONSE_OK))
		self.set_border_width(5)
		self.vbox.pack_start(main_notebook, gtk.TRUE, gtk.TRUE, 0)
		self.connect('delete-event', lambda x, y: self.emit('response', gtk.RESPONSE_CANCEL))
		self.show_all()

# === Initialization of PreferenceDialog === #
		prefmanager.load_database()
		self.set_prefs(prefmanager)

# === Start mainloop of Dialog === #
		while True:
			response = self.run()

			if response == gtk.RESPONSE_APPLY:
				prefmanager.save_database(self)
				self.app.notify_preferences_changed(self.get_prefs())
			elif response == gtk.RESPONSE_OK:
				prefmanager.save_database(self)
				self.app.notify_preferences_changed(self.get_prefs())
				self.close()
				break
			elif response == gtk.RESPONSE_CANCEL:
				self.close()
				break

	def open_fontselection(self, widget=None, entry=None):
		fontsel = gtk.FontSelectionDialog(unicode(_('Chose font...'), 'utf-8'))
		fontsel.show()
		fontsel.set_font_name(entry.get_text())
		res = fontsel.run()
		fontsel.hide()
		if res == gtk.RESPONSE_OK:
			entry.set_text(fontsel.get_font_name())
		fontsel.destroy()

	def open_colorselection(self, widget=None, entry=None):
		colorsel = gtk.ColorSelectionDialog(unicode(_('Chose color...'), 'utf-8'))
		colorsel.show()
		colorsel.colorsel.set_current_color(gtk.gdk.color_parse(entry.get_text()))
		res = colorsel.run()
		colorsel.hide()
		if res == gtk.RESPONSE_OK:
			color = colorsel.colorsel.get_current_color()
			color_str = '#%04x%04x%04x' % (color.red, color.green, color.blue)
			entry.set_text(color_str)
		colorsel.destroy()

	def refresh_luid_info(self, widget=None, data=None):
		luid_data = self.luid_info.get_all(True)

		if luid_data[LUIDInfo.LUID_INFO_GENERAL] and luid_data[LUIDInfo.LUID_INFO_DETAIL]:
			self.luidinfolist.freeze()
			self.luidinfolist.hide()
			self.luidinfolist.clear()
			self.luidinfosw.remove(self.luidinfolist)
			self.luidinfolist = gtk.CList(len(luid_data[LUIDInfo.LUID_INFO_GENERAL][0]),
										  luid_data[LUIDInfo.LUID_INFO_GENERAL][0])
			self.luidinfolist.column_titles_passive()
			self.luidinfolist.set_column_auto_resize(1, gtk.TRUE)
			self.luidinfosw.add(self.luidinfolist)
			for line in luid_data[LUIDInfo.LUID_INFO_GENERAL][1:]:
				self.luidinfolist.append(tuple(line))
			self.luidinfolist.columns_autosize()
			self.luidinfolist.thaw()
			self.luidinfolist.show()

			self.luidlist.freeze()
			self.luidlist.hide()
			self.luidlist.clear()
			self.luidsw.remove(self.luidlist)
			self.luidlist = gtk.CList(len(luid_data[LUIDInfo.LUID_INFO_DETAIL][0]),
									  luid_data[LUIDInfo.LUID_INFO_DETAIL][0])
			self.luidlist.column_titles_passive()
			self.luidlist.set_column_auto_resize(1, gtk.TRUE)
			self.luidsw.add(self.luidlist)
			for line in luid_data[LUIDInfo.LUID_INFO_DETAIL][1:]:
				self.luidlist.append(tuple(line))
			self.luidlist.columns_autosize()
			self.luidlist.thaw()
			self.luidlist.show()

	def close(self, widget=None, data=None):
		self.destroy()

	def get_prefs(self):
		prefs = {}

		# CheckButtons
		for name in [PreferenceManager.PREF_STARTUP_OPEN_LOG,
					 PreferenceManager.PREF_STARTUP_AUTO_JOIN,
					 PreferenceManager.PREF_AUTO_JOIN_ALL,
					 PreferenceManager.PREF_DOWNLOAD,
					 PreferenceManager.PREF_FORWARD,
					 PreferenceManager.PREF_FORWARD_ACCEPT_ALL,
					 PreferenceManager.PREF_FORWARD_ONLY_LISTED,
					 PreferenceManager.PREF_AUTO_EDITOR_CLEAR,
					 PreferenceManager.PREF_LOGGING_DISCONNECTED,
					 PreferenceManager.PREF_LOGGING_CURRENT_CLOSED,
					 PreferenceManager.PREF_LOGGING_COMPRESS,
					 PreferenceManager.PREF_GVIEWER_OPEN_PLAY,
					 PreferenceManager.PREF_FOCUS_TAB,
					 PreferenceManager.PREF_LOGGING_DOWNLOAD,
					 PreferenceManager.PREF_LOGGING_SENT,
					 PreferenceManager.PREF_GVIEWER_USE,
					 PreferenceManager.PREF_GVIEWER_STICKY] + DL_SUFFIX_PATTERNS.keys():
			prefs[name] = self.prefs[name].get_active()

		# SpinButtons
		for name in [PreferenceManager.PREF_GVIEWER_WAIT_W1_TAG,
					 PreferenceManager.PREF_GVIEWER_WAIT_1_CHAR]:
			prefs[name] = self.prefs[name].get_value_as_int()

		# Entries
		for name in [PreferenceManager.PREF_BROWSER_COMMAND,
					 PreferenceManager.PREF_COLOR_SCRIPT,
					 PreferenceManager.PREF_COLOR_SAKURA,
					 PreferenceManager.PREF_COLOR_KERO,
					 PreferenceManager.PREF_COLOR_SYNCHRONIZED,
					 PreferenceManager.PREF_COLOR_ERROR,
					 PreferenceManager.PREF_COLOR_URL,
					 PreferenceManager.PREF_INITIALSCRIPT,
					 PreferenceManager.PREF_LOGGING_PATH,
					 PreferenceManager.PREF_FONT_GENERAL,
					 PreferenceManager.PREF_FONT_SCRIPT]:
			prefs[name] = self.prefs[name].get_text()

		# RadioButtons
		for name in [POS_NEXT, POS_PREV,
					 POS_FIRST, POS_END]:
			if self.prefs[name].get_active():
				prefs[PreferenceManager.PREF_POSITION_PAGE] = name
				break
		for type in [gtk.POS_TOP, gtk.POS_BOTTOM, gtk.POS_LEFT, gtk.POS_RIGHT]:
			type = int(type)
			name = 'LOGTABPOS_%d' % type
			if self.prefs[name].get_active():
				prefs[PreferenceManager.PREF_POSITION_TAB] = type
				break

		return prefs

	def set_prefs(self, prefmanager):
		for key in prefmanager.rc.keys():
			if self.prefs.has_key(key):
				# Entries
				if key in [PreferenceManager.PREF_BROWSER_COMMAND,
						   PreferenceManager.PREF_COLOR_SCRIPT,
						   PreferenceManager.PREF_COLOR_SAKURA,
						   PreferenceManager.PREF_COLOR_KERO,
						   PreferenceManager.PREF_COLOR_SYNCHRONIZED,
						   PreferenceManager.PREF_COLOR_ERROR,
						   PreferenceManager.PREF_COLOR_URL,
						   PreferenceManager.PREF_INITIALSCRIPT,
						   PreferenceManager.PREF_LOGGING_PATH,
						   PreferenceManager.PREF_FONT_GENERAL,
						   PreferenceManager.PREF_FONT_SCRIPT]:
					self.prefs[key].set_text(prefmanager.rc[key])

				# SpinButtons
				elif key in [PreferenceManager.PREF_GVIEWER_WAIT_W1_TAG,
							 PreferenceManager.PREF_GVIEWER_WAIT_1_CHAR]:
					self.prefs[key].set_value(int(prefmanager.rc[key]))

				else:
					self.prefs[key].set_active(prefmanager.rc[key])

			if key == PreferenceManager.PREF_POSITION_PAGE:
				for type in [POS_NEXT, POS_PREV,
							 POS_FIRST, POS_END]:
					if prefmanager.rc[key] == type:
						self.prefs[type].set_active(gtk.TRUE)
						break
			elif key == PreferenceManager.PREF_POSITION_TAB:
				for type in [gtk.POS_TOP, gtk.POS_BOTTOM, gtk.POS_LEFT, gtk.POS_RIGHT]:
					type = int(type)
					name = 'LOGTABPOS_%d' % type
					if prefmanager.rc[key] == str(type):
						self.prefs[name].set_active(gtk.TRUE)
						break

		if not self.prefs[PreferenceManager.PREF_FORWARD].get_active():
			self.prefs[PreferenceManager.PREF_FORWARD_ACCEPT_ALL].set_sensitive(gtk.FALSE)
			self.prefs[PreferenceManager.PREF_FORWARD_ONLY_LISTED].set_sensitive(gtk.FALSE)

	def forward_toggled(self, widget=None, data=None):
		state = self.prefs[PreferenceManager.PREF_FORWARD].get_active()
		self.prefs[PreferenceManager.PREF_FORWARD_ACCEPT_ALL].set_sensitive(state)
		self.prefs[PreferenceManager.PREF_FORWARD_ONLY_LISTED].set_sensitive(state)

	def download_toggled(self, widget=None, data=None):
		state = self.prefs[PreferenceManager.PREF_DOWNLOAD].get_active()
		self.prefs[PreferenceManager.PREF_LOGGING_DOWNLOAD].set_sensitive(state)
		for p in DL_SUFFIX_PATTERNS.keys():
			self.prefs[p].set_sensitive(state)

	def logging_toggled(self, widget=None, data=None):
		logging_state = self.prefs[PreferenceManager.PREF_LOGGING_DISCONNECTED].get_active()
		logcurr_state = self.prefs[PreferenceManager.PREF_LOGGING_CURRENT_CLOSED].get_active()

		if not logging_state and not logcurr_state:
			self.prefs[PreferenceManager.PREF_LOGGING_PATH].set_sensitive(gtk.FALSE)
			self.prefs[PreferenceManager.PREF_LOGGING_COMPRESS].set_sensitive(gtk.FALSE)
		else:
			self.prefs[PreferenceManager.PREF_LOGGING_PATH].set_sensitive(gtk.TRUE)
			self.prefs[PreferenceManager.PREF_LOGGING_COMPRESS].set_sensitive(gtk.TRUE)

	def viewer_toggled(self, widget=None, data=None):
		viewer_state = self.prefs[PreferenceManager.PREF_GVIEWER_USE].get_active()
		self.prefs[PreferenceManager.PREF_GVIEWER_STICKY].set_sensitive(viewer_state)
		self.prefs[PreferenceManager.PREF_GVIEWER_OPEN_PLAY].set_sensitive(viewer_state)
		self.prefs[PreferenceManager.PREF_GVIEWER_WAIT_1_CHAR].set_sensitive(viewer_state)
		self.prefs[PreferenceManager.PREF_GVIEWER_WAIT_W1_TAG].set_sensitive(viewer_state)

	def create_general_frame(self):
		self.prefs[PreferenceManager.PREF_STARTUP_OPEN_LOG
				   ] = gtk.CheckButton(unicode(_('Open log window when startup')))
		self.prefs[PreferenceManager.PREF_STARTUP_AUTO_JOIN
				   ] = gtk.CheckButton(unicode(_('Connect bottle server when startup')))
		self.prefs[PreferenceManager.PREF_AUTO_JOIN_ALL
				   ] = gtk.CheckButton(unicode(_('Join all channels when connect')))
		self.prefs[PreferenceManager.PREF_AUTO_EDITOR_CLEAR
				   ] = gtk.CheckButton(unicode(_('Clear editor after send')))
		self.prefs[PreferenceManager.PREF_LOGGING_SENT
				   ] = gtk.CheckButton(unicode(_('Logging history of sent script')))

		genbox = gtk.VBox()
		for name in [PreferenceManager.PREF_STARTUP_OPEN_LOG,
					 PreferenceManager.PREF_STARTUP_AUTO_JOIN,
					 PreferenceManager.PREF_AUTO_JOIN_ALL,
					 PreferenceManager.PREF_AUTO_EDITOR_CLEAR,
					 PreferenceManager.PREF_LOGGING_SENT]:
			self.prefs[name].set_border_width(5)
			genbox.pack_start(self.prefs[name], gtk.FALSE, gtk.FALSE, 0)

		genframe = gtk.Frame(unicode(_('General')))
		genframe.set_border_width(5)
		genframe.add(genbox)
		return genframe

	def create_sstp_frame(self):
		sendbox = gtk.VBox()

		self.prefs[PreferenceManager.PREF_FORWARD] = gtk.CheckButton(unicode(_('Forward')))
		self.prefs[PreferenceManager.PREF_FORWARD_ACCEPT_ALL
				   ] = gtk.RadioButton(None, unicode(_('Accept All')))
		self.prefs[PreferenceManager.PREF_FORWARD_ONLY_LISTED
				   ] = gtk.RadioButton(self.prefs[PreferenceManager.PREF_FORWARD_ACCEPT_ALL],
									   unicode(_('Forward only listed Ghost')))

		for name in [PreferenceManager.PREF_FORWARD,
					 PreferenceManager.PREF_FORWARD_ACCEPT_ALL,
					 PreferenceManager.PREF_FORWARD_ONLY_LISTED]:
			self.prefs[name].set_border_width(5)
			sendbox.pack_start(self.prefs[name], gtk.FALSE, gtk.FALSE, 0)
		self.prefs[PreferenceManager.PREF_FORWARD].connect('toggled', self.forward_toggled)

		sendframe = gtk.Frame(unicode(_('SSTP server')))
		sendframe.set_border_width(5)
		sendframe.add(sendbox)
		return sendframe

	def create_initial_script_frame(self):
		self.prefs[PreferenceManager.PREF_INITIALSCRIPT] = gtk.Entry()

		iniscbox = gtk.VBox()
		iniscbox.pack_start(self.prefs[PreferenceManager.PREF_INITIALSCRIPT], gtk.FALSE, gtk.FALSE, 5)
		iniscbox.pack_start(gtk.Label(unicode(_('A first | is cursor position'), 'utf-8')),
								 gtk.FALSE, gtk.FALSE, 5)

		iniscframe = gtk.Frame(unicode(_('Initial script')))
		iniscframe.set_border_width(5)
		iniscframe.add(iniscbox)
		return iniscframe

	def create_general_tab(self):
		genframe   = self.create_general_frame()
		sendframe  = self.create_sstp_frame()
		iniscframe = self.create_initial_script_frame()

		general_tab = gtk.VBox()
		for item in [genframe, sendframe, iniscframe]:
			general_tab.pack_start(item, gtk.FALSE, gtk.FALSE, 0)
		return general_tab

	def create_font_frame(self):
		fonttable = gtk.Table(2, 3)
		list = [PreferenceManager.PREF_FONT_GENERAL, PreferenceManager.PREF_FONT_SCRIPT]
		for i in range(len(list)):
			self.prefs[list[i]] = gtk.Entry()
			fonttable.attach(gtk.Label(unicode(_(list[i]), 'utf-8')), 0, 1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
			fonttable.attach(self.prefs[list[i]], 1, 2, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
			btn = get_icon_button(gtk.STOCK_SELECT_FONT, size=gtk.ICON_SIZE_BUTTON, relief=gtk.RELIEF_NORMAL)
			btn.connect('clicked', self.open_fontselection, self.prefs[list[i]])
			fonttable.attach(btn, 2, 3, i, i+1, gtk.FILL, gtk.FILL, 0, 3)

		fontbox = gtk.VBox()
		fontbox.pack_start(fonttable, gtk.FALSE, gtk.FALSE, 0)

		fontframe = gtk.Frame(unicode(_('Font settings')))
		fontframe.set_border_width(5)
		fontframe.add(fontbox)
		return fontframe

	def create_color_frame(self):
		colortable = gtk.Table(6, 3)
		list = [PreferenceManager.PREF_COLOR_SCRIPT,
				PreferenceManager.PREF_COLOR_SAKURA,
				PreferenceManager.PREF_COLOR_KERO,
				PreferenceManager.PREF_COLOR_SYNCHRONIZED,
				PreferenceManager.PREF_COLOR_ERROR,
				PreferenceManager.PREF_COLOR_URL]
		for i in range(len(list)):
			self.prefs[list[i]] = gtk.Entry()
			colortable.attach(gtk.Label(unicode(_(list[i]), 'utf-8')), 0, 1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
			colortable.attach(self.prefs[list[i]], 1, 2, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
			btn = get_icon_button(gtk.STOCK_SELECT_COLOR, size=gtk.ICON_SIZE_BUTTON, relief=gtk.RELIEF_NORMAL)
			btn.connect('clicked', self.open_colorselection, self.prefs[list[i]])
			colortable.attach(btn, 2, 3, i, i+1, gtk.FILL, gtk.FILL, 0, 3)

		colorbox = gtk.VBox()
		colorbox.pack_start(colortable, gtk.FALSE, gtk.FALSE, 0)

		colorframe = gtk.Frame(unicode(_('Color settings')))
		colorframe.set_border_width(5)
		colorframe.add(colorbox)
		return colorframe

	def create_font_tab(self):
		font_tab = gtk.VBox()
		font_tab.pack_start(self.create_font_frame())
		font_tab.pack_start(self.create_color_frame())
		return font_tab

	def create_gviewer_tab(self):
		self.prefs[PreferenceManager.PREF_GVIEWER_USE] = gtk.CheckButton(unicode(_('Use GViewer'), 'utf-8'))
		self.prefs[PreferenceManager.PREF_GVIEWER_USE].set_border_width(2)
		self.prefs[PreferenceManager.PREF_GVIEWER_USE].connect('toggled', self.viewer_toggled)

		self.prefs[PreferenceManager.PREF_GVIEWER_STICKY] = gtk.CheckButton(unicode(_('Stick GViewer'), 'utf-8'))
		self.prefs[PreferenceManager.PREF_GVIEWER_STICKY].set_border_width(2)

		self.prefs[PreferenceManager.PREF_GVIEWER_OPEN_PLAY
				   ] = gtk.CheckButton(unicode(_('Auto open GViewer when play script'), 'utf-8'))
		self.prefs[PreferenceManager.PREF_GVIEWER_OPEN_PLAY].set_border_width(2)

		adj = gtk.Adjustment(1.0, 0.0, 500.0, 1.0, 5.0, 0.0)
		self.prefs[PreferenceManager.PREF_GVIEWER_WAIT_1_CHAR] = gtk.SpinButton(adj, 0, 0)

		adj = gtk.Adjustment(1.0, 0.0, 500.0, 1.0, 5.0, 0.0)
		self.prefs[PreferenceManager.PREF_GVIEWER_WAIT_W1_TAG] = gtk.SpinButton(adj, 0, 0)

		wait_table = gtk.Table(2, 3)
		wait_table.attach(gtk.Label(unicode(_('Wait for a character'), 'utf-8')),
						  0, 1, 0, 1, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(self.prefs[PreferenceManager.PREF_GVIEWER_WAIT_1_CHAR],
						  1, 2, 0, 1, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(gtk.Label(unicode(_('milli seconds'), 'utf-8')),
						  2, 3, 0, 1, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(gtk.Label(unicode(_('Wait for \\w1'), 'utf-8')),
						  0, 1, 1, 2, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(self.prefs[PreferenceManager.PREF_GVIEWER_WAIT_W1_TAG],
						  1, 2, 1, 2, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(gtk.Label(unicode(_('milli seconds'), 'utf-8')),
						  2, 3, 1, 2, gtk.FILL, gtk.FILL, 3, 3)

		viewbox = gtk.VBox()
		viewbox.pack_start(self.prefs[PreferenceManager.PREF_GVIEWER_USE], gtk.FALSE, gtk.FALSE, 0)
		viewbox.pack_start(self.prefs[PreferenceManager.PREF_GVIEWER_STICKY], gtk.FALSE, gtk.FALSE, 0)
		viewbox.pack_start(self.prefs[PreferenceManager.PREF_GVIEWER_OPEN_PLAY], gtk.FALSE, gtk.FALSE, 0)
		viewbox.pack_start(wait_table, gtk.FALSE, gtk.FALSE, 0)

		viewerframe = gtk.Frame(unicode(_('GViewer'), 'utf-8'))
		viewerframe.set_border_width(5)
		viewerframe.add(viewbox)

		viewer_tab = gtk.VBox()
		viewer_tab.pack_start(viewerframe, gtk.FALSE, gtk.FALSE, 0)
		self.viewer_toggled()
		return viewer_tab

	def create_logging_frame(self):
		self.prefs[PreferenceManager.PREF_LOGGING_DISCONNECTED
				   ] = gtk.CheckButton(unicode(_("Save `Current' log as XML when disconnect"), 'utf-8'))
		self.prefs[PreferenceManager.PREF_LOGGING_DISCONNECTED].set_border_width(2)
		self.prefs[PreferenceManager.PREF_LOGGING_DISCONNECTED].connect('toggled', self.logging_toggled)

		self.prefs[PreferenceManager.PREF_LOGGING_CURRENT_CLOSED
				   ] = gtk.CheckButton(unicode(_("Save log as XML when close `Current' tab"), 'utf-8'))
		self.prefs[PreferenceManager.PREF_LOGGING_CURRENT_CLOSED].set_border_width(2)
		self.prefs[PreferenceManager.PREF_LOGGING_CURRENT_CLOSED].connect('toggled', self.logging_toggled)

		self.prefs[PreferenceManager.PREF_LOGGING_COMPRESS] = gtk.CheckButton(unicode(_('GZIP Compress'),
																					  'utf-8'))
		self.prefs[PreferenceManager.PREF_LOGGING_COMPRESS].set_border_width(2)
		self.prefs[PreferenceManager.PREF_LOGGING_COMPRESS].set_sensitive(gtk.FALSE)

		self.prefs[PreferenceManager.PREF_LOGGING_PATH] = gtk.Entry()
		self.prefs[PreferenceManager.PREF_LOGGING_PATH].set_sensitive(gtk.FALSE)

		lbox = gtk.VBox()
		lbox.pack_start(self.prefs[PreferenceManager.PREF_LOGGING_PATH], gtk.FALSE, gtk.FALSE, 3)
		lbox.pack_start(gtk.Label(unicode(_('Can use time format e.g. %H, %M, %S and etc.'), 'utf-8')),
						gtk.FALSE, gtk.FALSE, 3)

		lframe = gtk.Frame(unicode(_('Filename'), 'utf-8'))
		lframe.set_shadow_type(gtk.SHADOW_ETCHED_OUT)
		lframe.set_border_width(5)
		lframe.add(lbox)

		logbox1 = gtk.VBox()
		for item in [self.prefs[PreferenceManager.PREF_LOGGING_DISCONNECTED],
					 self.prefs[PreferenceManager.PREF_LOGGING_CURRENT_CLOSED],
					 self.prefs[PreferenceManager.PREF_LOGGING_COMPRESS], lframe]:
			logbox1.pack_start(item, gtk.FALSE, gtk.FALSE, 2)

		logframe1 = gtk.Frame(unicode(_('Logs')))
		logframe1.set_border_width(5)
		logframe1.add(logbox1)
		return logframe1

	def create_logtab_behavior_frame(self):
		self.prefs[PreferenceManager.PREF_FOCUS_TAB
				   ] = gtk.CheckButton(unicode(_('Focus new page automatically'), 'utf-8'))
		self.prefs[PreferenceManager.PREF_FOCUS_TAB].set_border_width(2)

		pagepos_table = gtk.Table(2, 2)
		prev = None
		list = [[
			[POS_NEXT,  "To current tab's next"],
			[POS_PREV,  "To current tab's previous"]
			],
				[
			[POS_END,   'As end tab'],
			[POS_FIRST, 'As first tab']
			]]
		for i in range(len(list)):
			for j in range(len(list[i])):
				pref = list[i][j][0]
				description = list[i][j][1]

				self.prefs[pref]  = gtk.RadioButton(prev, unicode(_(description)))
				self.prefs[pref].set_border_width(2)
				pagepos_table.attach(self.prefs[pref], j, j+1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
				prev = self.prefs[pref]

		logbox2 = gtk.VBox()
		logbox2.pack_start(self.prefs[PreferenceManager.PREF_FOCUS_TAB], gtk.FALSE, gtk.FALSE, 2)
		logbox2.pack_start(pagepos_table, gtk.FALSE, gtk.FALSE, 2)

		logframe2 = gtk.Frame(unicode(_('When open new page...')))
		logframe2.set_border_width(5)
		logframe2.add(logbox2)
		return logframe2

	def create_logtab_position_frame(self):
		tabpos_table = gtk.Table(4, 1)
		prev = None
		list = [['Top', 'Bottom', 'Left', 'Right']]
		for i in range(len(list)):
			for j in range(len(list[i])):
				label = list[i][j]

				pref_name = None
				if label == 'Right':
					pref_name = 'LOGTABPOS_%d' % int(gtk.POS_RIGHT)
				elif label == 'Left':
					pref_name = 'LOGTABPOS_%d' % int(gtk.POS_LEFT)
				elif label == 'Bottom':
					pref_name = 'LOGTABPOS_%d' % int(gtk.POS_BOTTOM)
				else:
					pref_name = 'LOGTABPOS_%d' % int(gtk.POS_TOP)

				self.prefs[pref_name] = gtk.RadioButton(prev, unicode(_(label)))
				tabpos_table.attach(self.prefs[pref_name], j, j+1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
				prev = self.prefs[pref_name]

		logframe3 = gtk.Frame(unicode(_('Tab position')))
		logframe3.set_border_width(5)
		logframe3.add(tabpos_table)
		return logframe3

	def create_log_tab(self):
		log_tab = gtk.VBox()
		for item in [self.create_logging_frame(), self.create_logtab_behavior_frame(),
					 self.create_logtab_position_frame()]:
			log_tab.pack_start(item, gtk.FALSE, gtk.FALSE, 0)
		return log_tab

	def create_browser_frame(self):
		self.prefs[PreferenceManager.PREF_BROWSER_COMMAND] = gtk.Entry()

		brsbox = gtk.VBox()
		brsbox.pack_start(self.prefs[PreferenceManager.PREF_BROWSER_COMMAND], gtk.FALSE, gtk.FALSE, 5)
		brsbox.pack_start(gtk.Label(unicode(_('Replace %s to URL'), 'utf-8')), gtk.TRUE, gtk.TRUE, 3)

		brsframe = gtk.Frame(unicode(_('Command to run browser')))
		brsframe.set_border_width(5)
		brsframe.add(brsbox)
		return brsframe

	def create_download_frame(self):
		self.prefs[PreferenceManager.PREF_DOWNLOAD
				   ] = gtk.CheckButton(unicode(_('Auto download when receive script contained URL')))
		self.prefs[PreferenceManager.PREF_DOWNLOAD].set_border_width(5)
		self.prefs[PreferenceManager.PREF_DOWNLOAD].connect('toggled', self.download_toggled)

		self.prefs[PreferenceManager.PREF_LOGGING_DOWNLOAD] = gtk.CheckButton(unicode(_('Logging history')))
		self.prefs[PreferenceManager.PREF_LOGGING_DOWNLOAD].set_border_width(5)
		self.prefs[PreferenceManager.PREF_LOGGING_DOWNLOAD].set_sensitive(gtk.FALSE)

		# make table
		rows = 4
		ex_table = gtk.Table(7, rows)

		keys = DL_SUFFIX_PATTERNS.keys()
		keys.sort()
		length = len(keys) / rows + 1
		list = [keys[length*i:length*(i+1)] for i in range(rows)]

		for i in range(len(list)):
			for j in range(len(list[i])):
				self.prefs[list[i][j]] = gtk.CheckButton(list[i][j])
				self.prefs[list[i][j]].set_sensitive(gtk.FALSE)
				ex_table.attach(self.prefs[list[i][j]], j, j+1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)

		ex_frame = gtk.Frame(unicode(_('Supported file types'), 'utf-8'))
		ex_frame.set_shadow_type(gtk.SHADOW_ETCHED_OUT)
		ex_frame.set_border_width(5)
		ex_frame.add(ex_table)

		dlbox = gtk.VBox()
		dlbox.pack_start(self.prefs[PreferenceManager.PREF_DOWNLOAD], gtk.FALSE, gtk.FALSE, 0)
		dlbox.pack_start(self.prefs[PreferenceManager.PREF_LOGGING_DOWNLOAD], gtk.FALSE, gtk.FALSE, 0)
		dlbox.pack_start(ex_frame, gtk.FALSE, gtk.FALSE, 0)

		dlframe = gtk.Frame(unicode(_('Download'), 'utf-8'))
		dlframe.set_border_width(5)
		dlframe.add(dlbox)
		return dlframe

	def create_network_tab(self):
		net_tab = gtk.VBox()
		for item in [self.create_browser_frame(), self.create_download_frame()]:
			net_tab.pack_start(item, gtk.FALSE, gtk.FALSE, 0)
		return net_tab


class PreferenceManager:
	PREF_STARTUP_OPEN_LOG  = 'auto_open_log'
	PREF_STARTUP_AUTO_JOIN = 'auto_connect'

	PREF_AUTO_JOIN_ALL     = 'auto_join'
	PREF_AUTO_EDITOR_CLEAR = 'auto_clear'

	PREF_LOGGING_SENT = 'sent_log'

	PREF_FORWARD             = 'forward'
	PREF_FORWARD_ACCEPT_ALL  = 'accept_all'
	PREF_FORWARD_ONLY_LISTED = 'forward_listed'

	PREF_INITIALSCRIPT = 'initialscript'

	PREF_LOGGING_DISCONNECTED   = 'logging'
	PREF_LOGGING_CURRENT_CLOSED = 'logging_current_tab_closed'
	PREF_LOGGING_COMPRESS       = 'logging_compress'
	PREF_LOGGING_PATH           = 'logging_path'

	PREF_FOCUS_TAB     = 'logtabfocus'
	PREF_POSITION_TAB  = 'logtabpos'
	PREF_POSITION_PAGE = 'logpagepos'

	PREF_BROWSER_COMMAND = 'browser'

	PREF_DOWNLOAD         = 'autodl'
	PREF_LOGGING_DOWNLOAD = 'download_log'

	PREF_FONT_GENERAL = 'general font'
	PREF_FONT_SCRIPT  = 'script font'

	PREF_COLOR_SCRIPT       = 'script'
	PREF_COLOR_SAKURA       = 'sakura'
	PREF_COLOR_KERO         = 'kero'
	PREF_COLOR_SYNCHRONIZED = 'sync'
	PREF_COLOR_ERROR        = 'error'
	PREF_COLOR_URL          = 'URL'

	PREF_GVIEWER_USE         = 'use_gviewer'
	PREF_GVIEWER_STICKY      = 'sticky_gviewer'
	PREF_GVIEWER_OPEN_PLAY   = 'open_gviewer_when_play'
	PREF_GVIEWER_WAIT_1_CHAR = 'wait_1_char'
	PREF_GVIEWER_WAIT_W1_TAG = 'wait_w1_tag'

	DBFILE = 'bottlerc'

	def __init__(self):
		self.dbpath = os.path.join(open_bottlecase(), self.DBFILE)
		self.rc = {}
		self.initialize()
		self.load_database()

	def initialize(self):
		for k, v in [[self.PREF_STARTUP_OPEN_LOG, False],
					 [self.PREF_STARTUP_AUTO_JOIN, False],
					 [self.PREF_AUTO_JOIN_ALL, False],
					 [self.PREF_DOWNLOAD, False],
					 [self.PREF_LOGGING_DOWNLOAD, True],
					 [self.PREF_FORWARD, True],
					 [self.PREF_FORWARD_ACCEPT_ALL, True],
					 [self.PREF_FORWARD_ONLY_LISTED, False],
					 [self.PREF_AUTO_EDITOR_CLEAR, True],
					 [self.PREF_LOGGING_SENT, True],
					 [self.PREF_GVIEWER_USE, False],
					 [self.PREF_GVIEWER_STICKY, False],
					 [self.PREF_GVIEWER_OPEN_PLAY, False],
					 [self.PREF_INITIALSCRIPT, r'\t\u\s[10]\h\s[0]|\e'],
					 [self.PREF_GVIEWER_WAIT_1_CHAR, 47],
					 [self.PREF_GVIEWER_WAIT_W1_TAG, 50],
					 [self.PREF_LOGGING_DISCONNECTED, False],
					 [self.PREF_LOGGING_CURRENT_CLOSED, False],
					 [self.PREF_LOGGING_COMPRESS, False],
					 [self.PREF_LOGGING_PATH, os.path.normpath(open_bottlecase() + '/autolog/%Y%m%d/%H-%M-%S.xml')],
					 [self.PREF_FOCUS_TAB, True],
					 [self.PREF_POSITION_PAGE, POS_NEXT],
					 [self.PREF_POSITION_TAB, int(gtk.POS_TOP)],
					 [self.PREF_FONT_GENERAL, 'Sans 10'],
					 [self.PREF_FONT_SCRIPT, 'Monospace 10'],
					 [self.PREF_BROWSER_COMMAND, "mozilla -remote 'openFile(%s,new-tab)'"],
					 [self.PREF_COLOR_SCRIPT, 'dark green'],
					 [self.PREF_COLOR_SAKURA, 'black'],
					 [self.PREF_COLOR_KERO, 'brown'],
					 [self.PREF_COLOR_SYNCHRONIZED, 'blue'],
					 [self.PREF_COLOR_ERROR, 'red'],
					 [self.PREF_COLOR_URL, 'purple']] + [[i, True] for i in DL_SUFFIX_PATTERNS.keys()]:
			self.rc[k] = v

	def load_database(self):
		pat = re.compile('\s*:\s*')
		try:
			file = open(self.dbpath)
		except IOError:
			return
		while 1:
			line = unicode(file.readline(), 'utf-8')
			if not line:
				break
			elif line[0] in ['#', ';']:
				continue
			elif line[0:2] == '//':
				continue
			elif line[-2:] == '\r\n':
				line = line[:-2]
			elif line[-1] in ['\r', '\n']:
				line = line[:-1]

			[name, value] = pat.split(line, 1)
			if value == 'True':
				value = True
			elif value == 'False':
				value = False
			self.rc[name] = value
		file.close()

	def save_database(self, prefdialog):
		try:
			file = open(self.dbpath, 'w')
		except IOError:
			sys.stderr.write('Error: cannot write %s\n' % self.dbpath)
			return
		self.rc = prefdialog.get_prefs()
		file.write('# This is an auto-generated file.\n')
		file.write("# Don't edit this file while running %s!\n" % APP)
		for key in self.rc.keys():
			file.write('%s: %s\n' % (key.encode('utf-8'), self.rc[key]))
		file.close()

	def get(self, name):
		if not self.rc.has_key(name):
			raise ValueError('rc has no key %s\n' % name)
		return self.rc[name]

	def get_all(self):
		return self.rc
