
/*	
	GameboyAdvanceVM 
		- Nintendo GameboyAdvance Emulator
	Copyright 2002 Y_N y_n@users.sourceforge.jp
	Homepage https://sourceforge.jp/projects/gbaemu/
*/


#define	FOREG		(u8)((opcode>>16)&0xF)	/*1st operand register*/
#define	SOREG		(u8)(opcode&0xF)		/*2nd operand register*/
#define	DEREG		(u8)((opcode>>12)&0xF)	/*Destination register*/

#define	REG_0_4		(u8)((opcode)&0xF)
#define	REG_12_4	(u8)((opcode>>12)&0xF)
#define	REG_16_4	(u8)((opcode>>16)&0xF)

#define	IMM_OPERAND		(opcode & BIT_25_)	/*Immediate Operand (l)*/

#define	DPRINTF(str, cond)	sprintf(mne, "%08X %08X "str"%s", pc, opcode, cond_table[cond])
#define	GETCOND				(u8)((opcode>>28)&0x0F)

#define	OPCODE_		RMem32(pc)
#define	STR_BUF_SIZE			128
#define	STR_OFFSET_BUF_SIZE		32

extern u32 imm_shift(u32 opcode);
extern u32 imm_rotate(u32 opcode);

const char *cond_table[]={	/*RfBVe[u()*/
	"eq",	/*if Z set*/
	"ne",	/*if Z clear*/
	"cs",	/*if C set*/
	"cc",	/*if C clear*/
	"mi",	/*if N set*/
	"pl",	/*if N clear*/
	"vs",	/*if V set*/
	"vc",	/*if V clear*/
	"hi",	/*if C set and Z clear*/
	"ls",	/*if C clear and Z set*/
	"ge",	/*if ((N set and V set) or (N clear and V clear))*/
	"lt",	/*if ((N set and V clear) or (N clear and V set))*/
	"gt",	/*if (Z clear and(N or V set) or (N or V clear))*/
	"le",	/*if (Z set or(N set and V clear) or (N clear and V set))*/
	"",		/*always */
	"?",	/*`*/
};

void CDisassemblerDlg::disasm_arm_adc()
{	/*Rd := Rn + Op2 + Carry <Data Processing> (Z{L[)*/
	u32	opcode,	operand;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X adc%s r%d, r%d, %s", 
		pc, opcode, cond_table[cond], rd, rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_add()
{	/*Rd := Rn + Op2 <Data Processing> (Z)*/
	u32	opcode, operand;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	/*arm.reg[rd] = arm.reg[rn] + operand;*/
	sprintf(mne, "%08X %08X add%s r%d, r%d, %s", 
		pc, opcode, cond_table[cond], rd, rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_and()
{	/*Rd := Rn AND Op2 <Data Processing>*/
	u32	opcode, operand;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X and%s r%d, r%d, %s", 
		pc, opcode, cond_table[cond], rd, rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_b()
{	/*R15 := address ()*/
	u32	opcode, address;
	s32	offset;/*signed*/
	u8	cond;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;

	offset = ((opcode & 0x00FFFFFF)<<2);

	address = offset + pc + 8;

	/*ItZbg͈̔͂MSB^Ȃ}CiX*/
	if(offset & BIT_25_)address -= BIT_25_<<1;

	sprintf(mne, "%08X %08X b%s #0x%08X", 
		pc, opcode, cond_table[cond], address);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_bic()
{	/*Rd := Rn AND NOT Op2 <Data Processing> (rbgNA)*/
	u32	opcode, operand;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X bic%s r%d, r%d, %s", 
		pc, opcode, cond_table[cond], rd, rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_bl()
{	/*R14 := R15, R15 := address*/
	u32	opcode, address;
	s32	offset;
	u8	cond;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	offset	= (u32)((opcode & 0x00FFFFFF)<<2);

	/*ItZbg*/
	address = offset + pc + 8;

	/*ItZbg͈̔͂MSB^Ȃ}CiX*/
	if(offset & BIT_25_)address -= BIT_25_<<1;

	sprintf(mne, "%08X %08X bl%s #0x%08X", 
		pc, opcode, cond_table[cond], address);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_bx()
{	/*R15 := Rn (Ɩ߃Xe[g̕ύX)*/
	u32	opcode;
	u8	cond, rn;
	char mne[STR_BUF_SIZE];
	
	opcode	= OPCODE_;
	cond	= GETCOND;
	rn		= REG_0_4;

	/*R15 = arm.reg[rn];*/
	sprintf(mne, "%08X %08X bx%s r%d", pc, opcode, cond_table[cond], rn);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_cdp()
{	/*(Coprocessor-specific) (RvZbT[f[^)*/
	u32	opcode;
	u8	cond;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;

	sprintf(mne, "%08X %08X cdp%s r%d", pc, opcode, cond_table[cond]);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_cmn()
{	/*CPSR flags := Rn + Op2 <Data Processing> (tr)*/
	u32	opcode, operand;
	u8	cond, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immidiate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X cmn%s r%d, %s", 
		pc, opcode, cond_table[cond], rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_cmp()
{	/*CPSR flags := Rn - Op2 <Data Processing> (r)*/
	u32	opcode, operand;
	u8	cond, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X cmp%s r%d, %s", 
		pc, opcode, cond_table[cond], rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_eor()
{	/*Rd := (Rn AND NOT Op2) OR (op2 AND NOT Rn) <Data Processing> (rI_a)*/
	u32	opcode, operand;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X eor%s r%d, r%d, %s", 
		pc, opcode, cond_table[cond], rd, rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_ldc()
{	/*Coprocessor load (RvZbTւ̓])*/
	u32	opcode;
	u8	cond, rd, rn, cp;
	u8	offset;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= (u8)((opcode>>12)&0x0F);
	rn		= (u8)((opcode>>16)&0x0F);
	cp		= (u8)((opcode)&0x0F);

	offset = (u8)(opcode);

	sprintf(mne, "%08X %08X ldc%s r%d, r%d, #0x%X", 
		pc, opcode, cond_table[cond], rd, rn, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_ldm()
{	/*Stack manipulation (̃WX^̃[h,POP)*/
	u32	opcode,	reg_list;
	u32	bit, lst, flag;
	u8	cond, rn, mode;
	char mne[STR_BUF_SIZE*2], str_sp[16], str_mode[16], str_list[128*2], str[8];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rn		= (u8)((opcode>>16)&0x0F);	/*16-19 4bit*/
	mode	= (u8)((opcode>>23)&0x03);	/*23-24 2bit*/

	reg_list = opcode & 0xFFFF;

	if(rn==13){	/*sp==r13*/
		strcpy(str_sp, "sp");
		switch(mode){	/*PU*/
		case 0:	/*__*/
			strcpy(str_mode, "ea");
			break;
		case 1:	/*_U*/
			strcpy(str_mode, "ed");
			break;
		case 2:	/*P_*/
			strcpy(str_mode, "fa");
			break;
		case 3:	/*PU*/
			strcpy(str_mode, "fd");
			break;
		}
	}else{	/*PU*/
		sprintf(str_sp, "r%d", rn);
		switch(mode){
		case 0:	/*__*/
			strcpy(str_mode, "da");
			break;
		case 1:	/*_U*/
			strcpy(str_mode, "ia");
			break;
		case 2:	/*P_*/
			strcpy(str_mode, "db");
			break;
		case 3:	/*PU*/
			strcpy(str_mode, "ib");
			break;
		}
	}

	strcpy(str_list, "");
	flag=1;
	for(bit=BIT_15_,lst=15; bit; bit>>=1,lst--){
		if(opcode & bit){
			//if(flag){
				flag=0;
				sprintf(str, "r%d,", lst);
				strcat(str_list, str);
			//}
		}else{
			flag=1;
		}
	}

	sprintf(mne, "%08X %08X ldm%s%s %s%s, {%s}", 
		pc, opcode, str_mode, cond_table[cond], str_sp, (opcode&BIT_21_?"!":""), str_list);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_ldr()
{	/*Rd := (address) (WX^ւ̃[h)*/
	u32	opcode,	offset, address;
	u8	cond, rd, rn;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;
	rn		= (u8)((opcode>>16)&0xF);	/*base register*/

	address = arm.reg[rn];

	if(rn==15)address += 8;	/*ΏۂPC̏ꍇ*/

	if(opcode & BIT_25_){	/*1 = ItZbg̓WX^*/
		offset = imm_shift(opcode);
	}else{					/*0 = ItZbg̓WX^*/
		offset = opcode & 0xFFF;	/*0-11bit*/
	}

	if((opcode & BIT_24_)){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT_23_){	/*1 = Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*0 = Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}
	
	if(offset && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%08X", address);
	}else{		/*ItZbg*/
		sprintf(str_offset, "r%d", rn);
	}

	sprintf(mne, "%08X %08X ldr%s%s r%d, [%s]", 
		pc, opcode, ((opcode & BIT_22_)?"b":""), cond_table[cond], rd, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_ldrs()
{	/*Rd := (address) (WX^ւ̃[h) Register offset*/
	u32	opcode;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (u8)((opcode>>16)&0xF);	/*AhX̃x[XWX^*/
	rm		= (u8)(opcode&0xF);

	if(opcode & BIT_5_){	/*Halfwords*/
		if(opcode & BIT_6_){	/*Signed*/
			/*11SH: Signed Halfwords*/
			strcpy(str_offset, "sh");
		}else{
			/*01: Unsigned Halfwords*/
			strcpy(str_offset, "h");
		}
	}else{
		if(opcode & BIT_6_){	/*Signed*/
			/*10: Signed byte*/
			strcpy(str_offset, "sb");
		}else{
			/*00: SWP instruction ɖ肠I*/
			strcpy(str_offset, "<SWP>");
		}
	}

	sprintf(mne, "%08X %08X ldr%s%s r%d, [r%d, r%d]", pc, opcode, str_offset, cond_table[cond], rd, rn, rm);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_ldrs_imm()
{	/*Rd := (address) (WX^ւ̃[h) Immidiate offset*/
	u32	opcode,	offset;
	u8	cond, rd, rn;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (u8)((opcode>>16)&0xF);	/*AhX̃x[XWX^*/
	offset	= (opcode&0xF)|((opcode>>4)&0xF0);

	if(opcode & BIT_5_){	/*Halfwords*/
		if(opcode & BIT_6_){	/*Signed*/
			/*Signed Halfwords*/
			strcpy(str_offset, "sh");
		}else{
			/*Unsigned Halfwords*/
			strcpy(str_offset, "h");
		}
	}else{
		if(opcode & BIT_6_){	/*Signed*/
			/*Signed byte*/
			strcpy(str_offset, "sb");
		}else{
			/*SWP instruction*/
			strcpy(str_offset, "<SWP>");
		}
	}

	sprintf(mne, "%08X %08X ldr%s%s r%d, [r%d, #0x%02X]", pc, opcode, str_offset, cond_table[cond], rd, rn, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_mcr()
{	/*cRn := rRn {<op>cRm}*/
	u32	opcode;
	u8	cond;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;

	sprintf(mne, "%08X %08X mcr%s", pc, opcode, cond_table[cond]);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_mla()
{	/*Rd := (Rm * Rs) + Rn (ϘaZ)*/
	u32	opcode;
	u8	cond, rd, rm, rs, rn;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= (u8)((opcode>>16)&0x0F);
	rs		= (u8)((opcode>>8)&0x0F);
	rm		= (u8)((opcode)&0x0F);
	rn		= (u8)((opcode>>12)&0x0F);

	/*arm.reg[rd] = (arm.reg[rm] * arm.reg[rs]) + arm.reg[rn];*/

	sprintf(mne, "%08X %08X mla%s r%d, r%d, r%d, r%d", 
		pc, opcode, cond_table[cond], rd, rs, rm, rn);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_mlal()
{	/*Rd := (Rm * Rs) + Rn (64bitϘaZ)*/
	u32	opcode;
	u8	cond, rdh, rdl, rm, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rdh		= (u8)((opcode>>16)&0x0F);
	rdl		= (u8)((opcode>>12)&0x0F);
	rs		= (u8)((opcode>>8)&0x0F);
	rm		= (u8)((opcode)&0x0F);

	sprintf(mne, "%08X %08X mlal%s r%d, r%d, r%d, r%d", 
		pc, opcode, cond_table[cond], rdl, rdh, rm, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_mov()
{	/*Rd : = Op2 <Data Processing> (WX^͒萔̑)*/
	u32	opcode, operand;
	u8	cond, rd, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X mov%s r%d, %s", 
		pc, opcode, cond_table[cond], rd, str_offset);

	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_mrc()
{	/*Rn := cRn {<op>cRm} (RvZbTWX^CPUWX^ւ̓])*/
	u32	opcode;
	u8	cond;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;

	sprintf(mne, "%08X %08X mrc%s", 
		pc, opcode, cond_table[cond]);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_mrs()
{	/*Rn := PSR (Xe[^XWX^烌WX^ւ̓])*/
	u32	opcode;
	u8	cond;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;

	sprintf(mne, "%08X %08X mrs%s", pc, opcode, cond_table[cond]);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_msr()
{	/*PSR := Rm (WX^Xe[^XWX^ւ̓])*/
	u32	opcode;
	u8	cond;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;

	sprintf(mne, "%08X %08X msr%s", pc, opcode, cond_table[cond]);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_mul()
{	/*Rd := Rm * Rs (ώZ)*/
	u32	opcode;
	u8	cond, rd, rm, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= (u8)((opcode>>16)&0x0F);
	rs		= (u8)((opcode>>8)&0x0F);
	rm		= (u8)((opcode)&0x0F);

	/*arm.reg[rd] = arm.reg[rm] * arm.reg[rs];*/

	sprintf(mne, "%08X %08X mul%s r%d, r%d, r%d", 
		pc, opcode, cond_table[cond], rd, rm, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_mull()
{	/*Rd := Rm * Rs (64bitώZ)*/
	u32	opcode;
	u8	cond, rdh, rdl, rm, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rdh		= (u8)((opcode>>16)&0x0F);
	rdl		= (u8)((opcode>>12)&0x0F);
	rs		= (u8)((opcode>>8)&0x0F);
	rm		= (u8)((opcode)&0x0F);

	sprintf(mne, "%08X %08X mull%s r%d, r%d, r%d, r%d", 
		pc, opcode, cond_table[cond], rdl, rdh, rm, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_mvn()
{	/*Rd := 0xFFFFFFFF EOR Op2 <Data Processing> (1̕␔)*/
	u32	opcode, operand;
	u8	cond, rd, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}
	/*arm.reg[rd] = 0xFFFFFFFF ^ arm.reg[op2];*/

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X mvn%s r%d, %s", 
		pc, opcode, cond_table[cond], rd, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_orr()
{	/*Rd := Rn OR Op2 <Data Processing> (_a)*/
	u32	opcode, operand;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X orr%s r%d, r%d, %s", 
		pc, opcode, cond_table[cond], rd, rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_rsb()
{	/*Rd := Op2 - Rn <Data Processing> (tZ)*/
	u32	opcode, operand;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X rsb%s r%d, r%d, %s", 
		pc, opcode, cond_table[cond], rd, rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_rsc()
{	/*Rd := Op2 - Rn <Data Processing> (L[ttZ)*/
	u32	opcode, operand;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X rsc%s r%d, r%d, %s", 
		pc, opcode, cond_table[cond], rd, rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_sbc()
{	/*Rd := Rn - Op2 - 1 + Carry <Data Processing> (L[tZ)*/
	u32	opcode, operand;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X sbc%s r%d, r%d, %s", 
		pc, opcode, cond_table[cond], rd, rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_stc()
{	/*address := CRn (RvZbT̓e֊i[)*/
	u32	opcode;
	u8	cond;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;

	sprintf(mne, "%08X %08X stc%s", pc, opcode, cond_table[cond]);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_stm()
{	/*Stack manipulation (̃WX^֊i[,PUSH)*/
	u32	opcode;
	u8	cond, rn, mode;
	char mne[STR_BUF_SIZE*2], str_sp[16], str_mode[16], str_list[128*2], str[8];
	u32	bit, lst, flag;

	opcode	= OPCODE_;
	cond	= GETCOND;
	rn		= (u8)((opcode>>16)&0x0F);	/*16-19 4bit*/
	mode	= (u8)((opcode>>23)&0x03);	/*23-24 2bit*/

	if(rn==13){	/*sp==r13*/
		strcpy(str_sp, "sp");
		switch(mode){	/*PU*/
		case 0:	/*__*/
			strcpy(str_mode, "ed");
			break;
		case 1:	/*_U*/
			strcpy(str_mode, "ea");
			break;
		case 2:	/*P_*/
			strcpy(str_mode, "fd");
			break;
		case 3:	/*PU*/
			strcpy(str_mode, "fa");
			break;
		}
	}else{	/*PU*/
		sprintf(str_sp, "r%d", rn);
		switch(mode){
		case 0:	/*__*/
			strcpy(str_mode, "da");
			break;
		case 1:	/*_U*/
			strcpy(str_mode, "ia");
			break;
		case 2:	/*P_*/
			strcpy(str_mode, "db");
			break;
		case 3:	/*PU*/
			strcpy(str_mode, "ib");
			break;
		}
	}

	strcpy(str_list, "");
	flag=1;
	for(bit=1,lst=0; bit!=BIT_16_; bit<<=1,lst++){
		if(opcode & bit){
			//if(flag){
				flag=0;
				sprintf(str, "r%d,", lst);
				strcat(str_list, str);
			//}
		}else{
			flag=1;
		}
	}

	sprintf(mne, "%08X %08X stm%s%s %s%s, {%s}", 
		pc, opcode, str_mode, cond_table[cond], str_sp, (opcode&BIT_21_?"!":""), str_list);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_str()
{	/*<address> := Rd (WX^̓e֊i[)*/
	u32	opcode,	offset, address;
	u8	cond, rd, rn;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (u8)((opcode>>16)&0xF);	/*AhX̃x[XWX^*/

	if(opcode & BIT_25_){	/*1 = ItZbg̓WX^*/
		offset = imm_shift(opcode);
	}else{					/*0 = ItZbg͑l*/
		offset = opcode & 0xFFF;
	}

	if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
		address = arm.reg[rn] + offset;
	}else{					/*Down: ItZbgx[X猸Z*/
		address = arm.reg[rn] - offset;
	}

	if(offset && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%08X", address);
	}else{		/*ItZbg*/
		sprintf(str_offset, "r%d", rn);
	}

	sprintf(mne, "%08X %08X str%s%s r%d, [%s]", pc, opcode, ((opcode & BIT_22_)?"b":""), cond_table[cond], rd, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_strs()
{	/*<address> := Rd (WX^̓e֊i[) Register offset*/
	u32	opcode;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (u8)((opcode>>16)&0xF);	/*AhX̃x[XWX^*/
	rm		= (u8)(opcode&0xF);

	if(opcode & BIT_5_){	/*Halfwords*/
		if(opcode & BIT_6_){	/*Signed*/
			/*11SH: Signed Halfwords STR͖`H*/
			strcpy(str_offset, "<sh>");
		}else{
			/*01: Unsigned Halfwords*/
			strcpy(str_offset, "h");
		}
	}else{
		if(opcode & BIT_6_){	/*Signed*/
			/*10: Signed byte STR͖`H*/
			strcpy(str_offset, "<sb>");
		}else{
			/*00: SWP instruction ɖ肠I*/
			strcpy(str_offset, "<SWP>");
		}
	}

	sprintf(mne, "%08X %08X str%s%s r%d, [r%d, r%d]", pc, opcode, str_offset, cond_table[cond], rd, rn, rm);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_strs_imm()
{	/*<address> := Rd (WX^̓e֊i[) Immidiate offset*/
	u32	opcode,	offset;
	u8	cond, rn, rd;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (u8)((opcode>>16)&0xF);	/*AhX̃x[XWX^*/
	offset	= (opcode&0xF)|((opcode>>4)&0xF0);

	if(opcode & BIT_5_){	/*Halfwords*/
		if(opcode & BIT_6_){	/*Signed*/
			/*Signed Halfwords*/
			strcpy(str_offset, "sh");
		}else{
			/*Unsigned Halfwords*/
			strcpy(str_offset, "h");
		}
	}else{
		if(opcode & BIT_6_){	/*Signed*/
			/*Signed byte*/
			strcpy(str_offset, "sb");
		}else{
			/*SWP instruction*/
			strcpy(str_offset, "<SWP>");
		}
	}

	sprintf(mne, "%08X %08X str%s%s r%d, [r%d, #0x%02X]", pc, opcode, str_offset, cond_table[cond], rd, rn, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_sub()
{	/*Rd := Rn - Op2 <Data Processing> (Z)*/
	u32	opcode, operand;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X sub%s r%d, r%d, %s", 
		pc, opcode, cond_table[cond], rd, rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_swi()
{	/*OS call (\tgEFA荞)*/
	u32	opcode, comment;
	u8	cond;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	comment	= opcode & 0x00FFFFFF;

	sprintf(mne, "%08X %08X swi%s #0x%X", pc, opcode, cond_table[cond], comment);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_swp()
{	/*Rd := [Rn], [Rn] := Rm (Pf[^̌)*/
	u32	opcode;
	u8	cond, rd, rn, rm;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rd		= (u8)((opcode>>12)&0x0F);
	rn		= (u8)((opcode>>16)&0x0F);
	rm		= (u8)((opcode)&0x0F);

	/*arm.reg[rd] = arm.reg[rn];
	arm.reg[rn] = arm.reg[rm];*/

	sprintf(mne, "%08X %08X swp%s r%d, r%d, r%d", 
		pc, opcode, cond_table[cond], rd, rn, rm);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_teq()
{	/*CPSR flags := Rn EOR Op2 <Data Processing> (rbgp^[̔r)*/
	u32	opcode, operand;
	u8	cond, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];
	opcode	= OPCODE_;
	cond	= GETCOND;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}
	/*CPSR = arm.reg[rn] ^ arm.reg[op2];*/

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X teq%s r%d, %s", 
		pc, opcode, cond_table[cond], rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_tst()
{	/*CPSR flags := Rn AND Op2 <Data Processing> (w肵rbg̃eXg)*/
	u32	opcode, operand;
	u8	cond, rn, rm;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_;
	cond	= GETCOND;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}
	/*CPSR = arm.reg[rn] & arm.reg[op2];*/

	if(operand && (opcode & (BIT_25_ | 0xFF0))){
		sprintf(str_offset, "#0x%0X", operand);
	}else{		/*ItZbg*/
		rm = SOREG;
		sprintf(str_offset, "r%d", rm);
	}

	sprintf(mne, "%08X %08X tst%s r%d, %s", 
		pc, opcode, cond_table[cond], rn, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_undef()
{	/*Undefine (`)*/
	u32	opcode;
	char mne[STR_BUF_SIZE];

	opcode = OPCODE_;

	sprintf(mne, "%08X %08X <undefine>", pc, opcode);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_arm_unknow()
{	/*Undefine (`)*/
	u32	opcode;
	char mne[STR_BUF_SIZE];

	opcode = OPCODE_;

	sprintf(mne, "%08X %08X <unknow>", pc, opcode);
	m_lstDisasm.AddString(mne);
}
