#include "stdafx.h"
#include "resource.h"

#include "FWatchApp.hpp"

#include "Utility.hpp"

#include "SettingInfoFactory.hpp"
#include "WatchThreadGroup.hpp"

#include <assert.h>

#include <fstream>
#include <exception>


//////////////////////////////////////////////

CWatchThreadGroup::CWatchThreadGroup( const bool v_dryrun )
	: modified_(false)
	, dryrun_(v_dryrun)
{
}

void CWatchThreadGroup::ReleaseAllThreads()
{
	// ~߂ׂẴXbhɑM(ҋ@͂ȂB)
	for (CWatchThreadList::iterator ite = watchList_.begin();
		ite != watchList_.end();
		++ite) {
		CWatchThread* pWatchThread = *ite;
		pWatchThread->StopNoWait();
	}

	// ׂẴXbhjB(X̃Xbh̔jҋ@)
	for (CWatchThreadList::iterator ite = watchList_.begin();
		ite != watchList_.end();
		++ite) {
		CWatchThread* pWatchThread = *ite;
		delete pWatchThread;
	}

	watchList_.clear();
	modified_ = false;
}

CWatchThreadGroup::~CWatchThreadGroup()
{
	ReleaseAllThreads();
}

bool CWatchThreadGroup::isDryrRun() const
{
	return dryrun_;
}

bool CWatchThreadGroup::LoadInfData(const tstring& InfPath)
{
	if (InfPath.empty()) {
		return false;
	}

	ReleaseAllThreads();

	bool succeeded = false;
	try {
		std::basic_ifstream<tstring::value_type, std::char_traits<tstring::value_type> > fin;
		fin.imbue(std::locale("japanese"));
		fin.open(InfPath.c_str());
		if ( !fin.is_open()) {
			return false;
		}

		const CSettingInfo init;

		tstring line;
		while (std::getline(fin, line)) {

			CSettingInfoFactory factory(init);

			int col = 0;
			tstring::size_type lastpos = 0;
			for (;;) {
				const tstring::size_type pos = line.find('\t', lastpos);
				tstring arg;
				if (pos != tstring::npos) {
					arg = line.substr(lastpos, pos - lastpos);
				}
				else {
					arg = line.substr(lastpos);
				}

				switch(col)
				{
				case 0:
					factory.setWatchDir(arg);
					break;

				case 1:
					factory.setWatchFile(arg);
					break;

				case 2:
					factory.setAction(arg);
					break;

				case 3:
					factory.setAppName(arg);
					break;

				case 4:
					factory.setParam(arg);
					break;

				case 5:
					factory.setMaxProcess(atoi(arg));
					break;

				case 6:
					factory.setShowWindow( ShowWindowType::valueOf(atoi(arg)));
					break;

				case 7:
					factory.setWaitWrite(atoi(arg));
					break;

				case 8:
					factory.setTolerance(atoi(arg));
					break;

				case 9:
					factory.setDeletePending(atoi(arg));
					break;

				case 10:
					factory.setLog(arg);
					break;

				case 11:
					factory.setMaxDepth(atoi(arg));
					break;

				case 12:
					factory.setUsingDirNotificationAPI(atoi(arg) != 0);
					break;

				case 13:
					factory.setForceInterval(atoi(arg));
					break;

				case 14:
					factory.setDirNotificationAPIExpirySpan(atoi(arg));
					break;

				case 15:
					factory.setDirNotificationAPIRetryInterval(atoi(arg));
					break;

				case 16:
					factory.setLogLevel(atoi(arg));
					break;

				case 17:
					factory.setAppCurrentDir(arg);
					break;

				case 18:
					factory.setPersist(arg);
					break;

				default:
					break;
				}

				col++;
				lastpos = pos + 1;
				if (pos == std::string::npos) {
					break;
				}
			}

			append(factory.create());
		}
		fin.close();
		succeeded = true;
	}
	catch (const std::exception& v_exception) {
		CA2T mes(v_exception.what());
		app.showMessageBox(MB_ICONERROR | MB_OK, NULL, IDS_MAINPROC_INFLOADFAILED, mes);
	}

	modified_ = false;
	return succeeded;
}

bool CWatchThreadGroup::isModified() const
{
	return modified_;
}

bool CWatchThreadGroup::SaveInfData(const tstring& InfPath, bool v_bForce) const
{
	if (InfPath.empty()) {
		return false;
	}

	if ( !isModified() && ! v_bForce) {
		return true;
	}

	bool succeeded = false;
	try {
		std::basic_ofstream<tstring::value_type, std::char_traits<tstring::value_type> > fout;
		fout.exceptions(std::ios::failbit | std::ios::badbit);
		fout.imbue(std::locale("japanese"));
		fout.open(InfPath.c_str());
		if ( !fout.is_open()) {
			return false;
		}
		for (CWatchThreadList::const_iterator ite = watchList_.begin();
			ite != watchList_.end();
			++ite )
		{
			const CWatchThread* pWatchThread = *ite;
			const CSettingInfo& settingInfo = pWatchThread->getSettingInfo();

			fout << settingInfo.getWatchDir()	<< "\t";
			fout << settingInfo.getWatchFile()	<< "\t";
			fout << settingInfo.getAction()		<< "\t";
			fout << settingInfo.getAppName()	<< "\t";
			fout << settingInfo.getParam()		<< "\t";
			fout << settingInfo.getMaxProcess()	<< "\t";
			fout << settingInfo.getShowWindow()	<< "\t";
			fout << settingInfo.getWaitWrite()	<< "\t";
			fout << settingInfo.getTolerance()	<< "\t";
			fout << settingInfo.getDeletePending() << "\t";
			fout << settingInfo.getLog()		<< "\t";
			fout << settingInfo.getMaxDepth()	<< "\t";
			fout << ( settingInfo.isUsingDirNotificationAPI() ? 1 : 0 )	<< "\t";
			fout << settingInfo.getForceInterval()						<< "\t";
			fout << settingInfo.getDirNotificationAPIExpirySpan()		<< "\t";
			fout << settingInfo.getDirNotificationAPIRetryInterval()	<< "\t";
			fout << settingInfo.getLogLevel()	<< "\t";
			fout << settingInfo.getAppCurrentDir() << "\t";
			fout << settingInfo.getPersist() << std::endl;
		}

		modified_ = false;
		succeeded = true;
	}
	catch (const std::exception& v_exception) {
		CA2T mes(v_exception.what());
		app.showMessageBox(MB_ICONERROR | MB_OK, NULL, IDS_MAINPROC_INFSAVEFAILED, mes);
	}

	return succeeded;
}

unsigned int CWatchThreadGroup::size() const
{
	return watchList_.size();
}

const CSettingInfo& CWatchThreadGroup::getInfo(unsigned int idx) const
{
	//required:
	assert(idx >= 0 && idx < watchList_.size() && "݂ȂACeɂ̓ANZXł܂B");
	
	//do:
	return watchList_[idx]->getSettingInfo();
}

bool CWatchThreadGroup::append(const CSettingInfo& v_settingInfo)
{
	CWatchThread* pWatchThread = new CWatchThread(v_settingInfo);
	watchList_.push_back( pWatchThread );
	if ( ! dryrun_) {
		ClearLog(v_settingInfo);
		pWatchThread->Start();
	}
	modified_ = true;
	return true;
}

bool CWatchThreadGroup::replace(unsigned int idx, const CSettingInfo& v_settingInfo)
{
	//required:
	assert(idx >= 0 && idx < watchList_.size() && "݂ȂACeɂ̓ANZXł܂B");
	
	//do:

	delete watchList_[idx];
	CWatchThread *const pWatchThread = new CWatchThread(v_settingInfo);
	watchList_[idx] = pWatchThread;
	if ( !dryrun_) {
		ClearLog(v_settingInfo);
		pWatchThread->Start();
	}
	modified_ = true;
	return true;
}

bool CWatchThreadGroup::erase(unsigned int idx)
{
	//required:
	assert(idx >= 0 && idx < watchList_.size() && "͈͊OłB");

	//do:
	CWatchThreadList::iterator ite = watchList_.begin() + idx;
	delete *ite;
	watchList_.erase(ite);
	modified_ = true;
	return true;
}

bool CWatchThreadGroup::isRunning() const
{
	for (CWatchThreadList::const_iterator ite = watchList_.begin();
		ite != watchList_.end();
		++ite)
	{
		const CWatchThread* pWatchThread = *ite;
		if (pWatchThread->isRunning()) {
			return true;
		}
	}
	return false;
}

void CWatchThreadGroup::ClearLog(const CSettingInfo& settingInfo) const
{
	tstring logPath = settingInfo.getLog();
	if ( !settingInfo.isEnableLogging() || logPath.empty()) {
		// Oo͂Ȃꍇ͊֌WȂ
		return;
	}

	// gp̃bNIuWFNgΏB
	// (Ot@C̓ɎgpĂ̂̂ŁAꊇďB)
	// (ݒύXŃOt@CςꍇɁA
	// gȂȂOt@C̃bNIuWFNgĉh߁B)
	app.RemoveUnusedLock();

	// s̃Xbhœ̃OgpĂ̂邩肷B
	bool alreadyUsed = false;
	for (CWatchThreadList::const_iterator ite = watchList_.begin();
		ite != watchList_.end();
		++ite)
	{
		const CWatchThread* pWatchThread = *ite;
		if (pWatchThread->isRunning()) {
			const CSettingInfo& otherSettingInfo = pWatchThread->getSettingInfo();
			tstring otherLogPath = otherSettingInfo.getLog();
			if (otherSettingInfo.isEnableLogging()) {
				if (_tcsicmp(logPath.c_str(), otherLogPath.c_str()) == 0) {
					alreadyUsed = true;
					break;
				}
			}
		}
	}

	if (!alreadyUsed) {
		// ̊ĎXbhŃOgpĂ炸A
		// ݒt@CɂăOwĂꍇ̓O폜B
		if (app.IsDeleteLog()) {
			DeleteFile(logPath.c_str());
		}
	}
}
