#include "stdafx.h"

#include "FWatchApp.hpp"
#include "resource.h"

#include "Utility.hpp"

#include "WatchThread.hpp"

#include <process.h>
#include <assert.h>

#include <string>


#include <strsafe.h>
#pragma comment(lib, "strsafe.lib")

namespace
{

	class WatchHandle
	{
	private:
		const CSettingInfo& settingInfo_;

		DWORD creation_;

		DWORD lastChallenged_;

		HANDLE findHandle_;

		Logger* pLogger_;

	public:
		WatchHandle(const CSettingInfo& settingInfo)
			: settingInfo_(settingInfo)
			, creation_(0)
			, lastChallenged_(0)
			, findHandle_(INVALID_HANDLE_VALUE)
		{
		}

		~WatchHandle()
		{
			Close();
		}

		void SetLogger(Logger* pLogger)
		{
			pLogger_ = pLogger;
		}

		Logger* GetLogger() const
		{
			return pLogger_;
		}

		void Close()
		{
			if (findHandle_ != INVALID_HANDLE_VALUE) {
				DWORD err = 0;
				if (!::FindCloseChangeNotification(findHandle_)) {
					err = GetLastError();
				}

				findHandle_ = INVALID_HANDLE_VALUE;

				if (pLogger_) {
					pLogger_->LogMessage(4, NULL, err, IDS_LOGMES_WATCH_CLOSE);
				}
			}
		}

		bool checkAndRenew()
		{
			if ( !settingInfo_.isUsingDirNotificationAPI()) {
				// t@CĎnhgpȂꍇ͉Ȃ
				return false;
			}

			DWORD now = ::GetTickCount();
			DWORD retryInterval = settingInfo_.getDirNotificationAPIRetryInterval();
			DWORD expireSpan = settingInfo_.getDirNotificationAPIExpirySpan();

			// nh쐬ALԂ𒴂Ă΃nhN[YB
			bool renew = false;
			if (findHandle_ != INVALID_HANDLE_VALUE) {
				if (expireSpan > 0 && (now - creation_) > expireSpan * 1000L) {
					Close();
					renew = true;
				}
			}

			if (findHandle_ != INVALID_HANDLE_VALUE) {
				// nhLłΉ߂B
				return true;
			}

			if (!renew && expireSpan > 0 && (now - lastChallenged_) <= retryInterval * 1000L) {
				// ŌɃgCĂwbo߂ĂȂꍇ̓nh쐬ɖ߂
				return false;
			}

			// 쐬sݎɂ
			lastChallenged_ = now;

			// Ďnh̍쐬
			findHandle_ = ::FindFirstChangeNotification(
				settingInfo_.getWatchDir().c_str(),
				settingInfo_.getMaxDepth() > 0 ? TRUE : FALSE,
				FILE_NOTIFY_CHANGE_FILE_NAME
					| (settingInfo_.getMaxDepth() > 0 ? FILE_NOTIFY_CHANGE_DIR_NAME : 0)
					| FILE_NOTIFY_CHANGE_LAST_WRITE
					| FILE_NOTIFY_CHANGE_SIZE
				);
			if (findHandle_ != INVALID_HANDLE_VALUE) {
				// nh쐬ꂽꍇA쐬Zbg
				creation_ = now;

				if (pLogger_) {
					pLogger_->LogMessage(
						4,
						NULL,
						0,
						renew ? IDS_LOGMES_WATCH_RENEW : IDS_LOGMES_WATCH_CREATED
						);
				}

			} else {
				// nh쐬łȂꍇ
				if (pLogger_) {
					DWORD errcode = GetLastError();
					pLogger_->LogMessage(0, NULL, errcode, IDS_LOGMES_WATCHERROR);
				}
			}

			return findHandle_ != INVALID_HANDLE_VALUE;
		}

		operator HANDLE() const
		{
			return findHandle_;
		}

	};

}

void CWatchThreadInterruptedException::checkAndThrow(HANDLE hStopEvent)
{
	if (!hStopEvent) {
		throw CWatchThreadInterruptedException();
	}
	DWORD waitResult = WaitForSingleObject(hStopEvent, 0);
	if (waitResult != WAIT_TIMEOUT) {
		// ~Cxgɂً}E
		throw CWatchThreadInterruptedException();
	}
}


////

CWatchThread::CWatchThread(const CSettingInfo& v_settingInfo)
	: settingInfo_(v_settingInfo)
	, fileTimeComparator_(v_settingInfo.getTolerance())
	, nThreadID_(0)
	, hThread_(NULL)
{
	hStopEvent_ = ::CreateEvent( NULL, TRUE, FALSE, NULL );
	assert(hStopEvent_ != NULL && "~Cxgnh擾ł܂B");

	// {ݒ̎擾
	spanSweepDeletedEntries_ = app.GetLong(_T("CONFIG"), _T("SPAN_SWEEP_DELETED_ENTRIES"), 100);
	spanHeapCompaction_ = app.GetLong(_T("CONFIG"), _T("SPAN_HEAP_COMPACTION"), 60 * 1000);
	logCloseSpan_ = app.GetLogCloseSpan() * 1000;

	// t@CK[̍쐬
	FileLogger* pFileLogger = new FileLogger();
	pFileLogger->SetLogPath(settingInfo_.getLog().c_str());
	pFileLogger->SetEnableLogging(settingInfo_.isEnableLogging());
	pFileLogger->SetCloseAlways(logCloseSpan_ == 0);
	pFileLogger->SetLogThreshold(settingInfo_.getLogLevel());
	pLogger_.reset(pFileLogger);

	// ANVsIuWFNg̍쐬
	const ActionInvokerFactory actionInvokerFactory(settingInfo_);
	pActionInvoker_ = actionInvokerFactory.create();
	pActionInvoker_->SetLogger(pFileLogger);
}


CWatchThread::~CWatchThread()
{
	Stop();
	::CloseHandle(hStopEvent_);
}


const CSettingInfo& CWatchThread::getSettingInfo() const
{
	return settingInfo_;
}


bool CWatchThread::isRunning() const
{
	if (hThread_ == NULL) {
		return false;
	}
	return (::WaitForSingleObject( hThread_, 0 ) == WAIT_TIMEOUT);
}


void CWatchThread::StopNoWait()
{
	if (hThread_ == NULL) {
		return;
	}

	::SetEvent(hStopEvent_);
}


void CWatchThread::Stop()
{
	if (hThread_ == NULL) {
		return;
	}

	StopNoWait();
	::WaitForSingleObject(hThread_, INFINITE);
}


void CWatchThread::Start()
{
	//required:
	assert(hThread_ == NULL && "Xbh͂łɊJnĂ܂B");

	//do:

	// Xe[^X̐ݒƃXbh̊Jn
	::ResetEvent(hStopEvent_);
	hThread_ = (HANDLE) _beginthreadex(NULL, 0, FWatchThread, this, CREATE_SUSPENDED, (unsigned int*) &nThreadID_);
	if (hThread_ != NULL) {
		::ResumeThread(hThread_);
	}
}


bool CWatchThread::getFileSize(const tstring& v_path, unsigned __int64 *pSize)
{
	//required:
	assert( !v_path.empty() && "t@Cɋ󕶎͎wł܂B");
	assert(pSize && "TCY̊i[ϐNULL͎wł܂B");
	
	//do:
	ULARGE_INTEGER size = {0};
	HANDLE fileHandle = ::CreateFile(v_path.c_str(),
			GENERIC_READ,
			FILE_SHARE_READ | FILE_SHARE_WRITE,
			NULL,
			OPEN_EXISTING,
			FILE_ATTRIBUTE_NORMAL,
			NULL);
	if (fileHandle != INVALID_HANDLE_VALUE) {
		size.LowPart = ::GetFileSize(fileHandle, &size.HighPart);
		::CloseHandle(fileHandle);
		*pSize = size.QuadPart;
		return true;
	}
	return false;
}


void CWatchThread::EnumFile(const bool v_enableEvent)
{
	if (EnumFileRecursive(settingInfo_.getWatchDir(), 0, v_enableEvent)) {
		// ł́AJnfBNg̃t@C񋓂ꍇ̂ݍ폜ҋ@XgZbgĂ܂B
		// āAJnfBNgsꍇ͍폜͎sł܂B
		// (񋓂̂̂sꍇ͈ꎞIȏQR̉\At@C݂ȂǂsȈׁB)

		const DWORD now = ::GetTickCount();
		pendingList_.clear();

		for (CFileInfoMap::iterator ite_fileInfo = fileInfoMap_.begin();
			ite_fileInfo != fileInfoMap_.end();) {
			const tstring& absolutePath = ite_fileInfo->first;
			CFileInfo& fileInfo = ite_fileInfo->second;
			if ( !fileInfo.isExist()) {
				// Ggɑ݂邪̌ʁA݂Ȃ̂̏ꍇ
				if (settingInfo_.getDeletePending() > 0) {
					// 폜ҋ@̏ꍇ͍폜\L[ɓ
					fileInfo.setDeletePending(true);
					fileInfo.setDeleteBeginTickCount(now);
					pendingList_.push_back(absolutePath);

					// ڍ׃O
					if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 6) {
						pLogger_->LogMessage(6, NULL, 0, IDS_LOGMES_REMOVEPENDING, absolutePath.c_str(), fileInfo.str().c_str());
					}

				}
				else {
					// ڍ׃O
					if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 6) {
						pLogger_->LogMessage(6, NULL, 0, IDS_LOGMES_REMOVEENTRY, absolutePath.c_str());
					}
					// 폜ҋ@Ȃ̏ꍇ͑ɃGgėǂ
					ite_fileInfo = fileInfoMap_.erase(ite_fileInfo);
					continue;
				}
			}
			++ite_fileInfo;
		}
	}
}


void CWatchThread::sweepDeletedEntries()
{
	const DWORD now = ::GetTickCount();
	const DWORD pendingLimit = (DWORD) (settingInfo_.getDeletePending() * 100);

	for (FileNameList::iterator ite_pendingList = pendingList_.begin();
		ite_pendingList != pendingList_.end();) {
		const tstring& absoluteName = *ite_pendingList;
		CFileInfoMap::iterator ite_fileInfo = fileInfoMap_.find(absoluteName);
		
		bool removed = false;
		if (ite_fileInfo == fileInfoMap_.end()) {
			// 폜ҋ@XgɁAt@CGg}bvɑ݂Ȃt@Cw肳Ă
			// łɍ폜ς݂Ȃ̂ō폜ҋ@Xg폜B
			removed = true;
		}
		else {
			CFileInfo& fileInfo = ite_fileInfo->second;
			if (fileInfo.isExist()) {
				// t@Ĉ݂ō폜ҋ@Xg珜
				removed = true;
			}
			else if (fileInfo.isDeletePending()) {
				if ((settingInfo_.getDeletePending() <= 0) ||
					((now - fileInfo.getDeleteBeginTickCount()) > pendingLimit)) {
					// 폜ҋ@ΏۂłA폜Aw莞Ԃo߂Ă̂
					// t@CGg}bvƁA폜ҋ@Xg̑o폜B
					fileInfoMap_.erase(ite_fileInfo);
					removed = true;
				}
			}
		}

		if (removed) {
			// ڍ׃O
			if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 6) {
				pLogger_->LogMessage(6, NULL, 0, IDS_LOGMES_REMOVEENTRY, absoluteName.c_str());
			}

			// 폜
			ite_pendingList = pendingList_.erase(ite_pendingList);
			continue;
		}
		++ite_pendingList;
	}
}


bool CWatchThread::EnumFileRecursive(const tstring& v_baseDir, const int v_depth, const bool v_enableEvent)
{
	// ΏۃpX𐶐
	const tstring baseDir = EnsureLast(v_baseDir, _T("\\"));

	// ڍ׃O
	if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 8) {
		pLogger_->LogMessage(8, NULL, 0, IDS_LOGMES_SCAN, baseDir.c_str());
	}
	
	// ύXƎvt@C
	WIN32_FIND_DATA finddata = {0};
	HANDLE findHandle = ::FindFirstFile((baseDir + _T("*.*")).c_str(), &finddata);
	if (findHandle == INVALID_HANDLE_VALUE) {
		// ΏۂɃANZXs\ȂΉȂŖ߂B
		DWORD err = GetLastError();
		pLogger_->LogMessage(0, NULL, err, IDS_LOGMES_FINDERROR, v_baseDir.c_str());
		return false;
	}

	if (v_depth == 0) {
		// ݂Ȃt@C폜邽߁AׂẴt@C݃tOZbgB
		// (lbg[N̈ꎞIȐؒf̉\̂߁AΏۂɃANZXłꍇ̂ݎsB)
		for (CFileInfoMap::iterator ite = fileInfoMap_.begin();
			ite != fileInfoMap_.end();
			++ite)
		{
			CFileInfo& fileInfo = ite->second;
			fileInfo.setExist(false);
		}
	}

	const DWORD now = ::GetTickCount();
	for (;;) {
		// ~CxgĂ邩mFB
		CWatchThreadInterruptedException::checkAndThrow(hStopEvent_);

		// Ώۃt@C̐΃pX
		const tstring absolutePath = baseDir + finddata.cFileName;

		if (finddata.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
			// fBNg̏ꍇA
			if ( !IsMatch(finddata.cFileName, _T(".")) && !IsMatch(finddata.cFileName, _T(".."))) {
				if (v_depth < settingInfo_.getMaxDepth()) {
					// TufBNg̐[ɒBĂȂ΃TufBNg
					// (TufBNǧs̗L͕s)
					EnumFileRecursive(absolutePath, v_depth + 1, v_enableEvent);
				}
			}
		}
		else {
			// t@C̏ꍇ
			if (IsMatch(finddata.cFileName, settingInfo_.getWatchFile().c_str())) {
				// Ώۃt@CɃ}b`ꍇA

				CFileInfoMap::iterator ite = fileInfoMap_.find(absolutePath);
				if (ite != fileInfoMap_.end()) {
					// łɃGgɓo^ς݂̃t@C
					CFileInfo& fileInfo = ite->second;
					unsigned __int64 fileSize =
						(((unsigned __int64) finddata.nFileSizeHigh << 32) | ((unsigned __int64) finddata.nFileSizeLow & 0xffffffff));
					if (v_enableEvent && (
							!fileTimeComparator_(finddata.ftLastWriteTime, fileInfo.getLastModified()) ||
							fileSize != fileInfo.getSize())) {
						// Cxgʒm胂[hŁAAt@C̍XVA܂̓TCYύXĂꍇ
						// ύXmZbgB
						fileInfo.setDetectModified(true);
						fileInfo.setCountdownTickCount(now);

						// ύXʒmO
						pLogger_->LogMessage(5, NULL, 0, IDS_LOGMES_FILEMODIFIED, absolutePath.c_str());
					}
					// t@C̏XV.
					fileInfo.update(finddata);
					// 폜\łΐĂ邱Ƃ
					fileInfo.setExist(true);
					fileInfo.setDeletePending(false);

					// ڍ׃O
					if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 9) {
						pLogger_->LogMessage(9, NULL, 0, IDS_LOGMES_UPDATEENTRY, absolutePath.c_str(), fileInfo.str().c_str());
					}
				}
				else {
					// 炽ɃGgɒǉt@C
					CFileInfo fileInfo(finddata, v_enableEvent, true);
					fileInfo.setDeletePending(false);
					fileInfo.setCountdownTickCount(now);
					fileInfoMap_.insert(CFileInfoMap::value_type(absolutePath, fileInfo));

					// ڍ׃O
					if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 9) {
						pLogger_->LogMessage(9, NULL, 0, IDS_LOGMES_ADDENTRY, absolutePath.c_str(), fileInfo.str().c_str());
					}
				}
			}
		}
		if ( !::FindNextFile(findHandle, &finddata)) {
			DWORD err = GetLastError();
			if (err != ERROR_NO_MORE_FILES) {
				// ̃t@C̓ǂݎɎsꍇ
				pLogger_->LogMessage(0, NULL, err,IDS_LOGMES_FINDNEXTERROR, v_baseDir.c_str());
			}
			::FindClose(findHandle);
			break;
		}
	}

	return true;
}


void CWatchThread::run()
{
	// t@CɎgpfBNgpX̍쐬
	const tstring baseDir = EnsureLast(settingInfo_.getWatchDir(), _T("\\"));

	// JnbZ[W
	pLogger_->LogMessage(
		1,
		NULL,
		0,
		IDS_LOGMES_INIT,
		baseDir.c_str(),
		settingInfo_.getMaxDepth(),
		settingInfo_.getWatchFile().c_str()
		);

	try {
		DWORD lastSweep = 0;
		DWORD lastHeapCompact = 0;
		DWORD lastForceTickCount = 0;

		// Ďnh
		WatchHandle watchHandle(settingInfo_);
		watchHandle.SetLogger(pLogger_.get());

		// ʒmnh̍쐬(KvȂ)
		watchHandle.checkAndRenew();

		// Xg̏(ʒmȂ)
		EnumFile(false);
		lastForceTickCount = ::GetTickCount();
		pLogger_->LogMessage(7, NULL, 0, IDS_LOGMES_INITLIST, fileInfoMap_.size());

		pLogger_->LogMessage(1, NULL, 0, IDS_LOGMES_START);
		// t@CύXʒm[v̊Jn
		for (;;) {
			// ݎ̎擾
			const DWORD now = ::GetTickCount();

			// Gg̍폜
			if ((now - lastSweep ) > spanSweepDeletedEntries_) {
				sweepDeletedEntries();
				lastSweep = now;
			}

			// q[v̈k(s)
			if (spanHeapCompaction_ > 0 && (now - lastHeapCompact ) > spanHeapCompaction_) {
				_heapmin();
			}

			// Ot@CJꂽ܂܎gĂȂΕ
			if (logCloseSpan_ > 0 && (now - pLogger_->GetLastWrite()) > logCloseSpan_) {
				((FileLogger*) pLogger_.get())->Close(); // NOTE: RXgN^FileLoggerZbgĂ̂ŖȂ
			}

			// ANV̎s
			pActionInvoker_->sweepTerminatedProcess();
			const DWORD writeWaitLimit = (DWORD) (settingInfo_.getWaitWrite() * 100);
			for (CFileInfoMap::iterator ite = fileInfoMap_.begin();
				ite != fileInfoMap_.end();
				++ite) {
				if (settingInfo_.getMaxProcess() != 0
					&& pActionInvoker_->GetActiveProcessCount() >= settingInfo_.getMaxProcess()) {
					// vZXEɒB̂Ń`FbNssȂB
					// (ɓvZXlȉɌ܂ŒxB)
					break;
				}

				const tstring& absolutePath = ite->first;
				CFileInfo& fileInfo = ite->second;

				if (fileInfo.isDetectModified()) {
					// ύXmꂽACe
					const DWORD span = now - fileInfo.getCountdownTickCount();
					
					if (span >= writeWaitLimit) {
						// t@CTCY܂őҋ@Ԃo߂Ăꍇ
						unsigned __int64 nFileSize;
						if ( !getFileSize(absolutePath, &nFileSize) || (fileInfo.getSize() != nFileSize)) {
							// t@CJȂAt@CTCYύXĂ΃JEĝ蒼
							fileInfo.setSize(nFileSize);
							fileInfo.setCountdownTickCount(now);
						}
						else {
							// sς݂Ƃ݂Ȃ߁AύXmtOoffB
							fileInfo.setDetectModified(false);
							
							// O
							pLogger_->LogMessage(1, NULL, 0, IDS_LOGMES_FILEMODIFY_COMMITED, absolutePath.c_str());
							
							// ANVsB
							pActionInvoker_->createProcess(absolutePath, fileInfo);
						}
					}
				}
			}

			// Cxgҋ@nh
			// 0: ~Cxg
			// 1: tH_Ďnh(ȂNULL)
			HANDLE findHandle = (HANDLE) watchHandle;
			HANDLE handles[] = {
				hStopEvent_,
				NULL
			};
			int handleCount = 1;
			if (findHandle != INVALID_HANDLE_VALUE) {
				handles[1] = findHandle;
				handleCount = 2;
			}

			// tH_vtO
			bool bEnumerationRequired = false;

			// Cxgҋ@B
			const DWORD ret = ::WaitForMultipleObjects(handleCount, handles, FALSE, 10); // bZ[W10msȑOɖ߂\B
			if (ret == WAIT_OBJECT_0 || ret == WAIT_ABANDONED_0) {
				// ~CxgB
				throw CWatchThreadInterruptedException();
			}
			else if (ret == WAIT_OBJECT_0 + 1 || ret == WAIT_ABANDONED_0 + 1) {
				// tH_ĎCxgB
				bEnumerationRequired = true;

				// O
				pLogger_->LogMessage(3, NULL, 0, IDS_LOGMES_CHANGENOTIFY);

				// ̃tH_ĎCxg󗝉\ɂB
				if ( !::FindNextChangeNotification(findHandle)) {
					// tH_ĎCxg̎󗝍ĊJɎsꍇ
					DWORD err = GetLastError();
					pLogger_->LogMessage(0, NULL, err, IDS_LOGMES_WATCHNEXTERROR, baseDir.c_str());

					// nhU邪͌pB
					watchHandle.Close();
				}
			}

			// Ďnhč쐬B(KvȂ)
			watchHandle.checkAndRenew();

			if (settingInfo_.getForceInterval() > 0 && ((now - lastForceTickCount) >= (DWORD) (settingInfo_.getForceInterval() * 1000))) {
				// XVw肳ĂAŌɌĂw莞Ԃo߂Ă邽ߋsB
				bEnumerationRequired = true;

				// O
				pLogger_->LogMessage(3, NULL, 0, IDS_LOGMES_FORCEINTERVAL);
			}

			// tH_̑v΁AtH_𑖍B
			if (bEnumerationRequired) {
				lastForceTickCount = now;
				EnumFile(true);
				pLogger_->LogMessage(7, NULL, 0, IDS_LOGMES_UPDATELIST, fileInfoMap_.size());
			}
		}

	} catch (const CWatchThreadInterruptedException&) {
		// ~Cxgɂ銄荞݂ɂI

	} catch (...) {
		pLogger_->LogMessage(0, NULL, 0, IDS_LOGMES_UNKNOWNEXCEPTION);
		throw; // X[ăAvP[VG[ɂB(ɂVXeɏQOL^B)
	}

	// Ďt@CXgj
	fileInfoMap_.clear();

	// IbZ[W
	pLogger_->LogMessage(
		1,
		NULL,
		0,
		IDS_LOGMES_FINISHED,
		baseDir.c_str(),
		settingInfo_.getMaxDepth(),
		settingInfo_.getWatchFile().c_str()
		);
}


unsigned __stdcall CWatchThread::FWatchThread(void* pParam)
{
	CWatchThread* me = (CWatchThread*) pParam;
	if (SUCCEEDED(::CoInitializeEx( NULL, COINIT_APARTMENTTHREADED))) {
		me->run();
		::CoUninitialize();
	}
	else {
		app.showMessageBox(MB_SERVICE_NOTIFICATION | MB_ICONERROR | MB_OK, NULL, IDS_MAINPROC_COINITFAILED);
	}
	return 0;
}

