#include "stdafx.hpp"

#include "FWatchApp.hpp"
#include "resource.h"

#include "Utility.hpp"
#include "tstringUty.hpp"

#include "WatchTask.hpp"

#include <assert.h>
#include <atltime.h>

#include <string>
#include <vector>


WatchTask::WatchTask( const TaskSettingInfo& v_settingInfo )
	: EventDispatcher( _TEXT( "WatchTask EventDispatcher" ) )
	, settingInfo_( v_settingInfo )
	, dirChangeNotification_(
		v_settingInfo.getWatchDir(),
		v_settingInfo.getMaxDepth() > 0,
		v_settingInfo.getDirNotificationAPIRetryInterval(),
		v_settingInfo.getDirNotificationAPIExpirySpan(), 
		v_settingInfo.isUsingDirNotificationAPI()
		)
	, enumerationSignal_( v_settingInfo.getDelaySignalSpan() )
	, persistenceRequest_( v_settingInfo.getPersistenceDelaySpan() )
	, forceEnumerationTimer_(
		v_settingInfo.getForceInterval(),
		CFileTime::GetCurrentTime(),
		v_settingInfo.getForceInterval() > 0
		)
	, fileTimeComparator_( v_settingInfo.getTolerance() )
	, nameFilter_( false )
	//, actionListener_( v_settingInfo.getActionSettingInfo() )
{
	//appendEventListener( actionListener_ );

	std::vector<tstring> matches;
	split( settingInfo_.getWatchFile(), ';', std::back_insert_iterator<std::vector<tstring> >( matches ) );
	for( std::vector<tstring>::iterator ite_matches = matches.begin();
		ite_matches != matches.end();
		++ite_matches
		)
	{
		const tstring nameMatch = tstringuty::trimSpaceBoth( *ite_matches );
		if( ! nameMatch.empty() ) {
			nameFilter_.addPattern( SimplePatternMatch( nameMatch ) );
		}
	}
}


WatchTask::~WatchTask()
{
}

const TaskSettingInfo& WatchTask::getSettingInfo() const
{
	return settingInfo_;
}

bool WatchTask::getFileSize( const tstring& v_path, unsigned long long* v_pSize )
{
	//required:
	assert( ! v_path.empty() && "t@Cɋ󕶎͎wł܂B" );
	
	//do:
	HANDLE fileHandle = ::CreateFile( v_path.c_str(), GENERIC_READ, FILE_SHARE_READ|FILE_SHARE_WRITE,NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
	if( fileHandle != INVALID_HANDLE_VALUE ) {
		DWORD hi = 0;
		DWORD low = ::GetFileSize( fileHandle, &hi );
		*v_pSize = ( hi << 32 ) | low;
		::CloseHandle( fileHandle );
		return true;
	}
	return false;
}

void WatchTask::EnumFile( const bool v_enableEvent )
{
	// t@C̑
	if( EnumFileRecursive( settingInfo_.getWatchDir(), 0, v_enableEvent ) ) {
		// 폜ΏۃACe̐ݒ
		// JnfBNg̃t@C񋓂ꍇ̂ݍ폜ҋ@XgZbg܂B
		prepareDeleteEntries();
	}
}

void WatchTask::prepareDeleteEntries()
{
	const CFileTime now( CFileTime::GetCurrentTime() );

	deletePendingList_.clear();
	for( FileInfoMap::iterator ite_fileInfo = fileInfoMap_.begin();
		ite_fileInfo != fileInfoMap_.end(); )
	{
		const tstring& absolutePath = ite_fileInfo->first;
		FileInfo& fileInfo = ite_fileInfo->second;
		if( ! fileInfo.isExist() ) {
			if( settingInfo_.getDeletePending() > 0 ) {
				// 폜ҋ@
				fileInfo.setDeletePending( true );
				fileInfo.setDeletePendingStartTime( now );
				deletePendingList_.push_back( absolutePath );
				notifyFileEvent( EventListener::REMOVE_PENDING, absolutePath, fileInfo );
			}
			else {
				// 폜ҋ@Ȃ
				notifyFileEvent( EventListener::DETECT_REMOVED, absolutePath, fileInfo );
				ite_fileInfo = fileInfoMap_.erase( ite_fileInfo );
				continue;
			}
		}
		++ite_fileInfo;
	}
}

void WatchTask::sweepDeletedEntries()
{
	unsigned int pendingCount = 0;

	const CFileTime now( CFileTime::GetCurrentTime() );
	const CFileTimeSpan pendingLimit( settingInfo_.getDeletePending() );

	for( FileNameList::iterator ite_pendingList = deletePendingList_.begin();
		ite_pendingList != deletePendingList_.end(); )
	{
		const tstring& absolutePath = *ite_pendingList;
		FileInfoMap::iterator ite_fileInfo = fileInfoMap_.find( absolutePath );
		
		bool removed = false;
		if( ite_fileInfo == fileInfoMap_.end() ) {
			assert( false && "폜ҋ@XgɁAt@CGg}bvɑ݂Ȃt@Cw肳Ă܂B" );
			removed = true;
		}
		else {
			FileInfo& fileInfo = ite_fileInfo->second;
			if( fileInfo.isExist() ) {
				// t@Ĉ݂ō폜ҋ@Xg珜
				removed = true;
			}
			else if( fileInfo.isDeletePending() ) {
				if( ( settingInfo_.getDeletePending() <= 0 ) ||
					( ( now - fileInfo.getDeletePendingStartTime() ) > pendingLimit )	)
				{
					// 폜ҋ@ΏۂłA폜Aw莞Ԃo߂Ă̂
					// t@CGg}bvƁA폜ҋ@Xg̑o폜B
					notifyFileEvent( EventListener::DETECT_REMOVED, absolutePath, fileInfo );
					fileInfoMap_.erase( ite_fileInfo );
					removed = true;
					persistenceRequest_.setSignalDelay( now );
				}
			}
		}

		if( removed ) {
			ite_pendingList = deletePendingList_.erase( ite_pendingList );
			continue;
		}
		++ite_pendingList;
	}
}

bool WatchTask::EnumFileRecursive( const tstring& v_baseDir, const int v_depth, const bool v_enableEvent )
{
	// ΏۃpX𐶐
	const tstring baseDir = tstringuty::ensureEndsWith( v_baseDir, tstring( _TEXT("\\") ) );
	
	// ύXƎvt@C
	WIN32_FIND_DATA finddata = { 0 };
	HANDLE findHandle = ::FindFirstFile( ( baseDir + _TEXT("*.*") ).c_str(), &finddata );
	if( findHandle == INVALID_HANDLE_VALUE ) {
		// ΏۂɃANZXs\ȂΉȂŖ߂B
		return false;
	}

	if( v_depth == 0 ) {
		// ݂Ȃt@C폜邽߁AׂẴt@C݃tOZbgB
		// (lbg[N̈ꎞIȐؒf̉\̂߁AΏۂɃANZXłꍇ̂ݎsB)
		for( FileInfoMap::iterator ite = fileInfoMap_.begin();
			ite != fileInfoMap_.end();
			++ite)
		{
			FileInfo& fileInfo = ite->second;
			fileInfo.setExist( false );
		}
	}

	const CFileTime now( CFileTime::GetCurrentTime() );
	for(;;) {
		const tstring absolutePath = baseDir + finddata.cFileName;
		if( finddata.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ) {
			if( ! IsMatch( finddata.cFileName, _TEXT(".") ) && ! IsMatch( finddata.cFileName, _TEXT("..") ) ) {
				if( v_depth < settingInfo_.getMaxDepth() ) {
					EnumFileRecursive( absolutePath, v_depth + 1, v_enableEvent ); // TufBNǧs̗L͕s
				}
			}
		}
		else {
			if( nameFilter_.isMatch( finddata.cFileName ) ) {
				FileInfoMap::iterator ite = fileInfoMap_.find( absolutePath );
				if( ite != fileInfoMap_.end() ) {
					// Gg̍XV
					FileInfo& fileInfo = ite->second;
					const bool detectModified = ! fileTimeComparator_( finddata.ftLastWriteTime, fileInfo.getLastModified() );
					fileInfo.update( finddata );
					fileInfo.setExist( true );
					fileInfo.setDeletePending( false );
					if( v_enableEvent && detectModified ) {
						fileInfo.setDetectModified( true );
						fileInfo.setActionPendingStartTime( now );
						persistenceRequest_.setSignalDelay( now );
						notifyFileEvent( EventListener::UPDATE_ENTRY, absolutePath, fileInfo );
					}
				}
				else {
					// VKGg̒ǉ
					FileInfo fileInfo;
					fileInfo.update( finddata );
					fileInfo.setExist( true );
					fileInfo.setDeletePending( false );
					fileInfo.setDetectModified( v_enableEvent );
					fileInfo.setActionPendingStartTime( now );
					fileInfoMap_.insert( FileInfoMap::value_type( absolutePath, fileInfo ) );
					persistenceRequest_.setSignalDelay( now );
					if( v_enableEvent ) {
						notifyFileEvent( EventListener::NEW_ENTRY, absolutePath, fileInfo );
					}
				}
			}
		}
		if( ! ::FindNextFile( findHandle, &finddata ) ) {
			::FindClose( findHandle );
			break;
		}
	}

	return true;
}


void WatchTask::onThreadBinded()
{
	notifyStateEvent( EventListener::BIND_THREAD );

	bool initialSearch = false;
	if( fileInfoMap_.loadPersistence( settingInfo_.getPersistentFile() ) ) {
		initialSearch = true;
	}
	EnumFile( initialSearch );
}

void WatchTask::onThreadUnbinded()
{
	if( persistenceRequest_.isSignaledImmediate() ) {
		fileInfoMap_.savePersistence( settingInfo_.getPersistentFile() );
		persistenceRequest_.clear();
	}
	fileInfoMap_.clear();

	notifyStateEvent( EventListener::UNBIND_THREAD );
}

void WatchTask::onThreadSignal()
{
	notifyStateEvent( EventListener::THREAD_SIGNAL );

	enumerationSignal_.setSignalDelay( CFileTime::GetCurrentTime() );
	dirChangeNotification_.next();
}

void WatchTask::onThreadTick()
{
	// ݎ̎擾
	const CFileTime now( CFileTime::GetCurrentTime() );

	// ݂̊ĎXg̕ۑ(s)
	if( persistenceRequest_.isSignaled( now ) ) {
		fileInfoMap_.savePersistence( settingInfo_.getPersistentFile() );
		persistenceRequest_.clear();
	}

	// 폜ۗ̊؂ACȅ
	sweepDeletedEntries();

	// t@C̃gK

	if( forceEnumerationTimer_.isSignaled( now ) ) {
		// XVw肳ĂAŌɌĂw莞Ԃo߂Ă邽ߋsB
		forceEnumerationTimer_.reset( now );
		enumerationSignal_.setSignalImmediate();
		notifyStateEvent( EventListener::FORCE_INTERVAL );
	}

	if( enumerationSignal_.isSignaled( now ) ) {
		enumerationSignal_.clear();
		forceEnumerationTimer_.reset( now );
		notifyStateEvent( EventListener::ENUM_FILES );
		EnumFile(true);
	}

	// Cxg̒ʒm
	tickEventCycle();
	for( FileInfoMap::iterator ite = fileInfoMap_.begin();
		ite != fileInfoMap_.end();
		++ite)
	{
		const tstring& absolutePath = ite->first;
		FileInfo& fileInfo = ite->second;
		if( fileInfo.isDetectModified() ) {
			if( ( now - fileInfo.getActionPendingStartTime() ) >= settingInfo_.getWaitWrite() ) {
				unsigned long long fileSize;
				if( ! getFileSize( absolutePath, &fileSize ) ||	( fileInfo.getSize() != fileSize ) ) {
					// t@CJȂAt@CTCYύXĂ΃JEĝ蒼
					fileInfo.setSize( fileSize );
					fileInfo.setActionPendingStartTime( now );
				}
				else{
					fileInfo.setDetectModified( false );
					notifyFileEvent( EventListener::DETECT_MODIFIED, absolutePath, fileInfo );
				}
			}
		}
	}
}

HANDLE WatchTask::getWaitableHandle()
{
	return dirChangeNotification_.getHandle();
}
