/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>

#include "futaba.h"
#include "fb-conf.h"
#include "fb-viewer.h"
#include "fb-viewer-action.h"
#include "fb-image.h"
#include "file-utils.h"

#define DRAG_ICON_W 72
#define DRAG_ICON_H 72

static void     fb_viewer_class_init         (FbViewerClass    *klass);
static void     fb_viewer_init               (FbViewer         *viewer);
static void     fb_viewer_finalize           (GObject          *object);
static gboolean fb_viewer_focus_in_event     (GtkWidget        *widget,
					      GdkEventFocus    *event);
static void     fb_viewer_update_title       (FbViewer         *viewer);
/* callbacks */
static gboolean cb_button_press              (GtkWidget        *wg,
					      GdkEventButton   *event,
					      gpointer          data);
static gboolean cb_scroll_event              (GtkWidget        *wg,
					      GdkEvent         *event,
					      gpointer          data);
static void     cb_drag_begin                (GtkWidget *widget,
					      GdkDragContext *drag_context,
					      gpointer data);
static void     cb_drag_data_get             (GtkWidget *widget,
					      GdkDragContext *drag_context,
					      GtkSelectionData *selection_data,
					      guint info,
					      guint time,
					      gpointer data);
static void     cb_drag_data_received        (GtkWidget *icon_view,
					      GdkDragContext *drag_context,
					      gint x, gint y,
					      GtkSelectionData *selection_data,
					      guint info,
					      guint time,
					      gpointer data);
/* utils */
static void store_status (FbViewer *viewer);
static void restore_status (FbViewer *viewer);

static GtkWindowClass *parent_class = NULL;

GtkWidget *focus_viewer = NULL;
GList *viewer_list = NULL;

GType
fb_viewer_get_type (void)
{
	static GType object_type = 0;

	if (! object_type) {
		static const GTypeInfo object_info = {
			sizeof (FbViewerClass),
			NULL,
			NULL,
			(GClassInitFunc) fb_viewer_class_init,
			NULL,
			NULL,
			sizeof (FbViewer),
			32,
			(GInstanceInitFunc) fb_viewer_init,
		};

		object_type = g_type_register_static (GTK_TYPE_WINDOW, "FbViewer",
						      &object_info, 0);
	}

	return object_type;
}

static void
fb_viewer_class_init (FbViewerClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *) klass;
	object_class = (GtkObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;

	gobject_class->finalize = fb_viewer_finalize;

/* 	object_class->destroy = fb_viewer_destroy; */

	widget_class->focus_in_event = fb_viewer_focus_in_event;
}

static void
fb_viewer_init (FbViewer *viewer)
{
	GtkUIManager *ui_manager;
	GtkWidget *frame, *event_box;

	ui_manager = fb_viewer_ui_manager_get (viewer);

/* 	gtk_window_set_title(GTK_WINDOW(viewer), "- Futaba Viewer -"); */
	gtk_window_add_accel_group(GTK_WINDOW(viewer), gtk_ui_manager_get_accel_group(ui_manager));

	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_container_add(GTK_CONTAINER(viewer), frame);
	gtk_widget_show(frame);

	viewer->image = fb_image_new ();
	gtk_drag_source_set(viewer->image,
			    GDK_BUTTON1_MASK,
			    uri_drag_types, n_uri_drag_types,
			    GDK_ACTION_ASK  | GDK_ACTION_COPY
			    | GDK_ACTION_MOVE | GDK_ACTION_LINK);
	gtk_drag_dest_set(viewer->image, 
			  GTK_DEST_DEFAULT_HIGHLIGHT |
			  GTK_DEST_DEFAULT_MOTION | 
			  GTK_DEST_DEFAULT_DROP,
                          uri_drag_types,n_uri_drag_types,
			  GDK_ACTION_MOVE | GDK_ACTION_COPY);
	g_signal_connect(G_OBJECT(viewer->image), "scroll-event",
			 G_CALLBACK(cb_scroll_event), viewer);
	g_signal_connect(G_OBJECT(viewer->image), "button-press-event",
			 G_CALLBACK(cb_button_press), viewer);
	g_signal_connect (G_OBJECT (viewer->image), "drag-begin",
			  G_CALLBACK (cb_drag_begin), viewer);
	g_signal_connect (G_OBJECT (viewer->image), "drag-data-get",
			  G_CALLBACK (cb_drag_data_get), viewer);
	g_signal_connect (G_OBJECT (viewer->image), "drag-data-received",
			  G_CALLBACK (cb_drag_data_received), viewer);
	gtk_container_add(GTK_CONTAINER(frame), viewer->image);
	gtk_widget_show(viewer->image);

	viewer->popup = gtk_ui_manager_get_widget (ui_manager, "/Popup");
	viewer->layout = gtk_widget_create_pango_layout (viewer->image, NULL);
	viewer->size = VIEWER_NORMAL;
	viewer->path_list = NULL;
	viewer->position = -1;

	restore_status (viewer);
}

static void
restore_status (FbViewer *viewer)
{
	gint val_i, val_j;

	if (focus_viewer) {
		val_i = focus_viewer->allocation.width;
		val_j = focus_viewer->allocation.height;
	}
	else {
		fb_conf_get_value ("viewer width", &val_i, FB_CONF_TYPE_INT, "500");
		fb_conf_get_value ("viewer height", &val_j, FB_CONF_TYPE_INT, "500");
	}
	gtk_window_set_default_size(GTK_WINDOW(viewer), val_i, val_j);
}

static void
store_status (FbViewer *viewer)
{
	fb_conf_set_value ("viewer width", & GTK_WIDGET (viewer)->allocation.width, FB_CONF_TYPE_INT);
	fb_conf_set_value ("viewer height", & GTK_WIDGET (viewer)->allocation.height, FB_CONF_TYPE_INT);
}

static void
fb_viewer_finalize (GObject *object)
{
	gint i;
	GtkWidget *viewer = GTK_WIDGET (object);

	store_status (FB_VIEWER (object));

	i = MAX(0, g_list_index(viewer_list, viewer) - 1);
	viewer_list = g_list_remove(viewer_list, viewer);

	focus_viewer = g_list_nth_data(viewer_list, i);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gboolean
fb_viewer_focus_in_event(GtkWidget *widget,
			 GdkEventFocus *event)
{
/* 	g_print ("FbViewer::focus-in-event\n"); */

	focus_viewer = widget;

	return GTK_WIDGET_CLASS (parent_class)->focus_in_event (widget, event);
}

static gboolean
cb_button_press(GtkWidget *widget,
		GdkEventButton *event,
		gpointer data)
{
	FbViewer *viewer = FB_VIEWER (data);

	switch (event->button) {
	case 1: {
/* 		if (event->state == GDK_SHIFT_MASK) */
/* 			fb_viewer_prev_images (viewer, 1); */
/* 		else */
/* 			fb_viewer_next_images (viewer, 1); */
	}
		break;
	case 2: {
	}
		break;
	case 3: {
		gtk_menu_popup(GTK_MENU(viewer->popup), NULL, NULL, NULL, NULL,
			       event->button, event->time);
	}
		break;
	}

	return FALSE;
}

static gboolean
cb_scroll_event(GtkWidget *wg,
		GdkEvent *event,
		gpointer data)
{
	FbViewer *viewer = data;

/* 	g_print ("mouse scroll\n"); */
	switch (event->scroll.direction) {
	case 0:
		fb_viewer_prev_images(viewer, 1);
		break;
	case 1:
		fb_viewer_next_images(viewer, 1);
		break;
	}

	return TRUE;
}

static void
cb_drag_begin (GtkWidget *widget,
	       GdkDragContext *drag_context,
	       gpointer data)
{
	gchar *path;
	FbViewer *viewer = FB_VIEWER (data);

	path = fb_image_get_file_at_pointer (FB_IMAGE (viewer->image));
	if (path) {
		GdkPixbuf *pixbuf;

		if (viewer->drag_data) g_free (viewer->drag_data);
		viewer->drag_data = g_strdup_printf ("file://%s\r\n", path);

		pixbuf = gdk_pixbuf_new_from_file_at_size (path, DRAG_ICON_W, DRAG_ICON_H, NULL);
		if (pixbuf) {
			gtk_drag_set_icon_pixbuf (drag_context, pixbuf, -5, -5);
			g_object_unref (pixbuf);
		}

		g_free (path);
	}
}

static void
cb_drag_data_get (GtkWidget *widget,
		  GdkDragContext *drag_context,
		  GtkSelectionData *selection_data,
		  guint info,
		  guint time,
		  gpointer data)
{
	gchar *uri;
	FbViewer *viewer = FB_VIEWER (data);

/* 	g_print ("viewer:drag data get\n"); */

	uri = viewer->drag_data;

	switch (info) {
	case TARGET_TEXT_URI_LIST:
	case TARGET_TEXT_PLAIN:
		gtk_selection_data_set(selection_data, selection_data->target,
				       8, uri, strlen(uri));
		break;
	default:
		break;
	}

}

static void
cb_drag_data_received (GtkWidget *widget,
		       GdkDragContext *drag_context,
		       gint x, gint y,
		       GtkSelectionData *selection_data,
		       guint info,
		       guint time,
		       gpointer data)
{
	gchar *uri;
	FbViewer *viewer = FB_VIEWER (data);

/* 	g_print ("drag data received\n"); */

	uri = g_strndup (selection_data->data, selection_data->length);
	if (g_str_has_prefix (uri, "file:/")) {
		gint s_len, r_len;
		gchar *path, *ptr;
	
		s_len = strlen ("file:");
		r_len = 0;
		ptr = uri + strlen (uri) - 1;
		while (*ptr == '\r' || *ptr == '\n' || *ptr == '\0') {
			r_len++;
			ptr--;
		}

		path = g_strndup (uri + s_len, strlen (uri) - s_len - r_len);
		if (file_is_image (path)) {
			gint i, n_img, index;
		
			index = fb_image_set_file_at_pointer (FB_IMAGE (viewer->image), path);
			n_img = fb_image_get_n_images (FB_IMAGE (viewer->image));
			i = viewer->position + n_img - index;

			viewer->path_list = g_list_insert (viewer->path_list, path, i); 
			viewer->position++;
			fb_viewer_update_title (viewer);
		}
	}
	g_free (uri);

	gtk_drag_finish (drag_context, TRUE, TRUE, time);
}

GtkWidget *
fb_viewer_new (GList *path_list)
{
	GObject *object = g_object_new (FB_TYPE_VIEWER, NULL);
	GtkWidget *viewer = GTK_WIDGET (object);

	focus_viewer = viewer;
	viewer_list = g_list_append(viewer_list, viewer);

	FB_VIEWER (object)->path_list = path_list_dup (path_list);

	return viewer;
}

void
fb_viewer_start(GList *path_list,
		const gchar *path)
{
	gint i = 0;
	GList *p;
	FbViewer *viewer;

	if (! path_list) return;

	if (path) {
		for (p = path_list; p; p = p->next) {
			if (! strcmp (path, p->data)) break;
			i++;
		}
	}

	viewer = FB_VIEWER (fb_viewer_new (path_list));
	gtk_widget_show (GTK_WIDGET (viewer));

	viewer->position = MAX( 0, i);
	path = g_list_nth_data(viewer->path_list, viewer->position);
	if (path) {
		fb_image_add_image (FB_IMAGE (viewer->image), path, GTK_ORIENTATION_HORIZONTAL);
		fb_viewer_update_title (viewer);
	}
}

static void
fb_viewer_update_title (FbViewer *viewer)
{
	gint i;
	gchar *title, *path;
	GList *update_list = NULL;

	g_return_if_fail (FB_IS_VIEWER (viewer));

	for (i = 0; i < g_list_length (viewer->path_list); i++) {
		path = g_list_nth_data (viewer->path_list, i);
		if (! path) {
			if (viewer->position >= i)
				viewer->position--;
			continue;
		}

		update_list = g_list_append (update_list, path);
	}
	g_list_free (viewer->path_list);
	viewer->path_list = update_list;

	title = g_strdup_printf ("%d/%d - Futaba Viewer -",
				 viewer->position + 1, g_list_length (viewer->path_list));
	gtk_window_set_title(GTK_WINDOW(viewer), title);
	g_free (title);
}

void
fb_viewer_next_images (FbViewer *viewer,
		       gint n)
{
	gint i, n_img, nth, last;
	gchar *path;

	g_return_if_fail (FB_IS_VIEWER (viewer));

	last = g_list_length (viewer->path_list);
	nth = MIN ((viewer->position + n), last);
	if (nth == last) {
		gint x, y;

		fb_image_get_last_image_pos (FB_IMAGE (viewer->image), &x, &y);
		pango_layout_set_text (viewer->layout, "Last", -1);
		gdk_draw_layout (viewer->image->window,
				 viewer->image->style->fg_gc[GTK_STATE_NORMAL],
				 x, y,
				 viewer->layout);
 
		return;
	}

	gdk_draw_rectangle (viewer->image->window,
			    viewer->image->style->bg_gc[GTK_STATE_NORMAL],
			    TRUE,
/* 			    viewer->image->allocation.x, viewer->image->allocation.y, */
			    0, 0,
			    viewer->image->allocation.width, viewer->image->allocation.height);

	n_img = fb_image_get_n_images (FB_IMAGE (viewer->image));
	for (i = 0; i < n_img; i++) {
		path = g_list_nth_data (viewer->path_list, nth);
		if (! path_is_image (path)) {
			g_free (path);
			path = NULL;
			continue;
		}

		fb_image_set_file (FB_IMAGE (viewer->image), path);
		if (nth < last)	nth++;
	}
	viewer->position = nth - 1;

	fb_viewer_update_title (viewer);
}

void
fb_viewer_prev_images (FbViewer *viewer,
		       gint n)
{
	g_return_if_fail (FB_IS_VIEWER (viewer));

	viewer->position -= (2 * n * fb_image_get_n_images (FB_IMAGE (viewer->image)));
	if (viewer->position < -1) {
		gint x, y;

		viewer->position = -1;
		fb_viewer_next_images (viewer, n);

		fb_image_get_first_image_pos (FB_IMAGE (viewer->image), &x, &y);
		pango_layout_set_text (viewer->layout, "First", -1);
		gdk_draw_layout (viewer->image->window,
				 viewer->image->style->fg_gc[GTK_STATE_NORMAL],
				 x, y,
				 viewer->layout);
	}
	else {
		fb_viewer_next_images (viewer, n);
	}
}

void
fb_viewer_segmentalize (FbViewer *viewer,
			GtkOrientation way)
{
	gchar *path;

	g_return_if_fail (FB_IS_VIEWER (viewer));

	path = g_list_nth_data (viewer->path_list, viewer->position + 1);
	if (! path) return; /* FIXME (warning if failed) */

	fb_image_add_image (FB_IMAGE (viewer->image), path, way);
	viewer->position++;
}
