/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "fb-prefs.h"
#include "prefs-general.h"
#include "prefs-viewer.h"
#include "prefs-filter.h"
#include "prefs-thumbwin.h"

enum{
	CONTENTS,
	PREFERENCES
};

#define PREF_DIALOG_W 500
#define PREF_DIALOG_H 400

static void fb_prefs_win_class_init (FbPrefsWinClass *klass);
static void fb_prefs_win_init (FbPrefsWin *win);
static void fb_prefs_win_finalize (GObject *object);
static void fb_prefs_win_response (GtkDialog *dialog, gint arg);
static void fb_prefs_win_apply (FbPrefsWin *win);
/* utils */
static void prefs_page_register (FbPrefsWin *win,
				 FbPrefsPageEntry *page_list,
				 gint page_list_len);
/* call back */
static void cb_change_page(GtkTreeView *tree, gpointer data);

static FbPrefsPageEntry prefs_page[] = {
	{N_("General"), prefs_general_get_func},
	{N_("Viewer"), prefs_viewer_get_func},
	{N_("Filter"), prefs_filter_get_func},
	{N_("Thumbnail"), prefs_thumbwin_get_func},
};

static GtkDialogClass *parent_class = NULL;

GType
fb_prefs_win_get_type(void)
{
	static GType object_type = 0;

	if (!object_type) {
		static const GTypeInfo object_info = {
			sizeof (FbPrefsWinClass),
			NULL, /* base_init */
			NULL, /* base_finalize */
			(GClassInitFunc) fb_prefs_win_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
			sizeof (FbPrefsWin),
			32, /* n_preallocs */
			(GInstanceInitFunc) fb_prefs_win_init,
		};
		
		object_type = g_type_register_static(GTK_TYPE_DIALOG, "FbPrefsWin",
						     &object_info, 0);
	}

	return object_type;
}

static void
fb_prefs_win_class_init(FbPrefsWinClass *klass)
{
	GObjectClass *gobject_class;
	GtkDialogClass *dialog_class;

	parent_class = g_type_class_peek_parent(klass);

	gobject_class = (GObjectClass *) klass;
	dialog_class = (GtkDialogClass *) klass;

	gobject_class->finalize = fb_prefs_win_finalize;

	dialog_class->response = fb_prefs_win_response;
}

static void
cb_change_page(GtkTreeView *tree,
	       gpointer data)
{
	gint *n;
	gchar *content, *text;
	GtkTreeIter iter;
	GtkTreePath *tpath;
	GtkTreeModel *model;
	FbPrefsWin *win = data;

	gtk_tree_view_get_cursor(tree, &tpath, NULL);
	if (!tpath) tpath = gtk_tree_path_new_first();
	n = gtk_tree_path_get_indices(tpath);

	gtk_notebook_set_current_page(GTK_NOTEBOOK(win->pages), *n);

	model = gtk_tree_view_get_model(tree);
	if (gtk_tree_model_get_iter(model, &iter, tpath)) {
		content = NULL;
		gtk_tree_model_get(model, &iter, CONTENTS, &content, -1);
		if (content) {
			text = g_strconcat("<span weight=\"bold\" size=\"x-large\">", content,"</span>", NULL);
			gtk_label_set_markup(GTK_LABEL(win->title), text);
			g_free(content);
			g_free(text);
		}
	}

	gtk_tree_path_free(tpath);
}

static void
fb_prefs_win_init(FbPrefsWin *win)
{
	GList *funcs = NULL;
	GtkTreeViewColumn *column;
	GtkCellRenderer *render;
	GtkWidget  *main_hbox, *hbox, *vbox, *sepa;

	gtk_window_set_default_size(GTK_WINDOW(win), PREF_DIALOG_W, PREF_DIALOG_H);

	gtk_dialog_add_buttons(GTK_DIALOG(win),
			       GTK_STOCK_APPLY, GTK_RESPONSE_APPLY,
			       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			       GTK_STOCK_OK, GTK_RESPONSE_OK,
			       NULL);

	main_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(win)->vbox), main_hbox, TRUE, TRUE, 10);
	gtk_widget_show(main_hbox);

	/* contents list */
	win->title_list = gtk_tree_view_new();
	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(win->title_list), TRUE);
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(win->title_list), FALSE);
	g_signal_connect(G_OBJECT(win->title_list), "cursor-changed",
			 G_CALLBACK(cb_change_page), win);
	gtk_box_pack_start(GTK_BOX(main_hbox), win->title_list, FALSE, TRUE, 0);
	gtk_widget_show(win->title_list);

	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_max_width(column, 150);
	gtk_tree_view_column_set_min_width(column, 100);
	gtk_tree_view_append_column(GTK_TREE_VIEW(win->title_list), column);

	render = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(column, render, TRUE);
	gtk_tree_view_column_add_attribute(column, render, "text", CONTENTS);

	win->store = gtk_list_store_new(PREFERENCES,
					G_TYPE_STRING);
	gtk_tree_view_set_model(GTK_TREE_VIEW(win->title_list), GTK_TREE_MODEL(win->store));

	/* display */
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_hbox), vbox, TRUE, TRUE, 5);
	gtk_widget_show(vbox);

	/* title */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	win->title = gtk_label_new(NULL);
	gtk_box_pack_start(GTK_BOX(hbox), win->title, FALSE, FALSE, 5);
	gtk_widget_show(win->title);

	sepa = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), sepa, FALSE, FALSE, 0);
	gtk_widget_show(sepa);

	/* menu */
	win->pages = gtk_notebook_new();
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(win->pages), FALSE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(win->pages), FALSE);
	gtk_box_pack_end(GTK_BOX(vbox), win->pages, TRUE, TRUE, 0);
	gtk_widget_show(win->pages);

	prefs_page_register (win, prefs_page, G_N_ELEMENTS (prefs_page));
}

static void
prefs_page_register (FbPrefsWin *win,
		     FbPrefsPageEntry *page_list,
		     gint page_list_len)
{
	gint i;
	GtkTreeIter iter;
	FbPrefsFunc *func;

	for (i = 0; i < page_list_len; i++) {
		gtk_list_store_append(win->store, &iter);
		gtk_list_store_set(win->store, &iter,
				   CONTENTS, _(page_list[i].title),
				   -1);

		func = page_list[i].get ();
		gtk_notebook_append_page (GTK_NOTEBOOK (win->pages), func->create (), NULL);
		win->page_func_list = g_list_append (win->page_func_list, func);
	}
}

GtkWidget *
fb_prefs_win_new(void)
{
	GObject *object;

	object = g_object_new(FB_TYPE_PREFS_WIN,
			      "title", _("Preference - Futaba"),
			      NULL);

	return GTK_WIDGET(object);
}

static void
fb_prefs_win_finalize (GObject *object)
{
	GList *node;

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize (object);

	for (node = g_list_first (FB_PREFS_WIN (object)->page_func_list); node; node = node->next) {
		FbPrefsFunc *func = node->data;

		if (func->destroy) func->destroy();
	}
	g_list_free(FB_PREFS_WIN (object)->page_func_list);
}

static void
fb_prefs_win_apply(FbPrefsWin *win)
{
	GList *node;

	for (node = g_list_first (win->page_func_list); node; node = node->next) {
		FbPrefsFunc *func = node->data;

		if (func->apply) func->apply();
	}

}

static void
fb_prefs_win_response(GtkDialog *dialog,
		      gint arg)
{

	switch (arg) {
	case GTK_RESPONSE_APPLY:
		fb_prefs_win_apply(FB_PREFS_WIN(dialog));
		break;
	case GTK_RESPONSE_OK:
		fb_prefs_win_apply(FB_PREFS_WIN(dialog));
		gtk_widget_destroy(GTK_WIDGET(dialog));
		break;
	case GTK_RESPONSE_CANCEL:
		gtk_widget_destroy(GTK_WIDGET(dialog));
		break;
	default:
		break;
	}
}
