/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "fb-pathmenuitem.h"
#include "fb-window.h" /* widget_set_message */
#include "file-utils.h"

static void     fb_path_menu_item_class_init (FbPathMenuItemClass *klass);
static void     fb_path_menu_item_init       (FbPathMenuItem      *pitem);
static void     fb_path_menu_item_finalize   (GObject             *object);
static void     fb_path_menu_item_set_path   (FbPathMenuItem      *pitem,
					      const gchar         *path);

static GtkImageMenuItemClass *parent_class = NULL;

GType
fb_path_menu_item_get_type (void)
{
	static GType object_type = 0;

	if (! object_type) {
		static const GTypeInfo object_info = {
			sizeof (FbPathMenuItemClass),
			NULL,
			NULL,
			(GClassInitFunc) fb_path_menu_item_class_init,
			NULL,
			NULL,
			sizeof (FbPathMenuItem),
			32,
			(GInstanceInitFunc) fb_path_menu_item_init,
		};

		object_type = g_type_register_static (GTK_TYPE_IMAGE_MENU_ITEM, "FbPathMenuItem",
						      &object_info, 0);
	}

	return object_type;
}

static void
fb_path_menu_item_class_init (FbPathMenuItemClass *klass)
{
	GObjectClass *object_class;

	parent_class = g_type_class_peek_parent (klass);

	object_class = (GObjectClass *) klass;

	object_class->finalize = fb_path_menu_item_finalize;
}

static void
fb_path_menu_item_init (FbPathMenuItem *pitem)
{
	pitem->path = NULL;
	pitem->label = NULL;
	pitem->message = NULL;
}

static void
fb_path_menu_item_finalize (GObject *object)
{
	FbPathMenuItem *pitem = FB_PATH_MENU_ITEM (object);

	if (pitem->path) g_free (pitem->path);
	if (pitem->label) g_free (pitem->label);
	if (pitem->message) g_free (pitem->message);

	if (G_OBJECT_CLASS (parent_class)->finalize)
		G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
fb_path_menu_item_set_path (FbPathMenuItem *pitem,
			    const gchar *path)
{
	gchar *base, *icon_id;
	GtkWidget *label, *icon;

	g_return_if_fail (FB_IS_PATH_MENU_ITEM (pitem));
	g_return_if_fail (path_is_exists (path));

	base = g_path_get_basename (path);
	pitem->label = g_locale_to_utf8 (base, -1, NULL, NULL, NULL);
	if (! pitem->label) pitem->label = g_strdup (base);
	g_free (base);

	label = gtk_accel_label_new (pitem->label);
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_container_add (GTK_CONTAINER (pitem), label);
	gtk_accel_label_set_accel_widget (GTK_ACCEL_LABEL (label),
					  GTK_WIDGET (pitem));
	gtk_widget_show (label);

	if (file_is_archive (path)) {
		icon_id = g_strdup ("fb-archive");
	}
	else if (path_is_dir (path)) {
		icon_id = g_strdup ("fb-dir");
	}
	icon = gtk_image_new_from_stock (icon_id, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (pitem), icon);
	gtk_widget_show (icon);
	g_free (icon_id);

	pitem->path = g_strdup (path);

	pitem->message = g_strdup_printf (_("Go to \"%s\""), path);

	widget_set_message (GTK_WIDGET (pitem), pitem->message);
}

GtkWidget *
fb_path_menu_item_new (const gchar *path)
{
	GObject *object = g_object_new (FB_TYPE_PATH_MENU_ITEM, NULL);

	if (path || *path)
		fb_path_menu_item_set_path (FB_PATH_MENU_ITEM (object), path);

	return GTK_WIDGET (object);
}

G_CONST_RETURN gchar *
fb_path_menu_item_get_path (FbPathMenuItem *pitem)
{
	g_return_if_fail (FB_IS_PATH_MENU_ITEM (pitem));

	return pitem->path;
}
