/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "fb-locationbar.h"

enum {
	LOCATION_ACTIVATE,
	LAST_SIGNAL
};

static void     fb_locationbar_class_init     (FbLocationbarClass *klass);
static void     fb_locationbar_init           (FbLocationbar      *fb);
static void     fb_locationbar_finalize       (GObject          *object);

static GtkHBoxClass *parent_class = NULL;
static gint fb_locationbar_signals[LAST_SIGNAL] = {0};

GType
fb_locationbar_get_type (void)
{
	static GType object_type = 0;

	if (! object_type) {
		static const GTypeInfo object_info = {
			sizeof (FbLocationbarClass),
			NULL,
			NULL,
			(GClassInitFunc) fb_locationbar_class_init,
			NULL,
			NULL,
			sizeof (FbLocationbar),
			32,
			(GInstanceInitFunc) fb_locationbar_init,
		};

		object_type = g_type_register_static (GTK_TYPE_HBOX, "FbLocationbar",
						      &object_info, 0);
	}

	return object_type;
}

static void
fb_locationbar_class_init (FbLocationbarClass *klass)
{
	GObjectClass *object_class;

	parent_class = g_type_class_peek_parent (klass);

	object_class = (GObjectClass *) klass;

	fb_locationbar_signals[LOCATION_ACTIVATE]
		= g_signal_new ("location-activate",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
				G_STRUCT_OFFSET (FbLocationbarClass, location_activate),
				NULL, NULL,
				g_cclosure_marshal_VOID__STRING,
				G_TYPE_NONE, 1,
				G_TYPE_CHAR);

	object_class->finalize = fb_locationbar_finalize;

}

static void
cb_clicked (GtkToolButton *button,
	    gpointer data)
{
	FbLocationbar *lb = data;

	g_signal_emit_by_name (G_OBJECT (lb->entry), "activate", lb);
}

static void
cb_activate(GtkEntry *entry,
	    gpointer data)
{
	gchar *path_utf8, *path;
	FbLocationbar *lb = data;

	path_utf8 = gtk_editable_get_chars((GtkEditable *)lb->entry, 0, -1);
	path = g_locale_from_utf8 (path_utf8, -1, NULL, NULL, NULL);
	g_free (path_utf8);

	g_signal_emit (lb, fb_locationbar_signals[LOCATION_ACTIVATE],
		       0, path);

	g_free (path);
}

static void
fb_locationbar_init (FbLocationbar *lb)
{
	GtkTooltips *tips;
	GtkToolItem *button;

	lb->entry = gtk_entry_new ();

	button = gtk_tool_button_new_from_stock ("gtk-jump-to");
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (cb_clicked), lb);
	gtk_box_pack_start (GTK_BOX (lb), GTK_WIDGET (button), FALSE, FALSE, 0);
	gtk_widget_show (GTK_WIDGET (button));

	tips = gtk_tooltips_new ();
	gtk_tool_item_set_tooltip (button, tips, _("go to location"), "go to location");

	gtk_entry_set_max_length (GTK_ENTRY (lb->entry), 128);
	g_signal_connect (G_OBJECT (lb->entry), "activate",
			  G_CALLBACK (cb_activate), lb);
	gtk_box_pack_start (GTK_BOX (lb), lb->entry, TRUE, TRUE, 0);
	gtk_widget_show (lb->entry);
}

static void
fb_locationbar_finalize (GObject *object)
{
	FbLocationbar *lb = FB_LOCATIONBAR (object);

	if (G_OBJECT_CLASS (parent_class)->finalize)
		G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
fb_locationbar_new (void)
{
	GObject *object = g_object_new (FB_TYPE_LOCATIONBAR, NULL);

	return GTK_WIDGET (object);
}

void
fb_locationbar_set_text (FbLocationbar *lb,
			 const gchar *text)
{
	gchar *utf8;

	g_return_if_fail (FB_IS_LOCATIONBAR (lb));

	utf8 = g_locale_to_utf8 (text, -1, NULL, NULL, NULL);
	if (! utf8) utf8 = g_strdup (text);

	gtk_entry_set_text (GTK_ENTRY (lb->entry), utf8);

	g_free (utf8);
}
