/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <time.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "futaba.h"
#include "fb-dirview.h"
#include "fb-marshalers.h"
#include "file-utils.h"
#include "fb-pathmenuitem.h"

#define HISTORY_STOCK 25

enum {
	PATH_ACTIVATED,
	PATH_SETTED,
	HISTORY_MOVE,
	LAST_SIGNAL
};

/* utils */
static gchar   *get_absolute_path          (const gchar *path);
static gchar   *str_mode2str               (mode_t mode);
static gchar   *str_uid2str                (uid_t uid);
static gchar   *str_gid2str                (gid_t gid);
static gchar   *str_size2str               (off_t size);
static gchar   *str_entries2str            (gint count);
static gchar   *str_mtime2str              (time_t mtime);
static gint     count_dir_entries          (const gchar *path,
					    gboolean dot_filter,
					    gboolean other_filter);
static gchar   *get_icon_id                (FbFileType    type,
					    gboolean      readable);
static gboolean is_writable                (const gchar *path);
static gboolean is_readable                (const gchar *path);
/* dirview liststore func */
static GtkTreeIter dirlist_append          (GtkListStore *store,
					    const gchar  *path,
					    gboolean      dot_filter,
					    gboolean      other_filter,
					    gboolean set_size);
static void        current_dir_append      (GtkListStore *store,
					    const gchar  *path);
static void        parent_dir_append       (GtkListStore *store,
					    const gchar  *path);
static gint        dirlist_get_count       (GtkListStore *store,
					    FbFileType    type);
/* dirview sort func */
static gint     check_dot                  (GtkTreeModel *model,
					    GtkTreeIter  *a,
					    GtkTreeIter  *b);
static gint     sort_path                  (GtkTreeModel *model,
					    GtkTreeIter  *a,
					    GtkTreeIter  *b,
					    gpointer      data);
static gint     sort_type                  (GtkTreeModel *model,
					    GtkTreeIter  *a,
					    GtkTreeIter  *b,
					    gpointer      data);
static gint     sort_size                  (GtkTreeModel *model,
					    GtkTreeIter  *a,
					    GtkTreeIter  *b,
					    gpointer      data);
static gint     sort_mtime                 (GtkTreeModel *model,
					    GtkTreeIter  *a,
					    GtkTreeIter  *b,
					    gpointer      data);
/* futaba dirview funcs */
static void     fb_dirview_class_init      (FbDirviewClass    *klass);
static void     fb_dirview_init            (FbDirview         *klass);
static void     fb_dirview_row_activated   (GtkTreeView       *tview,
					    GtkTreePath       *tpath,
					    GtkTreeViewColumn *column);
static void     fb_dirview_finalize        (GObject           *object);
static void     fb_dirview_drag_begin      (GtkWidget         *widget,
					    GdkDragContext    *drag_context);
static void     fb_dirview_drag_data_get   (GtkWidget         *widget,
					    GdkDragContext    *context,
					    GtkSelectionData  *seldata,
					    guint              info,
					    guint              time);
static void     fb_dirview_drag_data_received (GtkWidget      *widget,
					       GdkDragContext *drag_context,
					       gint            x,
					       gint            y,
					       GtkSelectionData *data,
					       guint             info,
					       guint             time);
static void     fb_dirview_drag_leave         (GtkWidget      *widget,
					       GdkDragContext *drag_context,
					       guint           time);
static gboolean fb_dirview_drag_motion        (GtkWidget      *widget,
					       GdkDragContext *drag_context,
					       gint            x,
					       gint            y,
					       guint           time);
static gboolean fb_dirview_real_set_dir    (FbDirview         *dv,
					    const gchar       *path,
					    gboolean           set_history);
static void     fb_dirview_add_history     (FbDirview         *dv,
					    const gchar       *path);

static GtkTreeViewClass *parent_class = NULL;
static gint fb_dirview_signals[LAST_SIGNAL] = {0};
gboolean cdir_filter;
gboolean pdir_filter;
gint uid = -1;
gint gid = -1;

static gchar *
get_absolute_path(const gchar *path)
{
	gint p;
	gchar *new_path;
	const gchar *ptr = path;

	if (ptr[0] != '/') return NULL;

	/* remove ".", ".." from path */
	p = strlen (path);
	if (ptr[p] == '.') {
		p--;
		if (ptr[p] == '/') p--;

		if (ptr[p] == '.') {
			p -= 2;
			while ((ptr[p] != '/') && (p > 0)) p--;
		}
	}
	new_path = g_strndup (path, (guint)p);


	for (p= 1; ptr[p] == '/'; p++) {
		new_path[p - 1] = ' ';
	}
	if (new_path[0] == ' ') new_path = g_strchug (new_path);

	return new_path;
}


gchar *
replace_string(const gchar *src_path,
	       const gchar *pattern,
	       const gchar *replace)
{
	gint i, len;
	gchar *tmp, *new_path;

	if (!src_path || !pattern) return NULL;

	i = 0;
	len = 0;

	while (pattern[len] != '\0') len++;

	tmp = g_strdup (src_path);
	while (tmp[i] == pattern[i]) {
		tmp[i] = ' ';
		i++;

		if (tmp[i] == '\0' || pattern[i] == '\0') break;
	}
	if (i != len) return NULL;

	tmp = g_strchug (tmp);

	if (tmp && replace)
		new_path = g_strconcat (replace, tmp, NULL);
	else if (tmp)
		new_path = g_strdup (tmp);
	else
		new_path = g_strdup (replace);

	g_free (tmp);

	return new_path;
}

static gchar *
str_mode2str(mode_t mode)
{
	gchar permission[11] = {"----------"};

	switch (mode & S_IFMT) {
	case S_IFREG:
		permission[0] = '-';
		break;
	case S_IFLNK:
		permission[0] = 'l';
		break;
	case S_IFDIR:
		permission[0] = 'd';
		break;
	default:
		permission[0] = '?';
		break;
	}

	if (mode & S_IRUSR)
		permission[1] = 'r';
	if (mode & S_IWUSR)
		permission[2] = 'w';
	if (mode & S_IXUSR)
		permission[3] = 'x';

	if (mode & S_IRGRP)
		permission[4] = 'r';
	if (mode & S_IWGRP)
		permission[5] = 'w';
	if (mode & S_IXGRP)
		permission[6] = 'x';

	if (mode & S_IROTH)
		permission[7] = 'r';
	if (mode & S_IWOTH)
		permission[8] = 'w';
	if (mode & S_IXOTH)
		permission[9] = 'x';
   
	if (mode & S_ISUID)
		permission[3] = 'S';
	if (mode & S_ISGID)
		permission[6] = 'S';
	if (mode & S_ISVTX)
		permission[9] = 'T';

	permission[11] = 0;

	return g_strdup(permission);
}

static gchar *
str_uid2str (uid_t uid)
{
	struct passwd *pw = getpwuid(uid);

	if (pw) return g_strdup(pw->pw_name);

	return g_strdup_printf("%d", uid);
}

static gchar *
str_gid2str (gid_t gid)
{
	struct group *gr = getgrgid(gid);

	if (gr) return g_strdup(gr->gr_name);

	return g_strdup_printf("%d", gid);
}

static gchar *
str_size2str(off_t size)
{
	if (size > 1000000)
		return g_strdup_printf(_("%4.1f Mbyte"), (gfloat) size / 1000000);
	else if (size > 1000)
		return g_strdup_printf(_("%4.1f Kbyte"), (gfloat) size / 1000);

	return g_strdup_printf(_("%d byte"), size);
}

static gchar *
str_entries2str(gint count)
{
	if (count <= 0)
		return g_strdup(_("no item"));
	else if (count == 1)
		return g_strdup(_("1 item"));
	else if (count > 1)
		return g_strdup_printf(_("%d items"), count);
}

static gchar *
str_mtime2str(time_t mtime)
{
	/*   struct tm *jst = localtime(&time); */

	/*   return g_strdup_printf("%4d/%02d/%02d %s %02d:%02d", */
	/* 			 jst->tm_year + 1900, jst->tm_mon + 1, jst->tm_mday, */
	/* 			 _(week[jst->tm_wday]), jst->tm_hour, jst->tm_min); */
	return NULL;
}

static gint
count_dir_entries(const gchar *path,
		  gboolean dot_filter,
		  gboolean other_filter)
{
	gint len, count = 0;
	gchar *p;
	GList *path_list, *node;

	if (! path || ! *path) return -1;

	path_list = get_path_list(path, FALSE);
	for (node = path_list; node; node = node->next) {
		p = node->data;

		if (! p) continue;

		len = strlen (p);
		while (p[len] != '/') len--;
		len++;

		if (dot_filter && p[len] == '.') continue;

		if (other_filter &&
		    ! path_is_dir (p) &&
		    ! (file_is_image(p) || file_is_archive(p)))
			continue;

		count++;
	}

	path_list_free(path_list);

	return count;
}

static gchar *
get_icon_id(FbFileType type,
	    gboolean readable)
{
	if (type == OTHER) return NULL;

	if (readable) {
		if (type == DIRECTORY)
			return g_strdup("fb-dir");
		else if (type == IMAGE)
			return g_strdup("fb-image");
		else if (type == ARCHIVE)
			return g_strdup("fb-archive");
	}
	else {
		if (type == DIRECTORY)
			return g_strdup("fb-noread-dir");
		else if (type == IMAGE)
			return g_strdup("fb-noread-image");
		else if (type == ARCHIVE)
			return g_strdup("fb-noread-archive");
	}
	
	return NULL;
}

static gboolean
is_readable(const gchar *path)
{
	struct stat file;

	if (! path || ! *path) return FALSE; 
	if (stat(path, &file)) return FALSE;

	if (uid = -1) uid = getuid ();
	if (gid = -1) gid = getgid ();

	if (file.st_mode & S_IROTH)
		return TRUE;
	else if (gid == file.st_gid && (file.st_mode & S_IRGRP))
		return TRUE;
	else if (uid == file.st_uid && (file.st_mode & S_IRUSR))
		return TRUE;

	return FALSE;
}

static gboolean
is_writable(const gchar *path)
{
	struct stat file;

	if (! path || ! *path) return FALSE; 
	if (stat(path, &file)) return FALSE;

	if (uid = -1) uid = getuid ();
	if (gid = -1) gid = getgid ();

	if (file.st_mode & S_IWOTH)
		return TRUE;
	else if (gid == file.st_gid && (file.st_mode && S_IWGRP))
		return TRUE;
	else if (uid == file.st_uid && (file.st_mode && S_IWUSR))
		return TRUE;

	return FALSE;
}

static GtkTreeIter
dirlist_append(GtkListStore *list,
	       const gchar *path,
	       gboolean dot_filter,
	       gboolean other_filter,
	       gboolean set_size)
{
	struct stat file;
	gboolean readable;
	gint type;
	gchar *name, *utf8_name, *icon_id;
	gchar *permission, *owner, *group, *size, *mtime;
	GtkTreeIter iter;

	if (! path || ! *path) return iter;
	if (!GTK_IS_LIST_STORE(list)) return iter;

	name = g_path_get_basename(path);
	if (dot_filter && name[0] == '.') {
		g_free(name);
		return iter;
	}

	if (stat(path, &file)) {
		return iter;
	}

	if (S_ISDIR(file.st_mode))
		type = DIRECTORY;
	else if (S_ISREG(file.st_mode) && file_is_image(name))
		type = IMAGE;
	else if (S_ISREG(file.st_mode) && file_is_archive(name))
		type = ARCHIVE;
	else
		type = OTHER;

	if (other_filter && (type == OTHER)) {
		g_free(name);
		return iter;
	}

	utf8_name = g_locale_to_utf8 (name, -1, NULL, NULL, NULL);
	g_free(name);

	readable = is_readable(path);
/* 	is_writable (path); */

	if (type == DIRECTORY && set_size)
		size = str_entries2str(count_dir_entries(path, dot_filter, other_filter));
	else if (type == DIRECTORY && ! set_size)
		size = g_strdup ("...");
	else
		size = str_size2str(file.st_size);
	icon_id = get_icon_id(type, readable);
	permission = str_mode2str(file.st_mode);
	owner = str_uid2str(file.st_uid);
	group = str_gid2str(file.st_gid);
	mtime = str_mtime2str(file.st_mtime);

	gtk_list_store_append(list, &iter);
	gtk_list_store_set(list, &iter,
			   COLUMN_TYPE, type,
			   COLUMN_PATH, path,
			   COLUMN_SIZE, file.st_size,
			   COLUMN_MTIME, file.st_mtime,
			   COLUMN_READABLE, readable,
			   COLUMN_ICON, icon_id,
			   COLUMN_ICON_SIZE, GTK_ICON_SIZE_LARGE_TOOLBAR,
			   COLUMN_DISPLAY_PATH, utf8_name,
			   COLUMN_PERMISSION, permission,
			   COLUMN_OWNER, owner,
			   COLUMN_GROUP, group,
			   COLUMN_DISPLAY_SIZE, size,
			   COLUMN_DISPLAY_MTIME, mtime,
			   COLUMN_EDITABLE, FALSE,
			   -1);

	g_free (icon_id);
	g_free (utf8_name);
	g_free (permission);
	g_free (owner);
	g_free (group);
	g_free (size);
	g_free (mtime);

	return iter;
}

static void
current_dir_append (GtkListStore *store,
		    const gchar *path)
{
	GtkTreeIter iter;

	iter = dirlist_append (store, path, FALSE, FALSE, FALSE);
	gtk_list_store_set (store, &iter,
			    COLUMN_TYPE, WORKDIR,
			    COLUMN_DISPLAY_PATH, ".",
			    -1);
}

static void
parent_dir_append (GtkListStore *store,
		   const gchar *path)
{
	gchar *parent_path;
	GtkTreeIter iter;

	if (!strcmp (path, "/")) {
		parent_path = g_strdup (path);
	}
	else {
		parent_path = remove_level_from_path (path);
	}
	iter = dirlist_append (store, parent_path, FALSE, FALSE, FALSE);
	gtk_list_store_set (store, &iter,
			    COLUMN_TYPE, PARENTDIR,
			    COLUMN_DISPLAY_PATH, "..",
			    -1);

	g_free (parent_path);
}

static gint
dirlist_get_count(GtkListStore *list,
		  FbFileType type)
{
	gint cmp, count = 0;
	GtkTreeIter iter;

	if (type == DIRECTORY) count = 2;

	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list), &iter);
	while (gtk_tree_model_iter_next(GTK_TREE_MODEL(list), &iter)) {
		cmp = -1;
		gtk_tree_model_get(GTK_TREE_MODEL(list), &iter,
				   COLUMN_TYPE, &cmp,
				   -1);
		if (cmp == type) count++;
	}

	return count;
}

/*
 *------------------------------------------------
 *       Dirview Sort Func
 *------------------------------------------------
 */
static gint
check_dot(GtkTreeModel *model,
	  GtkTreeIter *a,
	  GtkTreeIter *b)
{
	gint type1, type2;

/* 	type1 = type2 = -1; */
	gtk_tree_model_get(model, a, COLUMN_TYPE, &type1, -1);
	gtk_tree_model_get(model, b, COLUMN_TYPE, &type2, -1);

	if (type1 == WORKDIR)
		return -1;
	else if (type2 == WORKDIR)
		return 1;
	else if (type1 == PARENTDIR)
		return -1;
	else if (type2 == PARENTDIR)
		return 1;
	else
		return 0;
}

static gint
sort_path(GtkTreeModel *model,
	  GtkTreeIter *a,
	  GtkTreeIter *b,
	  gpointer data)
{
	gint cmp;
	gchar *str1, *str2;

	cmp = check_dot (model, a, b);
	if (cmp != 0) return cmp;

	gtk_tree_model_get(model, a, COLUMN_PATH, &str1, -1);
	gtk_tree_model_get(model, b, COLUMN_PATH, &str2, -1);

	cmp = strcmp(str1, str2);

	g_free (str1);
	g_free (str2);

	return cmp;
}

static gint
sort_type(GtkTreeModel *model,
	  GtkTreeIter *a,
	  GtkTreeIter *b,
	  gpointer data)
{
	gint type1, type2;

	gtk_tree_model_get(model, a, COLUMN_TYPE, &type1, -1);
	gtk_tree_model_get(model, b, COLUMN_TYPE, &type2, -1);

	if (type1 < type2)
		return -1;
	else if (type2 > type1)
		return 1;
	else if (type1 == type2)
		return sort_path (model, a, b, data);
}

static gint
sort_size(GtkTreeModel *model,
	  GtkTreeIter *a,
	  GtkTreeIter *b,
	  gpointer data)
{
	gint size1, size2, cmp;

	cmp = check_dot(model, a, b);
	if (cmp) return cmp;

	gtk_tree_model_get(model, a, COLUMN_SIZE, &size1, -1);
	gtk_tree_model_get(model, b, COLUMN_SIZE, &size2, -1);

	if (size1 > size2)
		return -1;
	else if (size1 < size2)
		return 1;
	else if (size1 == size2)
		return sort_path (model, a, b, data);
}

static gint
sort_mtime(GtkTreeModel *model,
	   GtkTreeIter *a,
	   GtkTreeIter *b,
	   gpointer data)
{
	gint cmp;
	glong time1, time2;

	cmp = check_dot(model, a, b);
	if (cmp) return cmp;

	gtk_tree_model_get(model, a, COLUMN_MTIME, &time1, -1);
	gtk_tree_model_get(model, b, COLUMN_MTIME, &time2, -1);

	if (time1 > time2)
		return 1;
	else if (time1 < time2)
		return -1;
	else if (time1 == time2)
		return sort_path (model, a, b, data);
}

/*
 *------------------------------------------------
 *       Dirview
 *------------------------------------------------
 */
GType
fb_dirview_get_type (void)
{
	static GType object_type = 0;

	if (!object_type) {
		static const GTypeInfo object_info = {
			sizeof (FbDirviewClass),
			NULL,
			NULL,
			(GClassInitFunc) fb_dirview_class_init,
			NULL,
			NULL,
			sizeof (FbDirview),
			32,
			(GInstanceInitFunc) fb_dirview_init,
		};

		object_type = g_type_register_static (GTK_TYPE_TREE_VIEW, "FbDirview",
						      &object_info, 0);
	}

	return object_type;
}

static void
fb_dirview_class_init (FbDirviewClass *klass)
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;
	GtkTreeViewClass *tree_class;

	parent_class = g_type_class_peek_parent (klass);

	object_class = (GObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;
	tree_class = (GtkTreeViewClass *) klass;

	klass->path_activated = NULL;
	klass->path_setted = NULL;
	klass->history_move = NULL;

	fb_dirview_signals[PATH_ACTIVATED]
		= g_signal_new ("path-activated",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
				G_STRUCT_OFFSET (FbDirviewClass, path_activated),
				NULL, NULL,
				fb_marshal_VOID__STRING_INT,
				G_TYPE_NONE, 2,
				G_TYPE_CHAR,
				G_TYPE_INT);
	fb_dirview_signals[PATH_SETTED]
		= g_signal_new ("path-setted",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
				G_STRUCT_OFFSET (FbDirviewClass, path_setted),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID,
				G_TYPE_NONE, 0);
	fb_dirview_signals[HISTORY_MOVE]
		= g_signal_new ("history-move",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
				G_STRUCT_OFFSET (FbDirviewClass, history_move),
				NULL, NULL,
				g_cclosure_marshal_VOID__STRING,
				G_TYPE_NONE, 1,
				G_TYPE_CHAR);

	object_class->finalize = fb_dirview_finalize;

	widget_class->drag_begin = fb_dirview_drag_begin;
	widget_class->drag_data_get = fb_dirview_drag_data_get;
	widget_class->drag_data_received = fb_dirview_drag_data_received;
	widget_class->drag_leave = fb_dirview_drag_leave;
	widget_class->drag_motion = fb_dirview_drag_motion;

	tree_class->row_activated = fb_dirview_row_activated;

}

static void
cb_name_edited (GtkCellRendererText *cell,
		gchar *string,
		gchar *name,
		gpointer data)
{
	gchar *tmp, *src_path, *new_path;
	GtkTreeIter iter;
	GtkTreePath *tpath;
	FbDirview *dv = data;

	tpath = gtk_tree_path_new_from_string (string);
	gtk_tree_model_get_iter (GTK_TREE_MODEL (dv->store), &iter, tpath);
	gtk_list_store_set (dv->store, &iter,
			    COLUMN_EDITABLE, FALSE,
			    -1);
	gtk_tree_path_free (tpath);

	src_path = NULL;
	gtk_tree_model_get(GTK_TREE_MODEL(dv->store), &iter,
			   COLUMN_PATH, &src_path,
			   -1);
	/* FIX ME (add message dialog) */
	tmp = g_path_get_basename(src_path);
	if (!strcmp(name, tmp)) {
		g_free (src_path);
		g_free (tmp);
		return;
	}
	g_free(tmp);

	tmp = g_path_get_dirname(src_path);
	new_path = g_strconcat(tmp, "/", name, NULL);
	g_free(tmp);

	if (rename(src_path, new_path)) {
		g_free (src_path);
		g_free (new_path);
		return;
	}
	/* END */

	gtk_list_store_remove (dv->store, &iter);
	dirlist_append(dv->store, new_path, dv->dot_filter, dv->other_filter, TRUE);

	g_free (src_path);
	g_free (new_path);

}

static void
fb_dirview_init (FbDirview *dv)
{
	GtkCellRenderer *renderer;
	GtkTreeSortable *sortable;

	dv->next_history = gtk_menu_new ();
	dv->prev_history = gtk_menu_new ();
	dv->current = NULL;

	dv->workdir = NULL;
	dv->writable = FALSE;
	dv->image_num = -1;
	dv->source_id = -1;

	dv->dot_filter = TRUE;
	dv->other_filter = TRUE;

	dv->prev_path = NULL;

	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(dv), TRUE);
	gtk_tree_view_enable_model_drag_source (GTK_TREE_VIEW (dv),
						GDK_BUTTON1_MASK,
						uri_drag_types, n_uri_drag_types,
						GDK_ACTION_ASK  | GDK_ACTION_COPY
						| GDK_ACTION_MOVE | GDK_ACTION_LINK);
	gtk_tree_view_enable_model_drag_dest (GTK_TREE_VIEW (dv),
					      uri_drag_types, n_uri_drag_types,
					      GDK_ACTION_COPY | GDK_ACTION_MOVE);
/* 	{ */
/* 		GtkTreeSelection *selection; */

/* 		selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dv)); */
/* 		gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE); */
/* 	} */

	dv->column = gtk_tree_view_column_new();
	gtk_tree_view_append_column(GTK_TREE_VIEW(dv), dv->column);
	gtk_tree_view_column_set_resizable(dv->column, TRUE);
	gtk_tree_view_column_set_title(dv->column, _("Name"));

	renderer = gtk_cell_renderer_pixbuf_new();
	gtk_tree_view_column_pack_start(dv->column, renderer, FALSE);
	gtk_tree_view_column_set_attributes(dv->column, renderer,
					    "stock-id", COLUMN_ICON,
					    "stock-size", COLUMN_ICON_SIZE,
					    NULL);

	renderer = gtk_cell_renderer_text_new();
	g_signal_connect(G_OBJECT(renderer), "edited",
			 G_CALLBACK(cb_name_edited), dv);
	gtk_tree_view_column_pack_start(dv->column, renderer, TRUE);
	gtk_tree_view_column_set_attributes(dv->column, renderer,
					    "text", COLUMN_DISPLAY_PATH,
					    "editable", COLUMN_EDITABLE,
					    NULL);

	/* permission column */
	dv->column1 = gtk_tree_view_column_new();
	gtk_tree_view_column_set_resizable(dv->column1, TRUE);
	gtk_tree_view_column_set_title(dv->column1, _("Permission"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(dv), dv->column1);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(dv->column1, renderer, TRUE);
	gtk_tree_view_column_set_attributes(dv->column1, renderer,
					    "text", COLUMN_PERMISSION,
					    NULL);

	/* owner column */
	dv->column2 = gtk_tree_view_column_new();
	gtk_tree_view_column_set_resizable(dv->column2, TRUE);
	gtk_tree_view_column_set_title(dv->column2, _("Owner"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(dv), dv->column2);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(dv->column2, renderer, TRUE);
	gtk_tree_view_column_set_attributes(dv->column2, renderer,
					    "text", COLUMN_OWNER,
					    NULL);

	/* group column */
	dv->column3 = gtk_tree_view_column_new();
	gtk_tree_view_column_set_resizable(dv->column3, TRUE);
	gtk_tree_view_column_set_title(dv->column3, _("Group"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(dv), dv->column3);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(dv->column3, renderer, TRUE);
	gtk_tree_view_column_set_attributes(dv->column3, renderer,
					    "text", COLUMN_GROUP,
					    NULL);

	/* size column */
	dv->column4 = gtk_tree_view_column_new();
	gtk_tree_view_column_set_resizable(dv->column4, TRUE);
	gtk_tree_view_column_set_title(dv->column4, _("Size"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(dv), dv->column4);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(dv->column4, renderer, TRUE);
	gtk_tree_view_column_set_attributes(dv->column4, renderer,
					    "text", COLUMN_DISPLAY_SIZE,
					    NULL);

	/* mtime column */
	dv->column5 = gtk_tree_view_column_new();
	gtk_tree_view_column_set_resizable(dv->column5, TRUE);
	gtk_tree_view_column_set_title(dv->column5, _("Date Modified"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(dv), dv->column5);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(dv->column5, renderer, TRUE);
	gtk_tree_view_column_set_attributes(dv->column5, renderer,
					    "text", COLUMN_DISPLAY_MTIME,
					    NULL);

	dv->store = gtk_list_store_new(NUM_COLUMNS,
				       G_TYPE_INT,
				       G_TYPE_STRING,
				       G_TYPE_INT,
				       G_TYPE_LONG,
				       G_TYPE_BOOLEAN,
				       G_TYPE_STRING,
				       G_TYPE_INT,
				       G_TYPE_STRING,
				       G_TYPE_STRING,
				       G_TYPE_STRING,
				       G_TYPE_STRING,
				       G_TYPE_STRING,
				       G_TYPE_STRING,
				       G_TYPE_BOOLEAN,
				       -1);

	sortable = GTK_TREE_SORTABLE(dv->store);
	gtk_tree_sortable_set_sort_func(sortable,
					SORT_BY_NAME,
					sort_path, NULL, NULL);
	gtk_tree_sortable_set_sort_func(sortable,
					SORT_BY_SIZE,
					sort_size, NULL, NULL);
	gtk_tree_sortable_set_sort_func(sortable,
					SORT_BY_MTIME,
					sort_mtime, NULL, NULL);
	gtk_tree_sortable_set_sort_func(sortable,
					SORT_BY_TYPE,
					sort_type, NULL, NULL);

	gtk_tree_sortable_set_sort_column_id(sortable, SORT_BY_TYPE, GTK_SORT_ASCENDING);

	gtk_tree_view_set_model (GTK_TREE_VIEW (dv), GTK_TREE_MODEL (dv->store));

}

static void
fb_dirview_row_activated (GtkTreeView *tview,
			  GtkTreePath *tpath,
			  GtkTreeViewColumn *column)
{
	gint type = -1;
	gchar *path = NULL;
	GtkTreeIter iter;
	GtkTreeModel *model;
	FbDirview *dv = FB_DIRVIEW (tview);

	model = gtk_tree_view_get_model (tview);
	if (gtk_tree_model_get_iter (model, &iter, tpath)) {
		gtk_tree_model_get (model, &iter,
				    COLUMN_PATH, &path,
				    COLUMN_TYPE, &type,
				    -1);

		g_signal_emit (dv, fb_dirview_signals[PATH_ACTIVATED],
			       0, path, type);
	}
}

static void
fb_dirview_finalize (GObject *object)
{
	FbDirview *dv = FB_DIRVIEW (object);

	if (dv->workdir) g_free (dv->workdir);

	if (G_OBJECT_CLASS (parent_class)->finalize)
		G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
fb_dirview_drag_begin (GtkWidget *widget,
		       GdkDragContext *drag_context)
{
	GtkTreeIter iter;
	GtkTreePath *tpath;
	FbDirview *dv = FB_DIRVIEW (widget);

	gtk_tree_view_get_cursor (GTK_TREE_VIEW (widget), &tpath, NULL);
	if (gtk_tree_model_get_iter (GTK_TREE_MODEL (dv->store), &iter, tpath)) {
		gchar *icon_id = NULL;

		gtk_tree_model_get (GTK_TREE_MODEL (dv->store), &iter,
				    COLUMN_ICON, &icon_id,
				    -1);
		if (icon_id) {
			gtk_drag_set_icon_stock (drag_context, icon_id, -5, -5);
			g_free (icon_id);
		}
	}
	gtk_tree_path_free (tpath);
}

static void
fb_dirview_drag_data_get (GtkWidget        *widget,
			  GdkDragContext   *context,
			  GtkSelectionData *seldata,
			  guint             info,
			  guint             time)
{
	gchar *path, *uri;
	FbDirview *dv = FB_DIRVIEW (widget);

/* 	g_print ("dirview:drag data get\n"); */

	switch (info) {
	case TARGET_TEXT_URI_LIST:
	case TARGET_TEXT_PLAIN:
		if (fb_dirview_row_get (dv, COLUMN_PATH, &path, -1)) {
			uri = g_strdup_printf ("file://%s\r\n", path);
			gtk_selection_data_set (seldata, seldata->target,
						8, uri, strlen (uri));
			g_free (path);
			g_free (uri);
		}
		break;
	default:
		break;
	}

	GTK_WIDGET_CLASS (parent_class)->drag_data_get (widget, context, seldata, info, time);
}

static void
fb_dirview_drag_leave (GtkWidget *widget,
		       GdkDragContext *drag_context,
		       guint time)
{
	GtkTreeIter iter;
	FbDirview *dv = FB_DIRVIEW (widget);

	if (dv->prev_path) {
		if (gtk_tree_model_get_iter (GTK_TREE_MODEL (dv->store), &iter, dv->prev_path))
			gtk_list_store_set (dv->store, &iter,
					    COLUMN_ICON, "fb-dir",
					    -1);
		gtk_tree_path_free (dv->prev_path);
		dv->prev_path = NULL;
	}

	GTK_WIDGET_CLASS (parent_class)->drag_leave (widget, drag_context, time);
}

#define ICON_WIDTH 21

static void
fb_dirview_drag_data_received (GtkWidget *widget,
			       GdkDragContext *drag_context,
			       gint x, gint y,
			       GtkSelectionData *data,
			       guint info,
			       guint time)
{
	gchar *uri;
	GtkTreeIter iter;
	FbDirview *dv = FB_DIRVIEW (widget);

/* 	g_print ("dirview:drag data received\n"); */

	if (dv->prev_path) {
		if (gtk_tree_model_get_iter (GTK_TREE_MODEL (dv->store), &iter, dv->prev_path))
			gtk_list_store_set (dv->store, &iter,
					    COLUMN_ICON, "fb-dir",
					    -1);
		gtk_tree_path_free (dv->prev_path);
		dv->prev_path = NULL;
	}

	uri = g_strndup (data->data, data->length);
/* 	g_print ("%s", uri); */

	if (g_str_has_prefix (uri, "file:/")) {
		gint type, s_len, r_len;
		gchar *src, *dest, *base, *ptr, *sub;
		GtkTreeIter iter;
		GtkTreePath *tpath;

		s_len = strlen ("file:");
		r_len = 0;
		ptr = uri + strlen (uri) - 1;
		while (*ptr == '\r' || *ptr == '\n' || *ptr == '\0') {
			r_len++;
			ptr--;
		}

		src = g_strndup (uri + s_len, strlen (uri) - s_len - r_len);
		base = g_path_get_basename (src);
	
		type = -1;
		sub = NULL;
		gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (widget), x, y, &tpath, NULL, NULL, NULL);
		if (tpath) {
			if (gtk_tree_model_get_iter (GTK_TREE_MODEL (dv->store), &iter, tpath))
				gtk_tree_model_get (GTK_TREE_MODEL (dv->store), &iter,
						    COLUMN_TYPE, &type,
						    COLUMN_PATH, &sub,
						    -1);
			gtk_tree_path_free (tpath);
		}

		if ((type == DIRECTORY || type == WORKDIR || type == PARENTDIR)
		    && x <= ICON_WIDTH && sub)
			dest = g_strconcat (sub, "/", base, NULL);
		else
			dest = g_strconcat (dv->workdir, "/", base, NULL);

		g_free (base);
		if (sub) g_free (sub);

		if (! g_file_test (src, G_FILE_TEST_EXISTS) ||
		    g_file_test (dest, G_FILE_TEST_EXISTS))
			goto fin;

		switch (drag_context->suggested_action) {
		case GDK_ACTION_COPY: {
			if (path_is_dir (src)) {
				cpdir_recursive (src, dest);				
			}
			else {
				file_copy (src, dest);				
			}
		}
			break;
		case GDK_ACTION_MOVE: {
			if (path_is_dir (src)) {
				mvdir_recursive (src, dest);
			}
			else {
				file_move (src, dest);
			}
		}
			break;
		default:
			break;
		}

	fin:
		g_free (src);
		g_free (dest);
	}

	g_free (uri);
	
	GTK_WIDGET_CLASS (parent_class)->drag_data_received (widget, drag_context, x, y, data, info, time);
}

static gboolean
fb_dirview_drag_motion (GtkWidget *widget,
			GdkDragContext *drag_context,
			gint x, gint y,
			guint time)
{
	gchar *icon_id;
	GtkTreeIter iter;
	GtkTreePath *tpath;
	GtkTreeModel *model;
	FbDirview *dv = FB_DIRVIEW (widget);

	model = GTK_TREE_MODEL (dv->store);
	gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (widget), x, y, &tpath, NULL, NULL, NULL);
/* 	g_print ("pointer x %d y %d\n", x, y); */

        if (dv->prev_path &&
	    (x > ICON_WIDTH || ! tpath || gtk_tree_path_compare (tpath, dv->prev_path))) {
		if (gtk_tree_model_get_iter (model, &iter, dv->prev_path))
			gtk_list_store_set (dv->store, &iter,
					    COLUMN_ICON, "fb-dir",
					    -1);
		gtk_tree_path_free (dv->prev_path);
		dv->prev_path = NULL;
	}

	if (tpath && x <= ICON_WIDTH) {
		icon_id = NULL;
		if (gtk_tree_model_get_iter (model, &iter, tpath))
			gtk_tree_model_get (model, &iter,
					    COLUMN_ICON, &icon_id,
					    -1);
		if (icon_id) {
			if (! g_ascii_strcasecmp (icon_id, "fb-dir")) {
				gtk_list_store_set (dv->store, &iter,
						    COLUMN_ICON, "fb-dir-accept",
						    -1);
				dv->prev_path = tpath;
			}
			g_free (icon_id);
		}
	}

	return GTK_WIDGET_CLASS (parent_class)->drag_motion (widget, drag_context, x, y, time);
}

GtkWidget *
fb_dirview_new (void)
{
	GObject *object = g_object_new (FB_TYPE_DIRVIEW, NULL);

	return GTK_WIDGET (object);
}

GList *
fb_dirview_get_path_list (FbDirview *dv,
			  FbFileType type)
{
	gint cmp;
	gchar *path;
	GList *path_list = NULL;
	GtkTreeIter iter;

	g_return_val_if_fail (FB_IS_DIRVIEW (dv), NULL);

	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(dv->store), &iter);
	while (gtk_tree_model_iter_next(GTK_TREE_MODEL(dv->store), &iter)) {
		cmp = -1;
		path = NULL;
		gtk_tree_model_get(GTK_TREE_MODEL(dv->store), &iter,
				   COLUMN_TYPE, &cmp,
				   COLUMN_PATH, &path,
				   -1);
		if (cmp == type)
			path_list = g_list_append(path_list, path);
	}

	return path_list;
}

gboolean
fb_dirview_row_get (FbDirview *dv,
		    ...)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *select;
	va_list var_args;

	g_return_if_fail (FB_IS_DIRVIEW (dv));

	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (dv));
	if (! gtk_tree_selection_get_selected (select, &model, &iter))
		return FALSE;

	va_start (var_args, dv);
	gtk_tree_model_get_valist (model, &iter, var_args);
	va_end (var_args);

	return TRUE;
}

void
fb_dirview_row_remove (FbDirview *dv)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *select;

	g_return_if_fail (FB_IS_DIRVIEW (dv));

	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (dv));
	if (! gtk_tree_selection_get_selected (select, &model, &iter))
		return;

	gtk_list_store_remove(dv->store, &iter);
}

void
fb_dirview_row_edit (FbDirview *dv)
{
	GtkTreeIter iter;
	GtkTreePath *path;
	GtkTreeViewColumn *column;
	GtkTreeSelection *select;
	GtkTreeModel *model;

	g_return_if_fail (FB_IS_DIRVIEW (dv));

	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (dv));
	if (! gtk_tree_selection_get_selected (select, &model, &iter))
		return;

	gtk_list_store_set (GTK_LIST_STORE (model), &iter,
			    COLUMN_EDITABLE, TRUE,
			    -1);

	gtk_tree_view_get_cursor(GTK_TREE_VIEW(dv), &path, &column);
	gtk_tree_view_set_cursor(GTK_TREE_VIEW(dv), path, column, TRUE);

	gtk_tree_path_free(path);

/* 	fb_accelerator_lock(futaba); */
}

static gboolean
set_dir_entries (gpointer data)
{
	gint type;
	gchar *path, *size;
	GtkTreeIter iter;
	GtkTreeModel *model;
	FbDirview *dv = data;

	model = GTK_TREE_MODEL (dv->store);
	if (! gtk_tree_model_get_iter_first (model, &iter))
		return FALSE;

	do {
		while (gtk_events_pending ()) gtk_main_iteration ();	
		type = -1;
		gtk_tree_model_get (model, &iter,
				    COLUMN_TYPE, &type,
				    -1);
		if (type < WORKDIR || type > DIRECTORY) continue;

		path = NULL;
		gtk_tree_model_get (model, &iter,
				    COLUMN_PATH, &path,
				    -1);
		if (path) {
			size = str_entries2str(count_dir_entries(path, dv->dot_filter, dv->other_filter));
			gtk_list_store_set (dv->store, &iter,
					    COLUMN_DISPLAY_SIZE, size,
					    -1);
			g_free (path);
			g_free (size);
		}
	} while (gtk_tree_model_iter_next (model, &iter));

	dv->source_id = -1;

	return FALSE;
}

static gboolean
fb_dirview_real_set_dir (FbDirview *dv,
			 const gchar *path,
			 gboolean set_history)
{
	gchar *abs_path;
	GList *path_list, *s;
	GtkAdjustment *adjust;

	abs_path = get_absolute_path (path);
	if (! abs_path) return FALSE;

	if (dv->source_id > -1)
		g_source_remove (dv->source_id);

	gtk_list_store_clear (dv->store);

	if (! cdir_filter) {
		current_dir_append (dv->store, abs_path);
	}

	if (! pdir_filter) {
		parent_dir_append (dv->store, abs_path);
	}

	path_list = get_path_list (abs_path, FALSE);
	for (s = path_list; s != NULL; s = s->next) {
		dirlist_append (dv->store, s->data, dv->dot_filter, dv->other_filter, FALSE);
	}
	path_list_free (path_list);

	if (dv->workdir) g_free (dv->workdir);

	dv->workdir = g_strdup (abs_path);
	dv->writable = path_is_writable (abs_path);
	dv->image_num = dirlist_get_count (dv->store, IMAGE);
	dv->source_id = g_idle_add ((GSourceFunc) set_dir_entries, dv);

	if (set_history) fb_dirview_add_history (dv, abs_path);

	g_free (abs_path);

	adjust = gtk_tree_view_get_vadjustment (GTK_TREE_VIEW (dv));
	gtk_adjustment_set_value (adjust, 0.0);

	g_signal_emit (dv, fb_dirview_signals[PATH_SETTED], 0);

	return TRUE;
}

gboolean
fb_dirview_set_dir (FbDirview *dv,
		    const gchar *path)
{
	gchar *err_msg = NULL;

	g_return_val_if_fail (FB_IS_DIRVIEW (dv), FALSE);
	g_return_val_if_fail (path || *path, FALSE);

	if (! path_is_exists (path))
		err_msg = g_strdup (_("The path don't exists"));
	else if (! path_is_dir (path))
		err_msg = g_strdup (_("The path isn't directory"));
	else if (! path_is_readable (path))
		err_msg = g_strdup (_("The path isn't reabable"));

	if (err_msg) {
		GtkWidget *dlg;

		dlg = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL,
					      GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
					      err_msg);
		g_signal_connect_swapped (G_OBJECT (dlg), "response",
					  G_CALLBACK (gtk_widget_destroy), dlg);
		gtk_widget_show (dlg);

		return FALSE;
	}

	return fb_dirview_real_set_dir (dv, path, TRUE);
}

void
fb_dirview_update_dir (FbDirview *dv)
{
	g_return_if_fail (FB_IS_DIRVIEW (dv));

	fb_dirview_real_set_dir (dv, dv->workdir, FALSE);
}

G_CONST_RETURN gchar *
fb_dirview_get_workdir (FbDirview *dv)
{
	g_return_if_fail (FB_IS_DIRVIEW (dv));

	return dv->workdir;
}

void
fb_dirview_add_path (FbDirview *dv,
		     const gchar *path)
{
	gint type;
	gchar *store;
	GtkTreeIter iter;
	GtkTreePath *tpath;
	GtkTreeModel *model;

	g_return_if_fail (FB_IS_DIRVIEW (dv));
	g_return_if_fail (path || *path);

	model = GTK_TREE_MODEL (dv->store);

	gtk_tree_model_get_iter_first (model, &iter);
	do {
		store = NULL;
		gtk_tree_model_get (model, &iter,
				    COLUMN_PATH, &store,
				    -1);

		if (! strcmp (path, store)) {
			g_free (store);
			return;
		}
	} while (gtk_tree_model_iter_next (model, &iter));

	iter = dirlist_append (dv->store, path, dv->dot_filter, dv->other_filter, TRUE);

	type = -1;
	gtk_tree_model_get (model, &iter,
			    COLUMN_TYPE, &type,
			    -1);
	if (type == IMAGE) {
		dv->image_num++;
	}
/* 	else if (type == DIRECTORY) { */

/* 	} */

	tpath = gtk_tree_model_get_path (model, &iter);
	gtk_tree_view_set_cursor (GTK_TREE_VIEW (dv), tpath, NULL, FALSE);

	gtk_tree_path_free (tpath);
}

void
fb_dirview_add_new_dir (FbDirview *dv)
{
	gint nth;
	gchar *tmp, *path;

	g_return_if_fail (FB_IS_DIRVIEW (dv));

	tmp = g_strconcat (dv->workdir, _("/New1"), NULL);
	path = g_locale_from_utf8 (tmp, -1, NULL, NULL, NULL);
	g_free (tmp);
	for (nth = 2; path_is_exists (path); nth++) {
		g_free (path);
		tmp = g_strdup_printf (_("%s/New%d"), dv->workdir, nth);
		path = g_locale_from_utf8 (tmp, -1, NULL, NULL, NULL);
		g_free (tmp);
	}

	if (mkdir (path, S_IRWXU)) {
		g_free (path);
		return;
	}

	fb_dirview_add_path (dv, path);
	fb_dirview_row_edit (dv);

	g_free (path);
}

void
fb_dirview_set_dotfilter (FbDirview *dv,
			  gboolean set_filter)
{
	g_return_if_fail (FB_IS_DIRVIEW (dv));

	dv->dot_filter = set_filter;
}

void
fb_dirview_set_otherfilter (FbDirview *dv,
			    gboolean set_filter)
{
	g_return_if_fail (FB_IS_DIRVIEW (dv));

	dv->other_filter = set_filter;
}

gboolean
fb_dirview_get_dotfilter (FbDirview *dv)
{
	g_return_if_fail (FB_IS_DIRVIEW (dv));

	return dv->dot_filter;
}

gboolean
fb_dirview_get_otherfilter (FbDirview *dv)
{
	g_return_if_fail (FB_IS_DIRVIEW (dv));

	return dv->other_filter;
}

void
fb_dirview_set_sort_mode (FbDirview *dv,
			  FbSortMode mode)
{
	GtkSortType order;
	GtkTreeSortable *sortable;

	g_return_if_fail (FB_IS_DIRVIEW (dv));

	sortable = GTK_TREE_SORTABLE(dv->store);

	gtk_tree_sortable_get_sort_column_id(sortable, NULL, &order);
	gtk_tree_sortable_set_sort_column_id(sortable, mode, order);

}

static void
cb_set_history (GtkMenuItem *item,
		gpointer data)
{
	const gchar *path;
	GList *children;
	GtkWidget *comp;
	FbDirview *dv = data;

	comp = gtk_menu_get_active (GTK_MENU (dv->prev_history));
	if (comp == GTK_WIDGET (item)) { /* prev history item clicked */
		gtk_menu_shell_prepend (GTK_MENU_SHELL (dv->next_history), dv->current);

		children = gtk_container_get_children (GTK_CONTAINER (dv->prev_history));
		for (; children != NULL; children = children->next) {
			g_object_ref (children->data);
			gtk_container_remove (GTK_CONTAINER (dv->prev_history), children->data);
			if (comp == children->data) break;
			gtk_menu_shell_prepend (GTK_MENU_SHELL (dv->next_history), children->data);
		}
	}
	else { /* next history item clicked */
		comp = gtk_menu_get_active (GTK_MENU (dv->next_history));
		gtk_menu_shell_prepend (GTK_MENU_SHELL (dv->prev_history), dv->current);

		children = gtk_container_get_children (GTK_CONTAINER (dv->next_history));
		for (; children != NULL; children = children->next) {
			g_object_ref (children->data);
			gtk_container_remove (GTK_CONTAINER (dv->next_history), children->data);
			if (comp == children->data) break;
			gtk_menu_shell_prepend (GTK_MENU_SHELL (dv->prev_history), children->data);
		}
	}
	dv->current = comp;

	path = fb_path_menu_item_get_path (FB_PATH_MENU_ITEM (item));
	g_signal_emit (dv, fb_dirview_signals[HISTORY_MOVE], 0, path);
	fb_dirview_real_set_dir (dv, path, FALSE);
}

static void
fb_dirview_add_history (FbDirview *dv,
			const gchar *path)
{
	GList *children;
	GtkWidget *pitem;

	g_return_if_fail (path_is_exists (path));

	children = gtk_container_get_children (GTK_CONTAINER (dv->next_history));
	for (;children != NULL; children = children->next) {
		gtk_widget_destroy (GTK_WIDGET (children->data));
	}

	children = gtk_container_get_children (GTK_CONTAINER (dv->prev_history));
	if (g_list_length (children) > HISTORY_STOCK) {
		GList *last = g_list_last (children);
		gtk_widget_destroy (GTK_WIDGET (last->data));
	}				    

	pitem = fb_path_menu_item_new (path);
	g_signal_connect (G_OBJECT (pitem), "activate",
			  G_CALLBACK (cb_set_history), dv);
	gtk_widget_show (pitem);

	if (dv->current) {
		gtk_menu_shell_prepend (GTK_MENU_SHELL (dv->prev_history), dv->current);
	}
	dv->current = pitem;
}

void
fb_dirview_next_history (FbDirview *dv)
{
	const gchar *path;
	GList *children;

	g_return_if_fail (FB_IS_DIRVIEW (dv));

	children = gtk_container_get_children (GTK_CONTAINER (dv->next_history));
	if (! children) return;

	gtk_menu_shell_prepend (GTK_MENU_SHELL (dv->prev_history), dv->current);

	dv->current = g_list_nth_data (children, 0);
	g_object_ref (dv->current);
	gtk_container_remove (GTK_CONTAINER (dv->next_history), dv->current);

	path = fb_path_menu_item_get_path (FB_PATH_MENU_ITEM (dv->current));
	g_signal_emit (dv, fb_dirview_signals[HISTORY_MOVE], 0, path);
	fb_dirview_real_set_dir (dv, path, FALSE);
}

void
fb_dirview_prev_history (FbDirview *dv)
{
	const gchar *path;
	GList *children;

	g_return_if_fail (FB_IS_DIRVIEW (dv));

	children = gtk_container_get_children (GTK_CONTAINER (dv->prev_history));
	if (! children) return;

	gtk_menu_shell_prepend (GTK_MENU_SHELL (dv->next_history), dv->current);

	dv->current = g_list_nth_data (children, 0);
	g_object_ref (dv->current);
	gtk_container_remove (GTK_CONTAINER (dv->prev_history), dv->current);

	path = fb_path_menu_item_get_path (FB_PATH_MENU_ITEM (dv->current));
	g_signal_emit (dv, fb_dirview_signals[HISTORY_MOVE], 0, path);
	fb_dirview_real_set_dir (dv, path, FALSE);
}

void
fb_dirview_clear_history (FbDirview *dv)
{
	GList *children;

	g_return_if_fail (FB_IS_DIRVIEW (dv));

	children = gtk_container_get_children (GTK_CONTAINER (dv->prev_history));
	for (; children != NULL; children = children->next) {
		gtk_widget_destroy (GTK_WIDGET (children->data));
	}
	children = gtk_container_get_children (GTK_CONTAINER (dv->next_history));
	for (; children != NULL; children = children->next) {
		gtk_widget_destroy (GTK_WIDGET (children->data));
	}
       
}
