/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <glib-object.h>

#include "fb-conf.h"

GHashTable *fb_global_conf;

void
fb_conf_load (const gchar *path)
{
	FILE *fptr;
	gchar buf[1024];
	gchar **line;
	gchar *key, *value;

	fb_global_conf = g_hash_table_new_full ((GHashFunc) g_str_hash,
						(GEqualFunc) g_str_equal,
						g_free,
						g_free);

	if (! path || ! *path) return;

	fptr = fopen (path, "r");
	if (! fptr) return;

	while (fgets(buf, 1024, fptr)) {
		if (buf[0] == '#' || buf[0] == '\n') continue;

		line = g_strsplit (buf, ":", 2);
		key = g_strdup(*line);
		value = g_strndup(*(line + 1), strlen (*(line + 1)) - 1);
		g_strfreev (line);

		g_hash_table_insert (fb_global_conf, key, value);
	}
	fclose (fptr);
}

static void
conf_write (gpointer key,
	    gpointer value,
	    gpointer fptr)
{
	g_fprintf (fptr, "%s:%s\n", key, value);
}

void
fb_conf_save (const gchar *path)
{
	FILE *fptr;

	if (! path || ! *path) return;

	fptr = fopen (path, "w");
	g_fprintf (fptr, "# futaba config file\n");
	g_hash_table_foreach (fb_global_conf, (GHFunc) conf_write, fptr);
	fclose (fptr);
	
	g_hash_table_destroy (fb_global_conf);
}

void
fb_conf_get_value (const gchar *key,
		   gpointer value,
		   gint type,
		   const gchar *default_value)
{
	const gchar *value_str;

	g_return_if_fail (key || *key);

	value_str = g_hash_table_lookup (fb_global_conf, key);
	if (! value_str) {
		if (! default_value) {
			g_warning ("%s isn't setted default value\n", key);
			return;
		}
		value_str = default_value;
		fb_conf_set_value (key, value_str, FB_CONF_TYPE_STRING);
	}

	switch  (type) {
	case FB_CONF_TYPE_INT:
		*((gint *) value) = (gint) g_ascii_strtod (value_str, NULL);
		break;
	case FB_CONF_TYPE_FLOAT:
		*((gfloat *) value) = (gfloat) g_ascii_strtod (value_str, NULL);
		break;
	case FB_CONF_TYPE_BOOLEAN:
		if (! g_ascii_strcasecmp (value_str, "true"))
			*((gboolean *) value) = TRUE;
		else
			*((gboolean *) value) = FALSE;
		break;
	case FB_CONF_TYPE_STRING:
		*((gchar **)value) = g_strdup (value_str);
		break;
	default:
		break;
	}

}

void
fb_conf_set_value (const gchar *key,
		   gconstpointer value,
		   gint  type)
{
	gchar *value_str;

	g_return_if_fail (key || *key);

	switch (type) {
	case FB_CONF_TYPE_INT:
		value_str = g_strdup_printf ("%d", *((gint *) value));
		break;
	case FB_CONF_TYPE_FLOAT:
		value_str = g_strdup_printf ("%f", *((gfloat *) value));
		break;
	case FB_CONF_TYPE_BOOLEAN:
		if (*((gboolean *) value))
			value_str = g_strdup ("TRUE");
		else
			value_str = g_strdup ("FALSE");
		break;
	case FB_CONF_TYPE_STRING:
		value_str = g_strdup ((gchar *) value);
		break;
	default:
		break;
	}
	g_hash_table_replace (fb_global_conf,
			      g_strdup (key), value_str);
}
