/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>

#include "futaba.h"
#include "imageviewer.h"
#include "canvas.h"

#define ZOOMIN 0.77
#define ZOOMOUT 1.3

static void cb_quit(GtkAction *action, gpointer data);
static void cb_hide(GtkAction *action, gpointer data);
static void cb_first_page(GtkAction *action, gpointer data);
static void cb_next_page1(GtkAction *action, gpointer data);
static void cb_next_page2(GtkAction *action, gpointer data);
static void cb_next_page3(GtkAction *action, gpointer data);
static void cb_prev_page1(GtkAction *action, gpointer data);
static void cb_prev_page2(GtkAction *action, gpointer data);
static void cb_prev_page3(GtkAction *action, gpointer data);
static void cb_last_page(GtkAction *action, gpointer data);
static void cb_zoom_in(GtkAction *action, gpointer data);
static void cb_zoom_out(GtkAction *action, gpointer data);
static void cb_zoom_100(GtkAction *action, gpointer data);
static void cb_fit_image(GtkAction *action, gpointer data);
static void cb_right_rotation(GtkAction *action, gpointer data);
static void cb_left_rotation(GtkAction *action, gpointer data);
static void cb_mirror_rotation(GtkAction *action, gpointer data);
static void cb_r_zoom_in(GtkAction *action, gpointer data);
static void cb_r_zoom_out(GtkAction *action, gpointer data);
static void cb_r_zoom_100(GtkAction *action, gpointer data);
static void cb_r_fit_image(GtkAction *action, gpointer data);
static void cb_r_right_rotation(GtkAction *action, gpointer data);
static void cb_r_left_rotation(GtkAction *action, gpointer data);
static void cb_r_mirror_rotation(GtkAction *action, gpointer data);
static void cb_l_zoom_in(GtkAction *action, gpointer data);
static void cb_l_zoom_out(GtkAction *action, gpointer data);
static void cb_l_zoom_100(GtkAction *action, gpointer data);
static void cb_l_fit_image(GtkAction *action, gpointer data);
static void cb_l_right_rotation(GtkAction *action, gpointer data);
static void cb_l_left_rotation(GtkAction *action, gpointer data);
static void cb_l_mirror_rotation(GtkAction *action, gpointer data);

static GtkActionEntry actions[] = {
	{"Quit", "gtk-close", N_("viewer close"), "<release>q", NULL, G_CALLBACK(cb_quit)},
	{"Hide", NULL, "dummy", "<release>h", NULL, G_CALLBACK(cb_hide)},

	{"FirstPage", NULL, "dummy", "<release>Home", NULL, G_CALLBACK(cb_first_page)},
	{"NextPage1", NULL, "dummy", "<release>Right", NULL, G_CALLBACK(cb_next_page1)},
	{"NextPage2", NULL, "dummy", "<release>Down", NULL, G_CALLBACK(cb_next_page2)},
	{"NextPage3", NULL, "dummy", "<release>v", NULL, G_CALLBACK(cb_next_page3)},
	{"PrevPage1", NULL, "dummy", "<release>Left", NULL, G_CALLBACK(cb_prev_page1)},
	{"PrevPage2", NULL, "dummy", "<release>Up", NULL, G_CALLBACK(cb_prev_page2)},
	{"PrevPage3", NULL, "dummy", "<release>c", NULL, G_CALLBACK(cb_prev_page3)},
	{"LastPage", NULL, "dummy", "<release>e", NULL, G_CALLBACK(cb_last_page)},

	{"ZoomIn", "gtk-zoom-in", N_("zoom in"), "<release>J", NULL, G_CALLBACK(cb_zoom_in)},
	{"ZoomOut", "gtk-zoom-out", N_("zoom out"), "<release>L", NULL, G_CALLBACK(cb_zoom_out)},
	{"Zoom100", "gtk-zoom-100", N_("zoom 1:1"), "< release>O", NULL, G_CALLBACK(cb_zoom_100)},
	{"ZoomFit", "gtk-zoom-fit", N_("fit image to window"),"<release>K", NULL, G_CALLBACK(cb_fit_image)},
	{"RightRotation", NULL, N_("rigth rotation"), NULL, NULL, G_CALLBACK(cb_right_rotation)},
	{"LeftRotation", NULL, N_("left rotation"), NULL, NULL, G_CALLBACK(cb_left_rotation)},
	{"MirrorRotation", NULL, N_("mirror rotation"), NULL, NULL, G_CALLBACK(cb_mirror_rotation)},

	{"RightImage", NULL, N_("right image")},
	{"R_ZoomIn", "gtk-zoom-in", N_("zoom in"), "<alt>J", NULL, G_CALLBACK(cb_r_zoom_in)},
	{"R_ZoomOut", "gtk-zoom-out", N_("zoom out"), "<alt>L",  NULL, G_CALLBACK(cb_r_zoom_out)},
	{"R_Zoom100", "gtk-zoom-100", N_("zoom 1:1"), "<alt>O", NULL, G_CALLBACK(cb_r_zoom_100)},
	{"R_ZoomFit", "gtk-zoom-fit", N_("fit image to window"),"<alt>K", NULL, G_CALLBACK(cb_r_fit_image)},
	{"R_RightRotation", NULL, N_("rigth rotation"), NULL, NULL, G_CALLBACK(cb_r_right_rotation)},
	{"R_LeftRotation", NULL, N_("left rotation"), NULL, NULL, G_CALLBACK(cb_r_left_rotation)},
	{"R_MirrorRotation", NULL, N_("mirror rotation"), NULL, NULL, G_CALLBACK(cb_r_mirror_rotation)},

	{"LeftImage", NULL, N_("left image")},
	{"L_ZoomIn", "gtk-zoom-in", N_("zoom in"), "<control>J", NULL, G_CALLBACK(cb_l_zoom_in)},
	{"L_ZoomOut", "gtk-zoom-out", N_("zoom out"), "<control>L",  NULL, G_CALLBACK(cb_l_zoom_out)},
	{"L_Zoom100", "gtk-zoom-100", N_("zoom 1:1"), "<control>O", NULL, G_CALLBACK(cb_l_zoom_100)},
	{"L_ZoomFit", "gtk-zoom-fit", N_("fit image to window"),"<control>K", NULL, G_CALLBACK(cb_l_fit_image)},
	{"L_RightRotation", NULL, N_("rigth rotation"), NULL, NULL, G_CALLBACK(cb_l_right_rotation)},
	{"L_LeftRotation", NULL, N_("left rotation"), NULL, NULL, G_CALLBACK(cb_l_left_rotation)},
	{"L_MirrorRotation", NULL, N_("mirror rotation"), NULL, NULL, G_CALLBACK(cb_l_mirror_rotation)}
};
static guint actions_l = G_N_ELEMENTS(actions);

static void cb_two_pages(GtkAction *action, gpointer data);
static void cb_max_window(GtkAction *action, gpointer data);
static void cb_fullscreen(GtkAction *action, gpointer data);

static GtkToggleActionEntry toggle_actions[] = {
  {"TwoWindow", NULL, N_("two pages display"), "<release>space", NULL, G_CALLBACK(cb_two_pages), FALSE},
  {"MaxWindow", NULL, N_("Max Window"), "<release>M", NULL, G_CALLBACK(cb_max_window), FALSE},
  {"FullScreen", NULL, N_("full screen"), "<release>F", NULL, G_CALLBACK(cb_fullscreen), FALSE},
};
static guint toggle_actions_l = G_N_ELEMENTS(toggle_actions);

static void
cb_quit(GtkAction *action,
	gpointer data)
{
	FbImageViewer *viewer = data;

	g_signal_emit_by_name(G_OBJECT(viewer->top), "destroy", viewer);
}

static void
cb_hide(GtkAction *action,
	gpointer data)
{
	FbImageViewer *viewer = data;

	gtk_window_iconify(GTK_WINDOW(viewer->top));
}

static void
cb_first_page(GtkAction *action,
	      gpointer data)
{
	FbImageViewer *viewer = data;

	fb_imageviewer_first_page(viewer);
}

static void
cb_next_page1(GtkAction *action,
	      gpointer data)
{
	FbImageViewer *viewer = data;

	fb_imageviewer_next_pages(viewer, 1);
}

static void
cb_next_page2(GtkAction *action,
	      gpointer data)
{
	FbImageViewer *viewer = data;

	fb_imageviewer_next_pages(viewer, 10);
}

static void
cb_next_page3(GtkAction *action,
	      gpointer data)
{
	FbImageViewer *viewer = data;

	fb_imageviewer_next_pages(viewer, 50);
}

static void
cb_prev_page1(GtkAction *action,
	      gpointer data)
{
	FbImageViewer *viewer = data;

	fb_imageviewer_prev_pages(viewer, 1);
}

static void
cb_prev_page2(GtkAction *action,
	      gpointer data)
{
	FbImageViewer *viewer = data;

	fb_imageviewer_prev_pages(viewer, 10);
}

static void
cb_prev_page3(GtkAction *action,
	      gpointer data)
{
	FbImageViewer *viewer = data;

	fb_imageviewer_prev_pages(viewer, 50);
}

static void
cb_last_page(GtkAction *action,
	     gpointer data)
{
	FbImageViewer *viewer = data;

	fb_imageviewer_last_page(viewer);
}

static void
cb_zoom_in(GtkAction *action,
	   gpointer data)
{
        FbImageViewer *viewer = data;

	fb_imagecanvas_zooming(viewer->first, viewer->first->scale * ZOOMIN);

	if (viewer->two_pages)
		fb_imagecanvas_zooming(viewer->second, viewer->second->scale * ZOOMIN);
}

static void
cb_zoom_out(GtkAction *action,
	    gpointer data)
{
        FbImageViewer *viewer = data;

	fb_imagecanvas_zooming(viewer->first, viewer->first->scale * ZOOMOUT);

	if (viewer->two_pages)
		fb_imagecanvas_zooming(viewer->second, viewer->second->scale * ZOOMOUT);
}

static void
cb_zoom_100(GtkAction *action,
	    gpointer data)
{
        FbImageViewer *viewer = data;

	fb_imagecanvas_zooming(viewer->first, 1.0);

	if (viewer->two_pages)
		fb_imagecanvas_zooming(viewer->second, 1.0);
}

static void
cb_fit_image(GtkAction *action,
	     gpointer data)
{
        FbImageViewer *viewer = data;

	fb_imagecanvas_zoomfit(viewer->first);

	if (viewer->two_pages)
		fb_imagecanvas_zoomfit(viewer->second);
}

static void
cb_right_rotation(GtkAction *action,
		  gpointer data)
{
        FbImageViewer *viewer = data;

	fb_imagecanvas_rotate(viewer->first, FALSE);

	if (viewer->two_pages)
		fb_imagecanvas_rotate(viewer->second, FALSE);
}

static void
cb_left_rotation(GtkAction *action,
		 gpointer data)
{
        FbImageViewer *viewer = data;

	fb_imagecanvas_rotate(viewer->first, TRUE);

	if (viewer->two_pages)
		fb_imagecanvas_rotate(viewer->second, TRUE);
}

static void
cb_mirror_rotation(GtkAction *action,
		   gpointer data)
{
        FbImageViewer *viewer = data;

	fb_imagecanvas_mirror(viewer->first, TRUE, FALSE);

	if (viewer->two_pages)
		fb_imagecanvas_mirror(viewer->second, TRUE, FALSE);
}


static void
cb_r_zoom_in(GtkAction *action,
	     gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_zooming(viewer->first, viewer->first->scale * ZOOMIN);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_zooming(viewer->second,  viewer->second->scale * ZOOMIN);
}

static void
cb_r_zoom_out(GtkAction *action,
	      gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_zooming(viewer->first, viewer->first->scale * ZOOMOUT);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_zooming(viewer->second, viewer->second->scale * ZOOMOUT);
}

static void
cb_r_zoom_100(GtkAction *action,
	      gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_zooming(viewer->first, 1.0);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_zooming(viewer->second, 1.0);
}

static void
cb_r_fit_image(GtkAction *action,
	       gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_zoomfit(viewer->first);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_zoomfit(viewer->second);
}

static void
cb_r_right_rotation(GtkAction *action,
		    gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_rotate(viewer->first, FALSE);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_rotate(viewer->second, FALSE);
}

static void
cb_r_left_rotation(GtkAction *action,
		   gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_rotate(viewer->first, TRUE);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_rotate(viewer->second, TRUE);
}

static void
cb_r_mirror_rotation(GtkAction *action,
		     gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_mirror(viewer->first, TRUE, FALSE);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_mirror(viewer->second, TRUE, FALSE);
}


static void
cb_l_zoom_in(GtkAction *action,
	     gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_zooming(viewer->second, viewer->second->scale * ZOOMIN);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_zooming(viewer->first, viewer->first->scale * ZOOMIN);
}

static void
cb_l_zoom_out(GtkAction *action,
	      gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_zooming(viewer->second, viewer->second->scale * ZOOMOUT);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_zooming(viewer->first, viewer->first->scale * ZOOMOUT);
}

static void
cb_l_zoom_100(GtkAction *action,
	      gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_zooming(viewer->second, 1.0);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_zooming(viewer->first, 1.0);
}

static void
cb_l_fit_image(GtkAction *action,
	       gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_zoomfit(viewer->second);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_zoomfit(viewer->first);
}

static void
cb_l_right_rotation(GtkAction *action,
		    gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_rotate(viewer->second, FALSE);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_rotate(viewer->first, FALSE);
}

static void
cb_l_left_rotation(GtkAction *action,
		   gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_rotate(viewer->second, TRUE);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_rotate(viewer->first, TRUE);
}

static void
cb_l_mirror_rotation(GtkAction *action,
		     gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->open_way == RIGHT)
		fb_imagecanvas_mirror(viewer->second, TRUE, FALSE);
	else if (viewer->open_way == LEFT)
		fb_imagecanvas_mirror(viewer->first, TRUE, FALSE);
}

static void
cb_two_pages(GtkAction *action,
	     gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->two_pages)
		fb_imageviewer_fixup_window(viewer);
	else
		fb_imageviewer_split_window(viewer);
}

static void
cb_max_window(GtkAction *action,
	      gpointer data)
{
        FbImageViewer *viewer = data;

	if (viewer->size > WINDOW_FULL) return;

	if (viewer->size > WINDOW_MAX) {
		gtk_window_unmaximize(GTK_WINDOW(viewer->top));
		viewer->size ^= WINDOW_MAX;
	}
	else {
		gtk_window_maximize(GTK_WINDOW(viewer->top));
		gtk_window_move(GTK_WINDOW(viewer->top), 0, 0);
		viewer->size ^= WINDOW_MAX;
	}
}

static void
cb_fullscreen(GtkAction *action,
	      gpointer data)
{
	FbImageViewer *viewer = data;

	if (viewer->size > WINDOW_FULL) {
		gtk_window_unfullscreen(GTK_WINDOW(viewer->top));
		viewer->size ^= WINDOW_FULL;
	}
	else {
		gtk_window_fullscreen(GTK_WINDOW(viewer->top));
		viewer->size ^= WINDOW_FULL;
	}
}

GtkUIManager *
fb_viewer_popup_new(FbImageViewer *viewer)
{
	const gchar *viewer_ui = {
		"<ui>"
		"  <popup name=\"Popup1\">"
		"    <menuitem action=\"ZoomIn\"/>"
		"    <menuitem action=\"ZoomOut\"/>"
		"    <menuitem action=\"Zoom100\"/>"
		"    <menuitem action=\"ZoomFit\"/>"
		"    <separator/>"
		"    <menuitem action=\"RightRotation\"/>"
		"    <menuitem action=\"LeftRotation\"/>"
		"    <menuitem action=\"MirrorRotation\"/>"
		"    <separator/>"
		"    <menuitem action=\"TwoWindow\"/>"
		"    <menuitem action=\"FullScreen\"/>"
		"    <separator/>"
		"    <menuitem action=\"Quit\"/>"
		"  </popup>"
		"  <popup name=\"Popup2\">"
		"    <menu action=\"RightImage\">"
		"      <menuitem action=\"R_ZoomIn\"/>"
		"      <menuitem action=\"R_ZoomOut\"/>"
		"      <menuitem action=\"R_Zoom100\"/>"
		"      <menuitem action=\"R_ZoomFit\"/>"
		"      <separator/>"
		"      <menuitem action=\"R_RightRotation\"/>"
		"      <menuitem action=\"R_LeftRotation\"/>"
		"      <menuitem action=\"R_MirrorRotation\"/>"
		"    </menu>"
		"    <menu action=\"LeftImage\">"
		"      <menuitem action=\"L_ZoomIn\"/>"
		"      <menuitem action=\"L_ZoomOut\"/>"
		"      <menuitem action=\"L_Zoom100\"/>"
		"      <menuitem action=\"L_ZoomFit\"/>"
		"      <separator/>"
		"      <menuitem action=\"L_RightRotation\"/>"
		"      <menuitem action=\"L_LeftRotation\"/>"
		"      <menuitem action=\"L_MirrorRotation\"/>"
		"    </menu>"
		"    <separator/>"
		"    <menuitem action=\"ZoomIn\"/>"
		"    <menuitem action=\"ZoomOut\"/>"
		"    <menuitem action=\"Zoom100\"/>"
		"    <menuitem action=\"ZoomFit\"/>"
		"    <separator/>"
		"    <menuitem action=\"RightRotation\"/>"
		"    <menuitem action=\"LeftRotation\"/>"
		"    <menuitem action=\"MirrorRotation\"/>"
		"    <separator/>"
		"    <menuitem action=\"TwoWindow\"/>"
		"    <menuitem action=\"FullScreen\"/>"
		"    <separator/>"
		"    <menuitem action=\"Quit\"/>"
		"  </popup>"
		"  <popup name=\"Unvisible\">"
		"    <menuitem action=\"Hide\"/>"
		"    <menuitem action=\"MaxWindow\"/>"
		"    <menuitem action=\"FirstPage\"/>"
		"    <menuitem action=\"NextPage1\"/>"
		"    <menuitem action=\"NextPage2\"/>"
		"    <menuitem action=\"NextPage3\"/>"
		"    <menuitem action=\"PrevPage1\"/>"
		"    <menuitem action=\"PrevPage2\"/>"
		"    <menuitem action=\"PrevPage3\"/>"
		"    <menuitem action=\"LastPage\"/>"
      		"  </popup>"
		"</ui>"
	};
	GError *error = NULL;
	GtkActionGroup *action_group;
	GtkUIManager *ui_manager;

	action_group = gtk_action_group_new("Futaba Viewer");

	gtk_action_group_set_translation_domain(action_group, NULL);

	gtk_action_group_add_actions(action_group,
				     actions, actions_l,
				     viewer);

	gtk_action_group_add_toggle_actions(action_group,
					    toggle_actions, toggle_actions_l,
					    viewer);

	ui_manager = gtk_ui_manager_new();

	gtk_ui_manager_insert_action_group(ui_manager, action_group, 0);

	if (!gtk_ui_manager_add_ui_from_string(ui_manager, viewer_ui, -1, &error)) {
		g_print("Failed viewer popup init: %s\n", error->message);
		g_error_free(error);

		return NULL;
	}

	return ui_manager;
}
