/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>

#include "futaba.h"
#include "futaba-ui.h"
#include "action.h"
#include "dirview.h"

#define FB_ICON_SMALL "fb-small"
#define FB_ICON_SIZE_SMALL gtk_icon_size_from_name(FB_ICON_SMALL)

/*
 *------------------------------------------------------
 *          Utility
 *------------------------------------------------------
 */
static gboolean cb_show_comment(GtkWidget *widget, GdkEventCrossing *event, gpointer data);
static gboolean cb_remove_comment(GtkWidget *widget, GdkEventCrossing *event, gpointer data);

static gboolean
cb_show_comment(GtkWidget *widget,
		GdkEventCrossing *event,
		gpointer data)
{
	gchar *message = data;

	fb_statusbar_push(message);

	return FALSE;
}

static gboolean
cb_remove_comment(GtkWidget *widget,
		  GdkEventCrossing *evnet,
		  gpointer data)
{
	fb_statusbar_pop();

	return FALSE;
}

void
fb_widget_set_message(GtkWidget *widget,
		      gchar *message)
{
	if (!widget) return;

	g_signal_connect(G_OBJECT(widget), "enter-notify-event",
			 G_CALLBACK(cb_show_comment), message);
	g_signal_connect(G_OBJECT(widget), "leave-notify-event",
			 G_CALLBACK(cb_remove_comment), NULL);
}

void
fb_accelerator_lock(Futaba *futaba)
{
	GtkAccelGroup *accel;

	accel = gtk_ui_manager_get_accel_group(futaba->ui_manager);
	g_object_ref(accel);
	gtk_window_remove_accel_group(GTK_WINDOW(futaba->top), accel);
}

void
fb_accelerator_unlock(Futaba *futaba)
{
	GtkAccelGroup *accel;

	accel = gtk_ui_manager_get_accel_group(futaba->ui_manager);
	gtk_window_add_accel_group(GTK_WINDOW(futaba->top), accel);
}
/*
 *------------------------------------------------------
 *          Futaba Icon
 *------------------------------------------------------
 */
static GtkStockItem items[] = {
	{"fb-directory", N_("_Folder"), 0, 0, NULL},
	{"fb-archive", N_("_Archive"), 0, 0, NULL},
	{"fb-image", N_("_Image"), 0, 0, NULL},
	{"fb-noread-directory", N_("_Folder"), 0, 0, NULL},
	{"fb-noread-archive", N_("_Archive"), 0, 0, NULL},
	{"fb-noread-image", N_("_Image"), 0, 0, NULL},
	{"fb-preview", N_("_Preview"), 0, 0, NULL},
	{"fb-list", NULL, 0, 0, NULL},
	{"fb-dot", NULL, 0, 0, NULL},
	{"fb-another-file", N_("_Another"), 0, 0, NULL}
};
static guint items_len = G_N_ELEMENTS(items);

static void item_bind_icon(GtkIconFactory *factory, const gchar *stock_id, const gchar *name);

static void
item_bind_icon(GtkIconFactory *factory,
	       const gchar *stock_id,
	       const gchar *name)
{
	gchar *path;
	GdkPixbuf *icon;
	GtkIconSet *icon_set;

	path = g_strconcat(ICONDIR, "/", name, NULL);

	g_return_if_fail(g_file_test(path, G_FILE_TEST_EXISTS) == TRUE);

	icon = gdk_pixbuf_new_from_file(path, NULL);
	g_free(path);

	if (icon) {
		icon_set = gtk_icon_set_new_from_pixbuf(icon);
		gtk_icon_factory_add(factory, stock_id, icon_set);
		gtk_icon_set_unref(icon_set);
		gdk_pixbuf_unref(icon);
	}
}

void
fb_icon_add_stock(void)
{
	gchar *path;
	GtkIconFactory *factory;

	/* add new icon size */
	gtk_icon_size_register(FB_ICON_SMALL, 10, 10);

	/* add new icon */
	gtk_stock_add(items, items_len);

	factory = gtk_icon_factory_new();
	gtk_icon_factory_add_default(factory);

	item_bind_icon(factory, "fb-directory", "directory.png");
	item_bind_icon(factory, "fb-archive", "archive.png");
	item_bind_icon(factory, "fb-image", "image.png");
	item_bind_icon(factory, "fb-noread-directory", "noread-directory.png");
	item_bind_icon(factory, "fb-noread-archive", "noread-archive.png");
	item_bind_icon(factory, "fb-noread-image", "noread-image.png");
	item_bind_icon(factory, "fb-preview", "preview.png");
	item_bind_icon(factory, "fb-list", "list.png");
	item_bind_icon(factory, "fb-dot", "dot.png");
	item_bind_icon(factory, "fb-another-file", "another-file.png");

	g_object_unref(factory);
}
/*
 *--------------------------------------------------
 *          UI Manager
 *--------------------------------------------------
 */
void
fb_ui_set_message(GtkUIManager *ui_manager,
		  gchar *ui_path,
		  gchar *message)
{
	GtkWidget *item;

	g_return_if_fail(ui_path != NULL);

	item = gtk_ui_manager_get_widget(ui_manager, ui_path);
	g_return_if_fail(item != NULL);

	g_signal_connect(G_OBJECT(item), "enter-notify-event",
			 G_CALLBACK(cb_show_comment), message);
	g_signal_connect(G_OBJECT(item), "leave-notify-event",
			 G_CALLBACK(cb_remove_comment), NULL);
}

void
fb_ui_set_sensitive(GtkUIManager *ui_manager,
		    gchar *path,
		    gboolean sensitive)
{
	GtkWidget *widget;

	widget = gtk_ui_manager_get_widget(ui_manager, path);
	g_return_if_fail(widget != NULL);

	gtk_widget_set_sensitive(widget, sensitive);
}

GtkUIManager *
fb_ui_new(Futaba *futaba)
{
	const gchar *futaba_ui = {
		"<ui>"
		"  <menubar name=\"MenuBar\">"
		"    <menu action=\"FileMenu\">"
		"      <menuitem action=\"OpenViewer\"/>"
		"      <menuitem action=\"OpenThumb\"/>"
		"      <separator/>"
		"      <menuitem action=\"CloseViewer\"/>"
		"      <menuitem action=\"CloseThumb\"/>"
		"      <menuitem action=\"Quit\"/>"
		"    </menu>"
		"    <menu action=\"EditMenu\">"
		"      <menuitem action=\"Copy\"/>"
		"      <menuitem action=\"Cut\"/>"
		"      <menuitem action=\"Paste\"/>"
		"      <menuitem action=\"Delete\"/>"
		"      <separator/>"
		"      <menuitem action=\"Preferences\"/>"
		"    </menu>"
		"    <menu action=\"ViewMenu\">"
		"      <menuitem action=\"RefreshDir\"/>"
		"      <separator/>"
		"      <menuitem action=\"TBarShow\"/>"
		"      <menuitem action=\"LBarShow\"/>"
		"      <menuitem action=\"SBarShow\"/>"
		"      <separator/>"
		"      <menu action=\"Sort\">"
		"        <menuitem action=\"SortName\"/>"
		"        <menuitem action=\"SortSize\"/>"
		"        <menuitem action=\"SortMtime\"/>"
		"        <menuitem action=\"SortType\"/>"
		"        <separator/>"
		"        <menuitem action=\"Reverse\"/>"
		"      </menu>"
		"      <menu action=\"Column\">"
		"        <menuitem action=\"Col1\"/>"
		"        <menuitem action=\"Col2\"/>"
		"        <menuitem action=\"Col3\"/>"
		"        <menuitem action=\"Col4\"/>"
		"        <menuitem action=\"Col5\"/>"
		"        <separator/>"
		"        <menuitem action=\"ColTitle\"/>"
		"      </menu>"
		"      <menu action=\"Filter\">"
		"        <menuitem action=\"FilterDot\"/>"
		"        <menuitem action=\"FilterOther\"/>"
		"      </menu>"
		"    </menu>"
		"    <menu action=\"MoveMenu\">"
		"      <menuitem action=\"UpDir\"/>"
		"      <menuitem action=\"BackHistory\"/>"
		"      <menuitem action=\"NextHistory\"/>"
		"      <separator/>"
		"      <menuitem action=\"GoToHome\"/>"
		"      <separator/>"
		"      <menuitem action=\"ClearHistory\"/>"
		"    </menu>"
		"    <menu action=\"BookmarkMenu\">"
		"      <menuitem action=\"AddBookmark\"/>"
		"      <menuitem action=\"EditBookmark\"/>"
		"    </menu>"
		"    <menu action=\"HelpMenu\">"
		"      <menuitem action=\"About\"/>"
		"    </menu>"
		"  </menubar>"
		"  <toolbar name=\"ToolBar\">"
		"    <toolitem action=\"BackHistory\"/>"
		"    <toolitem action=\"ShowBackHistory\"/>"
		"    <toolitem action=\"NextHistory\"/>"
		"    <toolitem action=\"ShowNextHistory\"/>"
		"    <toolitem action=\"UpDir\"/>"
		"    <toolitem action=\"GoToHome\"/>"
		"    <toolitem action=\"RefreshDir\"/>"
		"    <separator/>"
		"    <toolitem action=\"SwitchDotFile\"/>"
		"    <toolitem action=\"SwitchFileDisplay\"/>"
		/*     "    <toolitem action=\"ChangeColumnType\"/>" */
		"    <separator/>"
		"    <toolitem action=\"SwitchPreview\"/>"
		"  </toolbar>"
		"  <popup name=\"DirViewPopup\">"
		"    <menuitem action=\"Open\"/>"
		"    <menuitem action=\"OpenViewer\"/>"
		"    <menuitem action=\"OpenThumb\"/>"
		"    <separator/>"
		"    <menuitem action=\"Rename\"/>"
		"    <menuitem action=\"Copy\"/>"
		"    <menuitem action=\"Cut\"/>"
		"    <menuitem action=\"Paste\"/>"
		"    <menuitem action=\"Delete\"/>"
		"    <separator/>"
		"    <menuitem action=\"MakeDir\"/>"
		/*   "    <menuitem action=\"ChangeBulk\"/>" */
		"    <menuitem action=\"Property\"/>"
		"  </popup>"
		"</ui>"
	};
	GError *error = NULL;
	GtkActionGroup *action_group;
	GtkUIManager *ui_manager;
	GtkWidget *item, *arrow;


	action_group = fb_actions_new(futaba);

	ui_manager = gtk_ui_manager_new();
	gtk_ui_manager_insert_action_group(ui_manager, action_group, 0);

	if (!gtk_ui_manager_add_ui_from_string(ui_manager, futaba_ui, -1, &error)) {
		g_print("Failed ui manager init: %s\n", error->message);
		g_error_free(error);

		return NULL;
	}
	/* File Menu */
	fb_ui_set_message(ui_manager, "/MenuBar/FileMenu/OpenViewer",
			  _("Open viewer"));
	fb_ui_set_message(ui_manager, "/MenuBar/FileMenu/OpenThumb",
			  _("Open thumbnail window"));
	fb_ui_set_message(ui_manager, "/MenuBar/FileMenu/CloseViewer",
			  _("Close viewer"));
	fb_ui_set_message(ui_manager, "/MenuBar/FileMenu/CloseThumb",
			  _("Close Thumbnail Window"));
	fb_ui_set_message(ui_manager, "/MenuBar/FileMenu/Quit", 
			  _("Quit futaba"));
	/* Edit Menu */
	fb_ui_set_message(ui_manager, "/MenuBar/EditMenu/Copy",
			  _("Copy the selected item"));
	fb_ui_set_message(ui_manager, "/MenuBar/EditMenu/Cut",
			  _("Cut the selected item"));
	fb_ui_set_message(ui_manager, "/MenuBar/EditMenu/Paste",
			  _("Paste the stocked item"));
	fb_ui_set_message(ui_manager, "/MenuBar/EditMenu/Delete",
			  _("Delete  the selected item"));
	fb_ui_set_message(ui_manager, "/MenuBar/EditMenu/Preferences", 
			  _("Edit preferences"));
	/* View Menu */
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/RefreshDir",
			  _("Refresh Directory"));

	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/TBarShow",
			  _("Change the visibility of toolbar"));
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/LBarShow",
			  _("Change the visibility of locationbar"));
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/SBarShow",
			  _("Change the visibility of statusbar"));

	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Sort/SortName",
			  _("Keep items sorted by name"));
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Sort/SortSize",
			  _("Keep items sorted by size"));
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Sort/SortMtime",
			  _("Keep items sorted by modification data"));
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Sort/SortType",
			  _("Keep items sorted by type"));
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Sort/Reverse",
			  _("Display items in the opposite order"));

	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Column/Col1",
			  _("Change the visibility of Permission column"));
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Column/Col2",
			  _("Change the visibility of Owner column"));
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Column/Col3",
			  _("Change the visibility of Group column"));
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Column/Col4",
			  _("Change the visibility of Size column"));
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Column/Col5",
			  _("Change the visibility of Mtime column"));
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Column/ColTitle",
			  _("Change the visibility of Column title"));

	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Filter/FilterDot",
			  _("Change the visibility of Dot file"));
	fb_ui_set_message(ui_manager, "/MenuBar/ViewMenu/Filter/FilterOther",
			  _("Change the visibility of Other file"));

	/* Move Menu */
	fb_ui_set_message(ui_manager, "/MenuBar/MoveMenu/UpDir",
			  _("Go up one level"));
	fb_ui_set_message(ui_manager, "/MenuBar/MoveMenu/BackHistory",
			  _("Go to the previous visited directory"));
	fb_ui_set_message(ui_manager, "/MenuBar/MoveMenu/NextHistory",
			  _("Go to the next visited directory"));
	fb_ui_set_message(ui_manager, "/MenuBar/MoveMenu/GoToHome",
			  _("Go to the home directory"));
	fb_ui_set_message(ui_manager, "/MenuBar/MoveMenu/ClearHistory",
			  _("Clear Back/Forward lists"));
	/* Bookmark Menu */
	fb_ui_set_message(ui_manager, "/MenuBar/BookmarkMenu/AddBookmark",
			  _("Add a bookmark for the current directory to this menu"));
	fb_ui_set_message(ui_manager, "/MenuBar/BookmarkMenu/EditBookmark",
			  _("Display a window editing the bookmarks"));
	/* Help Menu */
	fb_ui_set_message(ui_manager, "/MenuBar/HelpMenu/About",
			  _("Display credits"));

	/* DirView Popup */
	fb_ui_set_message(ui_manager, "/DirViewPopup/Open",
			  _("Open the selected item"));
	fb_ui_set_message(ui_manager, "/DirViewPopup/OpenViewer",
			  _("Open the selected item with viewer"));
	fb_ui_set_message(ui_manager, "/DirViewPopup/OpenThumb",
			  _("Open the current directory from thumbnail window"));
	fb_ui_set_message(ui_manager, "/DirViewPopup/Rename",
			  _("Rename the selected item"));
	fb_ui_set_message(ui_manager, "/DirViewPopup/Copy",
			  _("Copy the selected item"));
	fb_ui_set_message(ui_manager, "/DirViewPopup/Cut",
			  _("Cut the selected item"));
	fb_ui_set_message(ui_manager, "/DirViewPopup/Paste",
			  _("Paste the stocked item"));
	fb_ui_set_message(ui_manager, "/DirViewPopup/Delete",
			  _("Delete the selected item"));
	fb_ui_set_message(ui_manager, "/DirViewPopup/MakeDir",
			  _("Create a new directory"));
	/*   fb_ui_set_message(ui_manager, "/DirViewPopup/ChangeBulk", */
	/* 		    _("Change a lot of file name")); */
	fb_ui_set_message(ui_manager, "/DirViewPopup/Property",
			  _("Display the properties of selected item"));

	/* menu bar */
	item = gtk_ui_manager_get_widget(ui_manager, "/MenuBar/HelpMenu");
	gtk_menu_item_set_right_justified(GTK_MENU_ITEM(item), TRUE);

	item = gtk_ui_manager_get_widget(ui_manager, 
					 "/MenuBar/ViewMenu/Sort/Reverse");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), (gboolean) sort_direction);

	item = gtk_ui_manager_get_widget(ui_manager, 
					 "/MenuBar/ViewMenu/Column/ColTitle");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), show_coltitle);

	item = gtk_ui_manager_get_widget(ui_manager, 
					 "/MenuBar/ViewMenu/Column/Col1");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), show_col1);

	item = gtk_ui_manager_get_widget(ui_manager, 
					 "/MenuBar/ViewMenu/Column/Col2");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), show_col2);

	item = gtk_ui_manager_get_widget(ui_manager, 
					 "/MenuBar/ViewMenu/Column/Col3");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), show_col3);

	item = gtk_ui_manager_get_widget(ui_manager, 
					 "/MenuBar/ViewMenu/Column/Col4");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), show_col4);

	item = gtk_ui_manager_get_widget(ui_manager, 
					 "/MenuBar/ViewMenu/Column/Col5");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), show_col5);

	/* tool bar */
	arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_NONE);
	gtk_widget_show(arrow);

	item = gtk_ui_manager_get_widget(ui_manager, "/ToolBar/ShowBackHistory");
	gtk_tool_item_set_homogeneous(GTK_TOOL_ITEM(item), FALSE);
	gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(item), arrow);

	arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_NONE);
	gtk_widget_show(arrow);

	item = gtk_ui_manager_get_widget(ui_manager, "/ToolBar/ShowNextHistory");
	gtk_tool_item_set_homogeneous(GTK_TOOL_ITEM(item), FALSE);
	gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(item), arrow);

	/* add accelerator */
	action_connect_accelerator(action_group, "SwitchPreview");

	return ui_manager;
}
/*
 *--------------------------------------------------
 *          Tool Bar
 *--------------------------------------------------
 */
void
fb_toolbar_set_sensitive(Futaba *futaba)
{
	gint length;
	gboolean flag;

	length = MAX(0, g_list_length(futaba->history->record) - 1);

	flag = (futaba->history->position != length);
	fb_ui_set_sensitive(futaba->ui_manager, "/ToolBar/BackHistory", flag);
	fb_ui_set_sensitive(futaba->ui_manager, "/ToolBar/ShowBackHistory", flag);

	flag = (futaba->history->position != 0);
	fb_ui_set_sensitive(futaba->ui_manager, "/ToolBar/NextHistory", flag);
	fb_ui_set_sensitive(futaba->ui_manager, "/ToolBar/ShowNextHistory", flag);

	flag = g_ascii_strcasecmp(futaba->workdir, "/");
	fb_ui_set_sensitive(futaba->ui_manager, "/ToolBar/UpDir", flag);
}

/*
 *-----------------------------------------------
 *          Location Bar
 *-----------------------------------------------
 */
static void cb_clicked(GtkToolButton *button, gpointer data);
static void cb_activate(GtkEntry *entry, gpointer data);
static gboolean cb_lock_accel(GtkWidget *entry, GdkEventFocus *event, gpointer data);
static gboolean cb_unlock_accel(GtkWidget *entry, GdkEventFocus *event, gpointer data);

static void
cb_clicked(GtkToolButton *button,
	   gpointer data)
{
	Futaba *futaba = data;

	g_return_if_fail(GTK_IS_WIDGET(futaba->entry));

	g_signal_emit_by_name(G_OBJECT(futaba->entry), "activate", futaba);
}

static void
cb_activate(GtkEntry *entry,
	    gpointer data)
{
	gchar *path_utf8, *path;
	Futaba *futaba = data;

	path_utf8 = gtk_editable_get_chars((GtkEditable *)entry, 0, -1);
	path = str_utf8_to_local(path_utf8);

	if (!g_file_test(path, G_FILE_TEST_EXISTS)) {
		g_free(path);
		if (g_file_test(path_utf8, G_FILE_TEST_EXISTS))	 
			path = g_strdup(path_utf8);
		else
			return;
	}

	if (!g_ascii_strcasecmp(path, futaba->workdir)) {
		g_free(path);
		return;
	}

	fb_dirview_set_dir(futaba, path);

	g_free(path);
}

static gboolean
cb_lock_accel(GtkWidget *entry,
	      GdkEventFocus *event,
	      gpointer data)
{
	Futaba *futaba = data;

	fb_accelerator_lock(futaba);

	return TRUE;
}

static gboolean
cb_unlock_accel(GtkWidget *entry,
		GdkEventFocus *event,
		gpointer data)
{
	Futaba *futaba = data;

	fb_accelerator_unlock(futaba);

	return FALSE;
}

GtkWidget *
fb_locationbar_new(Futaba *futaba)
{
	GtkTooltips *tips;
	GtkToolItem *button;
	GtkWidget *hbox, *entry;

	hbox = gtk_hbox_new(FALSE, 0);

	button = gtk_tool_button_new_from_stock("gtk-jump-to");
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(cb_clicked), futaba);
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(button), FALSE, FALSE, 0);
	gtk_widget_show(GTK_WIDGET(button));

	tips = gtk_tooltips_new();
	gtk_tool_item_set_tooltip(button, tips, _("go to location"), "go to location");

	entry = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(entry), 128);
	g_signal_connect(G_OBJECT(entry), "activate",
			 G_CALLBACK(cb_activate), futaba);
	g_signal_connect(G_OBJECT(entry), "focus-in-event",
			 G_CALLBACK(cb_lock_accel), futaba);
	g_signal_connect(G_OBJECT(entry), "focus-out-event",
			 G_CALLBACK(cb_unlock_accel), futaba);
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
	gtk_widget_show(entry);

	futaba->entry = entry;

	return hbox;
}

void
fb_locationbar_set_text(Futaba *futaba,
			gchar *text)
{
	gchar *utf8;

	if (!text) return;

	utf8 = str_local_to_utf8(text);

	gtk_entry_set_text(GTK_ENTRY(futaba->entry), utf8);

	g_free(utf8);
}
/*
 *------------------------------------------------------------
 *            Status Bar
 *-------------------------------------------------------------
 */
#define CONTEXT "Futaba Status"

static guint statusbar_id;
static GtkWidget *statusbar = NULL;

GtkWidget *
fb_statusbar_new(void)
{
	statusbar = gtk_statusbar_new();
	statusbar_id = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar), "first statusbar");

	return statusbar;
}

void
fb_statusbar_push(gchar *string)
{
	gchar *message;

	if (!string) return;
	if (!statusbar) return;

	message = str_local_to_utf8(string);

	gtk_statusbar_push(GTK_STATUSBAR(statusbar), statusbar_id, message);
	g_free(message);
}

void
fb_statusbar_swap(gchar *string)
{
	gchar *message;

	if (!string) return;
	if (!statusbar) return;

	message = str_local_to_utf8(string);

	gtk_statusbar_pop(GTK_STATUSBAR(statusbar), statusbar_id);
	gtk_statusbar_push(GTK_STATUSBAR(statusbar), statusbar_id, message);

	g_free(message);
}

void
fb_statusbar_pop(void)
{
	if (!statusbar) return;

	gtk_statusbar_pop(GTK_STATUSBAR(statusbar), statusbar_id);
}
