# Copyright (C) 2006 by Aiwota Programmer
# aiwotaprog@tetteke.tk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import pygtk
pygtk.require('2.0')
import gtk
import itertools


class ThreadListModel(gtk.GenericTreeModel):
    column_types = (str, int, str, int, int, int, float, int)
    column_names = ["id", "num", "title", "res", "lineCount",
                    "lastModified", "average", "oldRes"]

    def __init__(self):
        gtk.GenericTreeModel.__init__(self)
        self.original_list = []
        self.list = []
        self.order_dict = {}
        self.sort_column_name = "num"
        self.sort_reverse = False
        self.filter_func = None
        self.compare_func = {}

        self.set_list([])

    def set_compare_func(self, name, func, key=None):
        self.compare_func[name] = (func, key)

    def del_compare_func(self, name):
        try:
            del self.filter_func[name]
        except:
            pass

    def get_dict(self, iter):
        path = self.get_path(iter)
        return self.list[path[0]]

    def modify_row(self, dict):
        id = dict["id"]
        if id in self.order_dict:
            index = self.order_dict[id]
            target_dict = self.list[index]
            for name in self.column_names:
                if name in dict:
                    target_dict[name] = dict[name]
            self.row_changed(index, self.get_iter(index))

    def refilter(self, filter_func):
        before_size = len(self.list)
        self.do_filter(filter_func)
        after_size = len(self.list)
        self.do_sort(self.sort_column_name, self.sort_reverse)
        self.list_modified(before_size, after_size)
        
    def set_list(self, newlist):
        self.original_list = newlist[:]
        self.refilter(self.filter_func)

    def build_order_dict(self):
        # key: thread id, value: index in self.list
        self.order_dict = dict(
            [(item["id"], index) for index, item in enumerate(self.list)])

    def list_modified(self, before_size, after_size):
        if before_size > after_size:
            print "remove", before_size-after_size, "threads"
            for i in range(before_size - after_size):
                self.row_deleted(after_size)
        elif after_size > before_size:
            print "insert", after_size-before_size, "threads"
            for i in range(before_size, after_size):
                self.row_inserted(i, self.get_iter(i))

    def do_filter(self, filter_func):
        # Set filter_func to self.filter_func after successful filtering.
        # successful means no exception.

        if not filter_func:
            self.list = self.original_list[:]
            self.filter_func = filter_func
            return

        predicate = lambda item: filter_func(self, item)
        iterable = itertools.ifilter(predicate, self.original_list)
        self.list = [item for item in iterable]
        self.filter_func = filter_func

    def compare(self, v1, v2, reverse):
        if not reverse:
            if v1 == 0 and v2 == 0:
                return 0
            elif v1 == 0:
                return 1
            elif v2 == 0:
                return -1
        if v1 > v2:
            return 1
        elif v1 < v2:
            return -1
        else:
            return 0

    def get_sort(self):
        return self.sort_column_name, self.sort_reverse

    def do_sort(self, column_name, reverse):
        sort_list = self.list[:]
        if column_name in self.compare_func:
            h = self.compare_func[column_name][0]
            key = self.compare_func[column_name][1]
            if not h:
                h = self.compare
            sort_list.sort(lambda x, y: h(x, y, reverse), key, reverse)
        else:
            if self.column_types[self.column_names.index(column_name)] == str:
                sort_list.sort(None, lambda dic: dic[column_name], reverse)
            else:
                h = lambda x, y: self.compare(x, y, reverse)
                sort_list.sort(h, lambda dic: dic[column_name], reverse)
        self.list = sort_list
        self.build_order_dict()

    def sort(self, column_name, order_specified=False, reverse=False):
        old_order_dict = self.order_dict

        if order_specified:
            sort_reverse = reverse
        else:
            if column_name == self.sort_column_name:
                # sort reverse
                sort_reverse = not self.sort_reverse
            else:
                sort_reverse = False

        self.do_sort(column_name, sort_reverse)
        self.sort_column_name = column_name
        self.sort_reverse = sort_reverse

        neworder = [old_order_dict[item["id"]] for item in self.list]
        self.rows_reordered(None, None, neworder)

    def on_get_flags(self):
        return gtk.TREE_MODEL_LIST_ONLY

    def on_get_n_columns(self):
        return len(self.column_types)

    def on_get_column_type(self, n):
        return self.column_types[n]

    def on_get_iter(self, path):
        try:
            return self.list[path[0]]
        except IndexError:
            return None

    def on_get_path(self, rowref):
        return self.order_dict[rowref["id"]]

    def on_get_value(self, rowref, column):
        return rowref[self.column_names[column]]

    def on_iter_next(self, rowref):
        try:
            i = self.order_dict[rowref["id"]] + 1
            return self.list[i]
        except IndexError:
            return None

    def on_iter_children(self, rowref):
        if rowref:
            return None
        return self.list[0]

    def on_iter_has_child(self, rowref):
        return False

    def on_iter_n_children(self, rowref):
        if rowref:
            return 0
        return len(self.list)

    def on_iter_nth_child(self, rowref, n):
        if rowref:
            return None
        try:
            return self.list[n]
        except IndexError:
            return None

    def on_iter_parent(self, child):
        return None
