# Copyright (C) 2006 by Aiwota Programmer
# aiwotaprog@tetteke.tk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import re
import urllib2
import codecs
import traceback
import os
import os.path

import config
from http_sub import HTTPRedirectHandler302, HTTPDebugHandler
from bookmark_core import BookmarkFormatError, Bookmark
from bookmark_editwindow import BookmarkEditWindow

bookmark_list = None

def init(bookmarks_path, bbsmenu_uri):
    global bookmark_list

    bookmark_list = BookmarkList(bookmarks_path, bbsmenu_uri)
    bookmark_list.load()


class BookmarkList(object):

    def __init__(self, bookmarks_path, bbsmenu_uri):
        self.__bookmarks_path = bookmarks_path
        self.__bbsmenu_uri = bbsmenu_uri
        self.__bookmark_list = []
        self.__view = None

    def set_view(self, view):
        self.__view = view

    def load(self):
        bookmark_list = []
        try:
            for line in file(self.__bookmarks_path):
                line = line.rstrip("\n")
                try:
                    bookmark = Bookmark(formatted=line)
                except BookmarkFormatError:
                    traceback.print_exc()
                else:
                    bookmark_list.append(bookmark)
        except IOError:
            traceback.print_exc()
        else:
            self.__bookmark_list = bookmark_list

    def save(self):
        try:
            basedir = os.path.dirname(self.__bookmarks_path)
            if not os.path.isdir(basedir):
                os.makedirs(basedir)
            f = file(self.__bookmarks_path, "w")
            for bookmark in self.__bookmark_list:
                line = bookmark.tabbed_text() + "\n"
                f.write(line)
        except:
            traceback.print_exc()
        else:
            f.close()

    def update_bbsmenu(self):
        reg = re.compile(
            "<A HREF=(?P<uri>http://[^\s]+).*>(?P<name>[^<]+)</A>")
        category_reg = re.compile(
            "(?:<BR>){1,2}<B>(?P<category>[^<]+)</B><BR>")

        new_bookmark_list = []
        bookmark_uri_dict = {}
        encoding = "cp932"

        current_category = ""
        opener = urllib2.build_opener(HTTPRedirectHandler302, HTTPDebugHandler)
        req = urllib2.Request(self.__bbsmenu_uri)
        req.add_header("User-agent", config.User_Agent)
        try:
            for line in opener.open(req):
                line = line.decode(encoding, "replace").strip()
                if line:
                    if not current_category:
                        m = category_reg.match(line)
                        if m:
                            current_category = m.group("category")
                            continue
                    m = reg.match(line)
                    if m:
                        name = m.group("name")
                        uri = m.group("uri")
                        category = current_category
                        if uri and uri in bookmark_uri_dict:
                            bookmark = bookmark_uri_dict[uri]
                            if category and \
                               category not in bookmark.categories:
                                bookmark.categories.append(category)
                        elif name and uri:
                            categories = ["bbsmenu"]
                            if category:
                                categories.append(category)
                            bookmark = Bookmark(uri=uri, name=name,
                                                categories=categories)
                            new_bookmark_list.append(bookmark)
                            bookmark_uri_dict[uri] = bookmark
                else:
                    current_category = ""
        except:
            traceback.print_exc()
        else:
            self._merge_bbsmenu(new_bookmark_list)

    def _merge_bbsmenu(self, new_bbsmenu_list):
        old_list = []
        for bookmark in self.__bookmark_list:
            if "bbsmenu" not in bookmark.categories:
                old_list.append(bookmark)
        new_bbsmenu_list += old_list
        self.__bookmark_list = new_bbsmenu_list
        self.save()
        category_list = []
        for bookmark in self.__bookmark_list:
            for category in bookmark.categories:
                if category and category not in category_list:
                    category_list.append(category)

    def get_category_list(self):
        categories = []
        for bookmark in self.__bookmark_list:
            for category in bookmark.categories:
                if category and category not in categories:
                    # avoid empty and duplicate
                    categories.append(category)
        return categories

    def get_bookmark_list_in_category(self, category):
        bookmarks = []
        for bookmark in self.__bookmark_list:
            if category in bookmark.categories:
                bookmarks.append(bookmark)
        return bookmarks

    def get_non_category_bookmark(self):
        bookmarks = []
        for bookmark in self.__bookmark_list:
            have_category = False
            for category in bookmark.categories:
                if category:
                    have_category = True
                    break
            if not have_category:
                bookmarks.append(bookmark)
        return bookmarks

    def add_new_bookmark(self, bookmark):
        self.__bookmark_list.append(bookmark)
        self.save()

    def delete_bookmark(self, bookmark):
        try:
            index = self.__bookmark_list.index(bookmark)
        except ValueError:
            pass
        else:
            del self.__bookmark_list[index]
            self.save()

    def add_bookmark_with_edit(self, name="New Bookmark", uri=None,
                               categories=None):

        def on_edit_complete(new_bookmark):
            self.add_new_bookmark(new_bookmark)
            self.new_bookmark_added()

        if not categories:
            categories = []
        bookmark = Bookmark(name=name, uri=uri, categories=categories)
        BookmarkEditWindow(on_edit_complete, bookmark)

    def new_bookmark_added(self):
        if self.__view:
            self.__view.update_categories()
