# Copyright (C) 2006 by Aiwota Programmer
# aiwotaprog@tetteke.tk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import os.path
import re
import time
import threading

import config
from BbsType import bbs_type_exception

REG_EXPR_HTTPDATE = re.compile("((?:Mon)|(?:Tue)|(?:Wed)|(?:Thu)|(?:Fri)|(?:Sat)|(?:Sun)), (\d{2}) ((?:Jan)|(?:Feb)|(?:Mar)|(?:Apr)|(?:May)|(?:Jun)|(?:Jul)|(?:Aug)|(?:Sep)|(?:Oct)|(?:Nov)|(?:Dec)) (\d{4}) (\d{2}):(\d{2}):(\d{2}) GMT")
WDAY_DICT = {"Mon":0, "Tue":1, "Wed":2, "Thu":3, "Fri":4, "Sat":5, "Sun":6}
MON_DICT = {"Jan":1, "Feb":2, "Mar":3, "Apr":4, "May":5, "Jun":6, "Jul":7,
            "Aug":8, "Sep":9, "Oct":10, "Nov":11, "Dec":12}

def _check_thread(bbs_type):
    """if bbs_type is not thread, raise BbsTypeError"""

    if not bbs_type.is_thread():
        raise bbs_type_exception.BbsTypeError, \
              "the bbs_type does not represent thread: " + bbs_type.uri

def get_logs_dir_path():
    return os.path.join(config.get_config_dir_path(), "logs")

def get_thread_dat_dir_path(bbs_type):
    """Returns dir path for saving thread dat file"""

    return os.path.join(get_board_dir_path(bbs_type), "dat")

def get_thread_idx_dir_path(bbs_type):
    """Returns dir path for saving thread index file"""

    return os.path.join(get_board_dir_path(bbs_type), "idx")

def get_thread_states_dir_path(bbs_type):
    """Returns dir path for saving thread states file"""

    return os.path.join(get_board_dir_path(bbs_type), "states")

def get_thread_dat_path(bbs_type):
    """Returns thread dat file path"""

    _check_thread(bbs_type)

    return os.path.join(get_thread_dat_dir_path(bbs_type),
                        bbs_type.thread + ".dat")

def get_board_subjecttxt_path(bbs_type):
    """Returns subject.txt file path"""

    return os.path.join(get_board_dir_path(bbs_type), "subject.txt")

def get_thread_states_path(bbs_type):
    """Returns thread states file path"""

    _check_thread(bbs_type)

    return os.path.join(get_thread_states_dir_path(bbs_type),
                        bbs_type.thread + ".states")

def get_board_states_path(bbs_type):
    """Returns board states file path"""

    return os.path.join(get_board_dir_path(bbs_type), "board.states")

def get_board_idx_path(bbs_type):
    """Returns board idx file path"""

    return os.path.join(get_board_dir_path(bbs_type), "subject.idx")

def get_board_dir_path(bbs_type):
    """Returns board dir path"""

    return os.path.join(get_logs_dir_path(), bbs_type.get_board_dir_path())

def get_thread_idx_path(bbs_type):
    """Returns idx file path of thread"""

    _check_thread(bbs_type)

    return os.path.join(get_thread_idx_dir_path(bbs_type),
                        bbs_type.thread + ".idx")

def get_board_cache_path(bbs_type):
    """Returns .cache file path of board"""

    return os.path.join(get_thread_idx_dir_path(bbs_type), ".cache")

def httpdate_to_secs(httpdate):
    """Returns the seconds since the epoch"""

    m = REG_EXPR_HTTPDATE.match(httpdate)
    if m:
        tm_wday = WDAY_DICT[m.group(1)]
        tm_mday = int(m.group(2))
        tm_mon = MON_DICT[m.group(3)]
        tm_year = int(m.group(4))
        tm_hour = int(m.group(5))
        tm_min = int(m.group(6))
        tm_sec = int(m.group(7))

        return int(time.mktime(
            (tm_year,tm_mon,tm_mday,tm_hour,tm_min,tm_sec,tm_wday,0,-1)) \
            - time.timezone)
    else:
        raise ValueError


class ThreadInvoker(threading.Thread):
    def __init__(self, on_end, *methods):
        super(ThreadInvoker, self).__init__()
        self.on_end = on_end
        self.methods = methods
    def run(self):
        try:
            for m in self.methods:
                m()
        finally:
            self.on_end()


class FileWrap:
    def __init__(self, path):
        self._file = None
        self._path = path
    def __del__(self):
        self.close()
    def seek(self, size):
        self.file().seek(size)
    def write(self, data):
        self.file().write(data)
    def close(self):
        if self._file:
            self._file.close()
            self._file = None
    def file(self):
        if not self._file:
            basedir = os.path.dirname(self._path)
            if not os.path.isdir(basedir):
                os.makedirs(basedir)
            self._file = file(self._path, "a+")
        return self._file
