# Copyright (C) 2006 by Aiwota Programmer
# aiwotaprog@tetteke.tk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import re
import os.path
import codecs
import fileinput

import misc

REG_EXPR_TITLE = re.compile(".*<>.*<>.*<>.*<>(.*)")
REG_EXPR_ELEM = re.compile( \
    "(?P<name>.*)<>(?P<mail>.*)<>(?P<date>.*)<>(?P<msg>.*)<>")

def get_dat_file_size(bbs, board, thread):
    """Returns size of dat file"""
    dat_path = misc.get_thread_dat_path(bbs, board, thread)
    if not os.path.exists(dat_path):
        return 0

    return os.path.getsize(dat_path)

def get_dat_line_count(bbs, board, thread):
    """Returns the number of line of a dat file specified by bbs, board
    and thread

    bbs: bbs id

    board: board id

    thread: thread id
    """
    dat_path = misc.get_thread_dat_path(bbs, board, thread)
    if not os.path.exists(dat_path):
        return 0

    f = fileinput.FileInput(dat_path)
    for l in f: -1
    return f.filelineno()

def do_get_title_from_dat(line):
    if line:
        m = REG_EXPR_TITLE.match(line.decode("cp932", "replace"))
        if m:
            return m.group(1)
    return ""
    
def get_title_from_dat(bbs, board, thread):
    """Returns thread title in dat file

    bbs: bbs id

    board: board id

    thread: thread id

    If failed, return None
    """
    dat_path = misc.get_thread_dat_path(bbs, board, thread)
    if not os.path.exists(dat_path):
        return None

    f = open(dat_path, "r")
    try:
        line = f.readline()
        return do_get_title_from_dat(line)
    finally:
        f.close()

def split_line_to_elems(line, func):
    """Splits a line to elements and invokes func

    line: represents one res

    func: is invoked after splitting
    format of user functon is:
    def some_func(name, mail, date, msg):
    where parameters represent corresponding elements of the res
    """
    m = REG_EXPR_ELEM.match(line)
    if m:
        name = m.group("name")
        mail = m.group("mail")
        date = m.group("date")
        msg = m.group("msg")
        func(name, mail, date, msg)

def load_dat(bbs, board, thread, func):
    """Loads entire dat and invokes func per one res

    bbs: bbs id

    board: board id

    thread: thread id

    func: is invoked per one res
    format of user function is:
    def some_func(line):
    where line is raw body of the res
    """
    dat_path = misc.get_thread_dat_path(bbs, board, thread)
    if not os.path.exists(dat_path):
        return

    f = fileinput.FileInput(dat_path)
    for line in f:
        func(line)
    f.close()

def load_dat_partly(bbs, board, thread, func, resnum):
    """Loads dat partly
    similar to load_dat, but load_dat_partly does not load entire dat.

    bbs: bbs id

    board: board id

    thread: thread id

    func: invoked per one res

    resnum: load downward resnum
    """
    dat_path = misc.get_thread_dat_path(bbs, board, thread)
    if not os.path.exists(dat_path):
        return

    f = fileinput.FileInput(dat_path)
    for line in f:
        num = f.filelineno()
        if num >= resnum:
            func(line)
    f.close()
