# Copyright (C) 2006 by Aiwota Programmer
# aiwotaprog@tetteke.tk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import pygtk
pygtk.require('2.0')
import gtk
import gtk.glade
import os
import time
import gobject
import gconf

import board_data
import uri_opener
import misc
from threadlistmodel import ThreadListModel
from BbsType import bbs_type_judge_uri
import config
import session

GLADE_DIR = os.path.join(os.path.dirname(os.path.realpath(__file__)),
                         "..", "data")
GLADE_FILENAME = "board_window.glade"

def open_board(uri, update=False):
    if not uri:
        raise ValueError, "parameter must not be empty"

    winwrap = session.get_window(uri)
    if winwrap:
        # already opened
        winwrap.window.present()
        if update:
            winwrap.load(update)
    else:
        winwrap = WinWrap(uri)
        session.window_created(uri, winwrap)
        winwrap.load(update)


class WinWrap:

    def __init__(self, uri):

        self.bbs_type = bbs_type_judge_uri.get_type(uri)
        self.bbs = self.bbs_type.bbs_type
        self.host = self.bbs_type.host
        self.board = self.bbs_type.board
        self.uri = self.bbs_type.uri
        
        glade_path = os.path.join(GLADE_DIR, GLADE_FILENAME)
        self.widget_tree = gtk.glade.XML(glade_path)

        self.window = self.widget_tree.get_widget("board_window")

        self.window.set_title(self.uri)

        self.treeview = self.widget_tree.get_widget("treeview")
        self.treeview.set_model(ThreadListModel())

        self.popupmenu = self.widget_tree.get_widget("popup_menu")
        self.toolbar = self.widget_tree.get_widget("toolbar")
        self.toolbar.unset_style()
        self.statusbar = self.widget_tree.get_widget("appbar")

        renderer = gtk.CellRendererText()

        self.treeviewcolumn = {}
        for i in range(1, len(ThreadListModel.column_names)):
            column_name = ThreadListModel.column_names[i]
            self.treeviewcolumn[column_name] = gtk.TreeViewColumn(
                column_name, renderer)
            self.treeviewcolumn[column_name].set_resizable(True)
            self.treeviewcolumn[column_name].set_reorderable(True)
            self.treeviewcolumn[column_name].set_clickable(True)
            self.treeviewcolumn[column_name].set_cell_data_func(
                renderer, self.on_cell_data, column_name)
            self.treeviewcolumn[column_name].connect(
                "clicked", self.on_column_clicked, column_name)
            self.treeview.append_column(self.treeviewcolumn[column_name])

        self.treeviewcolumn["lastModified"].set_cell_data_func(
            renderer, self.on_data_lastmodified)

        sigdic = {"on_board_window_destroy": self.on_board_window_destroy,
                  "on_quit_activate": self.on_quit_activate,
                  "on_refresh_activate": self.on_refresh_activate,
                  "on_treeview_row_activated":
                  lambda w,p,v: self.on_open_thread(w),
                  "on_treeview_button_press_event":
                  self.on_treeview_button_press_event,
                  "on_close_activate":
                  self.on_close_activate,
                  "on_toolbar_activate": self.on_toolbar_activate,
                  "on_statusbar_activate": self.on_statusbar_activate,
                  "on_board_window_delete_event":
                  self.on_board_window_delete_event,
                  "on_popup_menu_open_activate": self.on_open_thread}
        self.widget_tree.signal_autoconnect(sigdic)

        self.gconf_client = gconf.client_get_default()
        self.gconf_key_base = "/apps/" + config.APPNAME.lower() + \
                              "/board_states/"

        width = self.gconf_client.get_int(
            self.gconf_key_base + "window_width")
        height = self.gconf_client.get_int(
            self.gconf_key_base + "window_height")
        self.window.set_default_size(width, height)

        self.window.show()

        if not self.gconf_client.get_bool(self.gconf_key_base + "toolbar"):
            self.toolbar.parent.hide()
        if not self.gconf_client.get_bool(self.gconf_key_base + "statusbar"):
            self.statusbar.hide()

    def on_toolbar_activate(self, widget):
        if self.toolbar.parent.get_property("visible"):
            self.toolbar.parent.hide()
            self.gconf_client.set_bool(self.gconf_key_base + "toolbar", False)
        else:
            self.toolbar.parent.show()
            self.gconf_client.set_bool(self.gconf_key_base + "toolbar", True)

    def on_statusbar_activate(self, widget):
        if self.statusbar.get_property("visible"):
            self.statusbar.hide()
            self.gconf_client.set_bool(self.gconf_key_base+"statusbar", False)
        else:
            self.statusbar.show()
            self.gconf_client.set_bool(self.gconf_key_base + "statusbar", True)

    def updated_thread_highlight(self, column, cell, model, iter):

        def is_updated_thread():
            res = model.get_value(
                iter, ThreadListModel.column_names.index("res"))
            linecount = model.get_value(
                iter, ThreadListModel.column_names.index("lineCount"))
            return res != 0 and linecount != 0 and res > linecount

        if is_updated_thread():
            cell.set_property("weight", 800)
        else:
            cell.set_property("weight", 400)

    def on_cell_data(self, column, cell, model, iter, column_name):
        self.updated_thread_highlight(column, cell, model, iter)
        column_num = ThreadListModel.column_names.index(column_name)
        value = model.get_value(iter, column_num)
        if model.get_column_type(column_num) == gobject.TYPE_INT:
            if value == 0:
                cell.set_property("text", "")
            else:
                cell.set_property("text", str(value))
        else:
            cell.set_property("text", value)

    def on_data_lastmodified(self, column, cell, model, iter, user_data=None):
        self.updated_thread_highlight(column, cell, model, iter)
        lastmod = model.get_value(
            iter, ThreadListModel.column_names.index("lastModified"))
        if lastmod == 0:
            cell.set_property("text", "")
        else:
            cell.set_property("text", time.strftime(
                "%Y/%m/%d(%a) %H:%M:%S", time.localtime(lastmod)))

    def on_board_window_delete_event(self, widget, event):
        w, h = widget.get_size()
        self.gconf_client.set_int(self.gconf_key_base + "window_width", w)
        self.gconf_client.set_int(self.gconf_key_base + "window_height", h)

        return False

    def on_board_window_destroy(self, widget):
        pass

    def on_quit_activate(self, widget):
        session.main_quit()

    def on_close_activate(self, widget):
        self.window.destroy()

    def on_refresh_activate(self, widget):
        t = board_data.GetRemote(
            self.bbs, self.board, self.bbs_type.get_subject_txt_uri(),
            self.update_datastore)
        t.start()

    def on_column_clicked(self, treeviewcolumn, column_name):
        model = self.treeview.get_model()
        if model:
            model.sort(column_name)
            self.reset_sort_indicator()

    def reset_sort_indicator(self):
        model = self.treeview.get_model()
        if model:
            sort_column_name, sort_reverse = model.get_sort()
            for name,column in self.treeviewcolumn.iteritems():
                column.set_sort_indicator(False)
            if sort_column_name != "num" or sort_reverse:
                self.treeviewcolumn[sort_column_name].set_sort_indicator(True)
                if sort_reverse:
                    self.treeviewcolumn[sort_column_name].set_sort_order(
                        gtk.SORT_DESCENDING)
                else:
                    self.treeviewcolumn[sort_column_name].set_sort_order(
                        gtk.SORT_ASCENDING)
        
    def on_open_thread(self, widget):
        treeselection = self.treeview.get_selection()
        model, iter = treeselection.get_selected()
        if not iter:
            return

        thread = model.get_value(iter, ThreadListModel.column_names.index("id"))
        title = model.get_value(
            iter, ThreadListModel.column_names.index("title"))
        print thread + ':"' + title + '"', "activated"

        res = model.get_value(iter, ThreadListModel.column_names.index("res"))
        lineCount = model.get_value(
            iter, ThreadListModel.column_names.index("lineCount"))

        update = res > lineCount

        bbs_type_for_thread = self.bbs_type.clone_with_thread(thread)
        uri_opener.open_uri(bbs_type_for_thread.get_thread_uri(), update)

    def on_treeview_button_press_event(self, widget, event):
        if event.button == 3:
            x = int(event.x)
            y = int(event.y)
            time = event.time
            pthinfo = widget.get_path_at_pos(x, y)
            if pthinfo is not None:
                path, col, cellx, celly = pthinfo
                widget.grab_focus()
                widget.set_cursor(path, col, 0)
                self.popupmenu.popup(None, None, None, event.button, time)
            return 1

    def update_datastore(self, datalist, lastmod):
        print "reflesh datastore"

        try:
            lastmod = misc.httpdate_to_secs(lastmod)
        except:
            lastmod = 0

        time_start = time.time()
        list_list = []
        for id, dic in datalist.iteritems():
            dic["id"] = id

            # average
            if lastmod == 0 or dic["num"] == 0:
                dic["average"] = 0
            else:
                res = dic["res"]
                start = int(id)
                # avoid the Last-Modified time of subject.txt and
                # the build time of thread is equal (zero division)
                dur = lastmod - start
                if dur == 0:
                    dic["average"] = 999999
                else:
                    dic["average"] = int(res * 60 * 60 * 24 / dur)

            # lastModified
            httpdate = dic["lastModified"]
            try:
                secs = misc.httpdate_to_secs(httpdate)
                dic["lastModified"] = secs
            except ValueError:
                dic["lastModified"] = 0
            
            list_list.append(dic)

        model = self.treeview.get_model()
        model.set_list(list_list)

        # redraw visible area after set list to model
        self.treeview.queue_draw()

        self.reset_sort_indicator()

        print "end"
        time_end = time.time()
        print time_end - time_start

    def on_thread_idx_updated(self, thread_uri, idx_dic):
        if not thread_uri or not idx_dic:
            return

        # nothing to do if thread_uri does not belong to this board.
        bbs_type = bbs_type_judge_uri.get_type(thread_uri)
        if bbs_type.bbs_type != self.bbs \
           or bbs_type.board != self.board or not bbs_type.is_thread():
            return

        thread = bbs_type.thread

        model = self.treeview.get_model()
        if model:
            idx_dic["id"] = thread
            try:
                idx_dic["lastModified"] =  misc.httpdate_to_secs(
                    idx_dic["lastModified"])
            except ValueError:
                idx_dic["lastModified"] = 0
            model.modify_row(idx_dic)

    def load(self, update=False):
        sbj_path = misc.get_board_subjecttxt_path(
            self.bbs_type.bbs_type, self.bbs_type.board)
        sbj_exists = os.path.exists(sbj_path)

        if update or not sbj_exists:
            t = board_data.GetRemote(
                self.bbs, self.board, self.bbs_type.get_subject_txt_uri(),
                self.update_datastore)
            t.start()
        else:
            t = board_data.LoadLocal(
                self.bbs, self.board, self.update_datastore)
            t.start()
