//*****************************************************************************//
//fgstd.c
//Copyright (C) 2006 K.Tsuchiya all rights reserved.
//2006/09/4
//... fridge standard object library.
//*****************************************************************************//
#include <poll.h>
#include <regex.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <stdio.h>
#define __USE_GNU 
#include <fcntl.h>
#include "fridge.h"
#ifndef O_BINARY
#define O_BINARY 0
#endif 
/*------------------------------------------------------------------------------*/
/*                                                                              */
/*                       standard library access API                            */
/*                                                                              */
/*------------------------------------------------------------------------------*/
#define fgDefOffset "offset"
#define fgDefPid "pid"

//===============================================================================
//access function
//===============================================================================
typedef struct{
    char*name;
    fgData (*meth)(fgHandle,int self);
}fgStdFuncList;
//access
void* fgStdMethod(char*name,int flg,void*val){
    int i,size;
    void*p;
    /*default*/
    static cMap tbl;
    static fgStdFuncList list[]={
        {"sys"      ,(*fgStdObjSystem)},
        {"echo"     ,(*fgStdObjEcho)},
        {"open"     ,(*fgStdObjOpen)},
        {"close"    ,(*fgStdObjClose)},
        {"write"    ,(*fgStdObjWrite)},
        {"read"     ,(*fgStdObjRead)},
        {"readln"   ,(*fgStdObjReadln)},
        {"pwrite"   ,(*fgStdObjPwrite)},
        {"pread"    ,(*fgStdObjPread)},
        {"dwrite"   ,(*fgStdObjDwrite)},
        {"dappend"  ,(*fgStdObjDappend)},
        {"dread"    ,(*fgStdObjDread)},
        {"split"    ,(*fgStdObjSplit)},
        {"slice"    ,(*fgStdObjSlice)},
        {"replace"  ,(*fgStdObjReplace)},
        {"match"    ,(*fgStdObjMatch)},
        {"rslice"   ,(*fgStdObjRslice)},
        {"rreplace" ,(*fgStdObjRreplace)},
        {"len"      ,(*fgStdObjLen)},
        {"call"     ,(*fgStdObjCall)},
        {"talk"     ,(*fgStdObjTalk)},
    };
    switch(flg){
    case 0x0 : /*init*/
        tbl=cmapIni(sizeof(char*),sizeof(void*),(*cmapStrKey),(*cmapStrCmp));
        size=sizeof(list)/sizeof(fgStdFuncList);
        for(i=0;i<size;i++) cmapUpdate(tbl,&(list[i].name),&(list[i].meth));
        return tbl;
    case 0x1 : /*get*/
        p=cmapFind(tbl,&name);
        if(p) p=cioVoid(p,void*);
        return p;
    case 0x2 : /*set*/
        return cmapUpdate(tbl,&name,&val);
    case 0x3 : /*destruct*/ 
        return cmapDes(tbl);
    default : return NULL;
    }
}
/*------------------------------------------------------------------------------*/
/*                                                                              */
/*                             standard library                                 */
/*                                                                              */
/*------------------------------------------------------------------------------*/

//===============================================================================
//escape
//===============================================================================
char* fgStdEscape(char* str){
    int i,j,size;
    char*buf;
    size=strlen(str);
    cstrMakeBuf(buf,size+1,char,exit(-1));
    for(i=0,j=0;i<size;i++,j++){
        if(str[i]=='\\'){
            i++;
            if(i>=size) break;
            switch(str[i]){
            case '\\' : buf[j]='\\'; break;
            case 'a'  : buf[j]='\a'; break;
            case 'b'  : buf[j]='\b'; break;
            case 'f'  : buf[j]='\f'; break;
            case 'n'  : buf[j]='\n'; break;
            case 'r'  : buf[j]='\r'; break;
            case 't'  : buf[j]='\t'; break;
            case 'v'  : buf[j]='\v'; break;
            case '"'  : buf[j]='"' ; break;
            case '\'' : buf[j]='\''; break;
            }
        }else buf[j]=str[i];
    }
    buf[i]=CIO_ES;
    return buf;
}
//===============================================================================
//function for regex 
//2006/10/08 added.
//===============================================================================
//init object
static int fgStdObjRegInit(fgHandle fhdl,fgObj*obj){
    int size;
    fgObj*tmpobj;
    size=fgMakeObj(fhdl,fgDefSize,obj->id);
    tmpobj=fgGetObjLow(fhdl,size);
    (*tmpobj->valp)=fgSetValTypeInt(-1);
    return 0;
}
//getting flag
static int fgStdObjRegGetFlag(char*gen,int*gflag){
    int i,flag=REG_EXTENDED;
    for(i=0,*gflag=0;gen[i]!=CIO_ES;i++){
        switch(gen[i]){
        case 'i': flag|=REG_ICASE; break;
        case 'g': *gflag=1; break;
        default : break;
        }
    }
    return flag;
}
//regex compile
static regex_t fgStdObjRegCompile(char*gen,int flag){
    static tmpflg=0;
    static char* regex=NULL;
    static regex_t preg;
    if(!regex){
        cstrSet(regex,gen,exit(-1));
        regcomp(&preg,regex,flag);
        tmpflg=flag;
    }else if(strcmp(gen,regex) || (flag!=tmpflg)){
        free(regex);
        regfree(&preg);
        cstrSet(regex,gen,exit(-1));
        regcomp(&preg,regex,flag);
        tmpflg=flag;
    }
    return preg;
}
//===============================================================================
//replace regex 
//2006/10/08 speed up(by using static pointer).
//2006/10/10 bugfix.
//2006/10/16 bugfix.
//2006/10/21 setting garbage collection
//2006/10/22 bugfix.
//2006/10/25 bugfix.
//===============================================================================
fgData fgStdObjRreplace(fgHandle fhdl,int self){
    int offsetobj;
    int i,flag,gflag,rflag;
    int arg,index,outvar;
    int size,argsize;
    int offset,baseindex;
    fgData num,data;
    regmatch_t pmatch;
    fgObj*obj,*tmpobj;
    char*gen,*after,*buf;
    regex_t preg;
    cVector mlist;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    if(argsize>0){
        outvar=fgGetValObj(fgGetArryNum(fhdl,arg,0));
        obj=fgGetObjLow(fhdl,outvar);
    }
    else outvar=obj->id;
    /*init array size*/
    fgStdObjRegInit(fhdl,obj);
    /*checking arg size*/
    if(argsize<3){
        puts("rreplace <string> <regex> <after> [flag]");
        (*obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(outvar);
    }
     /*getting affer replace value*/
    data=fgGetArryNum(fhdl,arg,2);
    data=fgSysCast(fhdl,data,fgDataTypeStr);
    cstrSet(after,fgGetValStr(data),exit(-1));
    /*checking flag*/
    flag=REG_EXTENDED;
    if(argsize>3){
        data=fgGetArryNum(fhdl,arg,3);
        data=fgSysCast(fhdl,data,fgDataTypeStr);
        gen=fgGetValStr(data);
        flag=fgStdObjRegGetFlag(gen,&gflag);
    } else gflag=0;
    data=fgGetArryNum(fhdl,arg,1);
    data=fgSysCast(fhdl,data,fgDataTypeStr);
    gen=fgGetValStr(data);
    /*compile*/
    preg=fgStdObjRegCompile(gen,flag);
    data=fgGetArryNum(fhdl,arg,0);
    data=fgSysCast(fhdl,data,fgDataTypeStr);
    gen=fgGetValStr(data);
    /*init match list*/
    mlist=cvectIni(sizeof(regmatch_t),150);
    /*search*/
    for(i=0,baseindex=0,offset=0,size=0;
        rflag=(regexec(&preg,gen+baseindex,1,&pmatch,0)
            !=REG_NOMATCH);i++){
        data=fgSetValTypeInt((int)pmatch.rm_so+baseindex);
        offset=(int)pmatch.rm_eo-(int)pmatch.rm_so;
        pmatch.rm_so=fgGetValInt(data);
        pmatch.rm_eo=offset;
        baseindex=pmatch.rm_so+pmatch.rm_eo;
        size+=offset+1;
        cvectPush(mlist,&pmatch);
        if(!gflag) break;
    }
    if((!i) && (!rflag)){
        cvectDes(mlist);
        free(after);
        obj=fgGetObjLow(fhdl,self);
        (*obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    cstrMakeBuf(buf,size+1,char,exit(-1));
    fgGbgSet(fhdl,buf,size+1);
    /*replace*/
    if(!gflag){
        int size;
        char*tmpbuf;
        pmatch=cioVoid(cvectFind(mlist,0),regmatch_t);
        size=strlen(after);
        if(size>pmatch.rm_eo) cstrRemakeBuf(gen,
            strlen(gen)+size-pmatch.rm_eo+1,char,exit(-1));
        cstrSet(tmpbuf,(gen+pmatch.rm_so+pmatch.rm_eo),exit(-1));
        memcpy(gen+pmatch.rm_so,after,sizeof(char)*size);
        memcpy(gen+pmatch.rm_so+size,tmpbuf,strlen(tmpbuf)+1);
        free(tmpbuf);
    }else{
        int subsize;
        int afsize;
        int state,tmpstate;
        char*tmpbuf;
        afsize=strlen(after);
        cstrSet(tmpbuf,gen,exit(-1));
        for(i=0,subsize=0;i<cvectGetSize(mlist);i++){
            pmatch=cioVoid(cvectFind(mlist,i),regmatch_t);
            subsize+=afsize-pmatch.rm_eo;
        }
        if(subsize>0)cstrRemakeBuf(gen,
            strlen(gen)+subsize+1,char,exit(-1));
        for(i=0,tmpstate=0,state=0;i<cvectGetSize(mlist);i++){
            pmatch=cioVoid(cvectFind(mlist,i),regmatch_t);
            subsize=sizeof(char)*(pmatch.rm_so-tmpstate);
            memcpy(gen+state,tmpbuf+tmpstate,subsize);
            memcpy(gen+state+subsize,after,afsize);
            state+=afsize+subsize;
            tmpstate+=pmatch.rm_eo+subsize;
        }
        memcpy(gen+state,tmpbuf+tmpstate,strlen(tmpbuf)-tmpstate+1);
        free(tmpbuf);
    }
    *(obj->valp)=fgSetValTypeStr(gen);
    cvectDes(mlist);
    free(after);
    return fgSetValTypeObj(outvar);
}
//===============================================================================
//slice regex 
//2006/10/08 speed up(by using static pointer).
//2006/10/10 bugfix.
//2006/10/16 bugfix.
//===============================================================================
fgData fgStdObjRslice(fgHandle fhdl,int self){
    char*gen;
    char*buf;
    int i,flag,gflag;
    int arg,index,outvar;
    int size,argsize,rflag;
    int offset,baseindex;
    fgData num;
    fgData data;
    regmatch_t pmatch;
    fgObj*obj,*tmpobj;
    regex_t preg;
    cVector mlist;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    /*output variable*/
    if(argsize>3){
        outvar=fgGetValObj(fgGetArryNum(fhdl,arg,3));
        obj=fgGetObjLow(fhdl,outvar);
    }
    else outvar=obj->id;
    /*init array size*/
    fgStdObjRegInit(fhdl,obj);
    /*checking arg size*/
    if(argsize<2){
        puts("rslice <string> <regex> [flag] [outvar]");
        (*obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(outvar);
    }
    /*checking flag*/
    flag=REG_EXTENDED;
    if(argsize>2){
        data=fgGetArryNum(fhdl,arg,2);
        data=fgSysCast(fhdl,data,fgDataTypeStr);
        gen=fgGetValStr(data);
        flag=fgStdObjRegGetFlag(gen,&gflag);
    } else gflag=0;
    data=fgGetArryNum(fhdl,arg,1);
    data=fgSysCast(fhdl,data,fgDataTypeStr);
    gen=fgGetValStr(data);
    /*compile*/
    preg=fgStdObjRegCompile(gen,flag);
    data=fgGetArryNum(fhdl,arg,0);
    data=fgSysCast(fhdl,data,fgDataTypeStr);
    gen=fgGetValStr(data);
    /*init match list*/
    mlist=cvectIni(sizeof(regmatch_t),150);
    /*search*/
    for(i=0,baseindex=0,offset=0,size=0;
        rflag=(regexec(&preg,gen+baseindex,1,&pmatch,0)
            !=REG_NOMATCH);i++){
        data=fgSetValTypeInt((int)pmatch.rm_so+baseindex);
        offset=(int)pmatch.rm_eo-(int)pmatch.rm_so;
        pmatch.rm_so=fgGetValInt(data);
        pmatch.rm_eo=offset;
        baseindex=pmatch.rm_so+pmatch.rm_eo;
        size+=offset+1;
        cvectPush(mlist,&pmatch);
        if(!gflag) break;
    }
    if((!i) && (!rflag)){
        cvectDes(mlist);
        (*obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(outvar);
    }
    cstrMakeBuf(buf,size+1,char,exit(-1));
    fgGbgSet(fhdl,buf,size+1);
    /*slice*/
    if(!gflag){
         pmatch=cioVoid(cvectFind(mlist,0),regmatch_t);
         memcpy(buf,gen+pmatch.rm_so,pmatch.rm_eo);
         buf[pmatch.rm_eo]=CIO_ES;
         *(obj->valp)=fgSetValTypeStr(buf);
    }else{
        fgSetType(num,fgDataTypeInt);
        for(i=0;i<cvectGetSize(mlist);i++){
            pmatch=cioVoid(cvectFind(mlist,i),regmatch_t);
            memcpy(buf,gen+pmatch.rm_so,pmatch.rm_eo);
            buf[pmatch.rm_eo]=CIO_ES;
            fgSetArryNum(fhdl,outvar,i,fgSetValTypeStr(buf));
            buf+=pmatch.rm_eo+1;
        }
        *(obj->valp)=fgSetValTypeInt(0);
    }
    cvectDes(mlist);
    return fgSetValTypeObj(outvar);
}
//===============================================================================
//match regex
//2006/10/08 speed up(by using static pointer).
//2006/10/10 bugfix.
//2006/10/16 bugfix.
//===============================================================================
fgData fgStdObjMatch(fgHandle fhdl,int self){
    char*gen;
    fgObj*obj;
    fgObj*tmpobj;
    fgObj*offobj;
    fgData data;
    int size,argsize;
    int offset,baseindex;
    int i,flag,gflag;
    int arg,index,rflag;
    int offsetobj,outvar;
    regmatch_t pmatch;
    regex_t preg;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    /*output variable*/
    if(argsize>3){
        outvar=fgGetValObj(fgGetArgElm(fhdl,self,3));
        obj=fgGetObjLow(fhdl,outvar);
    }
    else outvar=obj->id;
    /*init*/
    fgStdObjRegInit(fhdl,obj);
    offsetobj=fgMakeObj(fhdl,fgDefOffset,outvar);
    tmpobj=fgGetObjLow(fhdl,offsetobj);
    (*tmpobj->valp)=fgSetValTypeInt(-1);
    /*checking argsize*/
    if(argsize<2){
        puts("match <string> <regex> [flag] [outvar]");
        (*obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(outvar);
    }
    /*checking flag*/
    if(argsize>2){
        data=fgGetArryNum(fhdl,arg,2);
        data=fgSysCast(fhdl,data,fgDataTypeStr);
        gen=fgGetValStr(data);
        flag=fgStdObjRegGetFlag(gen,&gflag);
    }else{
        flag=REG_EXTENDED;
        gflag=0;
    }
    /*compile*/
    data=fgGetArryNum(fhdl,arg,1);
    data=fgSysCast(fhdl,data,fgDataTypeStr);
    gen=fgGetValStr(data);
    preg=fgStdObjRegCompile(gen,flag);
    data=fgGetArryNum(fhdl,arg,0);
    data=fgSysCast(fhdl,data,fgDataTypeStr);
    gen=fgGetValStr(data);
    /*search*/
    for(i=0,baseindex=0,offset=0,size=0;
        rflag=(regexec(&preg,gen+baseindex,1,&pmatch,0)
            !=REG_NOMATCH);i++){
        data=fgSetValTypeInt((int)pmatch.rm_so+baseindex);
        offset=(int)pmatch.rm_eo-(int)pmatch.rm_so;
        baseindex=fgGetValInt(data)+offset;
        if(!gflag){
            (*obj->valp)=data;
            offsetobj=fgMakeObj(fhdl,fgDefOffset,outvar);
            offobj=fgGetObjLow(fhdl,offsetobj);
            (*offobj->valp)=fgSetValTypeInt(offset);
            break;
        }
        index=fgMakeArry(fhdl,outvar,fgSetValTypeInt(i));
        tmpobj=fgGetObjLow(fhdl,index);
        (*tmpobj->valp)=data;
        offsetobj=fgMakeObj(fhdl,fgDefOffset,index);
        offobj=fgGetObjLow(fhdl,offsetobj);
        (*offobj->valp)=fgSetValTypeInt(offset);
    }
    if((!i) && (!rflag)){
        (*obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(outvar);
    }
    if(i&&gflag){
        (*obj->valp)=fgSetValTypeInt(0);
        offsetobj=fgMakeObj(fhdl,fgDefOffset,outvar);
        offobj=fgGetObjLow(fhdl,offsetobj);
        (*offobj->valp)=fgSetValTypeInt(0);
    }
    return fgSetValTypeObj(outvar);
}
//===============================================================================
//direct write
//2006/10/20 added.
//===============================================================================
fgData fgStdObjDwrite(fgHandle fhdl,int self){
   FILE*pfp;
   fgObj*obj;
   fgData com;
   fgData data;
   char*outval;
   char outvalc;
   int i,arg,argsize;
   obj=fgGetObjLow(fhdl,self);
   arg=fgMakeObj(fhdl,fgDefArg,obj->id);
   //getting argsize
   argsize=fgSysGetSize(fhdl,arg);
   if(argsize<1){
       puts("dwrite <filename> <outval>...");
       return fgSetValTypeInt(-1);
   }
   //getting command
   com=fgGetArryNum(fhdl,arg,0);
   com=fgSysCast(fhdl,com,fgDataTypeStr);
   pfp=fopen(fgGetValStr(com),"w");
   if(!pfp) return fgSetValTypeInt(-1);
   //write data
   for(i=1;i<argsize;i++){
       data=fgGetArryNum(fhdl,arg,i);
       data=fgSysCast(fhdl,data,fgDataTypeStr);
       outval=fgGetValStr(data);
       fputs(outval,pfp);
   }
   outvalc=CIO_ES;
   fwrite(&outvalc,sizeof(char),1,pfp);
   fclose(pfp);
   return fgSetValTypeObj(obj->id);
}
//===============================================================================
//direct append
//2006/10/20 added.
//===============================================================================
fgData fgStdObjDappend(fgHandle fhdl,int self){
   FILE*pfp;
   fgObj*obj;
   fgData com;
   fgData data;
   char*outval;
   char outvalc;
   int i,arg,argsize;
   obj=fgGetObjLow(fhdl,self);
   arg=fgMakeObj(fhdl,fgDefArg,obj->id);
   //getting argsize
   argsize=fgSysGetSize(fhdl,arg);
   if(argsize<1){
       puts("dappend <filename> <outval>...");
       return fgSetValTypeInt(-1);
   }
   //getting command
   com=fgGetArryNum(fhdl,arg,0);
   com=fgSysCast(fhdl,com,fgDataTypeStr);
   pfp=fopen(fgGetValStr(com),"a");
   if(!pfp) return fgSetValTypeInt(-1);
   //write data
   for(i=1;i<argsize;i++){
       data=fgGetArryNum(fhdl,arg,i);
       data=fgSysCast(fhdl,data,fgDataTypeStr);
       outval=fgGetValStr(data);
       fputs(outval,pfp);
   }
   outvalc=CIO_ES;
   fwrite(&outvalc,sizeof(char),1,pfp);
   fclose(pfp);
   return fgSetValTypeObj(obj->id);
}
//===============================================================================
//direct read
//2006/10/20 added.
//2006/12/27 a erratum was corrected.
//===============================================================================
fgData fgStdObjDread(fgHandle fhdl,int self){
   FILE*pfp;
   char* com;
   fgObj*obj;
   fgData data;
   int outvar,arg;
   int i,argsize;
   char* outbuf=NULL;
   static char buf[FG_BUFSIZE];
   obj=fgGetObjLow(fhdl,self);
   arg=fgMakeObj(fhdl,fgDefArg,obj->id);
   //getting argsize
   argsize=fgSysGetSize(fhdl,arg);
   /*output variable*/
   if(argsize>1){
       outvar=fgGetValObj(fgGetArryNum(fhdl,arg,1));
       obj=fgGetObjLow(fhdl,outvar);
   }
   else outvar=self;
   /*checking args*/  
   if(argsize<1){
       puts("dread <filename> [outvar]");
       return fgSetValTypeInt(-1);
   }
   /*getting command*/
   data=fgGetArryNum(fhdl,arg,0);
   data=fgSysCast(fhdl,data,fgDataTypeStr);
   cstrSet(com,fgGetValStr(data),exit(-1));
   pfp=fopen(com,"r");
   if(!pfp) return fgSetValTypeInt(-1);
   if(!outbuf){
       cstrSet(outbuf,"",exit(-1));
   }else outbuf[0]=CIO_ES;
   while(fgets(buf,FG_BUFSIZE,pfp))
       cstrCat(outbuf,buf,exit(-1));
   data=fgSetValTypeStr(outbuf);
   *(obj->valp)=data;
   pclose(pfp);
   free(com);
   return fgSetValTypeObj(outvar);
}
//===============================================================================
//pipe write
//2006/10/08 speed up(by using static pointer).
//===============================================================================
fgData fgStdObjPwrite(fgHandle fhdl,int self){
   FILE*pfp;
   fgObj*obj;
   fgData com;
   fgData data;
   char*outval;
   char outvalc;
   int i,arg,argsize;
   obj=fgGetObjLow(fhdl,self);
   arg=fgMakeObj(fhdl,fgDefArg,obj->id);
   //getting argsize
   argsize=fgSysGetSize(fhdl,arg);
   if(argsize<1){
       puts("pwrite <command> <outval>...");
       return fgSetValTypeInt(-1);
   }
   //getting command
   com=fgGetArryNum(fhdl,arg,0);
   com=fgSysCast(fhdl,com,fgDataTypeStr);
   pfp=popen(fgGetValStr(com),"w");
   if(!pfp) return fgSetValTypeInt(-1);
   //write data
   for(i=1;i<argsize;i++){
       data=fgGetArryNum(fhdl,arg,i);
       data=fgSysCast(fhdl,data,fgDataTypeStr);
       outval=fgGetValStr(data);
       fputs(outval,pfp);
   }
   outvalc=CIO_ES;
   fwrite(&outvalc,sizeof(char),1,pfp);
   fclose(pfp);
   return fgSetValTypeObj(obj->id);
}
//===============================================================================
//pipe read
//2006/10/08 speed up(by using static pointer).
//===============================================================================
fgData fgStdObjPread(fgHandle fhdl,int self){
   FILE*pfp;
   char* com;
   fgObj*obj;
   fgData data;
   int outvar,arg;
   int i,argsize;
   char* outbuf=NULL;
   static char buf[FG_BUFSIZE];
   obj=fgGetObjLow(fhdl,self);
   arg=fgMakeObj(fhdl,fgDefArg,obj->id);
   //getting argsize
   argsize=fgSysGetSize(fhdl,arg);
   /*output variable*/
   if(argsize>1){
       outvar=fgGetValObj(fgGetArryNum(fhdl,arg,1));
       obj=fgGetObjLow(fhdl,outvar);
   }
   else outvar=self;
   /*checking args*/  
   if(argsize<1){
       puts("pread <command> [outvar]");
       return fgSetValTypeInt(-1);
   }
   /*getting command*/
   data=fgGetArryNum(fhdl,arg,0);
   data=fgSysCast(fhdl,data,fgDataTypeStr);
   cstrSet(com,fgGetValStr(data),exit(-1));
   pfp=popen(com,"r");
   if(!pfp) return fgSetValTypeInt(-1);
   if(!outbuf){
       cstrSet(outbuf,"",exit(-1));
   }else outbuf[0]=CIO_ES;
   while(fgets(buf,FG_BUFSIZE,pfp))
       cstrCat(outbuf,buf,exit(-1));
   data=fgSetValTypeStr(outbuf);
   *(obj->valp)=data;
   pclose(pfp);
   free(com);
   return fgSetValTypeObj(outvar);
}
//===============================================================================
//string length
//2006/10/08 speed up(by using static pointer).
//===============================================================================
fgData fgStdObjLen(fgHandle fhdl,int self){
    fgObj* obj;
    fgData data;
    int length,arg;
    int i,argsize;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    if(!argsize){
        puts("len <strval>");
        return fgSetValTypeInt(-1);
    }
    for(i=0,length=0;i<argsize;i++){
        data=fgGetArryNum(fhdl,arg,i);
        data=fgSysCast(fhdl,data,fgDataTypeStr);
        length+=strlen(fgGetValStr(data));
    }
    data=fgSetValTypeInt(length);
    (*obj->valp)=data;
    return fgSetValTypeObj(obj->id);
}
//===============================================================================
//slice string
//2006/10/08 speed up(by using static pointer).
//===============================================================================
fgData fgStdObjSlice(fgHandle fhdl,int self){
    char*str;
    fgObj*obj;
    fgData data;
    int i,argsize,arg;
    int index,size,tmpsize;
    static int bufsize=0;
    static char*buf=NULL;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    /*checking args*/
    if(argsize < 2){
        puts("slice <strval> <index> [offset] [outvar]");
        return fgSetValTypeInt(-1);
    }else if(argsize==2){
        data=fgGetArryNum(fhdl,arg,0);
        str=fgGetValStr(fgSysCast(fhdl,data,fgDataTypeStr));
        data=fgGetArryNum(fhdl,arg,1);
        index=fgGetValInt(fgSysCast(fhdl,data,fgDataTypeInt));
        size=1;
    }else if(argsize>=3){
        data=fgGetArryNum(fhdl,arg,0);
        str=fgGetValStr(fgSysCast(fhdl,data,fgDataTypeStr));
        data=fgGetArryNum(fhdl,arg,1);
        index=fgGetValInt(fgSysCast(fhdl,data,fgDataTypeInt));
        data=fgGetArryNum(fhdl,arg,2);
        size=fgGetValInt(fgSysCast(fhdl,data,fgDataTypeInt));
    }
    /*getting string*/
    if(obj->valp->type==fgDataTypeStr) buf=obj->valp->elm.s;
    tmpsize=strlen(str);
    if(tmpsize<index) index=tmpsize;
    while(index+size>tmpsize) size--;
    if(bufsize<=size) cstrRemakeBuf(buf,size+1,char,exit(-1));
    memcpy(buf,str+index,size);
    buf[size]=CIO_ES;
    (*obj->valp)=fgSetValTypeStr(buf);
    /*output*/
    if(argsize>3){
        fgData outdata;
        outdata=fgGetArryNum(fhdl,arg,3);
        if(fgGetType(outdata)==fgDataTypeObj){
            obj=fgGetObjLow(fhdl,fgGetValObj(outdata));
            if(obj->valp->type==fgDataTypeStr){
                int tmpsize=strlen(buf);
                if(strlen(obj->valp->elm.s)<tmpsize)
                    cstrRemakeBuf(obj->valp->elm.s,tmpsize+1,char,exit(-1));  
                memcpy(obj->valp->elm.s,buf,sizeof(char)*tmpsize+1);
            }else{
                cstrSet(str,buf,exit(-1));
                (*obj->valp)=fgSetValTypeStr(str);
            }
        }
    }
    return fgSetValTypeObj(obj->id);
}
//===============================================================================
//replace string
//2006/10/08 speed up(by using static pointer).
//===============================================================================
fgData fgStdObjReplace(fgHandle fhdl,int self){
    char*str;
    char*after;
    char*tmpbuf;
    fgData data;
    int argsize,index;
    int offset,outvar;
    int afsize,tmpsize,arg;
    fgObj*obj;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    if(argsize<2){
        puts("replace <strval> <after> [index] [offset]");
        return fgSetValTypeInt(-1);
    }
    data=fgGetArryNum(fhdl,arg,0);
    if(fgGetType(data)!=fgDataTypeObj) return fgSetValTypeInt(0);
    /*setting outvar*/
    outvar=fgGetValObj(data);
    obj=fgGetObjLow(fhdl,outvar);
    data=fgSysCast(fhdl,data,fgDataTypeStr);
    str=fgGetValStr(data);
    data=fgGetArryNum(fhdl,arg,1);
    data=fgSysCast(fhdl,data,fgDataTypeStr);
    after=fgGetValStr(data);
    /*checking option*/
    if(argsize>2){
        data=fgGetArryNum(fhdl,arg,2);
        data=fgSysCast(fhdl,data,fgDataTypeInt);
        index=fgGetValInt(data);
    }else index=0;
    if(argsize>3){
        data=fgGetArryNum(fhdl,arg,3);
        data=fgSysCast(fhdl,data,fgDataTypeInt);
        offset=fgGetValInt(data);
    }else offset=0;
    tmpsize=strlen(str);
    afsize=strlen(after);
    if(index>tmpsize)return fgSetValTypeInt(0);
    /*checking size*/
    while(tmpsize<(index+offset)) offset--;
    if(afsize>offset) cstrRemakeBuf(str,tmpsize+1+(afsize-offset),char,exit(-1));
    /*replace*/
    cstrSet(tmpbuf,(str+index+offset),exit(-1));
    memcpy(str+index,after,sizeof(char)*afsize);
    memcpy(str+index+afsize,tmpbuf,sizeof(char)*(strlen(tmpbuf)+1));
    /*setting output*/
    (*obj->valp)=fgSetValTypeStr(str);
    return fgSetValTypeObj(outvar);
}
//===============================================================================
//split string
//2006/10/08 speed up(by using static pointer).
//===============================================================================
fgData fgStdObjSplit(fgHandle fhdl,int self){
    char*str;
    char*token;
    char**spdata;
    int i,argsize,arg;
    fgObj* obj;
    fgData outobj;
    fgData data={fgDataTypeInt,0};
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    if(argsize<1){
        puts("split <strval> [delimiter]");
        return fgSetValTypeInt(-1);
    }
    else if(argsize<2){
        outobj=fgGetArryNum(fhdl,arg,0);
        obj=fgGetObjLow(fhdl,fgGetValObj(outobj));
        str=fgGetValStr(fgSysCast(fhdl,outobj,fgDataTypeStr));
        token=" ,\t\n";
    }
    else if(argsize<3){
        outobj=fgGetArryNum(fhdl,arg,0);
        obj=fgGetObjLow(fhdl,fgGetValObj(outobj));
        str=fgGetValStr(fgSysCast(fhdl,
            fgGetArryNum(fhdl,arg,1),fgDataTypeStr));
        cstrSet(token,str,exit(-1));
        str=fgGetValStr(fgSysCast(fhdl,outobj,fgDataTypeStr));
    }
    else{
        outobj=fgGetArryNum(fhdl,arg,0);
        obj=fgGetObjLow(fhdl,fgGetValObj(outobj));
        str=fgGetValStr(fgSysCast(fhdl,
            fgGetArryNum(fhdl,arg,2),fgDataTypeStr));
        cstrSet(token,str,exit(-1));
        str=fgGetValStr(fgSysCast(fhdl,
            fgGetArryNum(fhdl,arg,1),fgDataTypeStr));
    }
    spdata=cstrSplit(str,token);
    for(i=0;spdata[i]!=NULL;i++){
        fgGbgSet(fhdl,spdata[i],strlen(spdata[i]));
        fgSetArryNum(fhdl,
            fgGetValObj(outobj),i,fgSetValTypeStr(spdata[i]));
    }
    if(argsize>2)free(token);
    return outobj;
}
//===============================================================================
//system # == system(<args>)
//2006/10/08 speed up(by using static pointer).
//===============================================================================
fgData fgStdObjSystem(fgHandle fhdl,int self){
    int i,flg,arg;
    int argsize;
    char*buf;
    fgObj*obj;
    fgData data;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    cstrSet(buf,"",exit(-1));
    for(i=0;i<argsize;i++){
       data=fgGetArryNum(fhdl,arg,i);
       data=fgSysCast(fhdl,data,fgDataTypeStr);
       cstrCat(buf,data.elm.s,exit(-1));
       cstrCat(buf," ",exit(-1));
    }
    flg=system(buf);
    free(buf);
    /*return*/
    *(obj->valp)=fgSetValTypeInt(flg);
    return fgSetValTypeObj(obj->id);
}
//===============================================================================
//echo
//2006/10/07 speed up(by using static pointer).
//===============================================================================
fgData fgStdObjEcho(fgHandle fhdl,int self){
    int i,flg,arg;
    int argsize;
    char*buf;
    fgObj*obj;
    fgData data;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    cstrSet(buf,"",exit(-1));
    for(i=0;i<argsize;i++){
       data=fgGetArryNum(fhdl,arg,i);
       data=fgSysCast(fhdl,data,fgDataTypeStr);
       cstrCat(buf,data.elm.s,exit(-1));
       cstrCat(buf," ",exit(-1));
    }
    /*displaying value*/
    flg=puts(buf);
    free(buf);
    /*return*/
    *(obj->valp)=fgSetValTypeInt(flg);
    return fgSetValTypeObj(obj->id);
}
//===============================================================================
//read
//2006/10/07 speed up(by using static pointer).
//===============================================================================
/*function define*/
static int fgStdObjReadData(fgHandle fhdl,int fd,fgData* data);
//str
//2006/10/12 bugfix(It was be waiting a data forever).
//2006/10/13 bugfix(when fd was not open).
static int fgStdObjReadDataDef(fgHandle fhdl,int fd,fgData* data){
    fd_set fds;
    int size=0;
    int i,flg=0;
    int bufsize=FG_BUFSIZE*4;
    if(data->type !=fgDataTypeStr){
        data->type=fgDataTypeStr;
        data->elm.s=NULL;
    }
    FD_ZERO(&fds);
    FD_SET(fd,&fds);
    while(!(select(fd+1,&fds,NULL,NULL,NULL)<0))
        if(FD_ISSET(fd,&fds)) break;
    do{
        size+=bufsize;
        cstrRemakeBuf(data->elm.s,size,char,exit(-1));
        flg=read(fd,(data->elm.s+size-bufsize),sizeof(char)*bufsize);
        (data->elm.s+size-bufsize)[flg]=CIO_ES;
    }while(flg==sizeof(char)*bufsize);
    if(flg<0) *data=fgSetValTypeInt(flg);
    return flg;
}
//obj
static int fgStdObjReadDataObj(fgHandle fhdl,int fd,fgData* data){
    int flg;
    fgObj*obj;
    obj=fgGetObjLow(fhdl,data->elm.i);
    flg=fgStdObjReadData(fhdl,fd,obj->valp);
    return flg;
}
//getdata
static int fgStdObjReadData(fgHandle fhdl,int fd,fgData* data){
    static int (*func[])(fgHandle,int,fgData*)={
        fgStdObjReadDataDef, fgStdObjReadDataDef,
        fgStdObjReadDataDef, fgStdObjReadDataObj,
    };
    return func[data->type](fhdl,fd,data); 
}
//method
fgData fgStdObjRead(fgHandle fhdl,int self){
    int i,fd,arg,flg;
    int argsize;
    fgObj*obj;
    fgData data;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    if(argsize<2){
        puts("read <fd> <outvar>");
        *(obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    //getting arg 0(file const)
    data=fgGetArryNum(fhdl,arg,0);
    fd=fgGetValInt(fgSysCast(fhdl,data,fgDataTypeInt));
    if(fd<0){
        data=fgGetArryNum(fhdl,arg,1);
        obj=fgGetObjLow(fhdl,fgGetValObj(data));
        *(obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    //read data
    data=fgGetArryNum(fhdl,arg,1);
    flg|=fgStdObjReadData(fhdl,fd,&data);
    //return
    *(obj->valp)=fgSetValTypeInt(flg);
    return fgSetValTypeObj(obj->id);
}
//===============================================================================
//readln
//2006/10/07 speed up(by using static pointer).
//===============================================================================
/*function define*/
static int fgStdObjReadlnData(fgHandle fhdl,int fd,fgData* data);
//int
static int fgStdObjReadlnDataInt(fgHandle fhdl,int fd,fgData* data){
    return read(fd,&(data->elm.i),sizeof(int));
}
//flt
static int fgStdObjReadlnDataFlt(fgHandle fhdl,int fd,fgData* data){
    return read(fd,&(data->elm.f),sizeof(double));
}
//str
//2006/10/07 bugfix(operation when reading end of file).
static int fgStdObjReadlnDataStr(fgHandle fhdl,int fd,fgData* data){
    int i,flg=0;
    int bufsize=FG_BUFSIZE;
    cstrRemakeBuf(data->elm.s,bufsize,char,exit(-1));
    for(i=0;i<(bufsize-1);i++){
        flg|=read(fd,&(data->elm.s[i]),sizeof(char));
        if(flg<1) break;
        if(data->elm.s[i]==CIO_EL){i++;break;}
        if(data->elm.s[i]==CIO_ES){i++;break;}
        if(i>=(bufsize-2)){
            bufsize*=2;
            cstrRemakeBuf(data->elm.s,bufsize,char,exit(-1));
        }
    }
    data->elm.s[i]=CIO_ES;
    return flg;
}
//obj
static int fgStdObjReadlnDataObj(fgHandle fhdl,int fd,fgData* data){
    int flg;
    fgObj*obj;
    obj=fgGetObjLow(fhdl,data->elm.i);
    flg=fgStdObjReadlnData(fhdl,fd,obj->valp);
    return flg;
}
//getdata
static int fgStdObjReadlnData(fgHandle fhdl,int fd,fgData* data){
    static int (*func[])(fgHandle,int,fgData*)={
        fgStdObjReadlnDataInt, fgStdObjReadlnDataFlt,
        fgStdObjReadlnDataStr, fgStdObjReadlnDataObj,
    };
    return func[data->type](fhdl,fd,data); 
}
//method
fgData fgStdObjReadln(fgHandle fhdl,int self){
    int i,fd,arg,flg;
    int argsize;
    fgObj*obj;
    fgData data;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    if(argsize<2){
        puts("readln <fd> <outvar>");
        *(obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    //getting arg 0(file const)
    data=fgGetArryNum(fhdl,arg,0);
    fd=fgGetValInt(fgSysCast(fhdl,data,fgDataTypeInt));
    if(fd<0){
        data=fgGetArryNum(fhdl,arg,1);
        obj=fgGetObjLow(fhdl,fgGetValObj(data));
        *(obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    //read data
    data=fgGetArryNum(fhdl,arg,1);
    flg|=fgStdObjReadlnData(fhdl,fd,&data);
    *(obj->valp)=fgSetValTypeInt(flg);
    return fgSetValTypeObj(obj->id);
}
//===============================================================================
//write
//2006/10/07 speed up(by using static pointer).
//2006/10/07 bugfix(when writing binary data).
//===============================================================================
/*function define*/
static int fgStdObjWriteData(fgHandle fhdl,int fd,fgData data);
//int
static int fgStdObjWriteDataInt(fgHandle fhdl,int fd,fgData data){
    return write(fd,&data.elm.i,sizeof(int));
}
//flt
static int fgStdObjWriteDataFlt(fgHandle fhdl,int fd,fgData data){
    return write(fd,&data.elm.f,sizeof(double));
}
//str
static int fgStdObjWriteDataStr(fgHandle fhdl,int fd,fgData data){
    int flg;
    char*buf;
    cstrSet(buf,data.elm.s,exit(-1));
    flg= write(fd,buf,(sizeof(char)*strlen(buf)));
    free(buf);
    return flg;
}
//obj
static int fgStdObjWriteDataObj(fgHandle fhdl,int fd,fgData data){
    fgObj*obj=fgGetObjLow(fhdl,fgGetValObj(data));
    return fgStdObjWriteData(fhdl,fd,*(obj->valp));
}
//getdata
static int fgStdObjWriteData(fgHandle fhdl,int fd,fgData data){
    static int (*func[])(fgHandle,int,fgData)={
        fgStdObjWriteDataInt, fgStdObjWriteDataFlt,
        fgStdObjWriteDataStr, fgStdObjWriteDataObj,
    };
    return func[data.type](fhdl,fd,data); 
}
//method
//2006/10/13 bugfix(when fd was not open).
fgData fgStdObjWrite(fgHandle fhdl,int self){
    int i,fd,arg,flg;
    int argsize;
    fgObj*obj;
    fgData data;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    if(argsize<2){
        puts("write <fd> <value> ...");
        *(obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    //getting arg 0(file const)
    data=fgGetArryNum(fhdl,arg,0);
    fd=fgGetValInt(fgSysCast(fhdl,data,fgDataTypeInt));
    if(fd<0){
        *(obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    //write data
    for(i=1,flg=0;i<argsize;i++){
        data=fgGetArryNum(fhdl,arg,i);
        flg|=fgStdObjWriteData(fhdl,fd,data);
    }
    *(obj->valp)=fgSetValTypeInt(flg);
    return fgSetValTypeObj(self);
}
//===============================================================================
//close
//2006/10/07 speed up(by using static pointer).
//===============================================================================
fgData fgStdObjClose(fgHandle fhdl,int self){
    int flg;
    int arg;
    int argsize;
    fgObj*obj;
    fgData data;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    argsize=fgSysGetSize(fhdl,arg);
    /*checking args*/
    if(!argsize){
        puts("close <fd>");
        *(obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    data=fgGetArryNum(fhdl,arg,0);
    flg=fgGetValInt(fgSysCast(fhdl,data,fgDataTypeInt));
    *(obj->valp)=fgSetValTypeInt(close(flg));
    return fgSetValTypeObj(obj->id);
}
//===============================================================================
//open
//2006/10/07 speed up(by using static pointer).
//2006/10/14 bugfix(when read/write mode).
//2006/10/20 chmod(644) is added
//2006/10/14 bugfix(when write mode).
//===============================================================================
fgData fgStdObjOpen(fgHandle fhdl,int self){
    int fd,arg;
    int argsize;
    int flg,outvar;
    char*name,*s;
    fgData data;
    fgObj*obj;
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    /*output variable*/
    if(argsize>2) outvar=fgGetValObj(fgGetArryNum(fhdl,arg,2));
    else outvar=obj->id;
    /*checking args*/
    if(!argsize){
        puts("open <filename> [mode] [outvar]");
        *(obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    data=fgGetArryNum(fhdl,arg,0);
    cstrSet(name,fgSysCast(fhdl,data,fgDataTypeStr).elm.s,exit(-1));
    //gett arg 1(open flag)
    if(argsize>1){
        int i;
        char*tmps;
        data=fgGetArryNum(fhdl,arg,1);
        tmps=fgSysCast(fhdl,data,fgDataTypeStr).elm.s;
        for(i=0,flg=0;tmps[i]!=CIO_ES;i++){
            switch(tmps[i]){
            case 'w' : 
                flg|=O_WRONLY; 
                flg|=O_CREAT; 
                flg|=O_TRUNC; 
                break;
            case 'a' : flg|=O_APPEND; break;
            case 'r' : flg|=O_RDONLY; break;
            case 'b' : flg|=O_BINARY; break;
            }
        }
    }else flg=O_RDWR;
    if(flg==0) flg=O_RDWR;
    if((flg&O_RDONLY)&&(flg&O_WRONLY)){
        flg&=!(O_RDONLY);
        flg&=!(O_WRONLY);
        flg|=O_RDWR;
    }
    //open
    for(s=name;isspace(*s);s++) if(*s==CIO_ES) break;
    obj=fgGetObjLow(fhdl,outvar);
    fd=open(s,flg);
    if(fd<0){
        if((flg&O_WRONLY)||(flg&O_APPEND)){
            fd=open(s,flg|O_CREAT);
            fchmod(fd,S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
        }
    }
    (*obj->valp)=fgSetValTypeInt(fd);
    free(name);
    return fgSetValTypeObj(obj->id);
}
//===============================================================================
//talk
//2006/10/14 added.
//===============================================================================
//waiting I/O
//2006/10/18 bugfix
int fgStdObjTalkWaitIO(pid_t pid){
    FILE*pf;
    int i,flg;
    static char path[FG_BUFSIZE];
    static char buf[FG_BUFSIZE];
    if(pid<=0) return 0;
    sprintf(path,"/proc/%d/stat",pid);
    do{
        pf=fopen(path,"r");
        if(!pf) return -1;
        fgets(buf,FG_BUFSIZE,pf);
        for(i=0;buf[i]!=')';i++);
        while(isspace(buf[++i]));
        fclose(pf);
    }while((buf[i]!='S')&&(buf[i]!='D')&&(buf[i]!='Z'));
    return pid;
}
//method
fgData fgStdObjTalk(fgHandle fhdl,int self){
    fd_set fds;
    fgData data;
    fgObj*obj,*tmpobj;
    int fd,outvar,flag;
    int arg,argsize,pid;
    static char buf[FG_BUFSIZE];
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    /*checking args*/
    if(argsize<3){
        puts("talk <fd> <inbuf> <outbuf>");
        *(obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    /*getting fd and process id*/
    data=fgGetArryNum(fhdl,arg,0);
    fd=fgGetValInt(fgSysCast(fhdl,data,fgDataTypeInt));
    tmpobj=fgMakeObjLow(fhdl,fgDefPid,fgGetValObj(data));
    data=fgSysCast(fhdl,cioVoid((tmpobj->valp),fgData),fgDataTypeInt);
    pid=fgGetValInt(data);
    if(fd<0){
        data=fgGetArryNum(fhdl,arg,2);
        obj=fgGetObjLow(fhdl,fgGetValObj(data));
        *(obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    /*write*/
    fgStdObjTalkWaitIO(pid);
    FD_ZERO(&fds);
    FD_SET(fd,&fds);
    while(!(select(fd+1,NULL,&fds,NULL,NULL)<0))
        if(FD_ISSET(fd,&fds)) break;
    fsync(fd);
    data=fgGetArryNum(fhdl,arg,1);
    flag=fgStdObjWriteData(fhdl,fd,data);
    if(flag<0){
        data=fgGetArryNum(fhdl,arg,2);
        obj=fgGetObjLow(fhdl,fgGetValObj(data));
        *(obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    fsync(fd);
    /*read*/
    FD_ZERO(&fds);
    FD_SET(fd,&fds);
    while(!(select(fd+1,&fds,NULL,NULL,NULL)<0))
        if(FD_ISSET(fd,&fds)) break;
    fgStdObjTalkWaitIO(pid);
    fsync(fd);
    data=fgGetArryNum(fhdl,arg,2);
    obj=fgGetObjLow(fhdl,fgGetValObj(data));
    flag=fgStdObjReadData(fhdl,fd,&data);
    if(flag<0) *(obj->valp)=fgSetValTypeInt(-1);
    /*return*/
    return fgSetValTypeObj(obj->id);
}
//===============================================================================
//call
//2006/10/12 added.
//===============================================================================
//define
enum __fgStdObjCallType{
    __fgStdObjCallType_LOCAL=0,
    __fgStdObjCallType_TCP,
    __fgStdObjCallType_UDP,
};
//getting type from command
static int fgStdObjCallGetTypeFromCom(fgHandle fhdl,char*com){
    return __fgStdObjCallType_LOCAL;
}
//getting type
static int fgStdObjCallGetType(fgHandle fhdl,fgData val){
    val=fgSysCast(fhdl,val,fgDataTypeStr);
    if(strstr(fgGetValStr(val),"local"))
        return __fgStdObjCallType_LOCAL;
    if(strstr(fgGetValStr(val),"tcp"))
        return __fgStdObjCallType_TCP;
    if(strstr(fgGetValStr(val),"udp"))
        return __fgStdObjCallType_UDP;
    return __fgStdObjCallType_LOCAL;
}
//local method
static int fgStdObjCallLocal(char*com,int*pid){
    int sv[2];
    int fflag;
    /*create socket*/
    if(socketpair(AF_UNIX,SOCK_STREAM,0,sv)<0){
        perror("socket");
        return -1;
    }
    fcntl(sv[1],F_SETFL,O_RSYNC|O_SYNC);
    /*fork*/
    fflag=fork();
    if(fflag<0){
        perror("fork");
        return -1;
    }
    /*parent*/
    if(fflag){
        close(sv[1]);
        *pid=fflag;
        return sv[0];
    }
    /*son*/
    else{
        char**args;
        close(sv[0]);
        setbuf(stdout,NULL);
        dup2(sv[1],fileno(stdin));
        dup2(sv[1],fileno(stdout));
        dup2(sv[1],fileno(stderr));
        close(sv[1]);
        args=cstrSplit(com," \t\n");
        execvp(args[0],args);
        perror(args[0]);
        exit(-1);
    }
    return -1;
}
//tcp method
static int fgStdObjCallTcp(char*com,int*sv){
    return -1;
}
//udp method
static int fgStdObjCallUdp(char*com,int*sv){
    return -1;
}
//main method
fgData fgStdObjCall(fgHandle fhdl,int self){
    char*com;
    fgData data;
    fgObj*obj,*pobj;
    int outvar,type;
    int arg,argsize,flg;
    int pid=-1,timeout=-1;
    static int (*meth[])(char*,int*)={
        (*fgStdObjCallLocal),
        (*fgStdObjCallTcp),
        (*fgStdObjCallUdp),
    };
    obj=fgGetObjLow(fhdl,self);
    arg=fgMakeObj(fhdl,fgDefArg,obj->id);
    //getting argsize
    argsize=fgSysGetSize(fhdl,arg);
    /*checking args*/
    if(argsize<1){
        puts("call <command> [outvar] [timeout]");
        *(obj->valp)=fgSetValTypeInt(-1);
        return fgSetValTypeObj(obj->id);
    }
    /*setting outvar*/
    if(argsize<2)outvar=obj->id;
    else{
        outvar=fgGetValObj(fgGetArryNum(fhdl,arg,1));
        obj=fgGetObjLow(fhdl,outvar);
    }
    /*getting command*/
    data=fgGetArryNum(fhdl,arg,0);
    com=fgGetValStr(fgSysCast(fhdl,data,fgDataTypeStr));
    /*getting type*/
    #if 0
    if(argsize>3) type=fgStdObjCallGetType(fhdl,fgGetArryNum(fhdl,arg,3)); 
    else type=fgStdObjCallGetTypeFromCom(fhdl,com); 
    #else
    type=__fgStdObjCallType_LOCAL;
    #endif
    /*calling a process*/
    flg=meth[type](com,&pid);
    /*setting timeout*/
    if(argsize>2){
        data=fgGetArryNum(fhdl,arg,2);
        timeout=fgGetValInt(fgSysCast(fhdl,data,fgDataTypeStr));
    }
    if(flg>=0){
        struct timeval to={timeout};
        setsockopt(flg,SOL_SOCKET,SO_RCVTIMEO,&to,sizeof(struct timeval));
    }
    /*setting pid*/
    pobj=fgMakeObjLow(fhdl,fgDefPid,obj->id);
    *(obj->valp)=fgSetValTypeInt(flg);
    *(pobj->valp)=fgSetValTypeInt(pid);
    return fgSetValTypeObj(obj->id);
}

