<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.parameters
 */
/**
 * @file ArrayParameter.php
 * @brief ArrayParameter for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: ArrayParameter.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * ComplexParameter with Validator.
 */

require_once("ficus/lang/Assert.php");
require_once("ficus/parameters/ValidatableComplexParameter.php");
require_once("ficus/exception/IllegalTypeException.php");
require_once("ficus/exception/IllegalArgumentException.php");

/**
 * @class Ficus_ArrayParameter
 */
class Ficus_ArrayParameter extends Ficus_ValidatableComplexParameter
                          implements Iterator
{
	const TEMPLATE = "template";
	const ITEM = "item";
	const PATTERN = '/^([a-zA-Z0-9_:.]+)\[\]$/';
	/**
	 * type of this array contents
	 */
	private $arrayType = null;

	/**
	 * value parameters
	 */
	private $arrayValues = array();

	/**
	 * construct validiatable Parameter.
	 * @param $name string name
	 * @param $typename string name of type
	 * @param $value mixed value of this param
	 * @param $validator Ficus_Validator validator for validate parameter
	 */
	function __construct($name, $typename,
						 $value = null, $validator = array()){
		parent::__construct($name, $typename, $value);
		$this->setArrayType($typename);
		$this->setValidator($validator);
	}

	/**
	 * set array type
	 * @param $typename string type of Parameter
     * @throw Ficus_IllegalTypeException illegal type.
	 */
	private function setArrayType($typename){
		if(preg_match(self::PATTERN, $typename, $matches) === false){
			throw new Ficus_IllegalTypeException("{$matches[1]} is not exist");
		}
		$this->arrayType = $matches[1];
	}

	/**
	 * set Value
	 * @param $parameter Ficus_Parameter must be same type as $this->arrayType
	 */
	public function setValue($parameter){
		if(is_null($parameter)){
			return;
		}
		$this->append($parameter);
	}

	/**
	 * return value parameters
	 * @return array array of Ficus_ComplexParameter
	 */
	public function value(){
		return $this->arrayValues;
	}

	/**
	 * return index parameter
	 * @param $index integer index
     * @throw Ficus_IllegalArgumentException illegal index.
	 */
	public function at($index){
		if(count($this->arrayValues) > $index && $index >= 0){
			return $this->arrayValues[$index];
		}else{
			throw new Ficus_IllegalArgumentException("index $index is illegal index");
		}
	}

	/**
	 * return array count
	 * @return integer index length
	 */
	public function length(){
		return count($this->arrayValues);
	}
	
	/**
	 * append parameter
	 * @param $parameter Ficus_Parameter type of this parameter
     * @throw Ficus_IllegalTypeException no array.
	 */
	public function append($parameter){
		Ficus_Assert::TypeHinting("Ficus_Parameter", $parameter);
		if($parameter->typeName() == $this->arrayType){
			if(parent::getParameters()){
				$parameter->setName(self::ITEM);
				array_push($this->arrayValues, clone $parameter);
			}else{
				$parameter->setName(self::TEMPLATE);
				parent::append($parameter);
			}
		}else{
			throw new Ficus_IllegalTypeException("method parameter must be same type as {$this->arrayType}.");
		}
	}

	/**
	 * iteration function reset array pointer
	 */
	public function rewind() {
		reset($this->arrayValues);
	}

	/**
	 * iteration function returns current value
	 */
	public function current() {
		$var = current($this->arrayValues);
		return $var;
	}

	/**
	 * iteration function returns key of current array pointer
	 */
	public function key() {
	    $var = key($this->arrayValues);
		return $var;
	}

	/**
	 * iteration function get next array value
	 */
	public function next() {
		$var = next($this->arrayValues);
		return $var;
	}

	/**
	 * iteration function validate array pointer
	 */
	public function valid() {
		return (is_null(key($this->arrayValues)) === false);
	}

	/**
	 * clone method
	 */
	public function __clone(){
		$parameters = array();
		$arrayValues = array();
		foreach($this->parameters as $parameter){
			array_push($parameters, clone $parameter);
		}
		foreach($this->arrayValues as $value){
			array_push($arrayValues, clone $value);
		}
		$this->parameters = $parameters;
		$this->arrayValues = $arrayValues;
	}
}
?>
