<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.net
 */
/**
 * @file URIBuilder.php
 * @brief URIBuilder
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: URIBuilder.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * URIBuilder
 */
require_once("ficus/net/URIFactoryComponentFactory.php");
require_once("ficus/lang/Types.php");
require_once("ficus/exception/URISyntaxException.php");
/**
 * @class Ficus_URIBuilder
 */
class Ficus_URIBuilder {
    /**
     * uri patten
     */
    const URI_PATTERN = '/^(?:([a-zA-Z]|[a-zA-Z][a-zA-Z0-9\\+\\-\\.]*):)?([^#]*)(#(?:[;\\/\\?:@&=\\+\\$,]|[a-zA-Z0-9]|[\\-_\\.!~\\*\'\\(\\)])*)?/';

    /**
     * uri pattern that extractable to local part and namespace
     */
    const EXTRACTABLE_URI_PATTERN = '/[a-zA-Z][a-zA-Z0-9\+\-\.]*:.*[#\/:].*$/';
    
    /**
     * Build URI factory method.
     *
     * @param $uri string URI string.
     * @return mixed Ficus_URI or Ficus_URL or Ficus_URN.
     * @throw Ficus_URISyntaxException invalid URI.
     */
    public static function create($uri) {
        return self::build($uri);
    }

    /**
     * Create URI factory method.
     *
     * @param $uri string URI string.
     * @return mixed Ficus_URI or Ficus_URL or Ficus_URN.
     * @throw Ficus_URISyntaxException invalid URI.
     */
    public static function build($uri) {
        $uri = Ficus_Types::toStringOf($uri);
        if (preg_match(self::URI_PATTERN, $uri, $matches) === 0) {
            throw new Ficus_URISyntaxException("Syntax error: '$uri'");
        }
        $scheme = $matches[1];
        $specific = $matches[2];
        $fragment = isset($matches[3]) ? $matches[3] : null;

        if ($scheme == '') {
            // relative URI.
            return new Ficus_HierarchicalURI($uri);
        }

        try {
            $builder = Ficus_URIFactoryComponentFactory::getBuilderComponent($scheme);
            return $builder->createInstance($scheme, $uri);
        } catch (S2Container_ComponentNotFoundRuntimeException $e) {
            if ($specific{0} == '/' && $specific{1} == '/') {
                // unknown URL
                return new Ficus_URL($uri);
            } else if ($specific{0} == '/') {
                // unknown hierarchical URI
                return new Ficus_HierarchicalURI($uri);
            } else {
                // unknown URI
                return new Ficus_OpaqueURI($uri);
            }
        } catch (Ficus_ClassNotFoundException $e) {
            return null;
        } catch (Exception $e) {
            // TODO check out that S2Container's exception
            throw $e;
        }

    }

    /**
     * Create by parts.
     *
     * @param $scheme string scheme.
     * @param $authority string authority.
     * @param $path string path.
     * @param $query string query.
     * @param $fragment string fragment.
     * @return Ficus_URI URI.
     */
    public static function createByParts($scheme, $authority, $path,
                                         $query, $fragment) {
        $uri = self::createURIStringByParts($scheme, $authority, $path,
                                            $query, $fragment);
        return self::create($uri);
    }

    /**
     * Create URI string by parts.
     *
     * @param $scheme string scheme.
     * @param $authority string authority.
     * @param $path string path.
     * @param $query string query.
     * @param $fragment string fragment.
     * @return string URI.
     */
    public static function createURIStringByParts($scheme, $authority, $path,
                                         $query, $fragment) {
        $uri = '';
        $uri .= ($scheme) ? $scheme . ':' : '';
        $uri .= ($authority) ? '//' . $authority : '';
        $uri .= ($path) ? $path : '';
        $uri .= ($query) ? '?' . $query : '';
        $uri .= ($fragment) ? '#' . $fragment : '';
        return $uri;
    }

    /**
     * Check passed string is URI or URN and it is extractable.
     * @param $string string actual string
     * @return boolean true if is URI or URN
     */
    public static function isExtractableURI($string){
        return (preg_match(self::EXTRACTABLE_URI_PATTERN, $string) != false);
    }
}
?>
