<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file PHPVersion.php
 * @brief This class is the wrapper for phpversion
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: PHPVersion.php 2 2007-07-11 10:37:48Z ishitoya $
 */
require_once "ficus/lang/Assert.php";
require_once "ficus/exception/IllegalArgumentException.php";
/**
 * @class Ficus_PHPVersion
 */
class Ficus_PHPVersion
{
    const VERSION_PATTERN = '/(([0-9]+)\.([0-9]+)\.([0-9]+))-?(.*)/';
    const WHOLE = "whole";
    const MAJOR = "major";
    const MINOR = "minor";
    const RELEASE = "release";
    const INFO = "info";
    /** @var string phpversion information*/
    private static $version = null;

    /** @var array version array*/
    private static $versionArray = null;
    
    /**
     * constructor
     */
    public function __construct(){
        $this->initialize();
    }

    /**
     * is newer than param
     * @param $version string
     * @return boolean if current version newer than the version, return true
     */
    public function isNewerThan($version){
        self::initialize();
        return version_compare(self::$version, $version, ">");
    }

    /**
     * is older than param version
     * @param $version string
     * @return boolean if current version older than the version, return true
     */
    public function isOlderThan($version){
        self::initialize();
        return version_compare(self::$version, $version, "<");
    }

    /**
     * is same version of
     * @param $version string
     * @return boolean if current version same version, return true
     */
    public function isSame($version){
        self::initialize();
        return version_compare(self::$version, $version, "eq");
    }

    /**
     * is the version between $version1 and $version2
     * @param $version1 string
     * @param $version2 string
     * @return boolean return true if current version between 1 and 2.
     */
    public function isBetween($version1, $version2){
        self::initialize();
        $version1 = self::createVersionArray($version1);
        $version2 = self::createVersionArray($version2);
        $result1 = version_compare(self::$versionArray[self::WHOLE],
                                   $version1[self::WHOLE], ">=");
        $result2 = version_compare(self::$versionArray[self::WHOLE],
                                   $version2[self::WHOLE], "<=");
        return ($result1 && $result2);
    }

    /**
     * getMajorVersion
     * @param $version string version string
     * @return string major version
     */
    public function getMajor($version = null){
        self::initialize();
        if(is_null($version)){
            return self::$versionArray[self::MAJOR];
        }else{
            $versionArray = self::createVersionArray($version);
            return $versionArray[self::MAJOR];
        }
    }
    
    /**
     * getMinorVersion
     * @param $version string version string
     * @return string minor version
     */
    public function getMinor($version = null){
        self::initialize();
        if(is_null($version)){
            return self::$versionArray[self::MINOR];
        }else{
            $versionArray = self::createVersionArray($version);
            return $versionArray[self::MINOR];
        }
    }
    
    /**
     * getReleaseVersion
     * @param $version string version string
     * @return string major version
     */
    public function getRelease($version = null){
        self::initialize();
        if(is_null($version)){
            return self::$versionArray[self::RELEASE];
        }else{
            $versionArray = self::createVersionArray($version);
            return $versionArray[self::RELEASE];
        }
    }

    /**
     * get Info
     * @param $version string version string
     * @return string major version
     */
    public function getInfo($version = null){
        self::initialize();
        if(is_null($version)){
            return self::$versionArray[self::INFO];
        }else{
            $versionArray = self::createVersionArray($version);
            return $versionArray[self::INFO];
        }
    }

    /**
     * getMajorVersion
     * @return string current version
     */
    public function getCurrentVersion(){
        self::initialize();
        return self::$version;
    }
    
    /**
     * initialize
     */
    private static function initialize(){
        if(is_null(self::$version)){
            self::$version = phpversion();
            self::$versionArray = self::createVersionArray(self::$version);
        }
    }   

    /**
     * create Array
     * @param $version string version string
     * @return array array of version
     * @throw Ficus_IllegalArgumentException invalid version.
     */
    private function createVersionArray($version){
        $array = array();
        if(preg_match(self::VERSION_PATTERN, $version, $matches)){
            $array[self::WHOLE] = $matches[1];
            $array[self::MAJOR] = $matches[2];
            $array[self::MINOR] = $matches[3];
            $array[self::RELEASE] = $matches[4];
            if(isset($matches[5])){
                $array[self::INFO] = $matches[5];
            }else{
                $array[self::INFO] = null;
            }
        }else{
            throw new Ficus_IllegalArgumentException("specified version number is not a PHP standardized version number. : $version");
        }
        return $array;
    }
}
?>
