<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus
 */
/**
 * @file FicusPathInitializer.php
 * @brief path initializer for Ficus
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: FicusPathInitializer.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * ficus path initilaizer for ficus
 * use this class to use ficus package.
 * usage example :
 * require_once("path/to/FicusPathInitialize.php");
 * Ficus_FicusPathInitializer::setUp();
 *
 * if you want to use logger or config or depended library class, use set*
 * function before use them.
 */

/**
 * @class Ficus_FicusPathInitializer
 */
class Ficus_FicusPathInitializer
{
    /**
     * @var $configDir string config dir for PropertyFileReader.
     */
    private static $configDir = "";

    /**
     * @var $loggerDir string logger dir for Logger
     */
    private static $loggerDir = "";

    /**
     * @var $libDir string library dir
     */
    private static $libDir = "";

    /**
     * @var $paths array path
     */
    private static $paths = array();

    /**
     * @var $initialized boolean true if initialized
     */
    private static $initialized = false;
    
    /**
     * set config dir
     * @param $configDir string config dir
     */
    public static function setConfigDirectory($configDir){
        self::$configDir = $configDir;
    }

    /**
     * set logger dir
     * @param $loggerDir string logger dir
     */
    public static function setLoggerDirectory($loggerDir){
        self::$loggerDir = $loggerDir;
    }

    /**
     * set library dir
     * @param $libDir string library dir
     */
    public static function setLibraryDirectory($libDir){
        self::$libDir = $libDir;
    }

    /**
     * add paths
     */
    public static function addPath($path, $name = null){
        if(self::$initialized){
            require_once("ficus/lang/Assert.php");
            Ficus_Assert::isType($path, "Ficus_ClassPathElement");
            Ficus_ClassPath::add($path, $name);
        }else{
            self::$paths[] = $path;
        }
    }

    /**
     * add autoload paths
     */
    public static function addAutoLoadPath($path, $prefix = ""){
        if(self::$initialized){
            require_once("ficus/lang/Assert.php");
            Ficus_Assert::isType($path, "Ficus_ClassPathElement");
            Ficus_AutoClassLoader::add($path, $prefix);
        }else{
            throw new Exception("before call add Auto load class, must call setup method.");
        }            
    }

    public static function addIncludePath($paths){
        if(is_string($paths)){
            $paths = array($paths);
        }
        $systemIncludePaths = explode(PATH_SEPARATOR, get_include_path());
        $includePaths = array_merge($paths, $systemIncludePaths);
                        
        set_include_path(implode(PATH_SEPARATOR, $includePaths));
    }
    
    /**
     * initialize. set include path
     */
    private static function initialize(){
        $includePaths = array();
        $includePaths[] = self::convertDirectorySeparator(self::$libDir);
        $includePaths = array_merge($includePaths, self::$paths);
        self::addIncludePath($includePaths);    
        self::$initialized = true;
    }

    /**
     * set up ficus.
     */
    public static function setUp() {
        self::initialize();
        require_once("ficus/lang/ClassPath.php");
        require_once("ficus/lang/ClassPathElement.php");
        require_once("ficus/io/Dir.php");
        if(empty(self::$loggerDir) == false ||
           empty(self::$configDir) == false){
            require_once("ficus/io/PropertyFileReader.php");
            Ficus_ClassPath::add(
                new Ficus_ClassPathElement(
                    new Ficus_Dir(self::$configDir)), "conf");
        }
        if(empty(self::$loggerDir) == false){
            require_once("ficus/logging/Logger.php");
            Ficus_Logger::initialize(self::$loggerDir);
        }
        require_once("ficus/lang/AutoLoad.php");
        require_once("ficus/lang/S2ContainerAutoLoad.php");
        require_once("ficus/lang/ClassLoaderAutoLoad.php");
        Ficus_AutoLoad::add(new Ficus_S2ContainerAutoLoad());
        Ficus_AutoLoad::add(new Ficus_ClassLoaderAutoLoad());

        $pattern = new Ficus_PathPattern();
        $pattern->addInclude("**ficus");
        $dir = Ficus_Dir::createInstance(__FILE__, "ficus");
        $dir->setPattern($pattern);
        $src = new Ficus_ClassPathElement($dir);
        Ficus_ClassPath::add($src, "ficus");

        require_once("ficus/lang/AutoClassLoader.php");
    }

    /**
     * teardown
     */
    public static function tearDown(){
        Ficus_ClassPath::clear();
    }

    /**
     * convert DIRECTORY_SEPARATOR
     *
     * @param $path string path.
     */
    private static function convertDirectorySeparator($path){
        $needle = (DIRECTORY_SEPARATOR == '/') ? '\\' : '\?';
        return str_replace($needle, DIRECTORY_SEPARATOR, $path);
    }
}
?>
