<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages.scaffold.organizer
 */
/**
 * @file AbstractScaffoldOrganizer.php
 * @brief AbstractScaffoldOrganizer.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: AbstractScaffoldOrganizer.php 17 2007-07-19 13:33:15Z ishitoya $
 *
 */

require_once("ficus/lang/Assert.php");

/**
 * @class Ficus_AbstractScaffoldOrganizer
 */
abstract class Ficus_AbstractScaffoldOrganizer
implements Ficus_ScaffoldOrganizer, Ficus_ScaffoldConstants,
    Ficus_S2DaoModelConstants{
    const TEMPLATE_SUFFIX = ".tpl";
    
    /**
     * itemtemplate
     */
    protected $itemTempalte = "";

    /**
     * entity template
     */
    protected $entityTemplate = "";

    /**
     * result
     */
    protected $result = "";

    /**
     * mediator
     */
    protected $mediator = null;

    /**
     * constructor
     * @param $mediator Ficus_ScaffoldMediator mediator
     */
    public function __construct($mediator){
        $this->mediator = $mediator;
    }
    
    /**
     * prefix
     * @return string template prefix
     */
    public function prefix(){
        return "";
    }

    /**
     * get template file name
     * @param $type string type of template
     * @return string template filename
     */
    public function template($type){
        return $this->prefix() . $type . self::TEMPLATE_SUFFIX;
    }
        
    /**
     * item template
     * @param $template string template
     */
    public function setItemTemplate($itemTemplate){
        $this->itemTemplate = $itemTemplate;
    }

    /**
     * entity template
     * @param $template string template
     */
    public function setEntityTemplate($entityTemplate){
        $this->entityTemplate = $entityTemplate;
    }

    /**
     * get mediator
     */
    public function mediator(){
        return $this->mediator;
    }
    
    /**
     * visit part
     * @param $part Ficus_ScaffoldPart part
     */
    public function visit($part){
        Ficus_Assert::isInstanceOf($part, "Ficus_ScaffoldPart");
        $this->mediator->smarty()->clear("value");
        $this->mediator->smarty()->clear("qname");
        $this->mediator->smarty()->clear("nolabel");
        $result = $this->onVisit($part);
        $this->addResult($result);
    }

    /**
     * start visit
     * @param $container Ficus_ScaffoldPartsContainer
     */
    public function startVisit($container){
        Ficus_Assert::isInstanceOf($container, "Ficus_ScaffoldPartsContainer");
        $this->onStartVisit($container);
    }

    /**
     * end visit
     * @param $container Ficus_ScaffoldPartsContainer
     */
    public function endVisit($container){
        Ficus_Assert::isInstanceOf($container, "Ficus_ScaffoldPartsContainer");
        $this->onEndVisit($container);
        $this->mediator->smarty()->assign("result", $this->result);
        $this->result = $this->fetch($this->entityTemplate);
    }
    
    /**
     * result
     */
    public function result(){
        return $this->result;
    }

    /**
     * on visit
     * @param $part Ficus_ScaffoldPart part
     */
    abstract protected function onVisit($part);

    /**
     * on start visit
     * @param $container Ficus_ScaffoldPartsContainer
     */
    protected function onStartVisit($container){
        //do nothing
    }

    /**
     * on end visit
     * @param $container Ficus_ScaffoldPartsContainer
     */
    protected function onEndVisit($container){
        //do nothing
    }   
        
    /**
     * add result
     */
    protected function addResult($result){
        $this->result .= $result;
    }

    /**
     * assign
     * @param $name name of variable
     * @param $var mixed variable
     */
    protected function assign($name, $var){
        $this->mediator->smarty()->assign($name, $var);
    }
    
    /**
     * fetch
     * @param $template string template
     * @return string result
     */
    protected function fetch($type){
        return $this->mediator->smarty()->fetch($this->template($type));
    }

    /**
     * display
     * @param $template string template
     */
    protected function display($type){
        $this->mediator->smarty()->fetch($this->display($type));
    }

    /**
     * clear
     * @param $name string name of property
     */
    protected function clear($name){
        $this->mediator->smarty()->clear($name);
    }
}
?>