<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages
 */
/**
 * @file Pages.php
 * @brief abstract page
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: Page.php 19 2007-07-25 18:20:09Z ishitoya $
 */

require_once("ficus/pages/PageConstants.php");
require_once("ficus/pages/PageComponentFactory.php");
require_once("ficus/exception/PropertyNotFoundException.php");
require_once("ficus/exception/MethodNotFoundException.php");

/**
 * @class Ficus_Page
 */
abstract class Ficus_Page extends Ficus_Bean implements Ficus_PageConstants{
    /**
     * @var $package string package
     */
    protected $package = null;

    /**
     * @var $pagename string pagename
     */
    protected $pagename = null;

    /**
     * @var $request Ficus_PageRequest
     */
    protected $request = null;

    /**
     * @var $currentAction string action
     */
    protected $currentAction = null;

    /**
     * @var $nextAction string next action
     */
    protected $nextAction = null;
    
    /**
     * constructor
     * @param $package string package name
     * @param $pagename string pagename
     */
    public function __construct($package, $pagename){
        $this->package = $package;
        $this->pagename = $pagename;
        $this->onConstruct();
    }

    /**
     * check for method existance
     */
    protected function methodExists($method){
        return method_exists($this, $method);
    }

    /**
     * caller method
     * @param $name string name of method
     * @param $args array array of parameter
     * @return mixed method result
     */
    public function __call($name, $args){
        $method = "do" . ucfirst($name);
        if($this->methodExists($method) == false){
            return parent::__call($name, $args);
        }
        
        $this->currentAction = $name;
        try{
            $this->assign(self::CURRENT_PAGENAME, $this->pagename);
            $args = $args[0];
            Ficus_Assert::isInstanceOf($args, "Ficus_PageRequest");
            $this->request = $args;
            $this->onDo($name, $args);
            $result = call_user_func(array($this, $method), $args);
            $this->onDone($name, $args, $result);
        }catch(Exception $e){
            $this->assign(self::PAGE_EXCEPTION, $e);
            $this->onException($name, $args, $e);
            $result = "";
        }
        return $result;
    }

    /**
     * get default mode
     * @return string default mode name
     */
    abstract public function getDefaultMode();
    
    /**
     * on construct
     */
    abstract protected function onConstruct();

    /**
     * on exception
     * @param $mode string name of mode
     * @param $args array of args of do method
     * @param $exception Exception exception
     */
    abstract protected function onException($mode, $args, $exception);
    
    /**
     * on called
     * @param $mode string name of mode
     * @param $args array of args of do method
     */
    abstract protected function onDo($mode, $args);

    /**
     * on execute finished
     * @param $mode string name of mode
     * @param $args array of args of do method
     * @param $result mixed switch
     */
    protected function onDone($mode, $args, $result){
        if($result == self::NODISPLAY){
            return;
        }else if($result == self::MODE_TEMPLATE){
            $this->display($this->getModeTemplate($mode));
        }else if(is_string($result)){
            $this->display($result);
        }else if(is_array($result) &&
                 isset($result[self::PAGENAME])){
            if(isset($result[self::ACTIONNAME])){
                $this->forward($result[self::PAGENAME],
                               $result[self::ACTIONNAME]);
            }else{
                $this->forward($result[self::PAGENAME]);
            }
        }else{
            $this->display();
        }
    }

    /**
     * authorization
     */
    public function authorization(){
        $retval = $this->onAuthorization();
        if($retval !== self::FORWARDED &&
           $retval !== self::AUTHORIZATION_NOT_REQUIRED){
            throw new Ficus_InvalidImplementationException(get_class($this) . "::onAuthorization not valid. please check out it returns FORWARDED or AUTHORIZATION_NOT_REQUIRED.");
        }
        return $retval;
    }

    /**
     * on authorization
     */
    abstract protected function onAuthorization();

    /**
     * get modes
     */
    public function getModes(){
        $class = new ReflectionClass(get_class($this));
        $methods = $class->getMethods();

        foreach($methods as $method){
            if(preg_match('/^do(.+)$/', $method->getName(), $matches)){
                $mode = $matches[1];
                $mode = strtolower(substr($mode, 0, 1)) . substr($mode, 1);
                $modes[] = $mode;
            }
        }
        return $modes;
    }

    /**
     * forward
     * @param $pagename string target pagename
     * @param $actionname string actionname
     */
    protected function forward($pagename, $actionname = null, $buffering = false){
        if($pagename == $this->pagename()){
            return $this->pass($actionname);
        }
        $controller = Ficus_PageComponentFactory::getPageController();
        return $controller->execute($pagename, $actioname, $buffering);
    }

    /**
     * pass action in the page
     * @param $action name string actionname
     */
    protected function pass($actionname, $args = null){
        if(is_null($args)){
            return $this->{$actionname}($this->request);
        }
    }

    /**
     * move
     * @param $pagename string target pagename
     * @param $actionname string actionname
     */
    protected function move($pagename = null, $actionname = null, $time = 0){
        $self = $_SERVER["PHP_SELF"];
        if(preg_match('/(.*?)index.php$/', $self, $regs)){
            $self = $regs[1];
        }
        $location = "Location: ";
        if(is_null($pagename) && is_null($actionname)){
            $location .= $self;
        }else if(is_null($pagename) == false && is_null($actionname)){
            $location .= "$self?$pagename";
        }else if(is_null($pagename) && is_null($actionname) == false){
            $location .= "$self?$actionname";
        }else{
            $location .= "$self?$pagename&$actionname";
        }
        if($time > 0){
            sleep($time);
        }
        header($location);
    }
    
    /**
     * fetch template
     * @param $template string template name
     * @return string fetched string
     */
    protected function fetch($template = null){
        if(is_null($template)){
            $template =
                Ficus_PageComponentFactory::getTemplateResolver()->resolve($this);
        }
        return Ficus_PageComponentFactory::getSmarty()->fetch($template);
    }

    /**
     * display template
     * @param $template string template name
     * @return mixed return value of display
     */
    protected function display($template = null){
        if(is_null($template)){
            $template =
                Ficus_PageComponentFactory::getTemplateResolver()->resolve($this);
        }
        return Ficus_PageComponentFactory::getSmarty()->display($template);
    }

    /**
     * assign value to name
     * @param $name string name to assign
     * @param $value string value to assign
     * @return string return value of assign
     */
    protected function assign($name, $value){
        return  Ficus_PageComponentFactory::getSmarty()->assign($name, $value);
    }

    /**
     * assign reference  to name
     * @param $name string name to assign
     * @param $value string value to assign
     * @return string return value of assign
     */
    protected function assignReference($name, $value){
        return
            Ficus_PageComponentFactory::getSmarty()->assign_by_ref($name, $value);
    }
    
    /**
     * show error
     * @param $message string error message
     */
    public function error($message){
        $this->assign(self::PAGE_ERROR, $message);
    }

    /**
     * get property from registry
     * @param $name string name of key
     * @return string value
     */
    protected function property($name){
        $property = Ficus_Registry::search($name);
        if($property === false){
            throw new Ficus_PropertyNotFoundException("property $name is not registered in Ficus_Registry");
        }
        return $property;
    }

    /**
     * get env bean
     * @param $name string name of key
     * @return string value, if null specified in name then return whole values
     */
    protected function env($name = null){
        $env = Ficus_PageComponentFactory::getEnvBean()->values();
        if(is_null($name)){
            return $env;
        }
        if(isset($env[$name]) == false){
            throw new Ficus_PropertyNotFoundException("property $name is not found in _ENV and _SERVER");
        }
        return $env[$name];
    }

    /**
     * get visit bean
     * @return Ficus_PageVisitBean session information
     */
    protected function visit(){
        return Ficus_PageComponentFactory::getVisitBean();
    }

    /**
     * set next action
     * @param $action string mode name
     */
    protected function setNextAction($action){
        $this->assign("nextAction", $action);
        $this->nextAction = $action;
    }

    /**
     * get mode template
     * @param $mode string mode
     * @return string template name
     */
    protected function getModeTemplate($mode){
        return
            Ficus_PageComponentFactory::getTemplateResolver()->resolve($this,
                                                                       $mode);
    }
}
?>
