<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.analyzer
 */
/**
 * @file AnnotationChecker.php
 * @brief AnnotationChecker of PHP.
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: AnnotationChecker.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * Annotation checker of PHP.
 */
require_once('ficus/analyzer/AnalyzeData.php');
require_once('ficus/analyzer/AnnotationAnalyzeData.php');
require_once('ficus/analyzer/ClassAnalyzeData.php');
require_once('ficus/analyzer/FunctionAnalyzeData.php');
require_once('ficus/analyzer/ParameterAnalyzeData.php');
require_once('ficus/analyzer/ThrowAnalyzeData.php');
require_once('ficus/analyzer/SourceAnalyzer.php');
/**
 * @class Ficus_AnnotationChecker
 */
class Ficus_AnnotationChecker extends Ficus_SourceAnalyzer {

    /**
     * call each source.
     *
     * @param $classes classes in this source.
     * @see Ficus_SourceAnalyzer.
     */
    public function callSource($classes) {
        foreach ($classes as $class) {
            self::checkAnnotation($class);
        }
    }

    /**
     * check annotation.
     *
     * @param $class class.
     */
    protected function checkAnnotation($class) {
        self::checkClassAnnotation($class);
        foreach ($class->getFunctions() as $function) {
            self::checkFunctionAnnotation($class, $function);
        }
    }

    /**
     * check class annotation.
     *
     * @param $class class.
     */
    protected function checkClassAnnotation($class) {
        $ano = array();

        $atClass = '';
        foreach ($class->getAnnotations() as $annotation) {
            if ($annotation->getName() == '@class'
            || $annotation->getName() == '@interface') {
                $atClass = $annotation->getData();
            }
        }

        // no class
        if ($class->getName() == '* NO_CLASS *') {
            return;
        }

        if ($atClass == '') {
            echo "No declare annotation @class in 'class " . $class->getName() . "' at '" . $class->getPath() . "'\n";
        } else if ($class->getName() != $atClass) {
            echo "Bad annotation '@class " . $atClass . "' in 'class " . $class->getName() . "' at '" . $class->getPath() . "'\n";
        }
    }

    /**
     * Get annotation array.
     *
     * @param $annotations array of Ficus_AnnotattionAnalyzeData.
     * @param $type Annotation type.
     * @return array array of annotation strings.
     */
    protected function getAnnotationArray($annotations, $type) {
        $atParams = array();
        if (is_array($annotations[$type])) {
            foreach ($annotations[$type] as $paramData) {
                $split = preg_split('/ /', $paramData);
                $atParams[] = $split[0];
            }
        }
        return $atParams;
    }

    /**
     * Check function annotation.
     *
     * @param $class Ficus_ClassAnalyzeData.
     * @param $function Ficus_FunctionAnalyzeData function.
     */
    protected function checkFunctionAnnotation(Ficus_ClassAnalyzeData $class, Ficus_FunctionAnalyzeData $function) {

        $ano = array();
        foreach ($function->getAnnotations() as $annotation) {
            $ano[$annotation->getName()][] = $annotation->getData();
        }
        $params = array();
        foreach ($function->getParameters() as $parameter) {
            $params[] = $parameter[0]->getName();
        }
        $throws = array();
        foreach ($function->getThrows() as $throw) {
            $throws[] = $throw->getName();
        }

        $atParams = self::getAnnotationArray($ano, '@param');
        $atExceptions = self::getAnnotationArray($ano, '@exception');
        $atThrows = self::getAnnotationArray($ano, '@throw');
        $atThrows = array_merge($atExceptions, $atThrows);


        // no @params
        foreach (array_diff($params, $atParams) as $param) {
            echo "No declare annotation @param '" . $param . "' in '" . $class->getName() . "#" . $function->getName() . "()' at '" . $class->getPath() . "'\n";
        }
        // no using @params
        foreach (array_diff($atParams, $params) as $param) {
            if (strcasecmp($param, 'mixed') == 0 && sizeof($params) == 0) {
                // probably variable-length argument.
            } else {
                echo "No exist parameter '" . $param . "' in '" . $class->getName() . "#" . $function->getName() . "()' at '" . $class->getPath() . "'\n";
            }
        }
        // no @throw
        foreach (array_diff($throws, $atThrows) as $param) {
            echo "No declare annotation @throw '" . $param . "' in '" . $class->getName() . "#" . $function->getName() . "()' at '" . $class->getPath() . "'\n";
        }
        // no using @throw
        // possibility throw in deep function.
        // no check.

    }

}

/*
if ($argv[1] == null) {
    echo "need start path.";
    exit;
}
$analyze = new Ficus_AnnotationChecker();
$analyze->traverse($argv[1]);
*/

?>
