<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages.scaffold
 */
/**
 * @file ScaffoldManager.php
 * @brief index page of administrator site
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: ScaffoldManager.php 15 2007-07-18 15:02:48Z ishitoya $
 */

require_once("ficus/config/Registry.php");

/**
 * @class Ficus_ScaffoldManager
 */
class Ficus_ScaffoldManager extends Ficus_Bean
implements Ficus_ScaffoldConstants{
    /**
     * foreperson
     */
    protected $foreperson = null;

    /**
     * target
     */
    protected $target = null;
    
    /**
     * page
     */
    protected $page = null;

    /**
     * action
     */
    protected $action = null;

    /**
     * mediator
     */
    protected $mediator = null;

    /**
     * fuzzy
     */
    protected $fuzzy = false;

    /**
     * construct
     */
    public function __construct($page){
        $this->mediator = new Ficus_ScaffoldMediator();
        $this->guess($page);
    }

    /**
     * is guess able
     * @return boolean true if guessable
     */
    public function isGuessable(){
        if($this->page instanceof Ficus_ScaffoldPage &&
           $this->page->isEmptyTarget() == false){
            return true;
        }
            
        return $this->page->request()->has(self::KEY_TARGET);
    }
    
    /**
     * guess action
     * @param $args array of request
     */
    public function guess($page){
        $this->setPage($page);
        $request = $page->request();

        $form = new Ficus_ScaffoldFormBean();
        $form->setAction($page->nextAction());
        $form->setPage($page->pagename());
        $this->setFormFromRequest($form, "target", self::KEY_TARGET);
        $this->setFormFromRequest($form, "do",     self::KEY_DO);
        $this->setFormFromRequest($form, "method", self::KEY_METHOD);
        $this->setFormFromRequest($form, "message", self::KEY_MESSAGE);
        $this->setFormFromRequest($form, "transition", self::KEY_TRANSITION);
        $this->setFormFromRequest($form, "submitTitle",
                                  self::KEY_SUBMIT_TITLE);
        $this->setFormBean($form);

        if($this->isGuessable()){
            if($page instanceof Ficus_ScaffoldPage){
                $target = $page->target();
                $foreperson = $page->foreperson();
            }else{
                $target = $request->extractAsString(self::KEY_TARGET);
                $foreperson = $request->extractAsString(self::KEY_FOREPERSON);
            }
            $this->setTarget($target);
            $this->setAction($foreperson);
        }
    }

    /**
     * set bean data from request
     */
    private function setFormFromRequest($form, $property, $key){
        $value = $this->page->request()->extractAsString($key);
        if(is_null($value) == false){
            $form->set($property, $value);
        }
    }                       

    /**
     * set request
     */
    public function setPage($page){
        $this->page = $page;
    }
    
    /**
     * set Target
     * @param $target string target
     */
    public function setTarget($target){
        $this->initialize($target);
        $this->formBean()->setTarget($target);
    }

    /**
     * set Action
     */
    public function setAction($action){
        $this->isInitialized();
        $this->action = $action;
        $this->foreperson =
            $this->createForeperson($this->action);
        $this->formBean()->setForeperson($action);
    }

    /**
     * set form bean
     */
    public function setFormBean($bean){
        $this->mediator->setFormBean($bean);
    }

    /**
     * get form bean
     */
    public function formBean(){
        return $this->mediator->formBean();
    }

    /**
     * mediator
     */
    public function mediator(){
        return $this->mediator;
    }

    /**
     * organize a scaffold
     */
    public function organize(){
        $this->isInitialized();
        if(is_null($this->action)){
            throw new Ficus_NotReadyException("action is empty, you must run guess or setAction before runch organize method");
        }
        $scaffold = $this->foreperson->organize();
        $smarty = Ficus_ScaffoldComponentFactory::getSmarty();
        $form = $this->mediator->formBean();
        if(is_null($form) == false){
            $smarty->assign(self::KEY_PAGE,    $form->page());
            $smarty->assign(self::KEY_METHOD,  $form->method());
            $smarty->assign(self::KEY_ACTION,  $form->action());
            $smarty->assign(self::KEY_TARGET,  $form->target());
            $smarty->assign(self::KEY_DO,      $form->do());
            $smarty->assign(self::KEY_MESSAGE, $form->message());
            $smarty->assign(self::KEY_TRANSITION, $form->transition());
            $smarty->assign(self::KEY_FOREPERSON, $form->foreperson());
            $smarty->assign(self::KEY_SUBMIT_TITLE,
                            $form->submitTitle());
        }
        $smarty->assign(self::KEY_SCAFFOLD, $scaffold);
        $scaffold = $smarty->fetch($this->template($this->action));
        if($this->fuzzy){
            $header = $smarty->fetch(self::TEMPLATE_HEADER);
            $footer = $smarty->fetch(self::TEMPLATE_FOOTER);
            $scaffold = $header . $scaffold . $footer;
        }
        return $scaffold;
    }

    /**
     * set entity
     */
    public function setEntity($entity){
        $this->mediator->setEntity($entity);
    }

    /**
     * get deserialized entity
     */
    public function getEntityFromRequest(){
        return $this->mediator->getEntityFromRequest();
    }

    /**
     * fuzzy
     */
    public function setFuzzy($fuzzy){
        $this->fuzzy = $fuzzy;
    }
    
    /**
     * is initialized
     */
    protected final function isInitialized(){
        if(is_null($this->target)){
            throw new Ficus_NotReadyException("target is empty, you must run setTarget or guess before runch any public functions");
        }
    }

    /**
     * get template
     */
    protected function template($action){
        return "Scaffold" . ucfirst($action);
    }
    
    /**
     * initialize
     */
    protected function initialize($target){
        $this->target = $target;
        $this->mediator->setTarget($this->target);
        $this->mediator->setRequest($this->page->request());
    }
    
    /**
     * create foreperson
     * @param $action string action
     * @param $mediator Ficus_ScaffoldMediator mediator
     * @return UTil_ScaffoldForeperson
     */
    protected function createForeperson($action){
        switch($action){
        case(self::ACTION_VIEW):
            return new Ficus_ViewScaffoldForeperson($this->mediator); 
        case(self::ACTION_LIST):
            return new Ficus_ListScaffoldForeperson($this->mediator);
        case(self::ACTION_EDIT):
            return new Ficus_EditScaffoldForeperson($this->mediator);
        case(self::ACTION_NEW):
            return new Ficus_NewScaffoldForeperson($this->mediator);
        case(self::ACTION_DELETE):
            return new Ficus_DeleteScaffoldForeperson($this->mediator);
        default: throw new Ficus_NotSupportedException("scaffold foreperson $action is not available");
        }
    }
}
?>
