<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.net.mail
 */
/**
 * @file Mail.php
 * @brief Mail for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: Mail.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * Mail for php
 */

require_once("ficus/beans/Bean.php");
require_once("ficus/net/mail/MailAttachment.php");
require_once("Mail/Mail.php");
require_once("Mail/mimeDecode.php");

/**
 * @class Ficus_Mail
 */
class Ficus_Mail extends Ficus_Bean{
    const ENCODING_ISO_2022_JP = "iso-2022-jp";
    const ENCODING_UTF8 = "utf-8";

    /**
     * title
     */
    protected $subject = "";

    /**
     * to
     */
    protected $to = "";

    /**
     * from
     */
    protected $from = "";

    /**
     * date
     */
    protected $date = "";

    /**
     * charset
     */
    protected $charset = self::ENCODING_ISO_2022_JP;

    /**
     * attachments
     */
    protected $attachments = array();

    /**
     * body
     */
    protected $body = "";

    /**
     * encoding
     */
    protected $encoding = self::ENCODING_UTF8;

    /**
     * smtp host
     */
    protected $host = "localhost";

    /**
     * smtp port
     */
    protected $port = 25;

    /**
     * smtp auth
     */
    protected $auth = false;

    /**
     * smtp username
     */
    protected $username = "";

    /**
     * smtp password
     */
    protected $password = "";

    /**
     * send mail
     */
    public function send(){
        if($this->isEmptySubject()){
            throw new Ficus_NotReadyException("subject is not setted");
        }
        if($this->isEmptyBody()){
            throw new Ficus_NotReadyException("body is empty");
        }
        if($this->isEmptyTo()){
            throw new Ficus_NotReadyException("to address is empty");
        }
        if($this->isEmptyFrom()){
            throw new Ficus_NotReadyException("from address is empty");
        }

        $options = array("host" => $this->host,
                         "port" => $this->port,
                         "auth" => $this->auth,
                         "username" => $this->username,
                         "password" => $this->password,
                         "localhost" => "localhost");
                         
        $headers = array("From" => $this->from,
                         "To" => $this->to,
                         "Subject" => $this->getEncoded($this->subject));

        $mail = Mail::factory("SMTP", $options);
        $mail->send($this->to, $headers, $this->getEncoded($this->body));
    }
    
    /**
     * decode mail archive
     * @param $archive string mail archive
     */
    public function decode($archive){
        $params['include_bodies'] = true;
        $params['decode_bodies']  = true;
        $params['decode_headers'] = true;

        $decoder = new Mail_mimeDecode($archive);
        $mail = $decoder->decode($params);

        if(PEAR::isError($mail)){
            throw new Ficus_IllegalArgumentException($mail->getMessage());
        }
        
        if(isset($mail->ctype_parameters["boundary"])){
            $primary = $mail->parts[0];
            $charset = $primary->ctype_parameters["charset"];
            $this->body = $this->getDecoded($mail->parts[0]->body);
            for($i = 1; $i < count($mail->parts); $i++){
                $attachment = new Ficus_MailAttachment();
                $attachment->setCharset($charset);
                $attachment->decode($mail->parts[$i]);
                $attachment->setFilename($this->getDecoded($attachment->filename()));
                $this->attachments[] = $attachment;
            }
        }else{
            $this->charset = $mail->ctype_parameters["charset"];
            $this->body = $this->getDecoded($mail->body);
        }
        
        $this->subject = $this->getDecoded($mail->headers["subject"]);
        $this->to = $this->getDecoded($mail->headers["to"]);
        $this->from = $this->getDecoded($mail->headers["from"]);
        $this->date = $this->getDecoded($mail->headers["date"]);
    }

    /**
     * get decoded string
     * @param $target string target string to decode
     * @return string decoded string
     */
    protected function getDecoded($target){
        return mb_convert_encoding(trim($target),
                                   $this->encoding, $this->charset);
    }

    /**
     * get encoded string
     * @param $target string target string to encode
     * @return string encoded string
     */
    protected function getEncoded($target){
        return mb_convert_encoding(trim($target),
                                   $this->charset, $this->encoding);
    }

    /**
     * get attachment
     * @param $filename string filename
     * @return Ficus_MailAttachment attachment
     */
    public function getAttachment($filename){
        foreach($this->attachments as $attachment){
            if($attachment->filename() == $filename){
                return $attachment;
            }
        }
        throw new Ficus_IllegalArgumentException("$filename is not found");
    }   
}
?>
