<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.logging
 */
/**
 * @file Logger.php
 * @brief Logger for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: Logger.php 2 2007-07-11 10:37:48Z ishitoya $
 *
 * Util Ficus_Logger
 */

require_once("Log/Log.php");
require_once("ficus/lang/ClassPath.php");
require_once("ficus/io/PropertyFileReader.php");
require_once("ficus/exception/NotReadyException.php");

/**
 * @class Ficus_Logger
 */
class Ficus_Logger{
	//config file name
	const CONFIG_FILE = "logger";

    /**
     * log level
     * @see Log/Log.php
     */
    const LOG_EMERG = "ficus_logger_emerg";
    const LOG_ALERT = "ficus_logger_alert";
    const LOG_CRIT = "ficus_logger_crit";
    const LOG_ERR = "ficus_logger_err";
    const LOG_WARNING = "ficus_logger_warning";
    const LOG_NOTICE = "ficus_logger_notice";
    const LOG_INFO = "ficus_logger_info";
    const LOG_DEBUG = "ficus_logger_debug";
    const LOG_ALL = "ficus_logger_all";
    const LOG_NONE = "ficus_logger_none";

    
    /*
    const LOG_TYPE_SYSTEM = "system";
    const LOG_TYPE_MAIL = "mail";
    const LOG_TYPE_DEBUG ="debug";
    const LOG_TYPE_FILE = "file";
    */
	/**
	 * @var $log Log log object
	 */
	private static $log = null;

    /**
     * $handler
     */
    private static $handler = null;
    
    /**
     * $logLevel
     */
    private static $logLevel = PEAR_LOG_NOTICE;

    /**
     * $logLevels string log levels
     */
    private static $logLevels = array(self::LOG_EMERG   => PEAR_LOG_EMERG,
                                      self::LOG_ALERT   => PEAR_LOG_ALERT,
                                      self::LOG_CRIT    => PEAR_LOG_CRIT,
                                      self::LOG_ERR     => PEAR_LOG_ERR,
                                      self::LOG_WARNING => PEAR_LOG_WARNING,
                                      self::LOG_NOTICE  => PEAR_LOG_NOTICE,
                                      self::LOG_INFO    => PEAR_LOG_INFO,
                                      self::LOG_DEBUG   => PEAR_LOG_DEBUG,
                                      self::LOG_ALL     => PEAR_LOG_ALL,
                                      self::LOG_NONE    => PEAR_LOG_NONE);

    /**
	 * constructor
     * @param $base string directory
	 */
	public static function initialize($base){
		if(empty(self::$log)){
			$ini = Ficus_PropertyFileReader::read(self::CONFIG_FILE);
			//ignore mode. because its not work properly
			$conf = array('timeFormat' => $ini["timeformat"]);

			$file = $base . "/" . $ini["file"];
			self::$log = &Log::singleton(
				$ini["handler"],
				$file,
				$ini["ident"],
				$conf,
				$ini["level"]);
            self::$handler = $ini["handler"];
            self::$log->_timeFormat = "[" . self::$log->_timeFormat . "]";
		}
	}

	/**
	 * flush log
	 */
	public static function flush(){
        self::$log->flush();
	}

	/**
	 * clean log file
	 */
	public static function clean(){
		file_put_contents(self::$log->_filename, "");
	}

	/**
	 * log specific message to logger
	 * @param $msg+ string message to log.
	 * @param $level integer LOG_LEVEL
     * @throw Ficus_NotReadyException log is not ready.
	 */
	public static function log(){
        if(is_null(self::$log)){
            throw new Ficus_NotReadyException("Log is not ready. please call initialize() function before call log().");
        }
        $args = func_get_args();
        $count = func_num_args();
        $level = null;
        if($count == 0){
            throw new Ficus_IllegalArgumentException("There are no argument.");
        }else if($count >= 2){
            if(array_key_exists(end($args), self::$logLevels)){
                $level = self::$logLevels[array_pop($args)];
            }
        }

        foreach($args as $key => $arg){
            $args[$key] = self::convertVariableToString($arg);
        }
        $msg = implode($args, " , ");
        $location = self::getLocation();
        self::writeToFile($msg, $location, $level);
	}

	/**
	 * log and flush atonce
	 * @param $msg string message to log;
	 * @param $level integer log level
	 */
	public static function logAtOnce($msg, $level = null){
        if(is_null($level)){
            self::log($msg);
        }else{
            self::log($msg, $level);
        }
		self::flush();
	}

    /**
     * set mask
     */
    public static function setMask($level){
        $level = self::$logLevels[$level];
        self::$log->setMask(self::LOG_NONE | self::$log->UPTO($level));
    }
    
    /**
     * write to file
     * @param $msg misc argument to log
     * @param $location location where log method called
     * @param $level string log level
     */
    private static function writeToFile($msg, $location, $level = null){
        $msg .= $location;
		self::$log->log($msg, $level);
    }

    /**
     * get location
     */
    private static function getLocation(){
        $trace = debug_backtrace();
        $locate =  " at ";
        if ($trace[2]["class"] != null) {
            $locate .= $trace[2]["class"] . "::";
        }
        if ($trace[2]["function"] != null) {
            $locate .= $trace[2]["function"] . "()";
        }
        $locate .= " in " . $trace[1]["file"] . "#" . $trace[0]["line"];
        return $locate;
    }
    
    /**
     * Convert variable to string
     * @param $arg mixed argument to convert
     * @return string converted string
     */
    private static function convertVariableToString($arg){
		if(is_null($arg)){
			$arg = "NULL";
		}else if(is_bool($arg)){
            $arg = $arg ? "true" : "false";
        }else if(is_integer($arg) && $arg === 0){
            $arg = "0";
        }else if(is_string($arg) && $arg === ""){
            $arg = "empty string";
		}else if(is_string($arg) == false){
            $arg = self::$log->_extractMessage($arg);
        }
        return $arg;
    }
    
    /**
     * aliases of log level
     */
    public static function emerg(){
        self::log(func_get_args(), "ficus_logger_" . __FUNCTION__);
    }
    public static function alert(){
        self::log(func_get_args(), "ficus_logger_" . __FUNCTION__);
    }
    public static function crit(){
        self::log(func_get_args(), "ficus_logger_" . __FUNCTION__);
    }
    public static function err(){
        self::log(func_get_args(), "ficus_logger_" . __FUNCTION__);
    }
    public static function warning(){
        self::log(func_get_args(), "ficus_logger_" . __FUNCTION__);
    }
    public static function notice(){
        self::log(func_get_args(), "ficus_logger_" . __FUNCTION__);
    }
    public static function info(){
        self::log(func_get_args(), "ficus_logger_" . __FUNCTION__);
    }
    public static function debug(){
        self::log(func_get_args(), "ficus_logger_" . __FUNCTION__);
    }
}
?>
