<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */

/**
 * PHP Version 5
 *
 * LICENSE: This source file is subject to version 3.0 of the PHP license
 * that is available through the world-wide-web at the following URI:
 * http://www.php.net/license/3_0.txt.  If you did not receive a copy of
 * the PHP License and are unable to obtain it through the web, please
 * send a note to license@php.net so we can mail you a copy immediately.
 *
 * @category   PHP
 * @package      Ficus_ReflectionAnnotation
 * @author     Sebastian Bergmann <sb@sebastian-bergmann.de>
 * @copyright  2005 Sebastian Bergmann <sb@sebastian-bergmann.de>
 * @license    http://www.php.net/license/3_0.txt  PHP License 3.0
 * @version    CVS: $Id: ReflectionAnnotationClass.php 2 2007-07-11 10:37:48Z ishitoya $
 * @link       http://pear.php.net/package/  Ficus_ReflectionAnnotation
 * @since      File available since Release 1.0.0
 */

require_once 'ficus/lang/reflect/annotation/ReflectionAnnotationAnnotatedElement.php';
require_once 'ficus/lang/reflect/annotation/ReflectionAnnotationMethod.php';
require_once 'ficus/lang/reflect/annotation/ReflectionAnnotationParser.php';
require_once 'ficus/lang/reflect/annotation/ReflectionAnnotationProperty.php';
require_once 'ficus/lang/reflect/ReflectionAnnotation.php';

/**
 * Extension to ReflectionClass that is aware of Annotations.
 *
 * @category   PHP
 * @package      Ficus_ReflectionAnnotation
 * @author     Sebastian Bergmann <sb@sebastian-bergmann.de>
 * @copyright  2005 Sebastian Bergmann <sb@sebastian-bergmann.de>
 * @license    http://www.php.net/license/3_0.txt  PHP License 3.0
 * @version    Release: @package_version@
 * @link       http://pear.php.net/package/  Ficus_ReflectionAnnotation
 * @since      Class available since Release 1.0.0
 */
class   Ficus_ReflectionAnnotationClass extends ReflectionClass implements   Ficus_ReflectionAnnotationAnnotatedElement {
    private $annotations;

    /**
     * Constructor.
     *
     * @param  mixed $class
     * @throws InvalidArgumentException
     * @access public
     */
    public function __construct($class) {
        if (is_string($class)) {
            parent::__construct($class);
        }

        else if ($class instanceof ReflectionClass) {
            parent::__construct($class->getName());
        }

        else {
            throw new InvalidArgumentException;
        }

        $this->annotations =   Ficus_ReflectionAnnotationParser::parseAnnotations($this->getDocComment());
    }

    /**
     * Returns the class' annotations.
     *
     * @return   Ficus_ReflectionAnnotation[]
     * @access public
     */
    public function getAnnotations() {
        $annotations = array();

        foreach ($this->annotations as $name => $arguments) {
            $annotations[] =   Ficus_ReflectionAnnotation::factory(
              $this,
              $name,
              $arguments
            );
        }

        return $annotations;
    }

    /**
     * Returns the class' annotation specified by its name.
     *
     * @param  String $name
     * @return   Ficus_ReflectionAnnotation
     * @throws ReflectionException
     * @access public
     */
    public function getAnnotation($name) {
        if (isset($this->annotations[$name])) {
            return   Ficus_ReflectionAnnotation::factory(
              $this,
              $name,
              $this->annotations[$name]
            );
        } else {
            throw new ReflectionException(
              sprintf(
                'Class "%s" does not have annotation "%s"',

                $this->getName(),
                $name
              )
            );
        }
    }

    /**
     * Returns wether an annotation exists or not.
     *
     * @param  String $name
     * @return boolean
     * @access public
     */
    public function hasAnnotation($name) {
        return isset($this->annotations[$name]);
    }

    /**
     * Returns an array of this class' methods.
     *
     * @return   Ficus_ReflectionAnnotationMethod[]
     * @access public
     */
    public function getMethods() {
        $methods = array();

        foreach (parent::getMethods() as $method) {
            $methods[] = new   Ficus_ReflectionAnnotationMethod($this->getName(), $method->getName());
        }

        return $methods;
    }

    /**
     * Returns the class' method specified by its name.
     *
     * @param  string $name
     * @return   Ficus_ReflectionAnnotationMethod
     * @access public
     */
    public function getMethod($name) {
        return new   Ficus_ReflectionAnnotationMethod($this->getName(), $name);
    }

    /**
     * Returns an array of this class' properties.
     *
     * @return   Ficus_ReflectionAnnotationProperty[]
     * @access public
     */
    public function getProperties() {
        $properties = array();

        foreach (parent::getProperties() as $property) {
            $properties[] = new   Ficus_ReflectionAnnotationProperty($this->getName(), $property->getName());
        }

        return $properties;
    }

    /**
     * Returns the class' property specified by its name.
     *
     * @param  string $name
     * @return   Ficus_ReflectionAnnotationProperty
     * @access public
     */
    public function getProperty($name) {
        return new   Ficus_ReflectionAnnotationProperty($this->getName(), $name);
    }
}

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * c-hanging-comment-ender-p: nil
 * End:
 */
?>
