<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file ClassPath.php
 * @brief ClassPath for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: ClassPath.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * class path for php
 */

require_once("ficus/lang/Assert.php");
require_once("ficus/exception/IllegalArgumentException.php");
require_once("ficus/lang/ClassPathElement.php");
/**
 * @class Ficus_ClassPath
 */
class Ficus_ClassPath
{
	/** Package separator charactor */
	const PACKAGE_SEPARATOR = ".";
	const ESCAPED_DOT = "#####";

	/**
     * @var $classPaths array of Ficus_ClassPathElement class paths
     */
	private static $paths = array();

	/**
     * @var $defaultDir Ficus_Dir default directory
     **/
	private static $defaultDir = "";

    /**
     * @var $cache array of cache
     */
    private static $cache = array();

    /**
     * @var $expandedPaths array of cache
     */
    private static $expandedPaths = array();

    /**
     * get class Paths.
     * @return array array of class path
     */
    public static function paths(){
        return self::$paths;
    }
    
    /**
     * clear class Path
     * @return void
     */
    public static function clear(){
        self::$paths = array();
        self::$cache = array();
        self::$expandedPaths = array();
    }
    
    /**
     * add base directory
     * @param $path Ficus_ClassPathElement base directory
     * @return void
     * @throw Ficus_IllegalArgumentException no directory.
     */
    public static function add($path, $name = null){
        Ficus_Assert::isTypeAllowNull($path, "Ficus_ClassPathElement");
        if(is_null($name)){
            self::$paths[] = $path;
        }else{
            self::$paths[$name] = $path;
        }
        self::$cache = array();
        self::$expandedPaths = array();
    }

    /**
     * get named classpath
     * @param $name string path name
     * @return string class path
     * @throw Ficus_IllegalArgumentException directory not found.
     */
    public static function get($name){
        if(isset(self::$paths[$name])){
            return self::$paths[$name];
        }else{
            throw new Ficus_IllegalArgumentException("class path named $name is not found");
        }
    }
    
    /**
     * Package to directory.
     *
     * @param $package string package.
     * @return string directory.
     */
    public static function packageToDirectory($package) {
        return str_replace(self::PACKAGE_SEPARATOR, '/', $package);
    }

    /**
     * Directory to package.
     *
     * @param $directory string directory.
     * @return string package.
     */
    public static function directoryToPackage($directory) {
        return str_replace('/', self::PACKAGE_SEPARATOR, $directory);
    }

    /**
     * to String
     * @return string string
     */
    public static function __toString(){
        if(empty(self::$paths)){
            return "class paths is empty";
        }else{
            $expanded = self::getExpandedClassPaths();
            $string = implode(" ", $expanded);
            return $string;
        }
    }            

    /**
     * search paths for file
     * @param $path string search path
     * @return Array of found path
     */
    public static function search($file){
        if(isset(self::$cache[$file])){
            return self::$cache[$file];
        }
        if(empty(self::$paths)){
            throw new Ficus_NotReadyException("Class Path is not initialized. please check ficus/Ficus_ClassPath.php");
        }

        $result = array();
        foreach(self::$paths as $path){
            $ret = $path->search($file);
            $result = array_merge($ret, $result);
        }
        $result = array_unique($result);
        self::$cache[$file] = $result;
        return $result;
    }

    /**
     * get expanded class path
     * @return array of Ficus_Dir
     */
    public static function getExpandedClassPaths(){
        $classPaths = array();
        foreach(self::$paths as $element){
            $paths = $element->paths();
            $classPaths = array_merge($classPaths, $paths);
        }
        return $classPaths;
    }           
}
?>
