<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.db
 */
/**
 * @file PDODatabase.php
 * @brief
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: PDODatabase.php 12 2007-07-15 14:41:51Z ishitoya $
 * 
 *
 */

require_once("ficus/db/Database.php");
require_once("ficus/exception/SQLException.php");
require_once("ficus/exception/IllegalArgumentException.php");

/**
 * @class Ficus_PDODatabase
 */
class Ficus_PDODatabase extends Ficus_Database
{    
    /**
     * @var $database pdo object
     */
    protected $database = null;

    /**
     * @var $errorInfo string errorcode
     */
    protected $errorInfo = null;
    
    /**
     * connect to server
     * @param $dns String data source
     * @param $user String user name
     * @param $pass String password
     * @throw Ficus_SQLException database error.
     */
    public function connect($dsn, $user = null, $pass = null){
        if(empty($user) && empty($pass)){
            $this->database = new PDO($dsn);
        }else{
            $this->database = new PDO($dsn, $user, $pass);
        }
    }

    /**
	 * execute query
	 * @param $query String query
	 * @param $parameters array paramters
	 * @return query result
     * @throw Ficus_SQLException database error.
	 */
    public function query($query, $parameters = array()){
        $this->isConnected();
        $result = null;
        if(preg_match("/^SELECT/", $query)){
            $result = $this->executeQuery($query, $parameters);
        }else{
            try{
                $this->begin();
                $result = $this->executeQuery($query, $parameters);
                $this->commit();
            }catch(Ficus_SQLException $e){
                $this->rollBack();
                throw $e;
            }
        }
            
        return $result;
    }

    /**
     * check for multipul command
     */
    protected function isMultipleCommand($query){
        if(preg_match_all('/(?!--)(DELETE|INSERT|SELECT|UPDATE)(.+?);/ms', $query, $regs)){
            if(count($regs[0]) > 1){
                return true;
            }
        }
        return false;
    }

    /**
     * create statement and execute
	 * @param $query String query
	 * @param $parameters array paramters
	 * @return query result
     * @throw Ficus_SQLException database error.
	 */
    protected function executeQuery($query, $parameters){
        if($this->isMultipleCommand($query)){
            $result = $this->database->exec($query);
            $this->errorInfo = $this->database->errorInfo();
            if($result === false){
                throw new Ficus_SQLException($this->errorInfo[2]);
            }
            return $result;
        }
            
        $statement = $this->database->prepare($query);
        foreach($parameters as $index => $param){
            $index += 1;
            if(is_array($param) == false){
                $statement->bindParam($index, $param);
            }else if($param[self::TYPE_INDEX] == self::PARAM_TEXT){
                $statement->bindParam($index, $param[self::VALUE_INDEX]);
            }else if($param[self::TYPE_INDEX] == self::PARAM_LOB){
                $statement->bindParam($index, $param[self::VALUE_INDEX],
                                 PDO::PARAM_LOB);
            }else{
                throw new Ficus_IllegalArgumentException("type index of parameter is not text nor lob");
            }
        }
                
        $result = $statement->execute();
        $this->errorInfo = $statement->errorInfo();
        if($result === false){
            throw new Ficus_SQLException($this->errorInfo[2]);
        }
        return $statement;
    }

    /**
     * disconnect
     * pdo has no functionality to close connection.
     */
    public function disconnect(){
        $this->database = null;
    }

    /**
     * begin transaction
     */
    public function begin(){
        $this->isConnected();
        $this->database->beginTransaction();
    }

    /**
     * end transaction
     */
    public function commit(){
        $this->isConnected();
        $this->database->commit();
    }    

    /**
     * rollBack transaction
     */
    public function rollBack(){
        $this->isConnected(); 
        $this->database->rollBack();
    }

    /**
     * is connected
     * @throw Ficus_NotReadyException not connected
     */
    public function isConnected(){
        if(is_null($this->database)){
            throw new Ficus_NotReadyException("connection is not established");
        }
    }

    /**
     * has error
     * @return boolean there are some error, return true
     */
    public function hasError(){
        if(is_null($this->errorCode)){
            return false;
        }
        $class = substr($this->errorInfo[0], 0, 2);
        if($class === self::SQLSTATE_OK ||
           $class === self::SQLSTATE_WARN){
            return false;
        }
        return true;
    }           

    /**
     * get error info
     * @return array error info
     */
    public function errorInfo(){
        return $this->errorInfo;
    }
}
?>
