<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.parameters.visitors
 */
/**
 * @file ParameterToWSDLConverter.php
 * @brief ParameterToWSDLConverter for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: ParameterToWSDLConverter.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * ParameterToWSDLConverter.
 */

require_once("ficus/lang/Assert.php");
require_once("ficus/net/WSDL.php");
require_once("ficus/parameters/visitors/Acceptor.php");
require_once("ficus/parameters/visitors/Visitor.php");
require_once("ficus/validators/Validatable.php");
require_once("ficus/parameters/ValidatableComplexParameter.php");
require_once("ficus/parameters/ValidatableSimpleParameter.php");
require_once("ficus/exception/IllegalTypeException.php");

/**
 * @class Ficus_ParameterToWSDLConverter
 */
class Ficus_ParameterToWSDLConverter extends Ficus_Visitor
{
	/**
	 * @var $wsdl Ficus_WSDL
	 */
	private $wsdl = null;
	
	/**
	 * @var $dom DomDocument
	 */
	private $dom = null;

	/**
	 * @var $messages array array of message that DOMElement presentation
	 */
	private $messages = array();
	
	/**
	 * @var $types array array of types that DOMElement presentation
	 */
	private $types = array();

	/**
	 * constructor
	 * @param $wsdl Ficus_WSDL intstance of Util DSDL
	 * @param $dom array array of parameters to validate
	 */
	public function __construct($wsdl, $dom){
		$this->setWSDL($wsdl);
		$this->setDOMDocument($dom);
	}

	/**
	 * set WSDL class
	 * @param $wsdl Ficus_WSDL set WSDL class
     * @throw Ficus_IllegalTypeException no WSDL.
	 */
	private function setWSDL($wsdl){
		if($wsdl instanceof Ficus_WSDL){
			$this->wsdl = $wsdl;
		}else{
			throw new Ficus_IllegalTypeException("wsdl is not instance of Ficus_WSDL");
		}
	}

	/**
	 * set DOMDocument
	 * @param $dom DOMDocument set DOMDocument
     * @throw Ficus_IllegalTypeException no DOMDocument.
	 */
	private function setDOMDocument($dom){
		if($dom instanceof DOMDocument){
			$this->dom = $dom;
		}else{
			throw new Ficus_IllegalTypeException("dom is not instance of DOMDocument.");
		}
	}
	
	/**
	 * visit parameters and validate
	 * @param $parameter Ficus_Acceptor validatables
	 */
	public function visit($parameter){
		Ficus_Assert::isInstanceOf($parameter, "Ficus_Acceptor");
		if($parameter instanceof Ficus_ValidatableComplexParameter){
			//if type is null, the parameter is name of parameter
			if($parameter->typeName() === ""){
				$name = preg_replace("/\[\]/", "Array", $parameter->name());
				$this->messages[$name] =
					$this->createMessageElement($parameter, $name);
			}else if($parameter instanceof Ficus_ArrayParameter){
				$typename = preg_replace("/\[\]/", "Array",
										 $parameter->typeName());
				$this->types[$typename] =
					$this->createArrayTypesElement($parameter, $typename);
			}else{
				$this->types[$parameter->typeName()] =
					$this->createTypesElement($parameter);
			}

			//check for child parameters
			$children = $parameter->getParameters();
			foreach($children as $child){
				if($child instanceof Ficus_ValidatableComplexParameter &&
					isset($types[$child->name()]) === false){
					$child->accept($this);
				}
			}
		}
	}
	
	/**
	 * create Types Element
	 * @param $parameter Ficus_ValidatableComplexParameter target parameter
	 * @return DOMElement
	 */
	private function createTypesElement($parameter){
		$complexType = $this->dom->createElementNS(Ficus_WSDL::XMLNS_XSD,
												   "xsd:complexType");
		$complexType->setAttribute("name", $parameter->typeName());
		$all = $this->dom->createElementNS(Ficus_WSDL::XMLNS_XSD, "xsd:all");

		$params = $parameter->getParameters();
		foreach($params as $param){
			$element = $this->dom->createElementNS(Ficus_WSDL::XMLNS_XSD,
												   "xsd:element");
			$element->setAttribute("name", $param->name());
			$element->setAttribute("type", $param->typeName());
			$all->appendChild($element);
		}
		$complexType->appendChild($all);
		return $complexType;
	}

	/**
	 * create Types Element for Array
	 * @param $parameter Ficus_ValidatableComplexParameter target parameter
	 * @param $arrayName string array name
	 * @return DOMElement created types element for array
	 */
	private function createArrayTypesElement($parameter, $arrayName){
        $params = $parameter->getParameters();
		$checkparam = current($params);
		
		$complexType = $this->dom->createElementNS(Ficus_WSDL::XMLNS_XSD,
												   "xsd:complexType");
		$complexType->setAttribute("name", $arrayName);
		$complexContent = $this->dom->createElementNS(Ficus_WSDL::XMLNS_XSD,
													  "xsd:complexContent");
		$restriction = $this->dom->createElementNS(Ficus_WSDL::XMLNS_XSD,
												   "xsd:restriction");
		$restriction->setAttribute("base", Ficus_WSDL::P_SOAPENC . "Array");
		$attribute = $this->dom->createElementNS(Ficus_WSDL::XMLNS_XSD,
												 "xsd:attribute");
		$attribute->setAttribute("ref", Ficus_WSDL::P_SOAPENC . "arrayType");

		$typename = $parameter->typeName();
		if($checkparam instanceof Ficus_ComplexParameter){
			$typename = Ficus_WSDL::P_TYPENS . $typename;
		}else{
			$typename = Ficus_WSDL::P_XSD . $typename;
		}
		$attribute->setAttributeNS(Ficus_WSDL::XMLNS_WSDL, "wsdl:arrayType",
								   $typename);
		
		$restriction->appendChild($attribute);
		$complexContent->appendChild($restriction);
		$complexType->appendChild($complexContent);
		return $complexType;
	}

	/**
	 * create Message Element
	 * @param $parameter Ficus_ValidatableComplexParameter target parameter
	 * @return DOMElement created message element
	 */
	private function createMessageElement($parameter){
		$messageName = $parameter->name();
		$params      = $parameter->getParameters();

		$message = $this->dom->createElement("message");
		$message->setAttribute("name", $messageName);
		foreach($params as $param){
			$part = $this->dom->createElement("part");
			$part->setAttribute("name", $param->name());
		    $typename = preg_replace('/\[\]/', "Array", $param->typeName());
			if($param instanceof Ficus_ValidatableSimpleParameter){
				$part->setAttribute("type", Ficus_WSDL::P_XSD . $typename);
			}else{
				$part->setAttribute("type", Ficus_WSDL::P_TYPENS . $typename);
			}
			$message->appendChild($part);
		}
		return $message;
	}
	
	/**
	 * return array of messages
	 * @return array of DOMElement 
	 */
	public function getConvertedMessages(){
		return $this->messages;
	}

	/**
	 * return type Element
	 * @return DOMElement type Element
	 */
	public function getConvertedTypes(){
		$types = $this->dom->createElement("types");
		$schema = $this->dom->createElementNS(Ficus_WSDL::XMLNS_XSD,
											  "xsd:schema");
		$schema->setAttribute("targetNamespace",
							  $this->wsdl->targetNameSpace());
		foreach($this->types as $type){
			$types->appendChild($type);
		}
		return $types;
	}
}
?>
