<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.parameters.visitors
 */
/**
 * @file ParameterAccessor.php
 * @brief ParameterAccessor for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: ParameterAccessor.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * ParameterAccessor.
 */

require_once("ficus/lang/Assert.php");
require_once("ficus/parameters/Parameter.php");
require_once("ficus/parameters/ComplexParameter.php");
require_once("ficus/parameters/visitors/Acceptor.php");
require_once("ficus/parameters/visitors/Visitor.php");
require_once("ficus/validators/Validatable.php");
require_once("ficus/exception/IllegalArgumentException.php");
require_once("ficus/exception/PropertyNotFoundException.php");

/**
 * @class Ficus_ParameterAccessor
 */
class Ficus_ParameterAccessor extends Ficus_Visitor implements Iterator
{
	/**
	 * @var $parameters array array of parameters that passed to module
	 */
	private $parameters = array();

	/**
	 * @var $root Ficus_Parameter root parameter;
	 */
	private $root = null;
	/**
	 * constructor
	 * @param $parameter array array of parameters to validate
	 */
	public function __construct($parameter){
		if($parameter instanceof Ficus_ComplexParameter){
			$parameter->accept($this);
		}
	}
	
	/**
	 * visit parameters and sets parameters.
	 * @param $parameter Ficus_Acceptor validatables
	 */
	public function visit($parameter){
		Ficus_Assert::typeHinting("Ficus_Acceptor" , $parameter);
		Ficus_Assert::typeHinting("Ficus_Parameter", $parameter);
		if(empty($this->root) &&
		   $parameter instanceof Ficus_ComplexParameter){
			$this->root = $parameter;
		}else{
			$name = strtolower($parameter->name());
			$this->parameters[$name] = $parameter;
		}

		//check for child parameters
		if($parameter instanceof Ficus_ComplexParameter){
			$children = $parameter->getParameters();
			foreach($children as $child){
				$this->parameters[$child->name()] = $child;
			}
		}
	}

	/**
	 * get Root Parameter
	 * @return Ficus_ComplexParameter rootParameter
	 */
	public function getRootParameter(){
		return $this->root;
	}
	
	/**
	 * get parameters
	 * @param $name string name of this parameter
     * @throw Ficus_PropertyNotFoundException property not found.
	 */
	function __get($name){
		if(array_key_exists($name, $this->parameters) === false){
			throw new Ficus_PropertyNotFoundException("Property $name is not found in this parameter");
		}
		
		$result = $this->parameters[$name];
		//for nested parameter
		//$accessor->resource->id()
		if($result instanceof Ficus_ComplexParameter){
			$accessor = new Ficus_ParameterAccessor($result);
			return $accessor;
		}
		return $result->value();
	}

	/**
	 * get parameter value by name
	 * @param $name string name of this parameter
	 * @return mixed parameter value
	 */
	function get($name){
		return $this->callGetMethod($name);
	}
	
	/**
	 * set parameter value by name
	 * @param $name string name of target parameter
	 * @param $argument array argument of this param
	 * @return mixed result
	 */
	function set($name, $argument){
		return $this->__call($name, array($argument));
	}
	
	/**
	 * php magic method to get and set parameters like below
	 * $accessor->resource->id()
	 * $accessor->getResource->getId();
	 * $accessor->resource->setId("urn:soya:resource:1");
	 * $accessor->setResource($obj);
	 * @param $function string function name
	 * @param $arguments mixed arguments of this function
	 * @return mixed result
     * @throw Ficus_IllegalArgumentException illegal argument.
	 */
	function __call($function, $arguments){
		if(count($arguments) == 1){
			if(is_object($arguments[0])){
				return $this->callSetMethodWithObject(
					$function, $arguments[0]);
			}else{
				return $this->callSetMethod($function, $arguments[0]);
			}
		}else if(count($arguments) == 0){
			return $this->callGetMethod($function);
		}else{
			throw new Ficus_IllegalArgumentException("if argument specified, count must be 1, not specified then must be 0.");
		}
	}

	/**
	 * get method handler.
	 * @param $name string property name
	 * @return Ficus_ValidatableParameter
     * @throw Ficus_PropertyNotFoundException property not found.
	 */
	function callGetMethod($name){
		if(preg_match('/(?:get_?)([A-Z0-9_][a-zA-Z0-9_]+)/', $name, $match)){
			$name = strtolower($match[1]);
		}
		
		if(array_key_exists($name, $this->parameters) === false){
			throw new Ficus_PropertyNotFoundException("Property $name is not found in this parameter");
		}
		
		$param = $this->parameters[$name];
		if($name == Ficus_ArrayParameter::TEMPLATE){
			if($param instanceof Ficus_ComplexParameter){
				return new Ficus_ParameterAccessor(clone $param);
			}else{
				return $param;
			}
		}else if($param instanceof Ficus_ArrayParameter){
			return $param;
		}
		return $param->value();
	}

	/**
	 * set property
	 * @param $function string property name
	 * @param $argument mixed property value
     * @throw Ficus_PropertyNotFoundException property not found.
	 */
	private function callSetMethod($function, $argument){
		if(preg_match('/(?:set_?)([a-zA-Z0-9_]+)/', $function, $match)){
			$name = strtolower($match[1]);
		}
		
		if(array_key_exists($name, $this->parameters) === false){
			throw new Ficus_PropertyNotFoundException("Property $name is not found in this parameter");
        }

		$param = $this->parameters[$name];
		if($param instanceof Ficus_ArrayParameter){
			$this->callSetMethodForArray($name, $argument);
		}else{
		    $this->parameters[$name]->setValue($argument);
		}
	}

	/**
	 * set property
	 * @param $name string name of parameter
	 * @param $argument mixed value of parameter
     * @throw Ficus_PropertyNotFoundException property not found.
	 */
    private function callSetMethodWithObject($name, $argument){
        if(preg_match('/(?:set_?)([a-zA-Z0-9_]+)/i', $name, $match)){
			$name = strtolower($match[1]);
		}
		if(array_key_exists($name,  $this->parameters) === false){
			throw new Ficus_PropertyNotFoundException("Property $name is not found in this parameter");
        }
		
		$param = $this->parameters[$name];

		if($param instanceof Ficus_ArrayParameter){
			$this->callSetMethodForArray($name, $argument);
		    return;
		}

		if($param instanceof Ficus_ComplexParameter){
			$children = $param->getParameters();
			$class = new ReflectionClass($argument);
			
			foreach($children as $child){
				$childName = $child->name();
				$getMethod = "get" . ucfirst($childName);
				
				if(method_exists($argument, $getMethod)){
					$method = $class->getMethod($getMethod);
				}else if(method_exists($argument, $childName)){
					$method = $class->getMethod($childName);
				}else{
					continue;
				}
				$value = $method->invoke($argument);
				$accessor = new Ficus_ParameterAccessor(null);
				$child->accept($accessor);
				$accessor->{"set" . ucfirst($childName)}($value);
			}
		}
		$param->setValue($argument);
    }

	/**
	 * set value to ArrayParameter
	 * @param $name string name
	 * @param $argument mixed argument to set array
	 */
	private function callSetMethodForArray($name, $argument){
		$param = $this->parameters[$name];
		if($argument instanceof Ficus_ParameterAccessor){
			$value = clone $argument->getRootParameter();
		}else if($argument instanceof Ficus_Parameter){
			$value = clone $argument;
		}
		$param->append(clone($value));
	}
	
	/**
	 * if target parameter is array, then get index parameter
     *
     * @param $index int index.
	 */
	public function at($index){
		$target = $this->getRootParameter();
		if($target instanceof Ficus_ArrayParameter){
			try{
				$param = $target->at($index);
				if($param instanceof Ficus_ComplexParameter){
					return new Ficus_ParameterAccessor($param);
				}else{
					return $param->value();
				}
			}catch(Ficus_IllegalArgumentException $e){
				//do nothing, return null
			}
		}
		return null;
	}
	
	/**
	 * iteration function reset array pointer
	 */
	public function rewind() {
		reset($this->parameters);
	}

	/**
	 * iteration function returns current value
	 */
	public function current() {
		$var = current($this->parameters);
		if($var instanceof Ficus_ArrayParameter){
			return $var;
		}else if($var instanceof Ficus_ComplexParameter){
			$accessor = new Ficus_ParameterAccessor($var);
			return $accessor;
		}
		return $var;
	}

	/**
	 * iteration function returns key of current array pointer
	 */
	public function key() {
	    $var = key($this->parameters);
		return $var;
	}

	/**
	 * iteration function get next array value
	 */
	public function next() {
		$var = next($this->parameters);
		if($var instanceof Ficus_ArrayParameter){
			return $var;
		}else if($var instanceof Ficus_ComplexParameter){
			$accessor = new Ficus_ParameterAccessor($var);
			return $accessor;
		}
		return $var;
	}

	/**
	 * iteration function validate array pointer
	 */
	public function valid() {
		return (is_null(key($this->parameters)) === false);
	}
}
?>
